package Math::GSL::SF::Test;
use Math::GSL::Test qw/:all/;
use base q{Test::Class};
use Test::More tests => 1109;
use Math::GSL          qw/:all/;
use Math::GSL::Const   qw/:all/;
use Math::GSL::Errno   qw/:all/;
use Math::GSL::SF      qw/:all/;
use Math::GSL::Poly    qw/:all/;
use Math::GSL::Machine qw/:all/;
use Math::GSL::Sys     qw/$GSL_NAN/;
use Data::Dumper;
use strict;
use warnings;

BEGIN { gsl_set_error_handler_off() }

my $factor    = 100;
my $TOL1      = $factor*16.0*$GSL_DBL_EPSILON;
my $SQRT_TOL0 = 2.0*$GSL_SQRT_DBL_EPSILON;
my $DELTA     = 1.2246467991473531772e-16;
my $w         = 0.8*$GSL_LOG_DBL_MAX;
my $y         = 0.2*$GSL_DBL_MAX;

sub make_fixture : Test(setup) {
}
sub teardown : Test(teardown) {
}

sub TEST_RT66882 : Tests(1) {
    # This test was losing a small amount of precision with
    # perls that have DUSELONGDOUBLE, so give it some leeway
    # Where is the loss of precision coming from? Is it a bug in Perl, GSL or Math::GSL ?
    # A beer if you figure it out.
    local $TODO = "loss of precision on Perls with DUSELONGDOUBLE";

    # additional diagnostics
    local %ENV; $ENV{DEBUG} = 1;
    my $results = {
        'gsl_sf_fermi_dirac_m1_e(10.0, $r)' => 0.9999546021312975656,
    };
    verify_results($results, 'Math::GSL::SF');
}

sub TEST_THE_KITCHEN_SINK : Tests {
    my $results = { 
        'gsl_sf_airy_Ai_e(-5, $Math::GSL::GSL_MODE_DEFAULT, $r)'           => 0.3507610090241142,
        'gsl_sf_airy_Ai_e(-500,$Math::GSL::GSL_MODE_DEFAULT, $r)'         => 0.0725901201040411396,
        'gsl_sf_bessel_J0_e(0.1,$r)'         => 0.99750156206604003230,
        'gsl_sf_bessel_J0_e(2.0,$r)'         => 0.22389077914123566805,
        'gsl_sf_bessel_J0_e(5,$r)'           => -0.17759677131433830434739701,
        'gsl_sf_bessel_J0_e(100,$r)'         => 0.019985850304223122424,
        'gsl_sf_bessel_J0_e(1e10,$r)'        => 2.1755917502468917269e-06,
        'gsl_sf_erf_e(5,$r)'                 => 0.999999999998463,
        'gsl_sf_dilog_e(-3.0,$r)'            => -1.9393754207667089531,     
        'gsl_sf_dilog_e(-0.5,$r)'            => -0.4484142069236462024,     
        'gsl_sf_gamma_e(6.3,$r)'             =>  201.813275184748,
        'gsl_sf_dilog_e(-0.001,$r)'          => -0.0009997501110486510834,  
        'gsl_sf_dilog_e(0.1,$r)'             => 0.1026177910993911,        
        'gsl_sf_dilog_e(0.7,$r)'             => 0.8893776242860387386,     
        'gsl_sf_dilog_e(1.0,$r)'             => 1.6449340668482260,        
        'gsl_sf_clausen_e($M_PI/20.0,$r)'	=> 0.4478882448133546,
        'gsl_sf_clausen_e($M_PI/6.0,$r)'	=> 0.8643791310538927,
        'gsl_sf_clausen_e($M_PI/3.0,$r)'	=> 1.0149416064096535,
        'gsl_sf_clausen_e(  2.0*$M_PI + $M_PI/3.0,$r)'	=> 1.0149416064096535,
        'gsl_sf_clausen_e(100.0*$M_PI + $M_PI/3.0,$r)'	=> 1.0149416064096535,
        'gsl_sf_coupling_3j_e(0, 1, 1, 0,  1, -1,$r)'	=> sqrt(1.0/2.0),
        'gsl_sf_coupling_3j_e(1, 1, 2, 1, -1,  0,$r)'	=> sqrt(1.0/6.0),
        'gsl_sf_coupling_3j_e(2, 4, 6, 0,  2, -2,$r)'	=> sqrt(8.0/105.0),
        'gsl_sf_coupling_3j_e(4, 4, 8, 0,  0,  0,$r)'	=> sqrt(2.0/35.0),
        'gsl_sf_coupling_3j_e(4, 4, 8, 2, -2,  0,$r)'	=> 2.0/3.0*sqrt(2.0/35.0),
        'gsl_sf_coupling_3j_e(4, 4, 8, 4, -4,  0,$r)'	=> 1.0/(3.0*sqrt(70.0)),
        'gsl_sf_coupling_3j_e(1, 1, 2, 2, -1, 0,$r)'	=> 0,
        'gsl_sf_coupling_3j_e(1, 1, 2, 1, -2, 0,$r)'	=> 0,
        'gsl_sf_coupling_3j_e(1, 1, 2, 1, -1, 3,$r)'	=> 0,
        'gsl_sf_coupling_3j_e(1, 1, 3, 1, -1, 0,$r)'	=> 0,
        'gsl_sf_coupling_3j_e(1, 4, 2, 1, -1, 0,$r)'	=> 0,
        'gsl_sf_coupling_3j_e(4, 1, 2, 1, -1, 0,$r)'	=> 0,
        'gsl_sf_coupling_6j_e(2, 2, 4, 2, 2, 2,$r)'	=>  1.0/6.0,
        'gsl_sf_coupling_6j_e(4, 4, 2, 4, 4, 4,$r)'	=> -1.0/10.0,
        'gsl_sf_coupling_6j_e(4, 4, 2, 4, 4, 2,$r)'	=>  1.0/6.0,
        'gsl_sf_coupling_6j_e(4, 4, 2, 2, 2, 2,$r)'	=> -0.5/sqrt(5.0),
        'gsl_sf_coupling_6j_e(4, 4, 4, 2, 2, 2,$r)'	=>  sqrt(7.0/3.0)/10.0,
        'gsl_sf_coupling_6j_e(6, 6, 6, 4, 4, 4,$r)'	=> -sqrt(3.0/5.0)/14.0,
        'gsl_sf_coupling_6j_e(6, 6, 6, 4, 4, 2,$r)'	=> -sqrt(3.0/5.0)/7.0,
        'gsl_sf_coupling_6j_e(2, 2, 4, 2, 2, 7,$r)'	=> 0,
        'gsl_sf_coupling_6j_e(2, 2, 4, 2, 7, 2,$r)'	=> 0,
        'gsl_sf_coupling_6j_e(2, 2, 4, 7, 2, 2,$r)'	=> 0,
        'gsl_sf_coupling_6j_e(2, 2, 7, 2, 2, 2,$r)'	=> 0,
        'gsl_sf_coupling_6j_e(2, 7, 4, 2, 2, 2,$r)'	=> 0,
        'gsl_sf_coupling_6j_e(7, 2, 4, 2, 2, 2,$r)'	=> 0,
        'gsl_sf_coupling_9j_e(4, 2,  4, 3, 3, 2, 1, 1, 2,$r)'	=> -sqrt(1.0/6.0)/10.0,
        'gsl_sf_coupling_9j_e(8, 4, 10, 7, 3, 8, 1, 1, 2,$r)'	=>  sqrt(7.0/3.0)/60.0,
        'gsl_sf_coupling_9j_e(10, 2, 4, 3, 3, 2, 1, 1, 2,$r)'	=> 0,
        'gsl_sf_coupling_9j_e(4, 10, 4, 3, 3, 2, 1, 1, 2,$r)'	=> 0,
        'gsl_sf_coupling_9j_e(4, 2, 10, 3, 3, 2, 1, 1, 2,$r)'	=> 0,
        'gsl_sf_coupling_9j_e(4, 2, 4, 10, 3, 2, 1, 1, 2,$r)'	=> 0,
        'gsl_sf_coupling_9j_e(4, 2, 4, 3, 10, 2, 1, 1, 2,$r)'	=> 0,
        'gsl_sf_coupling_9j_e(4, 2, 4, 3, 3, 10, 1, 1, 2,$r)'	=> 0,
        'gsl_sf_coupling_9j_e(4, 2, 4, 3, 3, 2, 10, 1, 2,$r)'	=> 0,
        'gsl_sf_coupling_9j_e(4, 2, 4, 3, 3, 2, 1, 10, 2,$r)'	=> 0,
        'gsl_sf_coupling_9j_e(4, 2, 4, 3, 3, 2, 1, 1, 10,$r)'	=> 0,
        'gsl_sf_dawson_e(1.0e-15,$r)'	    => 1.0e-15,
        'gsl_sf_dawson_e(0.5,$r)'	        => 0.4244363835020222959,
        'gsl_sf_dawson_e(2.0,$r)'	        => 0.30134038892379196603,
        'gsl_sf_dawson_e(1000.0,$r)'	    => 0.0005000002500003750009,
        'gsl_sf_debye_1_e(0.1,$r)'	        =>  0.975277750004723276,
        'gsl_sf_debye_1_e(1.0,$r)'	        =>  0.777504634112248239,
        'gsl_sf_debye_1_e(10.0,$r)'	        => 0.164443465679946027,
        'gsl_sf_debye_2_e(0.1,$r)'	        =>  0.967083287045302664,
        'gsl_sf_debye_2_e(1.0,$r)'	        =>  0.70787847562782924,
        'gsl_sf_debye_2_e(10.0,$r)'	        => 0.0479714980201218708,
        'gsl_sf_debye_3_e(0.1,$r)'	        =>  0.962999940487211048,
        'gsl_sf_debye_3_e(1.0,$r)'	        =>  0.674415564077814667,
        'gsl_sf_debye_3_e(10.0,$r)'	        => 0.0192957656903454886,
        'gsl_sf_debye_4_e(0.1,$r)'	        =>  0.960555486124335944,
        'gsl_sf_debye_4_e(1.0,$r)'	        =>  0.654874068886737049,
        'gsl_sf_debye_4_e(10.0,$r)'	        => 0.00967367556027115896,
        'gsl_sf_debye_5_e(0.1,$r)'	        =>  0.95892849428310568745,
        'gsl_sf_debye_5_e(1.0,$r)'	        =>  0.6421002580217790246,
        'gsl_sf_debye_5_e(10.0,$r)'	        => 0.005701535852992908538,
        'gsl_sf_debye_6_e(0.1,$r)'	        =>  0.95776777382605465878,
        'gsl_sf_debye_6_e(1.0,$r)'	        =>  0.63311142583495107588,
        'gsl_sf_debye_6_e(10.0,$r)'	        => 3.7938493294615955279e-3,
        'gsl_sf_multiply_e(-3.0,2.0,$r)'	=> -6.0,
        "gsl_sf_multiply_e($y, 1.0/$y,\$r)"	=>  1.0,
        "gsl_sf_multiply_e($y, 0.2,\$r)"	    =>    0.04*$GSL_DBL_MAX,
        "gsl_sf_multiply_e($y, 4.0,\$r)"	    =>    0.8*$GSL_DBL_MAX,
        'gsl_sf_ellint_Kcomp_e( 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.3566005233611923760,
        'gsl_sf_ellint_Kcomp_e( 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.6857503548125960429,
        'gsl_sf_ellint_Kcomp_e(0.010, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.5708355989121522360,
        'gsl_sf_ellint_Ecomp_e(0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.0284758090288040010,
        'gsl_sf_ellint_Ecomp_e(0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.4674622093394271555,
        'gsl_sf_ellint_Ecomp_e(0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.5707570561503852873,
        'gsl_sf_ellint_Pcomp_e(0.99, 0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.13792612351836506315593,
        'gsl_sf_ellint_Pcomp_e(0.50, 0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.60455249360848890075108,
        'gsl_sf_ellint_Pcomp_e(0.01, 0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.49773208536003801277453,
        'gsl_sf_ellint_Dcomp_e(0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.375395076351788975665323192,
        'gsl_sf_ellint_Dcomp_e(0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.8731525818926755496456335628,
        'gsl_sf_ellint_Dcomp_e(0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.7854276176694868932799393751,
        'gsl_sf_ellint_F_e($M_PI/3.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.3065333392738766762,
        'gsl_sf_ellint_F_e($M_PI/3.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.0895506700518854093,
        'gsl_sf_ellint_F_e($M_PI/3.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.0472129063770918952,
        'gsl_sf_ellint_E_e($M_PI/3.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.8704819220377943536,
        'gsl_sf_ellint_E_e($M_PI/3.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.0075555551444720293,
        'gsl_sf_ellint_E_e($M_PI/3.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.0471821963889481104,
        'gsl_sf_ellint_P_e($M_PI/3.0, 0.99, 0.5, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.1288726598764099882,
        'gsl_sf_ellint_P_e($M_PI/3.0, 0.50, 0.5, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.9570574331323584890,
        'gsl_sf_ellint_P_e($M_PI/3.0, 0.01, 0.5, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.9228868127118118465,
        'gsl_sf_ellint_RF_e(5.0e-11, 1.0e-10, 1.0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 12.36441982979439,
        'gsl_sf_ellint_RF_e(1.0, 2.0, 3.0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.7269459354689082,
        'gsl_sf_ellint_RD_e(5.0e-11, 1.0e-10, 1.0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 34.0932594919337362,
        'gsl_sf_ellint_RD_e(1.0, 2.0, 3.0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.2904602810289906,
        'gsl_sf_ellint_RC_e(1.0, 2.0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.7853981633974482,
        'gsl_sf_ellint_RJ_e(2.0, 3.0, 4.0, 5.0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.1429757966715675,
        'gsl_sf_ellint_E_e($M_PI/2.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.02847580902880400098389,
        'gsl_sf_ellint_E_e($M_PI/2.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.46746220933942715545980,
        'gsl_sf_ellint_E_e($M_PI/2.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.57075705615038528733708,
        'gsl_sf_ellint_E_e(2*$M_PI/3.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.18646969601981364833972,
        'gsl_sf_ellint_E_e(2*$M_PI/3.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.92736886353438228163734,
        'gsl_sf_ellint_E_e(2*$M_PI/3.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.09433191591182246425715,
        'gsl_sf_ellint_E_e($M_PI, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.05695161805760800196777,
        'gsl_sf_ellint_E_e($M_PI, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.93492441867885431091959,
        'gsl_sf_ellint_E_e($M_PI, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.14151411230077057467416,
        'gsl_sf_ellint_E_e(4*$M_PI/3, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.92743354009540235559582,
        'gsl_sf_ellint_E_e(4*$M_PI/3, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.94247997382332634020184,
        'gsl_sf_ellint_E_e(4*$M_PI/3, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.18869630868971868509117,
        'gsl_sf_ellint_E_e(3*$M_PI/2.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.08542742708641200295166,
        'gsl_sf_ellint_E_e(3*$M_PI/2.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.40238662801828146637939,
        'gsl_sf_ellint_E_e(3*$M_PI/2.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.71227116845115586201123,
        'gsl_sf_ellint_E_e(5*$M_PI/3, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.24342131407742165030750,
        'gsl_sf_ellint_E_e(5*$M_PI/3, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.86229328221323659255693,
        'gsl_sf_ellint_E_e(5*$M_PI/3, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.23584602821259303893130,
        'gsl_sf_ellint_E_e(2*$M_PI, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.11390323611521600393555,
        'gsl_sf_ellint_E_e(2*$M_PI, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.86984883735770862183918,
        'gsl_sf_ellint_E_e(2*$M_PI, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 6.28302822460154114934831,
        'gsl_sf_ellint_E_e(7*$M_PI/3.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.98438515815301035756360,
        'gsl_sf_ellint_E_e(7*$M_PI/3.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 6.87740439250218065112143,
        'gsl_sf_ellint_E_e(7*$M_PI/3.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 7.33021042099048925976532,
        'gsl_sf_ellint_E_e(-$M_PI/2.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.02847580902880400098389,
        'gsl_sf_ellint_E_e(-$M_PI/2.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.46746220933942715545980,
        'gsl_sf_ellint_E_e(-$M_PI/2.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.57075705615038528733708,
        'gsl_sf_ellint_E_e(-2*$M_PI/3.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.18646969601981364833972,
        'gsl_sf_ellint_E_e(-2*$M_PI/3.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.92736886353438228163734,
        'gsl_sf_ellint_E_e(-2*$M_PI/3.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.09433191591182246425715,
        'gsl_sf_ellint_E_e(-$M_PI, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.05695161805760800196777,
        'gsl_sf_ellint_E_e(-$M_PI, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.93492441867885431091959,
        'gsl_sf_ellint_E_e(-$M_PI, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.14151411230077057467416,
        'gsl_sf_ellint_E_e(-4*$M_PI/3, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.92743354009540235559582,
        'gsl_sf_ellint_E_e(-4*$M_PI/3, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.94247997382332634020184,
        'gsl_sf_ellint_E_e(-4*$M_PI/3, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.18869630868971868509117,
        'gsl_sf_ellint_E_e(-3*$M_PI/2.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.08542742708641200295166,
        'gsl_sf_ellint_E_e(-3*$M_PI/2.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.40238662801828146637939,
        'gsl_sf_ellint_E_e(-3*$M_PI/2.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.71227116845115586201123,
        'gsl_sf_ellint_E_e(-5*$M_PI/3, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.24342131407742165030750,
        'gsl_sf_ellint_E_e(-5*$M_PI/3, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.86229328221323659255693,
        'gsl_sf_ellint_E_e(-5*$M_PI/3, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.23584602821259303893130,
        'gsl_sf_ellint_E_e(-2*$M_PI, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.11390323611521600393555,
        'gsl_sf_ellint_E_e(-2*$M_PI, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.86984883735770862183918,
        'gsl_sf_ellint_E_e(-2*$M_PI, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -6.28302822460154114934831,
        'gsl_sf_ellint_E_e(-7*$M_PI/3.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.98438515815301035756360,
        'gsl_sf_ellint_E_e(-7*$M_PI/3.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -6.87740439250218065112143,
        'gsl_sf_ellint_E_e(-7*$M_PI/3.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -7.33021042099048925976532,
        'gsl_sf_ellint_F_e($M_PI/2.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.35660052336119237603347,
        'gsl_sf_ellint_F_e($M_PI/2.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.68575035481259604287120,
        'gsl_sf_ellint_F_e($M_PI/2.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.57083559891215223602641,
        'gsl_sf_ellint_F_e(2*$M_PI/3.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.40666770744850807588478,
        'gsl_sf_ellint_F_e(2*$M_PI/3.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.28195003957330667648585,
        'gsl_sf_ellint_F_e(2*$M_PI/3.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.09445829144721257687207,
        'gsl_sf_ellint_F_e($M_PI, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 6.71320104672238475206694,
        'gsl_sf_ellint_F_e($M_PI, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.37150070962519208574241,
        'gsl_sf_ellint_F_e($M_PI, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.14167119782430447205281,
        'gsl_sf_ellint_F_e(4*$M_PI/3, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 8.01973438599626142824910,
        'gsl_sf_ellint_F_e(4*$M_PI/3, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.46105137967707749499897,
        'gsl_sf_ellint_F_e(4*$M_PI/3, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.18888410420139636723356,
        'gsl_sf_ellint_F_e(3*$M_PI/2.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 10.0698015700835771281004,
        'gsl_sf_ellint_F_e(3*$M_PI/2.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.05725106443778812861361,
        'gsl_sf_ellint_F_e(3*$M_PI/2.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.71250679673645670807922,
        'gsl_sf_ellint_F_e(5*$M_PI/3, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 12.1198687541708928279517,
        'gsl_sf_ellint_F_e(5*$M_PI/3, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.65345074919849876222825,
        'gsl_sf_ellint_F_e(5*$M_PI/3, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.23612948927151704892488,
        'gsl_sf_ellint_F_e(2*$M_PI, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 13.4264020934447695041339,
        'gsl_sf_ellint_F_e(2*$M_PI, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 6.74300141925038417148481,
        'gsl_sf_ellint_F_e(2*$M_PI, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 6.28334239564860894410562,
        'gsl_sf_ellint_F_e(7*$M_PI/3.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 14.7329354327186461803160,
        'gsl_sf_ellint_F_e(7*$M_PI/3.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 7.83255208930226958074138,
        'gsl_sf_ellint_F_e(7*$M_PI/3.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 7.33055530202570083928637,
        'gsl_sf_ellint_F_e(-$M_PI/2.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.35660052336119237603347,
        'gsl_sf_ellint_F_e(-$M_PI/2.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.68575035481259604287120,
        'gsl_sf_ellint_F_e(-$M_PI/2.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.57083559891215223602641,
        'gsl_sf_ellint_F_e(-2*$M_PI/3.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.40666770744850807588478,
        'gsl_sf_ellint_F_e(-2*$M_PI/3.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.28195003957330667648585,
        'gsl_sf_ellint_F_e(-2*$M_PI/3.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.09445829144721257687207,
        'gsl_sf_ellint_F_e(-$M_PI, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -6.71320104672238475206694,
        'gsl_sf_ellint_F_e(-$M_PI, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.37150070962519208574241,
        'gsl_sf_ellint_F_e(-$M_PI, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.14167119782430447205281,
        'gsl_sf_ellint_F_e(-4*$M_PI/3, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -8.01973438599626142824910,
        'gsl_sf_ellint_F_e(-4*$M_PI/3, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.46105137967707749499897,
        'gsl_sf_ellint_F_e(-4*$M_PI/3, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.18888410420139636723356,
        'gsl_sf_ellint_F_e(-3*$M_PI/2.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -10.0698015700835771281004,
        'gsl_sf_ellint_F_e(-3*$M_PI/2.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.05725106443778812861361,
        'gsl_sf_ellint_F_e(-3*$M_PI/2.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.71250679673645670807922,
        'gsl_sf_ellint_F_e(-5*$M_PI/3, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -12.1198687541708928279517,
        'gsl_sf_ellint_F_e(-5*$M_PI/3, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.65345074919849876222825,
        'gsl_sf_ellint_F_e(-5*$M_PI/3, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.23612948927151704892488,
        'gsl_sf_ellint_F_e(-2*$M_PI, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -13.4264020934447695041339,
        'gsl_sf_ellint_F_e(-2*$M_PI, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -6.74300141925038417148481,
        'gsl_sf_ellint_F_e(-2*$M_PI, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -6.28334239564860894410562,
        'gsl_sf_ellint_F_e(-7*$M_PI/3.0, 0.99, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -14.7329354327186461803160,
        'gsl_sf_ellint_F_e(-7*$M_PI/3.0, 0.50, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -7.83255208930226958074138,
        'gsl_sf_ellint_F_e(-7*$M_PI/3.0, 0.01, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -7.33055530202570083928637,
        'gsl_sf_ellint_P_e($M_PI/2.0, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.61678162163246646783050,
        'gsl_sf_ellint_P_e($M_PI/2.0, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.78030349465454812629168,
        'gsl_sf_ellint_P_e($M_PI/2.0, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.65580719756898353270922,
        'gsl_sf_ellint_P_e(2*$M_PI/3.0, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.88008918207571119911983,
        'gsl_sf_ellint_P_e(2*$M_PI/3.0, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.43655207300356008717867,
        'gsl_sf_ellint_P_e(2*$M_PI/3.0, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.23211110528200554950903,
        'gsl_sf_ellint_P_e($M_PI, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 7.23356324326493293566099,
        'gsl_sf_ellint_P_e($M_PI, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.56060698930909625258336,
        'gsl_sf_ellint_P_e($M_PI, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.31161439513796706541844,
        'gsl_sf_ellint_P_e(4*$M_PI/3, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 8.58703730445415467220216,
        'gsl_sf_ellint_P_e(4*$M_PI/3, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.68466190561463241798805,
        'gsl_sf_ellint_P_e(4*$M_PI/3, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.39111768499392858132786,
        'gsl_sf_ellint_P_e(3*$M_PI/2.0, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 10.8503448648973994034915,
        'gsl_sf_ellint_P_e(3*$M_PI/2.0, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.34091048396364437887504,
        'gsl_sf_ellint_P_e(3*$M_PI/2.0, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.96742159270695059812767,
        'gsl_sf_ellint_P_e(5*$M_PI/3, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 13.1136524253406441347808,
        'gsl_sf_ellint_P_e(5*$M_PI/3, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.99715906231265633976204,
        'gsl_sf_ellint_P_e(5*$M_PI/3, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.54372550041997261492747,
        'gsl_sf_ellint_P_e(2*$M_PI, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 14.4671264865298658713220,
        'gsl_sf_ellint_P_e(2*$M_PI, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 7.12121397861819250516672,
        'gsl_sf_ellint_P_e(2*$M_PI, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 6.62322879027593413083689,
        'gsl_sf_ellint_P_e(7*$M_PI/3.0, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 15.8206005477190876078631,
        'gsl_sf_ellint_P_e(7*$M_PI/3.0, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 8.24526889492372867057141,
        'gsl_sf_ellint_P_e(7*$M_PI/3.0, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 7.70273208013189564674630,
        'gsl_sf_ellint_P_e(-$M_PI/2.0, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.61678162163246646783050,
        'gsl_sf_ellint_P_e(-$M_PI/2.0, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.78030349465454812629168,
        'gsl_sf_ellint_P_e(-$M_PI/2.0, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.65580719756898353270922,
        'gsl_sf_ellint_P_e(-2*$M_PI/3.0, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.88008918207571119911983,
        'gsl_sf_ellint_P_e(-2*$M_PI/3.0, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.43655207300356008717867,
        'gsl_sf_ellint_P_e(-2*$M_PI/3.0, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.23211110528200554950903,
        'gsl_sf_ellint_P_e(-$M_PI, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -7.23356324326493293566099,
        'gsl_sf_ellint_P_e(-$M_PI, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.56060698930909625258336,
        'gsl_sf_ellint_P_e(-$M_PI, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.31161439513796706541844,
        'gsl_sf_ellint_P_e(-4*$M_PI/3, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -8.58703730445415467220216,
        'gsl_sf_ellint_P_e(-4*$M_PI/3, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.68466190561463241798805,
        'gsl_sf_ellint_P_e(-4*$M_PI/3, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.39111768499392858132786,
        'gsl_sf_ellint_P_e(-3*$M_PI/2.0, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -10.8503448648973994034915,
        'gsl_sf_ellint_P_e(-3*$M_PI/2.0, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.34091048396364437887504,
        'gsl_sf_ellint_P_e(-3*$M_PI/2.0, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.96742159270695059812767,
        'gsl_sf_ellint_P_e(-5*$M_PI/3, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -13.1136524253406441347808,
        'gsl_sf_ellint_P_e(-5*$M_PI/3, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.99715906231265633976204,
        'gsl_sf_ellint_P_e(-5*$M_PI/3, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.54372550041997261492747,
        'gsl_sf_ellint_P_e(-2*$M_PI, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -14.4671264865298658713220,
        'gsl_sf_ellint_P_e(-2*$M_PI, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -7.12121397861819250516672,
        'gsl_sf_ellint_P_e(-2*$M_PI, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -6.62322879027593413083689,
        'gsl_sf_ellint_P_e(-7*$M_PI/3.0, 0.99, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -15.8206005477190876078631,
        'gsl_sf_ellint_P_e(-7*$M_PI/3.0, 0.50, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -8.24526889492372867057141,
        'gsl_sf_ellint_P_e(-7*$M_PI/3.0, 0.01, -0.1, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -7.70273208013189564674630,
        'gsl_sf_ellint_D_e($M_PI/2.0, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.375395076351788975665323192,
        'gsl_sf_ellint_D_e($M_PI/2.0, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.8731525818926755496456335628,
        'gsl_sf_ellint_D_e($M_PI/2.0, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 0.7854276176694868932799393751,
        'gsl_sf_ellint_D_e(2*$M_PI/3.0, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.305885125424644860264320635,
        'gsl_sf_ellint_D_e(2*$M_PI/3.0, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.418324704155697579394036402,
        'gsl_sf_ellint_D_e(2*$M_PI/3.0, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.263755353901126149206022061,
        'gsl_sf_ellint_D_e($M_PI, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 4.750790152703577951330646444,
        'gsl_sf_ellint_D_e($M_PI, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.746305163785351099291267125,
        'gsl_sf_ellint_D_e($M_PI, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.570855235338973786559878750,
        'gsl_sf_ellint_D_e(4*$M_PI/3, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 5.195695179982511042396972113,
        'gsl_sf_ellint_D_e(4*$M_PI/3, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.074285623415004619188497818,
        'gsl_sf_ellint_D_e(4*$M_PI/3, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 1.877955116776821423913735408,
        'gsl_sf_ellint_D_e(3*$M_PI/2.0, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 7.126185229055366926995969476,
        'gsl_sf_ellint_D_e(3*$M_PI/2.0, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.619457745678026648936900687,
        'gsl_sf_ellint_D_e(3*$M_PI/2.0, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.356282853008460679839818125,
        'gsl_sf_ellint_D_e(5*$M_PI/3, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 9.056675278128222811594967044,
        'gsl_sf_ellint_D_e(5*$M_PI/3, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.164629867941048678685303509,
        'gsl_sf_ellint_D_e(5*$M_PI/3, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 2.834610589240099935765900794,
        'gsl_sf_ellint_D_e(2*$M_PI, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 9.501580305407155902661292832,
        'gsl_sf_ellint_D_e(2*$M_PI, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.492610327570702198582534249,
        'gsl_sf_ellint_D_e(2*$M_PI, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.141710470677947573119757500,
        'gsl_sf_ellint_D_e(7*$M_PI/3.0, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 9.946485332686088993727618315,
        'gsl_sf_ellint_D_e(7*$M_PI/3.0, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.820590787200355718479764901,
        'gsl_sf_ellint_D_e(7*$M_PI/3.0, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> 3.448810352115795210473614120,
        'gsl_sf_ellint_D_e(-$M_PI/2.0, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.375395076351788975665323192,
        'gsl_sf_ellint_D_e(-$M_PI/2.0, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -0.8731525818926755496456335628,
        'gsl_sf_ellint_D_e(-$M_PI/2.0, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -0.7854276176694868932799393751,
        'gsl_sf_ellint_D_e(-2*$M_PI/3.0, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.305885125424644860264320635,
        'gsl_sf_ellint_D_e(-2*$M_PI/3.0, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.418324704155697579394036402,
        'gsl_sf_ellint_D_e(-2*$M_PI/3.0, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.263755353901126149206022061,
        'gsl_sf_ellint_D_e(-$M_PI, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -4.750790152703577951330646444,
        'gsl_sf_ellint_D_e(-$M_PI, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.746305163785351099291267125,
        'gsl_sf_ellint_D_e(-$M_PI, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.570855235338973786559878750,
        'gsl_sf_ellint_D_e(-4*$M_PI/3, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -5.195695179982511042396972113,
        'gsl_sf_ellint_D_e(-4*$M_PI/3, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.074285623415004619188497818,
        'gsl_sf_ellint_D_e(-4*$M_PI/3, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -1.877955116776821423913735408,
        'gsl_sf_ellint_D_e(-3*$M_PI/2.0, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -7.126185229055366926995969476,
        'gsl_sf_ellint_D_e(-3*$M_PI/2.0, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.619457745678026648936900687,
        'gsl_sf_ellint_D_e(-3*$M_PI/2.0, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.356282853008460679839818125,
        'gsl_sf_ellint_D_e(-5*$M_PI/3, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -9.056675278128222811594967044,
        'gsl_sf_ellint_D_e(-5*$M_PI/3, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.164629867941048678685303509,
        'gsl_sf_ellint_D_e(-5*$M_PI/3, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -2.834610589240099935765900794,
        'gsl_sf_ellint_D_e(-2*$M_PI, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -9.501580305407155902661292832,
        'gsl_sf_ellint_D_e(-2*$M_PI, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.492610327570702198582534249,
        'gsl_sf_ellint_D_e(-2*$M_PI, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.141710470677947573119757500,
        'gsl_sf_ellint_D_e(-7*$M_PI/3.0, 0.99, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -9.946485332686088993727618315,
        'gsl_sf_ellint_D_e(-7*$M_PI/3.0, 0.50, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.820590787200355718479764901,
        'gsl_sf_ellint_D_e(-7*$M_PI/3.0, 0.01, 0, $Math::GSL::GSL_MODE_DEFAULT,$r)'	=> -3.448810352115795210473614120,
        'gsl_sf_erfc_e(-10.0,$r)'	=> 2.0,
        'gsl_sf_erfc_e(-5.0000002,$r)'	=> 1.9999999999984625433,
        'gsl_sf_erfc_e(-5.0,$r)'	=> 1.9999999999984625402,
        'gsl_sf_erfc_e(-1.0,$r)'	=> 1.8427007929497148693,
        'gsl_sf_erfc_e(-0.5,$r)'	=> 1.5204998778130465377,
        'gsl_sf_erfc_e(1.0,$r)'	=> 0.15729920705028513066,
        'gsl_sf_erfc_e(3.0,$r)'	=> 0.000022090496998585441373,
        'gsl_sf_erfc_e(7.0,$r)'	=> 4.183825607779414399e-23,
        'gsl_sf_erfc_e(10.0,$r)'	=> 2.0884875837625447570e-45,
        'gsl_sf_log_erfc_e(-1.0,$r)'	=> log(1.842700792949714869),
        'gsl_sf_log_erfc_e(-0.1,$r)'	=> 0.106576400586522485015,
        'gsl_sf_log_erfc_e(-1e-10,$r)'	=>  1.1283791670318505967e-10,
        'gsl_sf_log_erfc_e(0.0,$r)'	=> log(1.0),
        'gsl_sf_log_erfc_e(1e-10,$r)'	=> -1.128379167159174551e-10,
        'gsl_sf_log_erfc_e(0.001,$r)'	=> -0.0011290158896213548027,
        'gsl_sf_log_erfc_e(0.1,$r)'	=> -0.119304973737395598329,
        'gsl_sf_log_erfc_e(1.0,$r)'	=> log(0.15729920705028513066),
        'gsl_sf_log_erfc_e(10.0,$r)'	=> log(2.0884875837625447570e-45),
        'gsl_sf_erf_e(-10.0,$r)'	=> -1.0000000000000000000,
        'gsl_sf_erf_e(0.5,$r)'	=> 0.5204998778130465377,
        'gsl_sf_erf_e(1.0,$r)'	=> 0.8427007929497148693,
        'gsl_sf_erf_e(10.0,$r)'	=> 1.0000000000000000000,
        'gsl_sf_erf_Z_e(1.0,$r)'	=>  0.24197072451914334980,
        'gsl_sf_erf_Q_e(10.0,$r)'	=> 7.619853024160526066e-24,
        'gsl_sf_hazard_e(-20.0,$r)'	=> 5.5209483621597631896e-88,
        'gsl_sf_hazard_e(-10.0,$r)'	=> 7.6945986267064193463e-23,
        'gsl_sf_hazard_e(-1.0,$r)'	=> 0.28759997093917836123,
        'gsl_sf_hazard_e( 0.0,$r)'	=> 0.79788456080286535588,
        'gsl_sf_hazard_e( 1.0,$r)'	=> 1.5251352761609812091,
        'gsl_sf_hazard_e(10.0,$r)'	=> 10.098093233962511963,
        'gsl_sf_hazard_e(20.0,$r)'	=> 20.049753068527850542,
        'gsl_sf_hazard_e(30.0,$r)'	=> 30.033259667433677037,
        'gsl_sf_hazard_e(50.0,$r)'	=> 50.019984031905639809,
        'gsl_sf_hazard_e(80.0,$r)'	=> 80.012496096798234468,
        'gsl_sf_hazard_e(150.0,$r)'	=> 150.00666607420571802,
        'gsl_sf_hazard_e(300.0,$r)'	=> 300.00333325926337415,
        'gsl_sf_hazard_e(900.0,$r)'	=> 900.00111110836764382,
        'gsl_sf_hazard_e(1001.0,$r)'	=> 1001.0009989990049990,
        'gsl_sf_hazard_e(2000.0,$r)'	=> 2000.0004999997500003,
        'gsl_sf_exp_e(-10.0,$r)'	=> exp(-10.0),
        'gsl_sf_exp_e( 10.0,$r)'	=> exp( 10.0),
        "gsl_sf_exp_err_e(-10.0, $TOL1,\$r)"	=> exp(-10.0),
        "gsl_sf_exp_err_e( 10.0, $TOL1,\$r)"	=> exp( 10.0),
        'gsl_sf_exp_mult_e(-10.0,  1.0e-06,$r)'	=> 1.0e-06*exp(-10.0),
        'gsl_sf_exp_mult_e(-10.0,  2.0,$r)'	=>     2.0*exp(-10.0),
        'gsl_sf_exp_mult_e(-10.0, -2.0,$r)'	=>    -2.0*exp(-10.0),
        'gsl_sf_exp_mult_e( 10.0,  1.0e-06,$r)'	=> 1.0e-06*exp( 10.0),
        'gsl_sf_exp_mult_e( 10.0, -2.0,$r)'	=>    -2.0*exp( 10.0),
        "gsl_sf_exp_mult_e($w, 1.000001,\$r)"	=>     1.000001*exp($w),
        "gsl_sf_exp_mult_e($w, 1.000000001,\$r)"	=>  1.000000001*exp($w),
        "gsl_sf_exp_mult_e($w, 100.0,\$r)"	=>        100.0*exp($w),
        "gsl_sf_exp_mult_e($w, 1.0e+20,\$r)"	=>      1.0e+20*exp($w),
        "gsl_sf_exp_mult_e($w, exp(-$w)*exp($M_LN2),\$r)"	=> 2.0,
        "gsl_sf_exp_mult_err_e(-10.0, $SQRT_TOL0, 2.0, $SQRT_TOL0,\$r)" => 2.0*exp(-10.0),
        "gsl_sf_exp_mult_err_e($w, $SQRT_TOL0*$w, exp(-$w)*exp($M_LN2), $SQRT_TOL0*exp(-$w)*exp($M_LN2),\$r)"	=> 2.0,
        'gsl_sf_expm1_e(-10.0,$r)'	=> exp(-10.0)-1.0,
        'gsl_sf_expm1_e(-0.001,$r)'	=> -0.00099950016662500845,
        'gsl_sf_expm1_e(-1.0e-8,$r)'	=> -1.0e-08 + 0.5e-16,
        'gsl_sf_expm1_e( 1.0e-8,$r)'	=> 1.0e-08 + 0.5e-16,
        'gsl_sf_expm1_e( 0.001,$r)'	=> 0.0010005001667083417,
        'gsl_sf_expm1_e( 10.0,$r)'	=> exp(10.0)-1.0,
        'gsl_sf_exprel_e(-10.0,$r)'	=> 0.0999954600070237515,
        'gsl_sf_exprel_e(-0.001,$r)'	=> 0.9995001666250084,
        'gsl_sf_exprel_e(-1.0e-8,$r)'	=> 1.0 - 0.5e-08,
        'gsl_sf_exprel_e( 1.0e-8,$r)'	=> 1.0 + 0.5e-08,
        'gsl_sf_exprel_e( 0.001,$r)'	=> 1.0005001667083417,
        'gsl_sf_exprel_e( 10.0,$r)'	=> 2202.5465794806716517,
        'gsl_sf_exprel_2_e(-10.0,$r)'	=> 0.18000090799859524970,
        'gsl_sf_exprel_2_e(-0.001,$r)'	=> 0.9996667499833361107,
        'gsl_sf_exprel_2_e(-1.0e-8,$r)'	=> 0.9999999966666666750,
        'gsl_sf_exprel_2_e( 1.0e-8,$r)'	=> 1.0000000033333333417,
        'gsl_sf_exprel_2_e( 0.001,$r)'	=> 1.0003334166833361115,
        'gsl_sf_exprel_2_e( 10.0,$r)'	=> 440.3093158961343303,
        'gsl_sf_exprel_n_e(3, -1000.0,$r)'	=> 0.00299400600000000000,
        'gsl_sf_exprel_n_e(3, -100.0,$r)'	=> 0.02940600000000000000,
        'gsl_sf_exprel_n_e(3, -10.0,$r)'	=> 0.24599972760042142509,
        'gsl_sf_exprel_n_e(3, -3.0,$r)'	=> 0.5444917625849191238,
        'gsl_sf_exprel_n_e(3, -0.001,$r)'	=> 0.9997500499916678570,
        'gsl_sf_exprel_n_e(3, -1.0e-8,$r)'	=> 0.9999999975000000050,
        'gsl_sf_exprel_n_e(3,  1.0e-8,$r)'	=> 1.0000000025000000050,
        'gsl_sf_exprel_n_e(3,  0.001,$r)'	=> 1.0002500500083345240,
        'gsl_sf_exprel_n_e(3,  3.0,$r)'	=> 2.5745637607083706091,
        'gsl_sf_exprel_n_e(3,  3.1,$r)'	=> 2.6772417068460206247,
        'gsl_sf_exprel_n_e(3,  10.0,$r)'	=> 131.79279476884029910,
        'gsl_sf_exprel_n_e(3,  100.0,$r)'	=> 1.6128702850896812690e+38,
        'gsl_sf_exprel_n_e(50, -1000.0,$r)'	=> 0.04766231609253975959,
        'gsl_sf_exprel_n_e(50, -100.0,$r)'	=> 0.3348247572345889317,
        'gsl_sf_exprel_n_e(50, -10.0,$r)'	=> 0.8356287051853286482,
        'gsl_sf_exprel_n_e(50, -3.0,$r)'	=> 0.9443881609152163615,
        'gsl_sf_exprel_n_e(50, -1.0,$r)'	=> 0.980762245565660617,
        'gsl_sf_exprel_n_e(50, -1.0e-8,$r)'	=> 1.0 -1.0e-8/51.0,
        'gsl_sf_exprel_n_e(50,  1.0e-8,$r)'	=> 1.0 +1.0e-8/51.0,
        'gsl_sf_exprel_n_e(50,  1.0,$r)'	=> 1.01999216583666790,
        'gsl_sf_exprel_n_e(50,  3.0,$r)'	=> 1.0624205757460368307,
        'gsl_sf_exprel_n_e(50,  48.0,$r)'	=> 7.499573876877194416,
        'gsl_sf_exprel_n_e(50,  50.1,$r)'	=> 9.311803306230992272,
        'gsl_sf_exprel_n_e(50,  100.0,$r)'	=> 8.175664432485807634e+07,
        'gsl_sf_exprel_n_e(50,  500.0,$r)'	=> 4.806352370663185330e+146,
        'gsl_sf_exprel_n_e(500, -1000.0,$r)'	=> 0.3334815803127619256,
        'gsl_sf_exprel_n_e(500, -100.0,$r)'	=> 0.8335646217536183909,
        'gsl_sf_exprel_n_e(500, -10.0,$r)'	=> 0.9804297803131823066,
        'gsl_sf_exprel_n_e(500, -3.0,$r)'	=> 0.9940475488850672997,
        'gsl_sf_exprel_n_e(500, -1.0,$r)'	=> 0.9980079602383488808,
        'gsl_sf_exprel_n_e(500, -1.0e-8,$r)'	=> 1.0 -1.0e-8/501.0,
        'gsl_sf_exprel_n_e(500,  1.0e-8,$r)'	=> 1.0 +1.0e-8/501.0,
        'gsl_sf_exprel_n_e(500,  1.0,$r)'	=> 1.0019999920160634252,
        'gsl_sf_exprel_n_e(500,  3.0,$r)'	=> 1.0060240236632444934,
        'gsl_sf_exprel_n_e(500,  48.0,$r)'	=> 1.1059355517981272174,
        'gsl_sf_exprel_n_e(500,  100.0,$r)'	=> 1.2492221464878287204,
        'gsl_sf_exprel_n_e(500,  500.0,$r)'	=> 28.363019877927630858,
        'gsl_sf_exprel_n_e(500,  1000.0,$r)'	=> 2.4037563160335300322e+68,
        'gsl_sf_exprel_n_e(500,  1600.0,$r)'	=> 7.899293535320607403e+226,
        'gsl_sf_expint_E1_e(-1.0,$r)'	=> -1.8951178163559367555,
        'gsl_sf_expint_E1_e(1.0e-10,$r)'	=> 22.448635265138923980,
        'gsl_sf_expint_E1_e(1.0e-05,$r)'	=> 10.935719800043695615,
        'gsl_sf_expint_E1_e(0.1,$r)'	=> 1.82292395841939066610,
        'gsl_sf_expint_E1_e(1.0,$r)'	=> 0.21938393439552027368,
        'gsl_sf_expint_E1_e(10.0,$r)'	=> 4.156968929685324277e-06,
        'gsl_sf_expint_E1_e(50.0,$r)'	=> 3.783264029550459019e-24,
        'gsl_sf_expint_E1_e(300.0,$r)'	=> 1.710384276804510115e-133,
        'gsl_sf_expint_E2_e(-1.0,$r)'	=> 0.8231640121031084799,
        'gsl_sf_expint_E2_e(0.0,$r)'	=> 1.0,
        'gsl_sf_expint_E2_e(1.0/4294967296.0,$r)'	=> 0.9999999947372139168,
        'gsl_sf_expint_E2_e(1.0/65536.0,$r)'	=> 0.9998243233207178845,
        'gsl_sf_expint_E2_e(0.1,$r)'	=> 0.7225450221940205066,
        'gsl_sf_expint_E2_e(1.0,$r)'	=> 0.14849550677592204792,
        'gsl_sf_expint_E2_e(10.0,$r)'	=> 3.830240465631608762e-06,
        'gsl_sf_expint_E2_e(50.0,$r)'	=> 3.711783318868827367e-24,
        'gsl_sf_expint_E2_e(300.0,$r)'	=> 1.7047391998483433998e-133,
        'gsl_sf_expint_En_e(1,-1.0,$r)'	=> -1.8951178163559367555,
        'gsl_sf_expint_En_e(1,1.0e-10,$r)'	=> 22.448635265138923980,
        'gsl_sf_expint_En_e(1,1.0e-05,$r)'	=> 10.935719800043695615,
        'gsl_sf_expint_En_e(1,0.1,$r)'	=> 1.82292395841939066610,
        'gsl_sf_expint_En_e(1,1.0,$r)'	=> 0.21938393439552027368,
        'gsl_sf_expint_En_e(1,10.0,$r)'	=> 4.156968929685324277e-06,
        'gsl_sf_expint_En_e(1,50.0,$r)'	=> 3.783264029550459019e-24,
        'gsl_sf_expint_En_e(1,300.0,$r)'	=> 1.710384276804510115e-133,
        'gsl_sf_expint_En_e(2,-1.0,$r)'	=> 0.8231640121031084799,
        'gsl_sf_expint_En_e(2,0.0,$r)'	=> 1.0,
        'gsl_sf_expint_En_e(2,1.0/4294967296.0,$r)'	=> 0.9999999947372139168,
        'gsl_sf_expint_En_e(2,1.0/65536.0,$r)'	=> 0.9998243233207178845,
        'gsl_sf_expint_En_e(2,0.1,$r)'	=> 0.7225450221940205066,
        'gsl_sf_expint_En_e(2,1.0,$r)'	=> 0.14849550677592204792,
        'gsl_sf_expint_En_e(2,10.0,$r)'	=> 3.830240465631608762e-06,
        'gsl_sf_expint_En_e(2,50.0,$r)'	=> 3.711783318868827367e-24,
        'gsl_sf_expint_En_e(2,300.0,$r)'	=> 1.7047391998483433998e-133,
        'gsl_sf_expint_En_e(3,0.0,$r)'	=> 0.5,
        'gsl_sf_expint_En_e(3,1.0/4294967296.0,$r)'	=> 0.499999999767169356972,
        'gsl_sf_expint_En_e(3,1.0/65536.0,$r)'	=> 0.4999847426094515610,
        'gsl_sf_expint_En_e(3,0.1,$r)'	=> 0.4162914579082787612543,
        'gsl_sf_expint_En_e(3,1.0,$r)'	=> 0.10969196719776013683858,
        'gsl_sf_expint_En_e(3,10.0,$r)'	=> .000003548762553084381959981,
        'gsl_sf_expint_En_e(3,50.0,$r)'	=> 3.6429094264752049812e-24,
        'gsl_sf_expint_En_e(10,0.0,$r)'	=> 0.111111111111111111,
        'gsl_sf_expint_En_e(10,1.0/4294967296.0,$r)'	=> 0.111111111082007280658,
        'gsl_sf_expint_En_e(10,1.0/65536.0,$r)'	=> 0.11110920377910896018606,
        'gsl_sf_expint_En_e(10,0.1,$r)'	=> 0.099298432000896813567905,
        'gsl_sf_expint_En_e(10,1.0,$r)'	=> 0.036393994031416401634164534,
        'gsl_sf_expint_En_e(10,10.0,$r)'	=> 0.00000232530265702821081778968,
        'gsl_sf_expint_En_e(10,50.0,$r)'	=> 3.223296586749110919572e-24,
        'gsl_sf_expint_En_e(10,300.0,$r)'	=> 1.6608815083360041367294736e-133,
        'gsl_sf_expint_Ei_e(-1.0,$r)'	=> -0.21938393439552027368,
        'gsl_sf_expint_Ei_e(1.0/4294967296.0,$r)'	=> -21.603494112783886397,
        'gsl_sf_expint_Ei_e(1.0,$r)'	=> 1.8951178163559367555,
        'gsl_sf_expint_E1_scaled_e(-10000.0,$r)'	=> -0.00010001000200060024012,
        'gsl_sf_expint_E1_scaled_e(-1000.0,$r)'	=> -0.0010010020060241207251,
        'gsl_sf_expint_E1_scaled_e(-10.0,$r)'	=> -0.11314702047341077803,
        'gsl_sf_expint_E1_scaled_e(-1.0,$r)'	=> -0.69717488323506606877,
        'gsl_sf_expint_E1_scaled_e(1.0e-10,$r)'	=> 22.448635267383787506,
        'gsl_sf_expint_E1_scaled_e(1.0e-05,$r)'	=> 10.935829157788483865,
        'gsl_sf_expint_E1_scaled_e(0.1,$r)'	=> 2.0146425447084516791,
        'gsl_sf_expint_E1_scaled_e(1.0,$r)'	=> 0.59634736232319407434,
        'gsl_sf_expint_E1_scaled_e(10.0,$r)'	=> 0.091563333939788081876,
        'gsl_sf_expint_E1_scaled_e(50.0,$r)'	=> 0.019615109930114870365,
        'gsl_sf_expint_E1_scaled_e(300.0,$r)'	=> 0.0033222955652707070644,
        'gsl_sf_expint_E1_scaled_e(1000.0,$r)'	=> 0.00099900199402388071500,
        'gsl_sf_expint_E1_scaled_e(10000.0,$r)'	=> 0.000099990001999400239880,
        'gsl_sf_expint_E2_scaled_e(-10000.0,$r)'	=> -0.00010002000600240120072,
        'gsl_sf_expint_E2_scaled_e(-1000.0,$r)'	=> -0.0010020060241207250807,
        'gsl_sf_expint_E2_scaled_e(-10.0,$r)'	=> -0.13147020473410778034,
        'gsl_sf_expint_E2_scaled_e(-1.0,$r)'	=> 0.30282511676493393123,
        'gsl_sf_expint_E2_scaled_e(0.0,$r)'	=> 1.0,
        'gsl_sf_expint_E2_scaled_e(1.0/4294967296.0,$r)'	=> 0.99999999497004455927,
        'gsl_sf_expint_E2_scaled_e(1.0/65536.0,$r)'	=> 0.99983957954556245453,
        'gsl_sf_expint_E2_scaled_e(0.1,$r)'	=> 0.79853574552915483209,
        'gsl_sf_expint_E2_scaled_e(1.0,$r)'	=> 0.40365263767680592566,
        'gsl_sf_expint_E2_scaled_e(10.0,$r)'	=> 0.084366660602119181239,
        'gsl_sf_expint_E2_scaled_e(50.0,$r)'	=> 0.019244503494256481735,
        'gsl_sf_expint_E2_scaled_e(300.0,$r)'	=> 0.0033113304187878806691,
        'gsl_sf_expint_E2_scaled_e(1000.0,$r)'	=> 0.00099800597611928500004,
        'gsl_sf_expint_E2_scaled_e(10000.0,$r)'	=> 0.000099980005997601199281,
        'gsl_sf_expint_En_scaled_e(1,-10000.0,$r)'	=> -0.00010001000200060024012,
        'gsl_sf_expint_En_scaled_e(1,-1000.0,$r)'	=> -0.0010010020060241207251,
        'gsl_sf_expint_En_scaled_e(1,-10.0,$r)'	=> -0.11314702047341077803,
        'gsl_sf_expint_En_scaled_e(1,-1.0,$r)'	=> -0.69717488323506606877,
        'gsl_sf_expint_En_scaled_e(1,1.0e-10,$r)'	=> 22.448635267383787506,
        'gsl_sf_expint_En_scaled_e(1,1.0e-05,$r)'	=> 10.935829157788483865,
        'gsl_sf_expint_En_scaled_e(1,0.1,$r)'	=> 2.0146425447084516791,
        'gsl_sf_expint_En_scaled_e(1,1.0,$r)'	=> 0.59634736232319407434,
        'gsl_sf_expint_En_scaled_e(1,10.0,$r)'	=> 0.091563333939788081876,
        'gsl_sf_expint_En_scaled_e(1,50.0,$r)'	=> 0.019615109930114870365,
        'gsl_sf_expint_En_scaled_e(1,300.0,$r)'	=> 0.0033222955652707070644,
        'gsl_sf_expint_En_scaled_e(1,1000.0,$r)'	=> 0.00099900199402388071500,
        'gsl_sf_expint_En_scaled_e(1,10000.0,$r)'	=> 0.000099990001999400239880,
        'gsl_sf_expint_En_scaled_e(2,-10000.0,$r)'	=> -0.00010002000600240120072,
        'gsl_sf_expint_En_scaled_e(2,-1000.0,$r)'	=> -0.0010020060241207250807,
        'gsl_sf_expint_En_scaled_e(2,-10.0,$r)'	=> -0.13147020473410778034,
        'gsl_sf_expint_En_scaled_e(2,-1.0,$r)'	=> 0.30282511676493393123,
        'gsl_sf_expint_En_scaled_e(2,0.0,$r)'	=> 1.0,
        'gsl_sf_expint_En_scaled_e(2,1.0/4294967296.0,$r)'	=> 0.99999999497004455927,
        'gsl_sf_expint_En_scaled_e(2,1.0/65536.0,$r)'	=> 0.99983957954556245453,
        'gsl_sf_expint_En_scaled_e(2,0.1,$r)'	=> 0.79853574552915483209,
        'gsl_sf_expint_En_scaled_e(2,1.0,$r)'	=> 0.40365263767680592566,
        'gsl_sf_expint_En_scaled_e(2,10.0,$r)'	=> 0.084366660602119181239,
        'gsl_sf_expint_En_scaled_e(2,50.0,$r)'	=> 0.019244503494256481735,
        'gsl_sf_expint_En_scaled_e(2,300.0,$r)'	=> 0.0033113304187878806691,
        'gsl_sf_expint_En_scaled_e(2,1000.0,$r)'	=> 0.00099800597611928500004,
        'gsl_sf_expint_En_scaled_e(2,10000.0,$r)'	=> 0.000099980005997601199281,
        'gsl_sf_expint_En_scaled_e(3,0.0,$r)'	=> 0.5,
        'gsl_sf_expint_En_scaled_e(3,1.0/4294967296.0,$r)'	=> 0.4999999998835846787586,
        'gsl_sf_expint_En_scaled_e(3,1.0/65536.0,$r)'	=> 0.4999923718293796877864492,
        'gsl_sf_expint_En_scaled_e(3,0.1,$r)'	=> 0.4600732127235422583955,
        'gsl_sf_expint_En_scaled_e(3,1.0,$r)'	=> 0.298173681161597037170539,
        'gsl_sf_expint_En_scaled_e(3,10.0,$r)'	=> 0.07816669698940409380349,
        'gsl_sf_expint_En_scaled_e(3,50.0,$r)'	=> 0.0188874126435879566345,
        'gsl_sf_expint_En_scaled_e(3,300.0,$r)'	=> 0.00330043718181789963028657675,
        'gsl_sf_expint_En_scaled_e(10,0.0,$r)'	=> 0.111111111111111111,
        'gsl_sf_expint_En_scaled_e(10,1.0/4294967296.0,$r)'	=> 0.11111111110787735217158,
        'gsl_sf_expint_En_scaled_e(10,1.0/65536.0,$r)'	=> 0.1111108991839472074435,
        'gsl_sf_expint_En_scaled_e(10,0.1,$r)'	=> 0.1097417392579033988025,
        'gsl_sf_expint_En_scaled_e(10,1.0,$r)'	=> 0.09892913264064615521915,
        'gsl_sf_expint_En_scaled_e(10,10.0,$r)'	=> 0.0512181994376050593314159875,
        'gsl_sf_expint_En_scaled_e(10,50.0,$r)'	=> 0.0167118436335939556034579,
        'gsl_sf_expint_En_scaled_e(10,300.0,$r)'	=> 0.0032261400811599644878615,
        'gsl_sf_expint_Ei_scaled_e(-1000.0,$r)'	=> -0.00099900199402388071500,
        'gsl_sf_expint_Ei_scaled_e(-1.0,$r)'	=> -0.59634736232319407434,
        'gsl_sf_expint_Ei_scaled_e(1.0/4294967296.0,$r)'	=> -21.603494107753930958,
        'gsl_sf_expint_Ei_scaled_e(1.0,$r)'	=> 0.69717488323506606877,
        'gsl_sf_expint_Ei_scaled_e(1000.0,$r)'	=> 0.0010010020060241207251,
        'gsl_sf_Shi_e(-1.0,$r)'	=> -1.0572508753757285146,
        'gsl_sf_Shi_e(1.0/4294967296.0,$r)'	=> 2.3283064365386962891e-10,
        'gsl_sf_Shi_e(1.0/65536.0,$r)'	=> 0.00001525878906269737298,
        'gsl_sf_Shi_e(0.1,$r)'	=> 0.1000555722250569955,
        'gsl_sf_Shi_e(1.0,$r)'	=> 1.0572508753757285146,
        'gsl_sf_Shi_e(10.0,$r)'	=> 1246.1144901994233444,
        'gsl_sf_Shi_e(50.0,$r)'	=> 5.292818448565845482e+19,
        'gsl_sf_Shi_e(300.0,$r)'	=> 3.248241254044332895e+127,
        'gsl_sf_Chi_e(-1.0,$r)'	=> 0.8378669409802082409,
        'gsl_sf_Chi_e(1.0/4294967296.0,$r)'	=> -21.603494113016717041,
        'gsl_sf_Chi_e(1.0/65536.0,$r)'	=> -10.513139223999384429,
        'gsl_sf_Chi_e(1.0/8.0,$r)'	=> -1.4983170827635760646,
        'gsl_sf_Chi_e(1.0,$r)'	=> 0.8378669409802082409,
        'gsl_sf_Chi_e(10.0,$r)'	=> 1246.1144860424544147,
        'gsl_sf_Chi_e(50.0,$r)'	=> 5.292818448565845482e+19,
        'gsl_sf_Chi_e(300.0,$r)'	=> 3.248241254044332895e+127,
        'gsl_sf_expint_3_e(1.0e-10,$r)'	=> 1.0e-10,
        'gsl_sf_expint_3_e(1.0e-05,$r)'	=> 9.9999999999999975e-06,
        'gsl_sf_expint_3_e(0.1,$r)'	=> 0.09997500714119079665122,
        'gsl_sf_expint_3_e(0.5,$r)'	=> 0.48491714311363971332427,
        'gsl_sf_expint_3_e(1.0,$r)'	=> 0.80751118213967145285833,
        'gsl_sf_expint_3_e(2.0,$r)'	=> 0.89295351429387631138208,
        'gsl_sf_expint_3_e(5.0,$r)'	=> 0.89297951156924921121856,
        'gsl_sf_expint_3_e(10.0,$r)'	=> 0.89297951156924921121856,
        'gsl_sf_expint_3_e(100.0,$r)'	=> 0.89297951156924921121856,
        'gsl_sf_Si_e(-1.0,$r)'	=> -0.9460830703671830149,
        'gsl_sf_Si_e(1.0e-10,$r)'	=> 1.0e-10,
        'gsl_sf_Si_e(1.0e-05,$r)'	=> 9.999999999944444444e-06,
        'gsl_sf_Si_e(0.1,$r)'	=> 0.09994446110827695016,
        'gsl_sf_Si_e(1.0,$r)'	=> 0.9460830703671830149,
        'gsl_sf_Si_e(10.0,$r)'	=> 1.6583475942188740493,
        'gsl_sf_Si_e(50.0,$r)'	=> 1.5516170724859358947,
        'gsl_sf_Si_e(300.0,$r)'	=> 1.5708810882137495193,
        'gsl_sf_Si_e(1.0e+20,$r)'	=> 1.5707963267948966192,
        'gsl_sf_Ci_e(1.0/4294967296.0,$r)'	=> -21.603494113016717041,
        'gsl_sf_Ci_e(1.0/65536.0,$r)'	=> -10.513139224115799751,
        'gsl_sf_Ci_e(1.0/8.0,$r)'	=> -1.5061295845296396649,
        'gsl_sf_Ci_e(1.0,$r)'	=> 0.3374039229009681347,
        'gsl_sf_Ci_e(10.0,$r)'	=> -0.04545643300445537263,
        'gsl_sf_Ci_e(50.0,$r)'	=> -0.005628386324116305440,
        'gsl_sf_Ci_e(300.0,$r)'	=> -0.003332199918592111780,
        'gsl_sf_Ci_e(65536.0,$r)'	=> 0.000010560248837656279453,
        'gsl_sf_Ci_e(4294967296.0,$r)'	=> -1.0756463261957757485e-10,
        'gsl_sf_Ci_e(1099511627776.0,$r)'	=> -3.689865584710764214e-13,
        'gsl_sf_atanint_e(1.0e-10,$r)'	=> 1.0e-10,
        'gsl_sf_atanint_e(1.0e-05,$r)'	=> 9.99999999988888888889e-06,
        'gsl_sf_atanint_e(0.1,$r)'	=> 0.09988928686033618404,
        'gsl_sf_atanint_e(1.0,$r)'	=> 0.91596559417721901505,
        'gsl_sf_atanint_e(2.0,$r)'	=> 1.57601540344632342236,
        'gsl_sf_atanint_e(10.0,$r)'	=> 3.71678149306806859029,
        'gsl_sf_atanint_e(50.0,$r)'	=> 6.16499047850274874222,
        'gsl_sf_atanint_e(300.0,$r)'	=> 8.96281388924518959990,
        'gsl_sf_atanint_e(1.0e+5,$r)'	=> 18.084471031038661920,
        'gsl_sf_fermi_dirac_m1_e(-10.0,$r)'	=> 0.00004539786870243439450,
        'gsl_sf_fermi_dirac_m1_e( -1.0,$r)'	=> 0.26894142136999512075,
        'gsl_sf_fermi_dirac_m1_e(  1.0,$r)'	=> 0.7310585786300048793,
        'gsl_sf_fermi_dirac_0_e(-10.0,$r)'	=> 0.00004539889921686464677,
        'gsl_sf_fermi_dirac_0_e( -1.0,$r)'	=> 0.31326168751822283405,
        'gsl_sf_fermi_dirac_0_e(  1.0,$r)'	=> 1.3132616875182228340,
        'gsl_sf_fermi_dirac_0_e( 10.0,$r)'	=> 10.000045398899216865,
        'gsl_sf_fermi_dirac_1_e(-10.0,$r)'	=> 0.00004539941448447633524,
        'gsl_sf_fermi_dirac_1_e( -2.0,$r)'	=> 0.13101248471442377127,
        'gsl_sf_fermi_dirac_1_e( -1.0,$r)'	=> 0.3386479964034521798,
        'gsl_sf_fermi_dirac_1_e( -0.4,$r)'	=> 0.5825520806897909028,
        'gsl_sf_fermi_dirac_1_e(  0.4,$r)'	=> 1.1423819861584355337,
        'gsl_sf_fermi_dirac_1_e(  1.0,$r)'	=> 1.8062860704447742567,
        'gsl_sf_fermi_dirac_1_e(  1.5,$r)'	=> 2.5581520872227806402,
        'gsl_sf_fermi_dirac_1_e(  2.5,$r)'	=> 4.689474797599761667,
        'gsl_sf_fermi_dirac_1_e( 10.0,$r)'	=> 51.64488866743374196,
        'gsl_sf_fermi_dirac_1_e( 12.0,$r)'	=> 73.64492792264531092,
        'gsl_sf_fermi_dirac_1_e( 20.0,$r)'	=> 201.64493406478707282,
        'gsl_sf_fermi_dirac_1_e( 50.0,$r)'	=> 1251.6449340668482264,
        'gsl_sf_fermi_dirac_2_e(-10.0,$r)'	=> 0.00004539967212174776662,
        'gsl_sf_fermi_dirac_2_e( -2.0,$r)'	=> 0.13313272938565030508,
        'gsl_sf_fermi_dirac_2_e( -1.0,$r)'	=> 0.3525648792978077590,
        'gsl_sf_fermi_dirac_2_e( -0.4,$r)'	=> 0.6229402647001272120,
        'gsl_sf_fermi_dirac_2_e(  0.4,$r)'	=> 1.2915805581060844533,
        'gsl_sf_fermi_dirac_2_e(  1.0,$r)'	=> 2.1641656128127008622,
        'gsl_sf_fermi_dirac_2_e(  1.5,$r)'	=> 3.247184513920792475,
        'gsl_sf_fermi_dirac_2_e(  2.5,$r)'	=> 6.797764392735056317,
        'gsl_sf_fermi_dirac_2_e( 10.0,$r)'	=> 183.11605273482105278,
        'gsl_sf_fermi_dirac_2_e( 12.0,$r)'	=> 307.73921494638635166,
        'gsl_sf_fermi_dirac_2_e( 20.0,$r)'	=> 1366.2320146723590157,
        'gsl_sf_fermi_dirac_2_e( 50.0,$r)'	=> 20915.580036675744655,
        'gsl_sf_fermi_dirac_2_e(200.0,$r)'	=> 1.3336623201467029786e+06,
        'gsl_sf_fermi_dirac_mhalf_e(-10.0,$r)'	=> 0.00004539847236080549532,
        'gsl_sf_fermi_dirac_mhalf_e( -2.0,$r)'	=> 0.12366562180120994266,
        'gsl_sf_fermi_dirac_mhalf_e( -1.0,$r)'	=> 0.29402761761145122022,
        'gsl_sf_fermi_dirac_mhalf_e( -0.4,$r)'	=> 0.4631755336886027800,
        'gsl_sf_fermi_dirac_mhalf_e(  0.4,$r)'	=> 0.7654084737661656915,
        'gsl_sf_fermi_dirac_mhalf_e(  1.0,$r)'	=> 1.0270571254743506890,
        'gsl_sf_fermi_dirac_mhalf_e(  1.5,$r)'	=> 1.2493233478527122008,
        'gsl_sf_fermi_dirac_mhalf_e(  2.5,$r)'	=> 1.6663128834358313625,
        'gsl_sf_fermi_dirac_mhalf_e( 10.0,$r)'	=> 3.552779239536617160,
        'gsl_sf_fermi_dirac_mhalf_e( 12.0,$r)'	=> 3.897268231925439359,
        'gsl_sf_fermi_dirac_mhalf_e( 20.0,$r)'	=> 5.041018507535328603,
        'gsl_sf_fermi_dirac_mhalf_e( 50.0,$r)'	=> 7.977530858581869960,
        'gsl_sf_fermi_dirac_half_e(-10.0,$r)'	=> 0.00004539920105264132755,
        'gsl_sf_fermi_dirac_half_e( -2.0,$r)'	=> 0.12929851332007559106,
        'gsl_sf_fermi_dirac_half_e( -1.0,$r)'	=> 0.3277951592607115477,
        'gsl_sf_fermi_dirac_half_e( -0.4,$r)'	=> 0.5522452153690688947,
        'gsl_sf_fermi_dirac_half_e(  0.4,$r)'	=> 1.0386797503389389277,
        'gsl_sf_fermi_dirac_half_e(  1.0,$r)'	=> 1.5756407761513002308,
        'gsl_sf_fermi_dirac_half_e(  1.5,$r)'	=> 2.1448608775831140360,
        'gsl_sf_fermi_dirac_half_e(  2.5,$r)'	=> 3.606975377950373251,
        'gsl_sf_fermi_dirac_half_e( 10.0,$r)'	=> 24.084656964637653615,
        'gsl_sf_fermi_dirac_half_e( 12.0,$r)'	=> 31.540203287044242593,
        'gsl_sf_fermi_dirac_half_e( 20.0,$r)'	=> 67.49151222165892049,
        'gsl_sf_fermi_dirac_half_e( 50.0,$r)'	=> 266.09281252136259343,
        'gsl_sf_fermi_dirac_3half_e(-10.0,$r)'	=> 0.00004539956540456176333,
        'gsl_sf_fermi_dirac_3half_e( -2.0,$r)'	=> 0.13224678225177236685,
        'gsl_sf_fermi_dirac_3half_e( -1.0,$r)'	=> 0.3466747947990574170,
        'gsl_sf_fermi_dirac_3half_e( -0.4,$r)'	=> 0.6056120213305040910,
        'gsl_sf_fermi_dirac_3half_e(  0.4,$r)'	=> 1.2258236403963668282,
        'gsl_sf_fermi_dirac_3half_e(  1.0,$r)'	=> 2.0022581487784644573,
        'gsl_sf_fermi_dirac_3half_e(  1.5,$r)'	=> 2.9277494127932173068,
        'gsl_sf_fermi_dirac_3half_e(  2.5,$r)'	=> 5.768879312210516582,
        'gsl_sf_fermi_dirac_3half_e( 10.0,$r)'	=> 101.00510084332600020,
        'gsl_sf_fermi_dirac_3half_e( 12.0,$r)'	=> 156.51518642795728036,
        'gsl_sf_fermi_dirac_3half_e( 20.0,$r)'	=> 546.5630100657601959,
        'gsl_sf_fermi_dirac_3half_e( 50.0,$r)'	=> 5332.353566687145552,
        'gsl_sf_fermi_dirac_int_e(3,  -2.0,$r)'	=> 0.1342199155038680215,
        'gsl_sf_fermi_dirac_int_e(3,   0.0,$r)'	=> 0.9470328294972459176,
        'gsl_sf_fermi_dirac_int_e(3,   0.1,$r)'	=> 1.0414170610956165759,
        'gsl_sf_fermi_dirac_int_e(3,   1.0,$r)'	=> 2.3982260822489407070,
        'gsl_sf_fermi_dirac_int_e(3,   3.0,$r)'	=> 12.621635313399690724,
        'gsl_sf_fermi_dirac_int_e(3, 100.0,$r)'	=> 4.174893231066566793e+06,
        'gsl_sf_fermi_dirac_int_e(3, 500.0,$r)'	=> 2.604372285319088354e+09,
        'gsl_sf_fermi_dirac_int_e(5,  -2.0,$r)'	=> 0.13505242246823676478,
        'gsl_sf_fermi_dirac_int_e(5,   0.0,$r)'	=> 0.9855510912974351041,
        'gsl_sf_fermi_dirac_int_e(5,   0.1,$r)'	=> 1.0876519750101492782,
        'gsl_sf_fermi_dirac_int_e(5,   1.0,$r)'	=> 2.6222337848692390539,
        'gsl_sf_fermi_dirac_int_e(5,   3.0,$r)'	=> 17.008801618012113022,
        'gsl_sf_fermi_dirac_int_e(5, 100.0,$r)'	=> 1.3957522531334869874e+09,
        'gsl_sf_fermi_dirac_int_e(5, 500.0,$r)'	=> 2.1705672808114817955e+13,
        'gsl_sf_fermi_dirac_int_e(7,  -2.0,$r)'	=> 0.1352641105671255851,
        'gsl_sf_fermi_dirac_int_e(7,   0.0,$r)'	=> 0.9962330018526478992,
        'gsl_sf_fermi_dirac_int_e(7,   0.1,$r)'	=> 1.1005861815180315485,
        'gsl_sf_fermi_dirac_int_e(7,   1.0,$r)'	=> 2.6918878172003129203,
        'gsl_sf_fermi_dirac_int_e(7,   3.0,$r)'	=> 19.033338976999367642,
        'gsl_sf_fermi_dirac_int_e(7,  10.0,$r)'	=> 5654.530932873610014,
        'gsl_sf_fermi_dirac_int_e(7,  50.0,$r)'	=> 1.005005069985066278e+09,
        'gsl_sf_fermi_dirac_int_e(7, 500.0,$r)'	=> 9.691690268341569514e+16,
        'gsl_sf_fermi_dirac_int_e(9,  -2.0,$r)'	=> 0.1353174385330242691,
        'gsl_sf_fermi_dirac_int_e(9,   0.0,$r)'	=> 0.9990395075982715656,
        'gsl_sf_fermi_dirac_int_e(9,   0.1,$r)'	=> 1.1039997234712941212,
        'gsl_sf_fermi_dirac_int_e(9,   1.0,$r)'	=> 2.7113648898129249947,
        'gsl_sf_fermi_dirac_int_e(9,   3.0,$r)'	=> 19.768544008138602223,
        'gsl_sf_fermi_dirac_int_e(9,  10.0,$r)'	=> 10388.990167312912478,
        'gsl_sf_fermi_dirac_int_e(9,  50.0,$r)'	=> 2.85466960802601649e+10,
        'gsl_sf_fermi_dirac_int_e(9, 500.0,$r)'	=> 2.69273849842695876e+20,
        'gsl_sf_fermi_dirac_int_e(10,  -2.0,$r)'	=> 0.13532635396712288092,
        'gsl_sf_fermi_dirac_int_e(10,   0.0,$r)'	=> 0.9995171434980607541,
        'gsl_sf_fermi_dirac_int_e(10,   0.1,$r)'	=> 1.1045818238852612296,
        'gsl_sf_fermi_dirac_int_e(10,   1.0,$r)'	=> 2.7147765350346120647,
        'gsl_sf_fermi_dirac_int_e(10,   3.0,$r)'	=> 19.917151938411675171,
        'gsl_sf_fermi_dirac_int_e(10,  10.0,$r)'	=> 12790.918595516495955,
        'gsl_sf_fermi_dirac_int_e(10,  50.0,$r)'	=> 1.3147703201869657654e+11,
        'gsl_sf_fermi_dirac_int_e(10, 500.0,$r)'	=> 1.2241331244469204398e+22,
        'gsl_sf_fermi_dirac_int_e(11,  -2.0,$r)'	=> 0.1353308162894847149,
        'gsl_sf_fermi_dirac_int_e(11,   0.0,$r)'	=> 0.9997576851438581909,
        'gsl_sf_fermi_dirac_int_e(11,   0.1,$r)'	=> 1.1048751811565850418,
        'gsl_sf_fermi_dirac_int_e(11,   1.0,$r)'	=> 2.7165128749007313436,
        'gsl_sf_fermi_dirac_int_e(11,   3.0,$r)'	=> 19.997483022044603065,
        'gsl_sf_fermi_dirac_int_e(11,  10.0,$r)'	=> 14987.996005901818036,
        'gsl_sf_fermi_dirac_int_e(11,  50.0,$r)'	=> 5.558322924078990628e+11,
        'gsl_sf_fermi_dirac_int_e(11, 500.0,$r)'	=> 5.101293089606198280e+23,
        'gsl_sf_fermi_dirac_int_e(20,  -2.0,$r)'	=> 0.13533527450327238373,
        'gsl_sf_fermi_dirac_int_e(20,   0.0,$r)'	=> 0.9999995232582155428,
        'gsl_sf_fermi_dirac_int_e(20,   0.1,$r)'	=> 1.1051703357941368203,
        'gsl_sf_fermi_dirac_int_e(20,   1.0,$r)'	=> 2.7182783069905721654,
        'gsl_sf_fermi_dirac_int_e(20,   3.0,$r)'	=> 20.085345296028242734,
        'gsl_sf_fermi_dirac_int_e(20,  10.0,$r)'	=> 21898.072920149606475,
        'gsl_sf_fermi_dirac_int_e(20,  50.0,$r)'	=> 1.236873256595717618e+16,
        'gsl_sf_fermi_dirac_int_e(20, 500.0,$r)'	=> 9.358938204369557277e+36,
        'gsl_sf_gegenpoly_1_e(-0.2,   1.0,$r)'	=> -0.4,
        'gsl_sf_gegenpoly_1_e( 0.0,   1.0,$r)'	=> 2.0,
        'gsl_sf_gegenpoly_1_e( 1.0,   1.0,$r)'	=> 2.0,
        'gsl_sf_gegenpoly_1_e( 1.0,   0.5,$r)'	=> 1.0,
        'gsl_sf_gegenpoly_1_e( 5.0,   1.0,$r)'	=> 10.0,
        'gsl_sf_gegenpoly_1_e( 100.0, 0.5,$r)'	=> 100.0,
        'gsl_sf_gegenpoly_2_e(-0.2,   0.5,$r)'	=> 0.12,
        'gsl_sf_gegenpoly_2_e( 0.0,   1.0,$r)'	=> 1.00,
        'gsl_sf_gegenpoly_2_e( 1.0,   1.0,$r)'	=> 3.00,
        'gsl_sf_gegenpoly_2_e( 1.0,   0.1,$r)'	=> -0.96,
        'gsl_sf_gegenpoly_2_e( 5.0,   1.0,$r)'	=> 55.0,
        'gsl_sf_gegenpoly_2_e( 100.0, 0.5,$r)'	=> 4950.0,
        'gsl_sf_gegenpoly_3_e(-0.2,   0.5,$r)'	=> 0.112,
        'gsl_sf_gegenpoly_3_e( 0.0,   1.0,$r)'	=> -2.0/3.0,
        'gsl_sf_gegenpoly_3_e( 1.0,   1.0,$r)'	=> 4.000,
        'gsl_sf_gegenpoly_3_e( 1.0,   0.1,$r)'	=> -0.392,
        'gsl_sf_gegenpoly_3_e( 5.0,   1.0,$r)'	=> 220.000,
        'gsl_sf_gegenpoly_3_e( 100.0, 0.5,$r)'	=> 161600.000,
        'gsl_sf_gegenpoly_n_e(1,       1.0, 1.0,$r)'	=> 2.000              ,
        'gsl_sf_gegenpoly_n_e(10,      1.0, 1.0,$r)'	=> 11.000             ,
        'gsl_sf_gegenpoly_n_e(10,      1.0, 0.1,$r)'	=> -0.4542309376      ,
        'gsl_sf_gegenpoly_n_e(10,      5.0, 1.0,$r)'	=> 9.23780e+4         ,
        'gsl_sf_gegenpoly_n_e(10,    100.0, 0.5,$r)'	=> 1.5729338392690000e+13,
        'gsl_sf_gegenpoly_n_e(1000,  100.0, 1.0,$r)'	=> 3.3353666135627322e+232,
        'gsl_sf_gegenpoly_n_e(100,  2000.0, 1.0,$r)'	=> 5.8753432034937579e+202,
        'gsl_sf_gegenpoly_n_e(103,   207.0, 2.0,$r)'	=> 1.4210272202235983e+145,
        'gsl_sf_gegenpoly_n_e(103,    -0.4, 0.3,$r)'	=> -1.64527498094522e-04,
        'gsl_sf_laguerre_1_e(0.5, -1.0,$r)'	=> 2.5,
        'gsl_sf_laguerre_1_e(0.5,  1.0,$r)'	=> 0.5,
        'gsl_sf_laguerre_1_e(1.0,  1.0,$r)'	=> 1.0,
        'gsl_sf_laguerre_2_e( 0.5, -1.0,$r)'	=> 4.875,
        'gsl_sf_laguerre_2_e( 0.5,  1.0,$r)'	=> -0.125,
        'gsl_sf_laguerre_2_e( 1.0,  1.0,$r)'	=>  0.5,
        'gsl_sf_laguerre_2_e(-1.0,  1.0,$r)'	=> -0.5,
        'gsl_sf_laguerre_2_e(-2.0,  1.0,$r)'	=>  0.5,
        'gsl_sf_laguerre_2_e(-3.0,  1.0,$r)'	=>  2.5,
        'gsl_sf_laguerre_3_e(0.5, -1.0,$r)'	=> 8.479166666666666667,
        'gsl_sf_laguerre_3_e(0.5,  1.0,$r)'	=> -0.6041666666666666667,
        'gsl_sf_laguerre_3_e(1.0,  1.0,$r)'	=> -0.16666666666666666667,
        'gsl_sf_laguerre_3_e( 2.0,  1.0,$r)'	=> 2.3333333333333333333,
        'gsl_sf_laguerre_3_e(-2.0,  1.0,$r)'	=> 1.0/3.0,
        'gsl_sf_laguerre_3_e(-3.0,  1.0,$r)'	=> -1.0/6.0,
        'gsl_sf_laguerre_3_e(-4.0,  1.0,$r)'	=> -8.0/3.0,
        'gsl_sf_laguerre_n_e(1, 0.5, 1.0,$r)'	=> 0.5,
        'gsl_sf_laguerre_n_e(2, 1.0, 1.0,$r)'	=> 0.5,
        'gsl_sf_laguerre_n_e(3, 2.0, 1.0,$r)'	=> 2.3333333333333333333,
        'gsl_sf_laguerre_n_e(4, 2.0, 0.5,$r)'	=> 6.752604166666666667,
        'gsl_sf_laguerre_n_e(90, 2.0,  0.5,$r)'	=> -48.79047157201507897,
        'gsl_sf_laguerre_n_e(90, 2.0, -100.0,$r)'	=> 2.5295879275042410902e+63,
        'gsl_sf_laguerre_n_e(90, 2.0,  100.0,$r)'	=> -2.0929042259546928670e+20,
        'gsl_sf_laguerre_n_e(100, 2.0,  0.5,$r)'	=> -28.764832945909097418,
        'gsl_sf_laguerre_n_e(1000, 2.0, -0.5,$r)'	=> 2.4399915170947549589e+21,
        'gsl_sf_laguerre_n_e(1000, 2.0,  0.5,$r)'	=> -306.77440254315317525,
        'gsl_sf_laguerre_n_e(100000, 2.0, 1.0,$r)'	=> 5107.73491348319,
        'gsl_sf_laguerre_n_e(1e5, 2.5, 2.5,$r)'	    => -0.41491680394598644969113795e5,
        'gsl_sf_laguerre_n_e(1e5+1, 2.5, 2.5,$r)'	=> -0.41629446949552321027514888e5,
        'gsl_sf_laguerre_n_e(1e6+1, 2.5, 2.5,$r)'	=> -0.48017961545391273151977118e6,
        'gsl_sf_laguerre_n_e(5e6+1, 2.5, 2.5,$r)'	=> -0.15174037401611122446089494e7,
        'gsl_sf_laguerre_n_e(8e6+1, 2.5, 2.5,$r)'	=>  0.63251509472091810994286362e6,
        'gsl_sf_laguerre_n_e(1e7+1, 2.5, 2.5,$r)'	=>  0.15299484685632983178033887e7,
        'gsl_sf_laguerre_n_e(1e8+1, 2.5, 2.5,$r)'	=>  0.23645341644922756725290777e8,
        'gsl_sf_laguerre_n_e(1e9+1, 2.5, 2.5,$r)'	=> -0.17731002248958790286185878e8,
        'gsl_sf_laguerre_n_e(1, -2.0, 1.0,$r)'	    =>  -2.0,
        'gsl_sf_laguerre_n_e(2, -2.0, 1.0,$r)'	    =>   0.5,
        'gsl_sf_laguerre_n_e(3, -2.0, 1.0,$r)'	    =>   1.0/3.0,
        'gsl_sf_laguerre_n_e(10, -2.0, 1.0,$r)'	    => -0.04654954805996472663,
        'gsl_sf_laguerre_n_e(10, -5.0, 1.0,$r)'	    => -0.0031385030864197530864,
        'gsl_sf_laguerre_n_e(10, -9.0, 1.0,$r)'	    => -2.480158730158730159e-06,
        'gsl_sf_laguerre_n_e(10, -11.0,  1.0,$r)'	=> 2.7182818011463844797,
        'gsl_sf_laguerre_n_e(10, -11.0, -1.0,$r)'	=> 0.3678794642857142857,
        'gsl_sf_laguerre_n_e(100, -2.0,  1.0,$r)'	=>  -0.0027339992019526273866,
        'gsl_sf_laguerre_n_e(100, -2.0, -1.0,$r)'	=>   229923.09193402028290,
        'gsl_sf_laguerre_n_e(100, -10.0,  1.0,$r)'	=>  3.25966665871244092e-11,
        'gsl_sf_laguerre_n_e(100, -10.0, -1.0,$r)'	=>  0.00016484365618205810025,
        'gsl_sf_laguerre_n_e(100, -20.0, 1.0,$r)'	=>  5.09567630343671251e-21,
        'gsl_sf_laguerre_n_e(100, -30.0, 1.0,$r)'	=>  3.46063150272466192e-34,
        'gsl_sf_laguerre_n_e(100, -50.0,  1.0,$r)'	=>  1.20981872933162889e-65,
        'gsl_sf_laguerre_n_e(100, -50.0, -1.0,$r)'	=>  8.60763477742332922e-65,
        'gsl_sf_laguerre_n_e(100, -50.5,  1.0,$r)'	=>  4.84021010426688393e-31,
        'gsl_sf_laguerre_n_e(100, -50.5, -1.0,$r)'	=>  8.49861345212160618e-33,
        'gsl_sf_laguerre_n_e(100, -101.0,  1.0,$r)'	=> 2.7182818284590452354,
        'gsl_sf_laguerre_n_e(100, -101.0, -1.0,$r)'	=> 0.3678794411714423216,
        'gsl_sf_laguerre_n_e(100, -102.0,  1.0,$r)'	=> 271.8281828459045235,
        'gsl_sf_laguerre_n_e(100, -102.0, -1.0,$r)'	=> 37.52370299948711680,
        'gsl_sf_laguerre_n_e(100, -110.0,  1.0,$r)'	=> 1.0666955248998831554e+13,
        'gsl_sf_laguerre_n_e(100, -110.0, -1.0,$r)'	=> 1.7028306108058225871e+12,
        'gsl_sf_laguerre_n_e(100, -200.0,  1.0,$r)'	=> 7.47851889721356628e+58,
        'gsl_sf_laguerre_n_e(100, -200.0, -1.0,$r)'	=> 2.73740299754732273e+58,
        'gsl_sf_laguerre_n_e(100, -50.0,  10.0,$r)'	=> 4.504712811317745591e-21,
        'gsl_sf_laguerre_n_e(100, -50.0, -10.0,$r)'	=> 1.475165520610679937e-11,
        'gsl_sf_laguerre_n_e(100, 0.0, 0.5,$r)'	=> 0.18682260367692278801,
        'gsl_sf_laguerre_n_e(100, 0.0, 10.5,$r)'	=> 9.1796907354050059874,
        'gsl_sf_laguerre_n_e(100, 0.0, -10.5,$r)'	=> 5.6329215744170606488e24,
        'gsl_sf_laguerre_n_e(100, 0.0, 100.5,$r)'	=> -3.9844782875811907525e20,
        'gsl_sf_laguerre_n_e(100, 0.0, 150,$r)'	=> -1.4463204337261709595e31,
        'gsl_sf_lambert_W0_e(0.0,$r)'	=>  0.0,
        'gsl_sf_lambert_W0_e(1.0,$r)'	=>  0.567143290409783872999969,
        'gsl_sf_lambert_W0_e(2.0,$r)'	=>  0.852605502013725491346472,
        'gsl_sf_lambert_W0_e(20.0,$r)'	=> 2.205003278024059970493066,
        'gsl_sf_lambert_W0_e(1000.0,$r)'	=> 5.24960285240159622712606,
        'gsl_sf_lambert_W0_e(1.0e+6,$r)'	=> 11.38335808614005262200016,
        'gsl_sf_lambert_W0_e(1.0e+12,$r)'	=> 24.43500440493491313826305,
        'gsl_sf_lambert_W0_e(1.0e+308,$r)'	=> 702.641362034106812081125,
        'gsl_sf_lambert_W0_e(1.6849341956993852953416990,$r)'	=> 0.775706963944252869680440,
        'gsl_sf_lambert_W0_e(-1.0/$M_E - $GSL_DBL_EPSILON,$r)'	=> -1.0,
        'gsl_sf_lambert_W0_e(-1.0/$M_E + 1.0/(1024.0*1024.0*1024.0),$r)'	=> -0.999928845560308370714970,
        'gsl_sf_lambert_W0_e(-1.0/$M_E + 1.0/(1024.0*1024.0),$r)'	=> -0.997724730359774141620354,
        'gsl_sf_lambert_W0_e(-1.0/$M_E + 1.0/512.0,$r)'	=> -0.900335676696088773044678,
        'gsl_sf_lambert_W0_e(-1.0/$M_E + 0.25,$r)'	=> -0.1349044682661213545487599,
        'gsl_sf_lambert_Wm1_e(0.0,$r)'	=>  0.0,
        'gsl_sf_lambert_Wm1_e(1.0,$r)'	=>  0.567143290409783872999969,
        'gsl_sf_lambert_Wm1_e(2.0,$r)'	=>  0.852605502013725491346472,
        'gsl_sf_lambert_Wm1_e(20.0,$r)'	=> 2.205003278024059970493066,
        'gsl_sf_lambert_Wm1_e(-1.0/$M_E - $GSL_DBL_EPSILON,$r)'	=> -1.0,
        'gsl_sf_lambert_Wm1_e(-1.0/$M_E + 1.0/(1024.0*1024.0*1024.0),$r)'	=> -1.000071157815154608049055,
        'gsl_sf_lambert_Wm1_e(-1.0/$M_E + 1.0/(1024.0*1024.0),$r)'	=> -1.002278726118593023934693,
        'gsl_sf_lambert_Wm1_e(-1.0/$M_E + 1.0/512.0,$r)'	=> -1.106761200865743124599130,
        'gsl_sf_lambert_Wm1_e(-1.0/$M_E + 1.0/64.0,$r)'	=> -1.324240940341812125489772,
        'gsl_sf_lambert_Wm1_e(-1.0/$M_E + 0.25,$r)'	=> -3.345798131120112,
        'gsl_sf_log_e(0.1,$r)'	=> -2.3025850929940456840,
        'gsl_sf_log_e(1.1,$r)'	=> 0.09531017980432486004,
        'gsl_sf_log_e(1000.0,$r)'	=> 6.907755278982137052,
        'gsl_sf_log_abs_e(-0.1,$r)'	=> -2.3025850929940456840,
        'gsl_sf_log_abs_e(-1.1,$r)'	=> 0.09531017980432486004,
        'gsl_sf_log_abs_e(-1000.0,$r)'	=> 6.907755278982137052,
        'gsl_sf_log_abs_e(0.1,$r)'	=> -2.3025850929940456840,
        'gsl_sf_log_abs_e(1.1,$r)'	=> 0.09531017980432486004,
        'gsl_sf_log_abs_e(1000.0,$r)'	=> 6.907755278982137052,
        'gsl_sf_log_1plusx_e(1.0e-10,$r)'	=> 9.999999999500000000e-11,
        'gsl_sf_log_1plusx_e(1.0e-8,$r)'	=> 9.999999950000000333e-09,
        'gsl_sf_log_1plusx_e(1.0e-4,$r)'	=> 0.00009999500033330833533,
        'gsl_sf_log_1plusx_e(0.1,$r)'	=> 0.09531017980432486004,
        'gsl_sf_log_1plusx_e(0.49,$r)'	=> 0.3987761199573677730,
        'gsl_sf_log_1plusx_e(-0.49,$r)'	=> -0.6733445532637655964,
        'gsl_sf_log_1plusx_e(1.0,$r)'	=> $M_LN2,
        'gsl_sf_log_1plusx_e(-0.99,$r)'	=> -4.605170185988091368,
        'gsl_sf_log_1plusx_mx_e(1.0e-10,$r)'	=> -4.999999999666666667e-21,
        'gsl_sf_log_1plusx_mx_e(1.0e-8,$r)'	=> -4.999999966666666917e-17,
        'gsl_sf_log_1plusx_mx_e(1.0e-4,$r)'	=> -4.999666691664666833e-09,
        'gsl_sf_log_1plusx_mx_e(0.1,$r)'	=> -0.004689820195675139956,
        'gsl_sf_log_1plusx_mx_e(0.49,$r)'	=> -0.09122388004263222704,
        'gsl_sf_log_1plusx_mx_e(-0.49,$r)'	=> -0.18334455326376559639,
        'gsl_sf_log_1plusx_mx_e(1.0,$r)'	=> $M_LN2-1.0,
        'gsl_sf_log_1plusx_mx_e(-0.99,$r)'	=> -3.615170185988091368,
        'gsl_sf_pow_int_e(2.0, 3,$r)'	=> 8.0,
        'gsl_sf_pow_int_e(-2.0, 3,$r)'	=> -8.0,
        'gsl_sf_pow_int_e(2.0, -3,$r)'	=> 1.0/8.0,
        'gsl_sf_pow_int_e(-2.0, -3,$r)'	=> -1.0/8.0,
        'gsl_sf_pow_int_e(10.0, 4,$r)'	=> 1.0e+4,
        'gsl_sf_pow_int_e(10.0, -4,$r)'	=> 1.0e-4,
        'gsl_sf_pow_int_e(-10.0, 4,$r)'	=> 1.0e+4,
        'gsl_sf_pow_int_e(-10.0, -4,$r)'	=> 1.0e-4,
        'gsl_sf_pow_int_e(10.0, 40,$r)'	=> 1.0e+40,
        'gsl_sf_pow_int_e(8.0, -40,$r)'	=> 7.523163845262640051e-37,
        'gsl_sf_pow_int_e(-10.0, 40,$r)'	=> 1.0e+40,
        'gsl_sf_pow_int_e(-8.0, -40,$r)'	=> 7.523163845262640051e-37,
        'gsl_sf_pow_int_e(10.0, 41,$r)'	=> 1.0e+41,
        'gsl_sf_pow_int_e(8.0, -41,$r)'	=> 9.403954806578300064e-38,
        'gsl_sf_pow_int_e(-10.0, 41,$r)'	=> -1.0e+41,
        'gsl_sf_pow_int_e(-8.0, -41,$r)'	=> -9.403954806578300064e-38,
        'gsl_sf_psi_int_e(1,$r)'	=> -0.57721566490153286060,
        'gsl_sf_psi_int_e(2,$r)'	=> 0.42278433509846713939,
        'gsl_sf_psi_int_e(3,$r)'	=> 0.92278433509846713939,
        'gsl_sf_psi_int_e(4,$r)'	=> 1.2561176684318004727,
        'gsl_sf_psi_int_e(5,$r)'	=> 1.5061176684318004727,
        'gsl_sf_psi_int_e(100,$r)'	=> 4.600161852738087400,
        'gsl_sf_psi_int_e(110,$r)'	=> 4.695928024251535633,
        'gsl_sf_psi_int_e(5000,$r)'	=> 8.517093188082904107,
        'gsl_sf_psi_e(5000.0,$r)'	=> 8.517093188082904107,
        'gsl_sf_psi_e(5.0,$r)'	=> 1.5061176684318004727,
        'gsl_sf_psi_e(-10.5,$r)'	=>       2.3982391295357816134,
        'gsl_sf_psi_e(-100.5,$r)'	=>      4.615124601338064117,
        'gsl_sf_psi_e(-1.0e+5-0.5,$r)'	=> 11.512935464924395337,
        'gsl_sf_psi_e(-262144.0-0.5,$r)'	=> 12.476653064769611581,
        'gsl_sf_psi_1piy_e(0.8,$r)'	=> -0.07088340212750589223,
        'gsl_sf_psi_1piy_e(1.0,$r)'	=>  0.09465032062247697727,
        'gsl_sf_psi_1piy_e(5.0,$r)'	=>  1.6127848446157465854,
        'gsl_sf_psi_1piy_e(100.0,$r)'	=>  4.605178519404762003,
        'gsl_sf_psi_1piy_e(2000.0,$r)'	=> 7.600902480375416216,
        'gsl_sf_psi_1piy_e(-0.8,$r)'	=> -0.07088340212750589223,
        'gsl_sf_psi_1piy_e(-1.0,$r)'	=>  0.09465032062247697727,
        'gsl_sf_psi_1piy_e(-5.0,$r)'	=>  1.6127848446157465854,
        'gsl_sf_psi_1piy_e(-100.0,$r)'	=>  4.605178519404762003,
        'gsl_sf_psi_1piy_e(-2000.0,$r)'	=> 7.600902480375416216,
        'gsl_sf_psi_1_int_e(1,$r)'	=> 1.6449340668482264364,
        'gsl_sf_psi_1_int_e(2,$r)'	=> 0.64493406684822643647,
        'gsl_sf_psi_1_int_e(3,$r)'	=> 0.39493406684822643647,
        'gsl_sf_psi_1_int_e(4,$r)'	=> 0.28382295573711532536,
        'gsl_sf_psi_1_int_e(1,$r)'	=> 1.6449340668482264365,
        'gsl_sf_psi_1_int_e(5,$r)'	=> 0.22132295573711532536,
        'gsl_sf_psi_1_int_e(100,$r)'	=> 0.010050166663333571395,
        'gsl_sf_psi_1_int_e(110,$r)'	=> 0.009132356622022545705,
        'gsl_sf_psi_1_int_e(500,$r)'	=> 0.0020020013333322666697,
        'gsl_sf_psi_1_e(1.0/32.0,$r)'	=> 1025.5728544782377089,
        'gsl_sf_psi_1_e(1.0,$r)'	=> 1.6449340668482264365,
        'gsl_sf_psi_1_e(5.0,$r)'	=> 0.22132295573711532536,
        'gsl_sf_psi_1_e(100.0,$r)'	=> 0.010050166663333571395,
        'gsl_sf_psi_1_e(110.0,$r)'	=> 0.009132356622022545705,
        'gsl_sf_psi_1_e(500.0,$r)'	=> 0.0020020013333322666697,
        'gsl_sf_psi_1_e(-1.0 - 1.0/128.0,$r)'	=> 16386.648472598746587,
        'gsl_sf_psi_1_e(-1.50,$r)'	=> 9.3792466449891237539,
        'gsl_sf_psi_1_e(-10.5,$r)'	=> 9.7787577398148123845,
        'gsl_sf_psi_1_e(-15.5,$r)'	=> 9.8071247184113896201,
        'gsl_sf_psi_1_e(-50.5,$r)'	=> 9.8499971860824842274,
        'gsl_sf_psi_1_e(-1000.5,$r)'	=> 9.8686054001734414233,
        'gsl_sf_psi_n_e(1, 1,$r)'	=> 1.6449340668482264364,
        'gsl_sf_psi_n_e(1, 2,$r)'	=> 0.64493406684822643647,
        'gsl_sf_psi_n_e(1, 3,$r)'	=> 0.39493406684822643647,
        'gsl_sf_psi_n_e(1, 4,$r)'	=> 0.28382295573711532536,
        'gsl_sf_psi_n_e(1, 5,$r)'	=> 0.22132295573711532536,
        'gsl_sf_psi_n_e(1, 100,$r)'	=> 0.010050166663333571395,
        'gsl_sf_psi_n_e(1, 110,$r)'	=> 0.009132356622022545705,
        'gsl_sf_psi_n_e(1, 500,$r)'	=> 0.0020020013333322666697,
        'gsl_sf_psi_n_e(3, 5.0,$r)'	=> 0.021427828192755075022,
        'gsl_sf_psi_n_e(3, 500.0,$r)'	=> 1.6048063999872000683e-08,
        'gsl_sf_psi_n_e(10, 5.0,$r)'	=> -0.08675107579196581317,
        'gsl_sf_psi_n_e(10, 50.0,$r)'	=> -4.101091112731268288e-12,
        'gsl_sf_psi_n_e(0, -1.5,$r)'	=> 0.70315664064524318723,
        'gsl_sf_psi_n_e(1, -1.5,$r)'	=> 9.3792466449891237539,
        'gsl_sf_synchrotron_1_e(0.01,$r)'	=>  0.444972504114210632,
        'gsl_sf_synchrotron_1_e(1.0,$r)'	=>   0.651422815355364504,
        'gsl_sf_synchrotron_1_e(10.0,$r)'	=>  0.000192238264300868882,
        'gsl_sf_synchrotron_1_e(100.0,$r)'	=> 4.69759366592220221e-43,
        'gsl_sf_synchrotron_2_e(0.01,$r)'	=>  0.23098077342226277732,
        'gsl_sf_synchrotron_2_e(1.0,$r)'	=>   0.4944750621042082670,
        'gsl_sf_synchrotron_2_e(10.0,$r)'	=>  0.00018161187569530204281,
        'gsl_sf_synchrotron_2_e(256.0,$r)'	=> 1.3272635474353774058e-110,
        'gsl_sf_transport_2_e(1.0e-10,$r)'	=> 9.9999999999999999999e-11,
        'gsl_sf_transport_2_e(1.0,$r)'	=>     0.97303256135517012845,
        'gsl_sf_transport_2_e(3.0,$r)'	=>     2.41105004901695346199,
        'gsl_sf_transport_2_e(10.0,$r)'	=>    3.28432911449795173575,
        'gsl_sf_transport_2_e(100.0,$r)'	=>   3.28986813369645287294,
        'gsl_sf_transport_2_e(1.0e+05,$r)'	=> 3.28986813369645287294,
        'gsl_sf_transport_3_e(1.0e-10,$r)'	=> 4.999999999999999999997e-21,
        'gsl_sf_transport_3_e(1.0,$r)'	=>     0.479841006572417499939,
        'gsl_sf_transport_3_e(3.0,$r)'	=>     3.210604662942246772338,
        'gsl_sf_transport_3_e(5.0,$r)'	=>     5.614386613842273228585,
        'gsl_sf_transport_3_e(10.0,$r)'	=>    7.150322712008592975030,
        'gsl_sf_transport_3_e(30.0,$r)'	=>    7.212341416160946511930,
        'gsl_sf_transport_3_e(100.0,$r)'	=>   7.212341418957565712398,
        'gsl_sf_transport_3_e(1.0e+05,$r)'	=> 7.212341418957565712398,
        'gsl_sf_transport_4_e(1.0e-10,$r)'	=> 3.33333333333333333333e-31,
        'gsl_sf_transport_4_e(1.0e-07,$r)'	=> 3.33333333333333166666e-22,
        'gsl_sf_transport_4_e(1.0e-04,$r)'	=> 3.33333333166666666726e-13,
        'gsl_sf_transport_4_e(0.1,$r)'	=> 0.000333166726172109903824,
        'gsl_sf_transport_4_e(1.0,$r)'	=> 0.31724404523442648241,
        'gsl_sf_transport_4_e(3.0,$r)'	=> 5.96482239737147652446,
        'gsl_sf_transport_4_e(5.0,$r)'	=> 15.3597843168821829816,
        'gsl_sf_transport_4_e(10.0,$r)'	=> 25.2736676770304417334,
        'gsl_sf_transport_4_e(30.0,$r)'	=> 25.9757575220840937469,
        'gsl_sf_transport_4_e(100.0,$r)'	=> 25.9757576090673165963,
        'gsl_sf_transport_4_e(1.0e+05,$r)'	=> 25.9757576090673165963,
        'gsl_sf_transport_5_e(1.0e-10,$r)'	=> 2.49999999999999999999e-41,
        'gsl_sf_transport_5_e(1.0e-07,$r)'	=> 2.49999999999999861111e-29,
        'gsl_sf_transport_5_e(1.0e-04,$r)'	=> 2.49999999861111111163e-17,
        'gsl_sf_transport_5_e(0.1,$r)'	=> 0.000024986116317791487410,
        'gsl_sf_transport_5_e(1.0,$r)'	=> 0.236615879239094789259153,
        'gsl_sf_transport_5_e(3.0,$r)'	=> 12.77055769104415951115760,
        'gsl_sf_transport_5_e(5.0,$r)'	=> 50.26309221817518778543615,
        'gsl_sf_transport_5_e(10.0,$r)'	=> 116.3807454024207107698556,
        'gsl_sf_transport_5_e(30.0,$r)'	=> 124.4313279083858954839911,
        'gsl_sf_transport_5_e(100.0,$r)'	=> 124.4313306172043911597639,
        'gsl_sf_transport_5_e(1.0e+05,$r)'	=> 124.43133061720439115976,
        'gsl_sf_sin(-10.0)'	=>       0.5440211108893698134,
        'gsl_sf_sin(1.0)'	=>         0.8414709848078965067,
        'gsl_sf_sin(1000.0)'	=>      0.8268795405320025603,
        'gsl_sf_sin(1048576.75)'	=>  0.8851545351115651914,
        'gsl_sf_sin(62831853.75)'	=> 0.6273955953485000827,
        'gsl_sf_sin(1073741822.5)'	=> -0.8284043541754465988,
        'gsl_sf_sin(1073741824.0)'	=> -0.6173264150460421708,
        'gsl_sf_sin(1073741825.5)'	=>  0.7410684679436226926,
        'gsl_sf_sin(1099511627776.0)'	=> -0.4057050115328287198,
        'gsl_sf_cos(-10.0)'	    => -0.8390715290764524523,
        'gsl_sf_cos(1.0)'	        =>  0.5403023058681397174,
        'gsl_sf_cos(1000.0)'	    =>  0.5623790762907029911,
        'gsl_sf_cos(1048576.75)'	=>  0.4652971620066351799,
        'gsl_sf_cos(62831853.75)'	=>  0.7787006914966116436,
        'gsl_sf_cos(1073741822.5)'	=> -0.5601305436977716102,
        'gsl_sf_cos(1073741824.0)'	=>  0.7867071229411881196,
        'gsl_sf_cos(1099511627776.0)'	=> -0.9140040719915570023,
        'gsl_sf_sinc_e(1.0/1024.0,$r)'	=> 0.9999984312693665404,
        'gsl_sf_sinc_e(1.0/2.0    ,$r)'	=> 2.0/$M_PI,
        'gsl_sf_sinc_e(80.5       ,$r)'	=> 0.0039541600768172754,
        'gsl_sf_sinc_e(100.5      ,$r)'	=> 0.0031672625490924445,
        'gsl_sf_sinc_e(1.0e+06 + 0.5,$r)'	=> 3.18309727028927157e-07,
        'gsl_sf_lnsinh_e(0.1,$r)'	=>  -2.3009189815304652235,
        'gsl_sf_lnsinh_e(1.0,$r)'	=>   0.16143936157119563361,
        'gsl_sf_lnsinh_e(5.0,$r)'	=>   4.306807418479684201,
        'gsl_sf_lnsinh_e(100.0,$r)'	=> 99.30685281944005469,
        'gsl_sf_lncosh_e(0.125,$r)'	=> 0.007792239318898252791,
        'gsl_sf_lncosh_e(1.0,$r)'	=>   0.4337808304830271870,
        'gsl_sf_lncosh_e(5.0,$r)'	=>   4.306898218339271555,
        'gsl_sf_lncosh_e(100.0,$r)'	=> 99.30685281944005469,
        'gsl_sf_angle_restrict_pos_err_e(2.0*$M_PI,$r)'	=> 2*$M_PI,
        'gsl_sf_angle_restrict_pos_err_e(-2.0*$M_PI,$r)'	=> 2*$DELTA,
        'gsl_sf_angle_restrict_pos_err_e(1e9,$r)'	=> 0.5773954235013851694,
        'gsl_sf_angle_restrict_pos_err_e(1e12,$r)'	=> 5.625560548042800009446,
        'gsl_sf_angle_restrict_pos_err_e(-1e9,$r)'	=> 5.7057898836782013075,
        'gsl_sf_angle_restrict_pos_err_e(-1e12,$r)'	=> 0.6576247591367864674792517289,
        'gsl_sf_angle_restrict_symm_err_e(1e9,$r)'	=> 0.5773954235013851694,
        'gsl_sf_angle_restrict_symm_err_e(1e12,$r)'	=> -0.6576247591367864674792517289,
        'gsl_sf_angle_restrict_symm_err_e(-1e9,$r)'	=> -0.5773954235013851694,
        'gsl_sf_angle_restrict_symm_err_e(-1e12,$r)'	=> 0.6576247591367864674792517289,
        'gsl_sf_zeta_int_e(-61.0,$r)'	=> -3.30660898765775767257e+34,
        'gsl_sf_zeta_int_e(-8,$r)'	=> 0.0,
        'gsl_sf_zeta_int_e(-6,$r)'	=> 0.0,
        'gsl_sf_zeta_int_e(-5.0,$r)'	=>  -0.003968253968253968253968,
        'gsl_sf_zeta_int_e(-4,$r)'	=> 0.0,
        'gsl_sf_zeta_int_e(-3,$r)'	=> 1.0/120.0,
        'gsl_sf_zeta_int_e(-2,$r)'	=> 0.0,
        'gsl_sf_zeta_int_e(-1,$r)'	=> -1.0/12.0,
        'gsl_sf_zeta_int_e( 5.0,$r)'	=> 1.0369277551433699263313655,
        'gsl_sf_zeta_int_e(31.0,$r)'	=> 1.0000000004656629065033784,
        'gsl_sf_zetam1_int_e(-61.0,$r)'	=> -3.30660898765775767257e+34,
        'gsl_sf_zetam1_int_e(-5.0,$r)'	=>  -1.003968253968253968253968,
        'gsl_sf_zetam1_int_e(-8,$r)'	=> -1.0,
        'gsl_sf_zetam1_int_e(-6,$r)'	=> -1.0,
        'gsl_sf_zetam1_int_e(-4,$r)'	=> -1.0,
        'gsl_sf_zetam1_int_e(-3,$r)'	=> -119.0/120.0,
        'gsl_sf_zetam1_int_e(-2,$r)'	=> -1.0,
        'gsl_sf_zetam1_int_e(-1,$r)'	=> -13.0/12.0,
        'gsl_sf_zetam1_int_e( 5.0,$r)'	=> 0.0369277551433699263313655,
        'gsl_sf_zetam1_int_e(31.0,$r)'	=> 0.0000000004656629065033784,
        'gsl_sf_zeta_e(-151,$r)'	=> 8.195215221831378294e+143,
        'gsl_sf_zeta_e(-51,$r)'	=> 9.68995788746359406565e+24,
        'gsl_sf_zeta_e(-5,$r)'	=> -0.003968253968253968253968,
        'gsl_sf_zeta_e(-8,$r)'	=> 0.0,
        'gsl_sf_zeta_e(-6,$r)'	=> 0.0,
        'gsl_sf_zeta_e(-4,$r)'	=> 0.0,
        'gsl_sf_zeta_e(-3,$r)'	=> 1.0/120.0,
        'gsl_sf_zeta_e(-2,$r)'	=> 0.0,
        'gsl_sf_zeta_e(-1,$r)'	=> -1.0/12.0,
        'gsl_sf_zeta_e(-0.5,$r)'	=> -0.207886224977354566017307,
        'gsl_sf_zeta_e(-1e-10,$r)'	=> -0.49999999990810614668948,
        'gsl_sf_zeta_e(0.0,$r)'	=>    -0.5,
        'gsl_sf_zeta_e(1e-10,$r)'	=>  -0.50000000009189385333058,
        'gsl_sf_zeta_e(0.5,$r)'	=> -1.460354508809586812889499,
        'gsl_sf_zeta_e(1.0-1.0/1024.0,$r)'	=> -1023.4228554489429787,
        'gsl_sf_zeta_e(1.0+1.0/1048576,$r)'	=> 1.0485765772157343441e+06,
        'gsl_sf_zeta_e(5.0,$r)'	=> 1.036927755143369926331365,
        'gsl_sf_zeta_e(25.5,$r)'	=> 1.000000021074106110269959,
        'gsl_sf_zetam1_e(-8,$r)'	=> -1.0,
        'gsl_sf_zetam1_e(-6,$r)'	=> -1.0,
        'gsl_sf_zetam1_e(-4,$r)'	=> -1.0,
        'gsl_sf_zetam1_e(-3,$r)'	=> -119.0/120.0,
        'gsl_sf_zetam1_e(-2,$r)'	=> -1.0,
        'gsl_sf_zetam1_e(-1,$r)'	=> -13.0/12.0,
        'gsl_sf_zetam1_e(-0.5,$r)'	=> -1.207886224977354566017307,
        'gsl_sf_zetam1_e(-1e-10,$r)'	=> -1.49999999990810614668948,
        'gsl_sf_zetam1_e(0.0,$r)'	=>    -1.5,
        'gsl_sf_zetam1_e(1e-10,$r)'	=>  -1.50000000009189385333058,
        'gsl_sf_zetam1_e(0.5,$r)'	=> -2.460354508809586812889499,
        'gsl_sf_zetam1_e(2.0,$r)'	=>  0.64493406684822643647,
        'gsl_sf_zetam1_e(3.0,$r)'	=>  0.20205690315959428540,
        'gsl_sf_zetam1_e(5.0,$r)'	=>  0.0369277551433699263314,
        'gsl_sf_zetam1_e(9.5,$r)'	=>  0.0014125906121736622712,
        'gsl_sf_zetam1_e(10.5,$r)'	=> 0.000700842641736155219500,
        'gsl_sf_zetam1_e(12.5,$r)'	=> 0.000173751733643178193390,
        'gsl_sf_zetam1_e(13.5,$r)'	=> 0.000086686727462338155188,
        'gsl_sf_zetam1_e(15.5,$r)'	=> 0.000021619904246069108133,
        'gsl_sf_zetam1_e(16.5,$r)'	=> 0.000010803124900178547671,
        'gsl_sf_zetam1_e(25.5,$r)'	=> 0.000000021074106110269959,
        'gsl_sf_hzeta_e(2,  1.0,$r)'	=>  1.6449340668482264365,
        'gsl_sf_hzeta_e(2, 10.0,$r)'	=>  0.1051663356816857461,
        'gsl_sf_hzeta_e(5,  1.0,$r)'	=>  1.0369277551433699263,
        'gsl_sf_hzeta_e(5, 10.0,$r)'	=>  0.000030413798676470276,
        'gsl_sf_hzeta_e(9,  0.1,$r)'	=>  1.0000000004253980e+09,
        'gsl_sf_hzeta_e(30, 0.5,$r)'	=>  1.0737418240000053e+09,
        'gsl_sf_hzeta_e(30, 0.9,$r)'	=>  2.3589824880264765e+01,
        'gsl_sf_hzeta_e(75, 0.25,$r)'	=> 1.4272476927059599e+45,
        'gsl_sf_eta_int_e(-91,$r)'	=> -4.945598888750002040e+94,
        'gsl_sf_eta_int_e(-51,$r)'	=> -4.363969073121683116e+40,
        'gsl_sf_eta_int_e(-5,$r)'	=> 0.25,
        'gsl_sf_eta_int_e(-1,$r)'	=> 0.25,
        'gsl_sf_eta_int_e( 0,$r)'	=> 0.5,
        'gsl_sf_eta_int_e( 5,$r)'	=> 0.9721197704469093059,
        'gsl_sf_eta_int_e( 6,$r)'	=> 0.9855510912974351041,
        'gsl_sf_eta_int_e( 20,$r)'	=> 0.9999990466115815221,
        'gsl_sf_eta_int_e( 1000,$r)'	=> 1.0,
        'gsl_sf_eta_e(-51.5, $r)'	=> -1.2524184036924703656e+41 ,
        'gsl_sf_eta_e(-5, $r)'	    => 0.25,
        'gsl_sf_eta_e(0.5, $r)'	    => 0.6048986434216303702,
        'gsl_sf_eta_e(0.999, $r)'	=> 0.6929872789683383574,
        'gsl_sf_eta_e(1.0, $r)'	    => 0.6931471805599453094,
        'gsl_sf_eta_e(1.0+1.0e-10, $r)'	=> 0.6931471805759321998,
        'gsl_sf_eta_e( 5, $r)'	=> 0.9721197704469093059,
        'gsl_sf_eta_e( 5.2, $r)'	=> 0.9755278712546684682,
        'gsl_sf_eta_e( 6, $r)'	=> 0.9855510912974351041,
        'gsl_sf_eta_e( 20,$r)'	=> 0.9999990466115815221, 
    };

    verify_results($results, 'Math::GSL::SF');

}
sub TEST_NAN_STRING : Tests(1)
{
    my $self = shift;
    my $results = {
        'gsl_sf_gamma_e(-1,$r)'                                 => $GSL_NAN,
    };
    verify_results($results, 'Math::GSL::SF');
}

sub TEST_NAN : Tests(18)
{
    my $results = {
        'gsl_sf_coupling_3j_e(-1, 1, 2, 1, -1, 0,$r)'	        => $GSL_NAN,
        'gsl_sf_coupling_3j_e(1, -1, 2, 1, -1, 0,$r)'	        => $GSL_NAN,
        'gsl_sf_coupling_3j_e(1, 1, -2, 1, -1, 0,$r)'	        => $GSL_NAN,
        'gsl_sf_coupling_6j_e(-2, 2, 4, 2, 2, 2,$r)'	        => $GSL_NAN,
        'gsl_sf_coupling_6j_e(2, -2, 4, 2, 2, 2,$r)'	        => $GSL_NAN,
        'gsl_sf_coupling_6j_e(2, 2, -4, 2, 2, 2,$r)'	        => $GSL_NAN,
        'gsl_sf_coupling_6j_e(2, 2, 4, -2, 2, 2,$r)'	        => $GSL_NAN,
        'gsl_sf_coupling_6j_e(2, 2, 4, 2, -2, 2,$r)'	        => $GSL_NAN,
        'gsl_sf_coupling_6j_e(2, 2, 4, 2, 2, -2,$r)'	        => $GSL_NAN,
        'gsl_sf_coupling_9j_e(-4, 2, 4, 3, 3, 2, 1, 1, 2,$r)'	=> $GSL_NAN,
        'gsl_sf_coupling_9j_e(4, -2, 4, 3, 3, 2, 1, 1, 2,$r)'	=> $GSL_NAN,
        'gsl_sf_coupling_9j_e(4, 2, -4, 3, 3, 2, 1, 1, 2,$r)'	=> $GSL_NAN,
        'gsl_sf_coupling_9j_e(4, 2, 4, -3, 3, 2, 1, 1, 2,$r)'	=> $GSL_NAN,
        'gsl_sf_coupling_9j_e(4, 2, 4, 3, -3, 2, 1, 1, 2,$r)'	=> $GSL_NAN,
        'gsl_sf_coupling_9j_e(4, 2, 4, 3, 3, -2, 1, 1, 2,$r)'	=> $GSL_NAN,
        'gsl_sf_coupling_9j_e(4, 2, 4, 3, 3, 2, -1, 1, 2,$r)'	=> $GSL_NAN,
        'gsl_sf_coupling_9j_e(4, 2, 4, 3, 3, 2, 1, -1, 2,$r)'	=> $GSL_NAN,
        'gsl_sf_coupling_9j_e(4, 2, 4, 3, 3, 2, 1, 1, -2,$r)'	=> $GSL_NAN,
    };
    verify_results($results, 'Math::GSL::SF');
}
sub TEST_ZZZ_OLD_BUGS : Test(1)
{
    my $r= Math::GSL::SF::gsl_sf_result_struct->new;
    ok_similar( [ gsl_sf_expint_En_e(3,300.0,$r) ] , [ 1.6608815083360041367294736e-133 ] );
}

sub TEST_J0_RESULT_STRUCT: Tests(2) {
    my $result = Math::GSL::SF::gsl_sf_result_struct->new;
    my ($status) = gsl_sf_bessel_J0_e(2.0,$result); 
    ok( defined $result->{err}, '$result->{err}' );
    ok( is_similar($result->{val}, gsl_sf_bessel_J0(2.0), $result->{err}) , '$result->{val}' );
}

Test::Class->runtests;
