#!/usr/bin/perl -w

# This script has been heavily modified from the Device::Cdio Build.PL 
# Jonathan Leto <jonathan@leto.net>
#    Copyright (C) 2006 Rocky Bernstein <rocky@cpan.org>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
use strict;
use warnings;
use Config;
use Data::Dumper;
use Module::Build;
use lib 'inc';
use GSLBuilder;

BEGIN {
    eval { require ExtUtils::PkgConfig };
    if ($@) {
        print "\nI see that you are a CPANTester, you really should install ExtUtils::PkgConfig !\n"
            if $ENV{AUTOMATED_TESTING};
        print <<EXIT;
ExtUtils::PkgConfig is currently needed to find GSL for the compilation of this module.
It may be bundled with Math::GSL in the future.
EXIT
        exit 0;
    }
}



sub try_compile {
    my ($c, %args) = @_;

    my $ok = 0;
    my $tmp = "tmp$$";
    local(*TMPC);

    my $obj_ext = $Config{_o} || ".o";
    unlink("$tmp.c", "$tmp$obj_ext");

    if (open(TMPC, ">", "$tmp.c")) {
	print TMPC $c;
	close(TMPC);

	my $cccmd = $args{cccmd};
	my $errornull;
	my $ccflags =  $Config{'ccflags'};
    $ccflags .= " $args{ccflags}" if $args{ccflags};

    if ($args{silent} ) {
	    $errornull = "2>/dev/null" unless defined $errornull;
	} else {
	    $errornull = '';
	}

    $cccmd = "$Config{'cc'} -o $tmp $ccflags $tmp.c $errornull"
	    unless defined $cccmd;

	printf "cccmd = $cccmd\n" if $args{verbose};
	my $res = system($cccmd);
	$ok = defined($res) && $res == 0;

	if ( !$ok ) {
	    my $errno = $? >> 8;
	    local $! = $errno;
	    print "
		
*** The test compile of '$tmp.c' failed: status $?
*** (the status means: errno = $errno or '$!')
*** DO NOT PANIC: this just means that *some* you may get some innocuous
*** compiler warnings.
";
	}
	unlink("$tmp.c");

    }
    return $ok;
}

sub try_cflags ($) {
    my ($ccflags) = @_;
    my $c_prog = "int main () { return 0; }\n";
    print "Checking if $Config{cc} supports \"$ccflags\"...";
    my $result = try_compile($c_prog, ccflags=>$ccflags);
    if ($result) {
	print "yes\n";
	return " $ccflags";
    }
    print "no\n";
    return '';
	
}

print "Checking for GSL..";
my %gsl_pkgcfg = ExtUtils::PkgConfig->find ('gsl');

my $MIN_GSL_VERSION = "1.8";
my $gv = $gsl_pkgcfg{'modversion'};
my $current_minor_version;

if (defined $gv) {
  if ($gv =~ m{\A(\d+(?:\.\d+)+)}) {
    my @current= split /\./, $1;
    my @min= split /\./, $MIN_GSL_VERSION;
    $current_minor_version = $current[1];
    unless ($current[0] >= $min[0] && $current[1] >= $min[1]) {
	    printf "
*** 
*** You need to have GSL %s or greater installed. (You have $gv).
*** Get GSL at http://www.gnu.org/software/gsl\n", $MIN_GSL_VERSION;
	    exit 1;
	} else {
	    print "Found GSL version $gv\n";
	    
	}
    } else {
	print "
*** 
*** Can't parse GSL version $gv.
*** Will continue and keep my fingers crossed for luck.
";
    }
} else {
    print "
*** 
*** Can't find GSL configuration info. Is GSL installed?
*** Get GSL at http://www.gnu.org/software/gsl
";
    exit 1;
}


my $ccflags = $gsl_pkgcfg{cflags};

## Swig produces a number of GCC warnings. Turn them off if we can.
$ccflags .= try_cflags("-Wno-strict-aliasing");
$ccflags .= try_cflags("-Wno-unused-function");
$ccflags .= try_cflags("-Wno-unused-value");
$ccflags .= try_cflags("-Wno-unused-function");
$ccflags .= try_cflags("-Wno-unused-variable");

my $ldflags = "$gsl_pkgcfg{libs} -gsl";
my $swig_flags= "-Wall $gsl_pkgcfg{cflags}";

if ('cygwin' eq $Config{osname} && 
    $Config{shrpenv} =~ m{\Aenv LD_RUN_PATH=(.*)\Z} ) {
    $ldflags .= " -L$1 -lperl";
    # Should we check the 32-ness?
    $swig_flags = '-DNEED_LONG';
} elsif ('darwin' eq $Config{osname}) {
    $ldflags .= " -bundle -flat_namespace";
}
my @Subsystems = sort qw/
            BLAS         Diff          Machine    Statistics
            Eigen        Matrix        Poly       Wavelet2D
            BSpline      Errno         PowInt     Wavelet
            CBLAS        FFT           Min          
            CDF          Fit           QRNG
            Chebyshev    Monte         RNG
            Heapsort     Multifit      Randist      Sum
            Combination  Histogram     Multimin     Roots      
            Complex      Histogram2D   Multiroots   SF         
            Const        Siman         IEEEUtils    Sys
            Integration  NTuple        Sort                  
            DHT          Interp        ODEIV        Vector
            Deriv        Linalg        Permutation  Spline
/;

# BSplines appeared in 1.9
if ($current_minor_version < 9 ) {
   @Subsystems = grep { ! /BSpline/ } @Subsystems;
}
my $cleanup = qq{core *.core Makefile Math-GSL-* tmp* pod2ht*.tmp _build blib *.so *.orig };

my $builder = GSLBuilder->new( 
    module_name         => 'Math::GSL',
    add_to_cleanup      => [ $cleanup ],
    create_makefile_pl  => 'passthrough',
    dist_abstract       => 'Interface to the GNU Scientific Library using SWIG',
    dist_author         => 'Jonathan Leto <jonathan@leto.net>',
    dist_version_from   => 'lib/Math/GSL.pm',
    include_dirs	    => q{},
    extra_linker_flags  => '-shared ' . $ldflags,
    extra_compiler_flags=> $ccflags,
    # Set to true for CPAN releases to prevent breakage
    swig_disabled       => 1,
    swig_flags          => $swig_flags, 
    license             => 'gpl',
    requires => {
        'ExtUtils::PkgConfig'   =>  '1.03',
        'Scalar::Util'          =>       0,
        'Test::More'            =>       0,
        'Test::Exception'       =>    0.21,
        'Test::Class'           =>    0.12,
        version                 =>       0,
    	perl 			        =>  '5.8.0',
    },
    sign                  => 0,
    swig_source           => [  
                                map { [ "$_.i" ] } @Subsystems
                             ],
);
$builder->add_build_element('swig');
$builder->create_build_script();
print "Have a great day!\n";
