use Dist::Iller::Standard;

our $VERSION = '0.0100'; # VERSION:
# PODNAME: Dist::Iller::Config::Author::CSSON
# ABSTRACT: Dist::Iller config for Csson

class Dist::Iller::Config::Author::CSSON using Moose with Dist::Iller::Role::Config {

    use Path::Tiny;

    has filepath => (
        is => 'ro',
        isa => Path,
        default => 'author-csson.yaml',
        coerce => 1,
    );
    has is_task => (
        is => 'ro',
        isa => Bool,
        default => 0,
    );
    has installer => (
        is => 'rw',
        isa => Str,
        lazy => 1,
        default => 'MakeMaker',
    );
    has is_private => (
        is => 'rw',
        isa => Int,
        lazy => 1,
        default => 0,
    );
    has homepage => (
        is => 'rw',
        isa => Str,
        lazy => 1,
        builder => 1,
    );
    has splint => (
        is => 'rw',
        isa => Int,
        default => 0,
    );

    method _build_homepage {
        return sprintf 'https://metacpan.org/release/' . $self->distribution_name;
    }

    method build_file {
        return $self->installer =~ m/MakeMaker/ ? 'Makefile.PL' : 'Build.PL';
    }

    method is_private_release {
        return !$ENV{'FAKE_RELEASE'} && $self->is_private ? 1 : 0;
    }
    method is_cpan_release {
        return !$ENV{'FAKE_RELEASE'} && $self->is_private ? 0 : 1;
    }
    method add_default_github {
        # check git config
        my $add_default_github = 0;
        my $git_config = path('.git/config');
        if($git_config->exists) {
            my $git_config_contents = $git_config->slurp_utf8;
            if($git_config_contents =~ m{github\.com:([^/]+)/(.+)\.git}) {
                $add_default_github = 1;
            }
            else {
                say ('[DI/@Author::CSSON] No github url found');
            }
        }
        return $add_default_github;
    }
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Dist::Iller::Config::Author::CSSON - Dist::Iller config for Csson

=head1 VERSION

Version 0.0100, released 2015-11-24.

=head1 SYNOPSIS

    # in iller.yaml
    + config: Author::CSSON

=head1 DESCRIPTION

Dist::Iller::Config::Author::Csson is a L<Dist::Iller> configuration. The plugin list is in C<share/author-csson.yaml>.

=head1 SOURCE

L<https://github.com/Csson/p5-Dist-Iller-Config-Author-CSSON>

=head1 HOMEPAGE

L<http://metacpan.org/release/Dist-Iller-Config-Author-CSSON>

=head1 AUTHOR

Erik Carlsson <info@code301.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Erik Carlsson.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
