use 5.008;

use strict;
use warnings;

use ExtUtils::MakeMaker;
use Config;

my $libdirs = $ENV{JS_LIBS} || '';
my $incdirs = $ENV{JS_INC} || '';
my $defines = $ENV{JS_DEFINES} || '';
my $CC = $Config{cc};
my $myextlib = '';
my $needtest = 1;

if($^O eq 'MSWin32') {
    # Search in PATH for xulrunner SDK
    my $xulsdk = $ENV{XUL_PATH};
    unless($xulsdk) {
	my @dirs = split ';', $ENV{PATH};
	for(@dirs) {
	    if(-x "$_/xulrunner.exe" && -e "$_/../include/jsapi.h") {
		$xulsdk = $_;
		last;
	    }
	}
	unless($xulsdk) {
	    warn "Can't found your XulRunner SDK directory in PATH\n";
	    exit 0;
	}
	$xulsdk =~ s/bin$//;
    }
    warn "XulRunner SDK found in '$xulsdk'\n";

    my($dll) = glob "$xulsdk\\bin\\js*.dll";
    my($libname) = $dll =~ /(js\d+)\.dll/;
    $libdirs = "-L${xulsdk}lib -l$libname -lnspr4";
    $incdirs = "-I${xulsdk}include";
    $defines = "-DXP_WIN=1 -D_CRT_SECURE_NO_WARNINGS -TP";
    my $mozver;
    {
	open(my $fd, "<${xulsdk}include/mozilla-config.h");
	while(my $line = <$fd>) {
	    chomp $line;
	    if($line =~ /#define MOZILLA_VERSION "(.+)"/) {
		$mozver = $1; last
	    }
	}
    }
    die "Can't found my MOZILLA_VERSION\n" if(!$mozver);
    my $JSOP = $mozver gt '1.9.1' ? '#define' : '#undef';
    $JSOP .= " JS_HAS_JSOP_TRACE";
    open(my $fd, ">JS_Env.h") or die "Can't create JS_Env.h: $!\n";
    print $fd <<EOF;
/* Autogenerated, don't edit */
#define JS_THREADSAFE	1
$JSOP
EOF

    warn "Will use\n\tJS_LIBS=$libdirs\n\tJS_INC=$incdirs\n\tJS_DEFINES=$defines\n";
    $needtest=0;
}

if((my $src = $ENV{JS_SRC}) && $CC eq 'gcc') {
    # For developers only.
    # SM is uninstalled, so uses the static library.
    $incdirs = "-I$ENV{JS_SRC} -I$ENV{JS_SRC}/..";
    if(-x "$src/js-config") { # A recent SM
	my $cflags = qx($src/js-config --cflags);
	chomp $cflags;
	$incdirs .= ' ' . (split ' ', $cflags)[1];
	$libdirs = qx($src/js-config --libs);
	chomp $libdirs;
	my @libs = split(' ', $libdirs);
	$libdirs = join(' ', @libs[2 .. $#libs]); 
    } else {
	$libdirs = "-lm -lnspr4";
    }
    $defines = '-Wno-unused-variable -DXP_UNIX';
    ($myextlib) = glob "$ENV{JS_SRC}/libjs*.a";
    my($vhead) = grep -f $_, glob "$ENV{JS_SRC}/../js{version,config}.h";

    die "Your SpiderMonkey doesn't seems to be compiled yet.\nI was looking for $myextlib\n"
	unless(-f $myextlib);

    my $ver = (split ' ',`grep "#define JS_VERSION " $vhead`)[2];
    if($ver >= 185) {
	$CC = 'c++';
    }

    warn "Found SpiderMonkey $ver (for static build)\n";
    warn "Will use\n\tJS_LIBS=$libdirs\n\tJS_INC=$incdirs\n\tJS_DEFINES=$defines\n\tEXTLIB=$myextlib\n";
}

my $pkgn = '';
unless($libdirs && $incdirs) {
PKGCONF: {
    # In the common case we depend on a pkg-config file.
    #warn "Will try with pkg-config files.\n";
    my @build_reqs = (
	'mozilla-js',
	'libjs'
    );
    unshift @build_reqs, $ARGV[0] if $ARGV[0];

    for(@build_reqs) {
	#warn "Testing for $_...\n";
	my $res = system 'pkg-config', '--exists',  "$_ >= 1.7";
	if($res == -1) {
	    warn "Can't run pkg-config\n";
	    last PKGCONF;
	}
	if(!$res) {
	    $pkgn = $_;
	    last;
	}
    }

    unless($pkgn) {
	warn "Sad, I can't find any pkg-config file for SpiderMonkey\n";
	last;
    }
    my $vers = `pkg-config --modversion $pkgn`;
    chomp $vers;
    warn "Found $pkgn ($vers)\n";
    $libdirs = `pkg-config --libs $pkgn`;
    chomp $libdirs;
    $defines = `pkg-config --cflags-only-other $pkgn`;
    chomp $defines;
    $incdirs = `pkg-config --cflags-only-I $pkgn`;
    chomp $incdirs;

    $defines .= ' -DXP_UNIX' unless $defines =~ /-DXP/; # Just in case.

    warn qq{Will use\n\tJS_LIBS="$libdirs"\n\tJS_INC="$incdirs"\n\tJS_DEFINES="$defines"\n};
}}

unless($incdirs && $libdirs) {
    warn "Will try a brute-force search, review the results...\n";
    my @lpaths = qw(/usr/lib64 /usr/lib /usr/local/lib64 /usr/local/lib /opt/local/lib);
    my @ipaths = qw(
	/usr/include/mozjs /usr/include/smjs /usr/include/js
        /usr/local/include/mozjs /usr/local/include/smjs /usr/local/include/js
        /usr/include /usr/local/include /opt/local/include/js
    );
    my(@incs, @libs, $lib);
    unless($incdirs) {
	for my $p (@ipaths) {
	    if(-f("$p/jsapi.js") && -f("$p/jsdbgapi.h")) {
		push @incs, "-I$p";
	    }
	    if(-f "$p/nspr/pratom.h") {
		push @incs, "-I$p/nspr";
	    } elsif(-f "$p/nspr4/pratom.h") {
		push @incs, "-I$p/nspr4";
	    }
	}
	$incdirs = join(' ', @incs);
    }
    ALL:
    for my $path (@lpaths) {
	if(!$lib) {
	    SEARCHLIB:
	    for my $l ( qw(mozjs smjs js) ) {
		if(-f "$path/lib$l.so") {
		    push @libs, "-L$path -l$l";
		    last SEARCHLIB;
		}
	    }
	}
        if(-f "$path/libnspr.so") {
	    push @libs, "-L$path -lnspr";
	} elsif( -f "$path/libnspr4.so") {
	    push @libs, "-L$path -lnspr4";
	}
    }
    $libdirs = join(' ', @libs);
    $defines = "-DXP_UNIX";

    warn qq{Will use\n\tJS_LIBS="$libdirs"\n\tJS_INC="$incdirs"\n\tJS_DEFINES="$defines"\n};
}

unless($incdirs && $libdirs) {
    warn <<EOF;
SpiderMonkey headers and libraries not found.

Try manually defining JS_LIBS, JS_INC and JS_DEFINES environment variables like:
export JS_LIBS="-L/path/to/spidermonkey/library -l<libname>"
export JS_INC="-I/path/to/spidermonkey/include"
export JS_DEFINE="-DXP_UNIX"
EOF
    exit 0;
}

# Compile a test to determine if we can find libs and headers and
# to generate the dynamic header file
if($needtest) {
    require File::Temp;
    my $exe = File::Temp::tmpnam();
    my $cc = join(" ", $CC, $defines, $incdirs, $libdirs, "-ldl -o", $exe);
    #warn "$cc testjs.c\n";
    qx($cc utils/testjs.c $myextlib);
    if($?) {
	warn "Test compile failed, check JS_INC, JS_LIBS and JS_DEFINE for errors.\n";
	warn "Your package-config file '$pkgn.pc' can be broken!\n" if $pkgn;
	exit 1; # I want smokers reports
    }

    my $res = qx($exe);
    if(!$res) {
	warn "Unusable library, please make sure it's available in your LD_LIBRARY_PATH.\n";
	exit 1; # I want smokers reports
    }
    open(my $fd, ">JS_Env.h") or die "Can't create JS_Env.h: $!\n";
    print $fd "/* Autogenerated, don't edit */\n$res\n";
    unlink($exe);
}

# Write makefile
WriteMakefile(
    NAME            => "JSP",
    VERSION_FROM    => "lib/JSP.pm",
    PREREQ_PM       => {
        "Test::More"      => 0,
        "Test::Exception" => 0,
    },
    ABSTRACT_FROM   => "lib/JSP.pm", # retrieve abstract from module
    AUTHOR          => "Salvador Ortiz <sortiz\@cpan.org>",
    DEFINE          => "$defines",
    LIBS            => "$libdirs",
    INC             => $incdirs,
    LICENSE         => "perl",
    EXE_FILES	    => [ 'bin/jsp' ],
    OBJECT          => q/$(O_FILES)/,
    CC              => $CC,
    DL_FUNCS	    => {
	JSP => [], 'JSP::RawObj' => [],
	'JSP::PerlClass' => [], 'JSP::TrapHandler' => [], 
    },
    MYEXTLIB        => $myextlib,
    ($CC eq 'c++' ? (XSOPT => '-C++', LD => $CC) : ()),
    clean => { FILES => 'JS_Env.h' },
);

package MY;

use File::Spec;

sub post_initialize {
    my($self) = shift;

    my @headers = <*.h>;
    for (@headers, 'typemap' ) {
        $self->{PM}->{$_} = File::Spec->catfile($self->{INST_ARCHAUTODIR}, $_);
    }
    return '';
}
