use strict;
use warnings;

BEGIN {
    if ($ENV{'PERL_CORE'}){
        chdir 't';
        unshift @INC, '../lib';
    }
    use Config;
    if (! $Config{'useithreads'}) {
        print("1..0 # SKIP Perl not compiled with 'useithreads'\n");
        exit(0);
    }
}

use ExtUtils::testlib;

sub ok {
    my ($id, $ok, $name) = @_;

    # You have to do it this way or VMS will get confused.
    if ($ok) {
        print("ok $id - $name\n");
    } else {
        print("not ok $id - $name\n");
        printf("# Failed test at line %d\n", (caller)[2]);
    }

    return ($ok);
}

BEGIN {
    $| = 1;
    print("1..28\n");   ### Number of tests that will be run ###
};

my $test = 1;

use threads;
use threads::shared;
ok($test++, 1, 'Loaded');

### Start of Testing ###

{
    # Scalar
    my $x = shared_clone(14);
    ok($test++, $x == 14, 'number');

    $x = shared_clone('test');
    ok($test++, $x eq 'test', 'string');
}

{
    my %hsh = ('foo' => 2);
    eval {
        my $x = shared_clone(%hsh);
    };
    ok($test++, $@ =~ /Usage:/, '1 arg');

    threads->create(sub {})->join();  # Hide leaks, etc.
}

{
    my $x = 'test';
    my $foo :shared = shared_clone($x);
    ok($test++, $foo eq 'test', 'cloned string');

    $foo = shared_clone(\$x);
    ok($test++, $$foo eq 'test', 'cloned scalar ref');

    threads->create(sub {
        ok($test++, $$foo eq 'test', 'cloned scalar ref in thread');
    })->join();

    $test++;
}

{
    my $foo :shared;
    $foo = shared_clone(\$foo);
    ok($test++, ref($foo) eq 'REF', 'Circular ref typ');
    ok($test++, threads::shared::_id($foo) == threads::shared::_id($$foo), 'Circular ref');

    threads->create(sub {
        ok($test++, threads::shared::_id($foo) == threads::shared::_id($$foo), 'Circular ref in thread');

        my ($x, $y, $z);
        $x = \$y; $y = \$z; $z = \$x;
        $foo = shared_clone($x);
    })->join();

    $test++;

    ok($test++, threads::shared::_id($$foo) == threads::shared::_id($$$$$foo),
                    'Cloned circular refs from thread');
}

{
    my @ary = (qw/foo bar baz/);
    my $ary = shared_clone(\@ary);

    ok($test++, $ary->[1] eq 'bar', 'Cloned array');
    $ary->[1] = 99;
    ok($test++, $ary->[1] == 99, 'Clone mod');
    ok($test++, $ary[1] eq 'bar', 'Original array');

    threads->create(sub {
        ok($test++, $ary->[1] == 99, 'Clone mod in thread');

        $ary[1] = 'bork';
        $ary->[1] = 'thread';
    })->join();

    $test++;

    ok($test++, $ary->[1] eq 'thread', 'Clone mod from thread');
    ok($test++, $ary[1] eq 'bar', 'Original array');
}

{
    my $scalar = 'zip';

    my $obj = {
        'ary' => [ 1, 'foo', [ 86 ], { 'bar' => [ 'baz' ] } ],
        'ref' => \$scalar,
    };

    $obj->{'self'} = $obj;

    bless($obj, 'Foo');

    my $copy :shared;

    threads->create(sub {
        $copy = shared_clone($obj);

        ok($test++, ${$copy->{'ref'}} eq 'zip', 'Obj ref in thread');
        ok($test++, threads::shared::_id($copy) == threads::shared::_id($copy->{'self'}), 'Circular ref in cloned obj');
        ok($test++, is_shared($copy->{'ary'}->[2]), 'Shared element in cloned obj');
    })->join();

    $test += 3;

    ok($test++, ref($copy) eq 'Foo', 'Obj cloned by thread');
    ok($test++, ${$copy->{'ref'}} eq 'zip', 'Obj ref in thread');
    ok($test++, threads::shared::_id($copy) == threads::shared::_id($copy->{'self'}), 'Circular ref in cloned obj');
    ok($test++, $copy->{'ary'}->[3]->{'bar'}->[0] eq 'baz', 'Deeply cloned');
    ok($test++, ref($copy) eq 'Foo', 'Cloned object class');
}

{
    my $hsh :shared = shared_clone({'foo' => [qw/foo bar baz/]});
    ok($test++, is_shared($hsh), 'Shared hash ref');
    ok($test++, is_shared($hsh->{'foo'}), 'Shared hash ref elem');
    ok($test++, $$hsh{'foo'}[1] eq 'bar', 'Cloned structure');
}

exit(0);

# EOF
