package blx::xsdsql::generator;
use strict;
use warnings;
use Carp;
use blx::xsdsql::ut qw(nvl ev);

use constant {
	STREAM_CLASS => 'blx::xsdsql::OStream'
};


sub _check_table_filter {
	my ($self,$table,$level,%params)=@_;
	if (defined $self->{LEVEL_FILTER}) {
		return 0  if $level != $self->{LEVEL_FILTER};
	}
	if  (defined $self->{TABLES_FILTER}) {
		return 1 if $self->{TABLES_FILTER}->{uc($table->get_sql_name)};
		my $path=$table->get_attrs_value qw(PATH);
		return 0 unless defined $path;
		return 1 if $self->{TABLES_FILTER}->{$path};
		return 0;
	}
	return 1;
}

sub _parser {
	my ($self,$table,%params)=@_;
	my $handle=$self->{HANDLE};
	if ($self->_check_table_filter($table,$params{LEVEL})) {
		$handle->table_header($table,%params);
		for my $col($table->get_columns) {
			$handle->column($col,%params,TABLE => $table);
		}
		$handle->table_footer($table,%params);
	}
	for my $t($table->get_child_tables) {
		$self->_parser($t,%params,LEVEL => $params{LEVEL} + 1);
	}
	if (nvl($table->get_attrs_value qw(PATH)) eq '/') {
		for my $t($table->get_attrs_value qw(TYPES)) {
			$self->_parser($t,%params,LEVEL => $params{LEVEL} + 1);
		}
		$handle->footer($table,%params);
	}
	return undef; 
}


sub generate {
	my ($self,%params)=@_;
	my $table=nvl(delete $params{ROOT_TABLE},$self->{ROOT_TABLE});
	croak "ROOT_TABLE param not set " unless defined $table;
	my $command=nvl(delete $params{COMMAND},$self->{COMMAND});
	croak "COMMAND param not set" unless defined $command;
	my $handle_class='blx::xsdsql::generator::'.$self->{OUTPUT_NAMESPACE}.'::'.$self->{DB_NAMESPACE}.'::handle::'.$command;
	croak 'LEVEL_FILTER param not valid - must \d{1,11} ' unless nvl($params{LEVEL_FILTER},0)=~/^\d{1,11}$/;  
	if (defined $params{TABLES_FILTER}) {
		$params{TABLES_FILTER}=[ $params{TABLES_FILTER} ] if ref($params{TABLES_FILTER}) eq '';
		croak "TABLES_FILTER param type not valid  - must be an array of scalar or a scalar not null" 
			unless ref($params{TABLES_FILTER}) eq 'ARRAY';
		for my $e(@{$params{TABLES_FILTER}}) {
			croak "TABLES_FILTER param type not valid  - must be an array of scalar or a scalar not null" 
				unless defined $e && ref($e) eq '';
		}
		$params{TABLES_FILTER}={ map { (uc($_),1); } @{$params{TABLES_FILTER}} }; #transform into hash
	}
	unless (defined $self->{HANDLE_OBJECTS}->{$handle_class}) {
		ev('use',$handle_class);
		$self->{HANDLE_OBJECTS}->{$handle_class}=$handle_class->new(%params);
	}

	$self->{HANDLE}=$self->{HANDLE_OBJECTS}->{$handle_class};
	$self->{HANDLE}->header($table,%params) unless $params{NO_HEADER_COMMENT};
	for my $p qw( LEVEL_FILTER TABLES_FILTER) {
		$self->{$p}=delete $params{$p};
	}
	$self->_parser($table,%params,LEVEL => 0,TABLENAME_LIST => []);
	return $self;
}

sub new {
	my ($class,%params)=@_;
	$params{OUTPUT_NAMESPACE}='sql' unless defined $params{OUTPUT_NAMESPACE};
	croak "no param DB_NAMESPACE spec" unless defined $params{DB_NAMESPACE};
	my $fd=nvl(delete $params{FD},*STDOUT);
	my $self=bless \%params,$class;

	if (ref($fd) ne STREAM_CLASS) {
		if (ref($self->{OUTPUT_STREAM}) eq STREAM_CLASS) {
			$self->{OUTPUT_STREAM}->set_output_descriptor($fd);
		}
		else {
		  ev('use ',STREAM_CLASS);
			$self->{OUTPUT_STREAM}=STREAM_CLASS->new(OUTPUT_STREAM => $fd);
		}
	}
	else {
		$self->{OUTPUT_STREAM}=$fd;
	}

	return $self;
}

sub get_namespaces {
	my @n=();
	for my $i(@INC) {
		my $dirgen=File::Spec->catdir($i,'blx','xsdsql','generator');
		next unless  -d "$dirgen";
		next if $dirgen=~/^\./;
		next unless opendir(my $fd,$dirgen);
		while(my $d=readdir($fd)) {
			my $dirout=File::Spec->catdir($dirgen,$d);
			next unless -d $dirout;
			next if $d=~/^\./;
			next unless opendir(my $fd1,$dirout);
			while(my $d1=readdir($fd1)) {
				my $dirout=File::Spec->catdir($dirgen,$d,$d1);
				next unless -d $dirout;
				next if $d1=~/^\./;
				push @n,$d.'::'.$d1;
			}
			closedir $fd1;
		}
		closedir($fd);
	}
	return wantarray ? @n : \@n;
}

1;

__END__

=head1 NAME

blx::xsdsql::generator  -  generate the files for create table ,drop table ,add primary key 

=head1 SYNOPSIS

use blx::xsdsql::generator


=head1 DESCRIPTION

this package is a class - instance it with the method new

=cut


=head1 FUNCTIONS


new - constructor

	PARAMS:
		ROOT_TABLE - tree of objects generated by blx::xsdsql::parser::parse
		OUTPUT_NAMESPACE => default sql
		DB_NAMESPACE     => default <none>
		FD  => streamer class, file descriptor  , array or string  (default *STDOUT)



generate - generate a file

	PARAMS:
		ROOT_TABLE   => <root_object> - tree of objects generated by blx::xsdsql::parser::parse
		COMMAND      => create_table|drop_table|addpk|drop_sequence|create_sequence 
		LEVEL_FILTER  => <n> -  produce code only for tables at level <n> (n >= 0) - root has level 0  (default none)
		TABLES_FILTER  => [<name>] - produce code only for tables in  <name> - <name> is a table_name or a xml_path 


	the method return a self to object



get_namespaces  - static method  
	
	the method return an array of namespace founded 




=head1 EXPORT

None by default.


=head1 EXPORT_OK

None

=head1 SEE ALSO


See blx::xsdsql::parser  for parse a xsd file (schema file) and blx::xsdsql::xml for read/write a xml file into/from a database

=head1 AUTHOR

lorenzo.bellotti, E<lt>bellzerozerouno@tiscali.itE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2010 by lorenzo.bellotti

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

See http://www.perl.com/perl/misc/Artistic.html

=cut
 

