=head1 NAME

Sys::Statistics::Linux::SockStats - Collect linux socket statistics.

=head1 SYNOPSIS

   use Sys::Statistics::Linux::SockStats;

   my $lxs   = new Sys::Statistics::Linux::SockStats;
   my $stats = $lxs->get;

=head1 DESCRIPTION

This module collects socket statistics from the F</proc> filesystem. It is tested on x86 hardware
with the distributions SuSE (SuSE on s390 and s390x architecture as well), Red Hat, Debian
and Mandrake on kernel versions 2.4 and 2.6 but should also running on other linux distributions
with the same kernel release number. To run this module it is necessary to start it as root or
another user with the authorization to read the F</proc> filesystem.

=head1 SOCKET STATISTICS

Generated by F</proc/net/sockstat>.

   used    -  Total number of used sockets.
   tcp     -  Number of tcp sockets in use.
   udp     -  Number of udp sockets in use.
   raw     -  Number of raw sockets in use.
   ipfrag  -  Number of ip fragments in use.

=head1 METHODS

=head2 All methods

   C<new()>
   C<init()>
   C<get()>

=head2 new()

Call C<new()> to create a new object.

   my $lxs = new Sys::Statistics::Linux::CpuStats;

=head2 get()

Call C<get()> to get the statistics. C<get()> returns the statistics as a hash reference.

   my $stats = $lxs->get;

=head1 EXPORTS

No exports.

=head1 SEE ALSO

B<proc(5)>

=head1 REPORTING BUGS

Please report all bugs to <jschulz.cpan(at)bloonix.de>.

=head1 AUTHOR

Jonny Schulz <jschulz.cpan(at)bloonix.de>.

=head1 COPYRIGHT

Copyright (c) 2006, 2007 by Jonny Schulz. All rights reserved.

This program is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=cut

package Sys::Statistics::Linux::SockStats;
our $VERSION = '0.01';

use strict;
use warnings;
use IO::File;
use Carp qw(croak);

sub new {
   return bless {
      files => {
         sockstats  => '/proc/net/sockstat',
      },
      stats => {},
   }, shift;
}

sub get {
   my $self = shift;
   $self->{stats} = $self->_load;
   return $self->{stats};
}

#
# private stuff
#

sub _load {
   my $self  = shift;
   my $class = ref $self;
   my $file  = $self->{files};
   my $fh    = new IO::File;
   my %socks;

   $fh->open($file->{sockstats}, 'r') or croak "$class: unable to open $file->{sockstats} ($!)";

   while (my $line = <$fh>) {
      if ($line =~ /sockets: used (\d+)/) {
         $socks{used} = $1;
      } elsif ($line =~ /TCP: inuse (\d+)/) {
         $socks{tcp} = $1;
      } elsif ($line =~ /UDP: inuse (\d+)/) {
         $socks{udp} = $1;
      } elsif ($line =~ /RAW: inuse (\d+)/) {
         $socks{raw} = $1;
      } elsif ($line =~ /FRAG: inuse (\d+)/) {
         $socks{ipfrag} = $1;
      }
   }

   $fh->close;
   return \%socks;
}

1;
