=head1 NAME

Sys::Statistics::Linux::FileStats - Collect linux file statistics.

=head1 SYNOPSIS

   use Sys::Statistics::Linux::FileStats;

   my $lxs   = new Sys::Statistics::Linux::FileStats;
   my $stats = $lxs->get;

=head1 DESCRIPTION

This module collects file statistics from the F</proc> filesystem. It is tested on x86 hardware
with the distributions SuSE (SuSE on s390 and s390x architecture as well), Red Hat, Debian
and Mandrake on kernel versions 2.4 and 2.6 but should also running on other linux distributions
with the same kernel release number. To run this module it is necessary to start it as root or
another user with the authorization to read the F</proc> filesystem.

=head1 FILE STATISTICS

Generated by F</proc/sys/fs/file-nr>, F</proc/sys/fs/inode-nr> and F</proc/sys/fs/dentry-state>.

   fhalloc    -  Number of allocated file handles.
   fhfree     -  Number of free file handles.
   fhmax      -  Number of maximum file handles.
   inalloc    -  Number of allocated inodes.
   infree     -  Number of free inodes.
   inmax      -  Number of maximum inodes.
   dentries   -  Dirty directory cache entries.
   unused     -  Free diretory cache size.
   agelimit   -  Time in seconds the dirty cache entries can be reclaimed.
   wantpages  -  Pages that are requested by the system when memory is short.

=head1 METHODS

=head2 All methods

   C<new()>
   C<get()>

=head2 new()

Call C<new()> to create a new object.

   my $lxs = new Sys::Statistics::Linux::CpuStats;

=head2 get()

Call C<get()> to get the statistics. C<get()> returns the statistics as a hash reference.

   my $stats = $lxs->get;

=head1 EXPORTS

No exports.

=head1 SEE ALSO

B<proc(5)>

=head1 REPORTING BUGS

Please report all bugs to <jschulz.cpan(at)bloonix.de>.

=head1 AUTHOR

Jonny Schulz <jschulz.cpan(at)bloonix.de>.

=head1 COPYRIGHT

Copyright (c) 2006, 2007 by Jonny Schulz. All rights reserved.

This program is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=cut

package Sys::Statistics::Linux::FileStats;
our $VERSION = '0.01';

use strict;
use warnings;
use IO::File;
use Carp qw(croak);

sub new {
   return bless {
      files => {
         file_nr    => '/proc/sys/fs/file-nr',
         inode_nr   => '/proc/sys/fs/inode-nr',
         dentries   => '/proc/sys/fs/dentry-state',
      },
      stats => {},
   }, shift;
}

sub get {
   my $self  = shift;
   $self->{stats} = $self->_load;
   return $self->{stats};
}

#
# private stuff
#

sub _load {
   my $self  = shift;
   my $class = ref $self;
   my $file  = $self->{files};
   my $fh    = new IO::File;
   my %fstats;

   $fh->open($file->{file_nr}, 'r') or croak "$class: unable to open $file->{file_nr} ($!)";
   @fstats{qw(fhalloc fhfree fhmax)} = (split /\s+/, <$fh>)[0..2];
   $fh->close;

   $fh->open($file->{inode_nr}, 'r') or croak "$class: unable to open $file->{inode_nr} ($!)";
   @fstats{qw(inalloc infree)} = (split /\s+/, <$fh>)[0..1];
   $fstats{inmax} = $fstats{inalloc} + $fstats{infree};
   $fh->close;

   $fh->open($file->{dentries}, 'r') or croak "$class: unable to open $file->{dentries} ($!)";
   @fstats{qw(dentries unused agelimit wantpages)} = (split /\s+/, <$fh>)[0..3];
   $fh->close;

   return \%fstats;
}

1;
