#!/usr/bin/perl

# $Id: session_cleanup.pl,v 1.2 2003/12/01 14:22:21 lachoy Exp $

# session_cleanup.pl

use strict;
use OpenInteract::Startup;
use Storable qw( thaw );
use MIME::Base64;

my $TEMP_TRACK_FILE = 'tmp_session_id';

{
    $| = 1;
    my $usage = "Usage: $0 --website_dir=/path/to/my_site days-old";

    my ( $OPT_analyze, $OPT_debug, $OPT_help, $OPT_decode, $OPT_force );
    my $R = OpenInteract::Startup->setup_static_environment_options(
                    $usage,
                    { 'analyze'  => \$OPT_analyze,
                      'debug'    => \$OPT_debug,
                      'decode=s' => \$OPT_decode,
                      'help'     => \$OPT_help,
                      'force'    => \$OPT_force },
                    { temp_lib => 'lazy' } );
    if ( $OPT_help ) { 
        print usage(); exit();
    }

    if ( $OPT_analyze ) {
        print "Analysis mode on; no changes will be made\n";
    }

    $OPT_decode ||= 'base64';

    my ( $days_old ) = @ARGV;
    $days_old ||= 60;
    my ( $removed, $kept, $count );

    my $initial_time = time;

    my $db = $R->db;

    # Grab all the session_ids and print them to a file, one per line,
    # so we only have to keep one handle open

    my ( $id );
    my $sql = qq/ SELECT id FROM sessions /;
    my $sth = $db->prepare( $sql );
    $sth->execute;

    open( IDLIST, "+> $TEMP_TRACK_FILE" );
    $sth->bind_col( 1, \$id );
    while ( $sth->fetch ) {
        print IDLIST "$id\n";
        $count++;
    }
    $sth->finish;
    print "Scanning [$count] sessions...\n";

    seek( IDLIST, 0, 0 );

    my $below_thresh = time - ( $days_old * 86400 );
    print "Removing those before ", scalar( localtime( $below_thresh ) ), "\n";

    my $current = 0;
ID:
    while ( <IDLIST> ) {
        chomp;
        my $sth_d = $db->prepare( "SELECT a_session FROM sessions WHERE id = ?" );
        $sth_d->execute( $_ );
        my ( $raw_data ) = $sth_d->fetchrow_array;

        my $do_remove = 0;
        my $session = eval { decode( $OPT_decode, $raw_data ) } || {};
        if ( $@ ) {
            $OPT_debug && print "Caught error decoding id '$_': $@\n";
            $do_remove++ if ( $OPT_force );
        }

        my $timestamp = $session->{timestamp} || 0;
        if ( $do_remove || scalar keys %{ $session } == 0 ) {
            $do_remove++;
        }
        elsif ( $timestamp < $below_thresh ) {
            $do_remove++;
        }

        if ( $do_remove ) {
            $OPT_analyze || $db->do( "DELETE FROM sessions WHERE id = '$_'" );
            $removed++;
        }
        else {
            $kept++;
        }
        print " $current" if ( $current > 0 && $current % 500 == 0 );
        if ( $current > 0 && $current % 2500 == 0 ) {
            print "\n";
            my $time_show = ( $timestamp == 0 ) ? "n/a" : scalar( localtime( $timestamp ) );
            $OPT_debug && print "[Item: $current] ",
                                "[$time_show] ",
                                "[Keys: ", join( ', ', keys %{ $session } ), "]\n";
        }
        $current++;
    }
    close( IDLIST );
    print "\n",
          "Results:\n",
          "   Sessions: $count\n",
          "   Removed:  $removed\n",
          "   Kept:     $kept\n",
          "   Time:     ", ( time - $initial_time ), " seconds\n";
    unlink( $TEMP_TRACK_FILE )
}

sub decode {
    my ( $type, $data ) = @_;
    if ( $type eq 'base64' ) {
        return thaw( decode_base64( $data ) );
    }
    elsif ( $type eq 'storable' ) {
        return thaw( $data );
    }
    else {
        die "Cannot decode type [$type] unknown\n";
    }
}

sub usage {
    return <<USAGE;
$0 [ days-previous ] options

Cleanup old/stale sessions from your database. Depends on the
'timestamp' field being set.

Options:

 --help
     Print this message

 --analyze
     Show a report on the number of sessions and those
     that would be removed, but don't actually remove

 --decode=type
     Type of session encoding. Types are:
        base64   - Storable + Base64
        storable - Storable

 --debug
     Display extra debugging along the way

 --force
     If we catch an error while decoding the session data (e.g., the
     data was serialized with an older version of Storable) and
     '--force' is active, we'll go ahead and remove the session. If
     you also have '--debug' turned on you'll see the error messages
     as they occur.

USAGE
}
