package Amazon::API::Botocore::Pod;

use strict;
use warnings;

use parent qw{ Exporter };

use Pod::Usage;
use Pod::Find qw(pod_where);

our @EXPORT_OK = qw{ pod };

our $VERSION = '2.0.13'; ## no critic (RequireInterpolationOfMetachars)

########################################################################
sub pod {
########################################################################
  my ($exit_val) = @_;

  return pod2usage(
    -exitval => $exit_val,
    -input   => pod_where( { -inc => 1 }, __PACKAGE__ ),
    -output  => \*STDOUT,
  );
}

1;

## no critic (RequirePackageMatchesPodName)

__END__

=pod

=head1 NAME

Amazon::API::Botocore - Automagic AWS API class generator

=head1 SYNOPSIS

 amazon-api options command

 Options
 -------
 --botocore-path, -b  path to git botocore project, default: . or $BOTOCORE_PATH
 --help, -h           help
 --module-name, -m    module name suffix, Example: Route53 (see Notes below)
 --output-path, -o    path for output, - for STDOUT
 --pager, --nopager   turn pager on or off, default is on
 --pod, --nopod       generate pod, default is on
 --service, -s        Botocore API service, Example: route53
 --tidy, -t           perltidy output, default is true, use --no-tidy to disable

 Commands
 --------
 describe           dump the Botocore metadata for a service
 create-stub(s)     create the Amazon::API::{service-name} from the Botocore
                    metadata
 create-shape(s)    create the Amazon::API::Botocore::Shape objects from the
                    Botocore metadata
 help method|shape  show help for method or shape

 Notes
 -----
 1. By default the API class created for a service will be named
    Amazon::API::{service-name}. "service-name" will be derived from
    the service option (-s) using the following rules:

    a. upper case all letters in service name if the length of the
       name is 3 (i.e SNS, SQS, etc.)
    b. upper case the first letter if the the length > 3

    * if you don't like these rules, use the -m option to change the name ;-)

2. The current working directory will be used as the root directory
   for creating the API and shape classes. Use the -o option to create
   the API in a different directory.s Directories will be created if
   they do not exist.

3. Use this script to get help for specific APIs, methods
   or shapes.

   amazon-api -s sqs help
   amazon-api -s sqs help CreateQueue
   amazon-api -s sqs help CreateQueueRequest

   Note that if you created a custom name for your API module
   (example: SecretsManager), then use the API module short name for
   the service.

   amazon-api -s SecretsManager help GetSecretValue

4. Dump the entire Botocore description for an API

   amazon-api -s sqs describe

5. Don't like our pod?

   amazon-api -s sqs describe | jq -r .sqs.operations.CreateQueue

See also - perldoc Amazon::API::Botocore::Pod

=head1 DESCRIPTION

Generates Perl classes for Amazon APIs from Botocore specfications.
These classes are typically used to invoke specific AWS APIs.  For
example:

 my $rt53 = Amazon::API::Route53->new;

 my $hosted_zone = $rt53->ListHostedZones;

 print Dumper([$hosted_zones]);

See L<Amazon::API> for more details.

=head1 NOTES

First build the project from either the GitHub repository or CPAN.

I<Note that building the project from the GitHub repository requires
additional tooling you may not have in your environment. See
L<README-BUILD.md|https://github.com/rlauer6/perl-Amazon-API/blob/master/README-BUILD.md>
for more details.>

Assuming you have built the L<Amazon::API> project...

 git clone https://github.com/rlauer6/perl-Amazon-API.git
 export PROJECT_HOME="$(pwd)/perl-Amazon-API"
 cd $PROJECT_HOME
 ./bootstrap
 ./configure && make && make install

or installed L<Amazon::API> from CPAN...

 cpanm -v Amazon::API

...now clone the Botocore repository.

 git clone https://github.com/boto/botocore.git
 export BOTOCORE_PATH=$(pwd)/botocore

...and now generate the API class and its data classes (shapes)

 amazon-api -s sqs create-stubs
 amazon-api -s sqs create-shapes

...get some help

 amazon-api -s sqs help
 amazon-api -s sqs help CreateQueue
 amazon-api -s sqs help CreateQueueRequest

=head1 METHODS AND SUBROUTINES

No public methods are exported by this class. It's a modulino!

=head1 AUTHOR

Rob Lauer - <rlauer6@comcast.net>

=head1 SEE OTHER

L<Amazon::API>, L<Amazon::API::Botocore::Pod>, L<Amazon::API::Botocore::Shape>

=cut
