#!../../bin/wish -f
# --------------------------------------------------------------------------
# Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
#
# You can use and distribute this software under the terms of the license
# version 1 you should have received along with this software.
# If not or if you want additional information, write to
# Forschungszentrum Informatik, "OBST", Haid-und-Neu-Strasse 10-14,
# D-76131 Karlsruhe, Germany.
# --------------------------------------------------------------------------
#
# mta2edge - 20/8/93 - dietmar theobald
#
# mta2edge (<schema> | (+|-)std | (+|-)gen | (+|-)inst | (+|-)r)...
#
# Writes a grl description of the inheritance structure between the classes
# defined in the given schemas to the standard output which may then be
# displayed using the extensible graph editor EDGE developed by F. Newberry.
#
# Arguments are processed from left to right and are either schema names
# or options which specify which kind of classes are to be processed.
#
# The '+r|-r' option specifies if schemas are to be processed recursively, i.e.
# all (transitively) imported schemas of a given schema will be processed, too.
#
# The remaining options specify if particular kinds of classes are to be
# processed (+<opt>), or not (-<opt>).
# These kinds are: generic classes ('gen'), generic instantiations ('inst'),
# and standard classes ('std'), i.e. neither generic classes nor generic
# instantiations.
#
# Defaults are: -r, +std, +gen, +inst
#

#
# Definition of grl-style output.
#   
proc grl_header {} {
   global self argv

   return "/*\n * Generated by $self on [exec date]\n * arguments: \{ [set argv] \}\n */\ngraph: \{\n  orientation: bottom_to_top\n\n"
}
proc grl_footer {} {
   return "\} /* graph */\n"
}
proc grl_node {tp} {
   return "  node: { title: \"[tpname $tp id]\" label: \"[tpname $tp]\"}\n"
}
proc grl_edge {from to} {
   return "  edge: { sourcename: \"[tpname $from id]\" targetname: \"[tpname $to id]\" }\n"
}


#
# Auxiliary procedures for computing type names (and caching them once they
# are computed for the first time).
#
proc name {obj} {
   return [[$obj get_name] make_Cstring]
}

proc tpname {tp {kind label}} {
#
# tpname($tp) ... label for the given type
# tpid($tp)   ... id of the given type as used externally
#
   global tplbl tpid

   if {[catch {set tpid($tp)} id]} {
      if {[OBST isa $tp sos_Class_type]} {
	 if {[$tp is_instantiation] == "TRUE"} {
	    set sep ""
	    set id  "[name [$tp get_generic_class]]<"
	    agg loop [$tp get_actual_gen_params] {
	       append id "$sep[tpname [[agg current] make_type] id]"
	       set sep ","
	    }
	    set tpid($tp)  "$id>"
	    set tplbl($tp) [set tpid($tp)]

	 } elseif {[$tp is_generic_class] == "TRUE"} {
	    set sep 	   ""
	    set tpid($tp)  "[name $tp]<"
	    set tplbl($tp) [set tpid($tp)]
	    agg loop [$tp get_formal_gen_params] {
	       set pnm [name [agg current]]

	       append tpid($tp)  "$sep$pnm"
	       append tplbl($tp) "$sep$pnm:[tpname [[agg current] make_type] id]"
	       set sep ","
	    }
	    append tpid($tp)  ">"
	    append tplbl($tp) ">"
	 } else {
	    set tpid($tp)  [name $tp]
	    set tplbl($tp) [set tpid($tp)]
	 }
	 if {[$tp get_is_abstract] == "TRUE"} {
	    append tplbl($tp) " (*)"
	 }
      } else {
	 set tpid($tp)  [name $tp]
	 set tplbl($tp) [set tpid($tp)]
      }
   }
   return [expr {$kind == "label" ? [set tplbl($tp)] : [set tpid($tp)]}]
}

#
# Procedure to register the types of interest, i.e. the types which will
# appear in the output.
#
proc register_tp {tp} {
#
# globals:
#    types      ... characteristic function of the set of types of interest
#    show(std)  ... "FALSEFALSE" iff standard class types are to be processed
#    show(gen)  ... "TRUE" iff generic class types are to be processed
#    show(inst) ... "TRUE" iff generic instantiations are to be processed
#
   global types show 

   if {[OBST isa $tp sos_Class_type]} {
      set is_gen  [$tp is_generic_class]
      set is_inst [$tp is_instantiation]

      if {($is_gen == $show(gen) || $is_inst == $show(inst)
	   			 || "$is_gen$is_inst" == $show(std))} {
	 set types($tp) ""
      }
   }
}

proc do_schema {schema} {
#
# globals:
#    schemas   ... characteristic function of the set of processed schemas
#    recursive ... flag if imported schemas are to be processed, too
#
   global schemas recursive

   if {[catch {set schema($schema)}]} {
      agg loop [$schema get_types] {
	 register_tp [agg current]
      }
      if {$recursive} {
	 agg loop [$schema get_imports] {
	    do_schema [[agg current] get_module]
	 }
      }
   }
}


# ------ main -------

tclOBST bind

set self       "mta2edge.tcl"
set str	       [mcall sos_String::create [OBST const TEMP_CONTAINER]]
set schemadir  [mcall sos_Schema_module::schema_dir]

set show(gen)  "TRUE"
set show(inst) "TRUE"
set show(std)  "FALSEFALSE"
set recursive  0

foreach arg $argv {
   case $arg in {
      +gen    { set show(gen)  "TRUE"		}
      -gen    { set show(gen)  ""		}
      +inst   { set show(inst) "TRUE"		}
      -inst   { set show(inst) ""		}
      +std    { set show(std)  "FALSEFALSE"	}
      -std    { set show(std)  "" 		}
      +r      { set recursive  1		}
      -r      { set recursive  0		}
      default {
	 $str assign_Cstring $arg
	 
	 if {[set schema [$schemadir {[]} $str]] == [OBST const NO_OBJECT]} {
	    puts stderr "*** $self: no schema '$arg' (ignored)"
	 } else {
	    do_schema $schema
	 }
      }
   }
}

puts -nonewline stdout [grl_header]

if {![catch {array startsearch types} scan]} {
   while {[set tp [array nextelement types $scan]] != ""} {
      puts -nonewline stdout [grl_node $tp]

      agg loop [$tp get_super_classes] {
	 if {[[agg current] get_is_direct] == "TRUE"} {
	    set sc [[agg current] get_super_class]
	    if {[OBST isa $sc sos_Generic_instantiation]
		&& [$tp is_generic_class] == "TRUE"} {
	       set gen 1
	       agg loop [$sc get_act_gen_params] {
		  if {![OBST isa [agg current] sos_Gen_param]} {
		     set gen 0
		     break
		  }
	       }
	       set sctp [expr {$gen ? [$sc get_gen_class] : [$sc make_type]}]
	    } else {
	       set sctp [$sc make_type]
	    }
	    if {![catch {set types($tp);set types($sctp)}]} {
	       append edges [grl_edge $tp $sctp]
	    }
	 }
      }
   }
   array donesearch types $scan
}
puts stdout ""
catch {puts -nonewline stdout [set edges]}
puts -nonewline stdout [grl_footer]

catch {destroy .}
catch {exit 0}
