#!../../bin/wish -f
# --------------------------------------------------------------------------
# Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
#
# You can use and distribute this software under the terms of the license
# version 1 you should have received along with this software.
# If not or if you want additional information, write to
# Forschungszentrum Informatik, "OBST", Haid-und-Neu-Strasse 10-14,
# D-76131 Karlsruhe, Germany.
# --------------------------------------------------------------------------
#
# dirTool_DoCopy.tcl - 8/2/93 - dietmar theobald
#
# TclOBST script to perform the `copy_container' operation of dirTool.
#
# This must be performed in a separate process, since the container
# directory is changed across the invocation.
#
# The script takes three arguments:
#  - path of the directory from which the container is to be copied
#  - path of the object to be copied within the OBST directory hierarchy
#  - encoding of the object to be copied
#
# The copy is performed by copying the object's container to the active
# container directory. Additionally, the object is inserted into the active
# directory hierarchy with the given name.
#
# Neither the (OBST) parent directory, the given object path, nor a container
# with the given number must yet exist.
#
# A successful copy depends on some preconditions on the copied objects (i.e.
# all reachable objects in the copied containers). It might be necessary to
# copy several containers/objects in a specific order to fulfill these
# conditions:
#  - The copy must be complete, i.e. there must be no dangling references
#    starting from copied objects which refer to objects only existing in the
#    source container directory.
#  - The copied objects must not reference objects which reside in as well
#    the source and the target container directories but at other physical
#    locations.
#
#    This will be often fulfilled in the following case: two initially
#    identical databases were separated and independently modified by adding
#    distinct sections. The only container/object modified by both parties
#    is a directory object where a named reference to the distinct sections
#    was inserted.
#
# One situation which fulfills the above requirements is the development of 
# independent schemata in local databases which are then to be merged into
# a central database.
# (Expressed in the above used terms: the seperate sections are made up by
#  the seperate meta data, and the common directory is the schema directory
#  "/sos_schemas".)
# In this case, each of the added schemata has to be inserted in turn -
# preferably using `dirTool'.

proc do_err {msg} {
   puts stderr "error: $msg"
   exit 1
}
tclOBST bind

set SELF "dirTool_DoCopy.tcl"
puts stdout "$SELF: started\n"

set fromDir [lindex $argv 0]
set toDir   $env(OBSTCONTAINER)
set objPath [lindex $argv 1]
set objName [file tail $objPath]
set dirPath [file dirname $objPath]
set newObj  [lindex $argv 2]

tclOBST bind $newObj
set newCnt  [$newObj container]

set strObj  [OBST tmpstr $dirPath]

puts stdout "from: $fromDir\nto  : $toDir\npath: $objPath\ncnt : $newCnt\n"

if {$objPath == "/"} {
   do_err "can't copy root directory"
}
if {[catch "mcall sos_Directory::lookup $strObj" dirObj]} {
   do_err "can't access parent directory \"$dirPath\""
}
if {$dirObj == [OBST const NO_OBJECT]} then {
   do_err "no parent directory \"$dirPath\""
}
set dirCnt [$dirObj container]

if {[cnt open WRITING TESTING $dirCnt] != "OPENED"} {
   do_err "no write access to directory \"$dirPath\" ($dirCnt)"
}
$strObj assign_Cstring $objPath

if {[catch "mcall sos_Directory::lookup $strObj" _newObj]} {
   do_err "can't check for existence of \"$objPath\""
}
if {$_newObj != [OBST const NO_OBJECT]} {
   do_err "object \"$objPath\" does already exist"
}
if {[file exists $toDir/$newCnt]} {
   do_err "container $newCnt does already exist"
}
if {[catch "exec cp $fromDir/$newCnt $toDir/$newCnt"]} {
   do_err "can't copy container $newCnt"
}
puts stdout "copied container $newCnt"

if {[file exists $toDir/1.lock]} {
   set lock_file $toDir/1.lock
} else if {[file exists $toDir/1.lck]} {
   set lock_file $toDir/1.lck
} else {
   set lock_file ""
}
if {$lock_file != ""} {
   if {[catch "exec touch $toDir/$newCnt.lock"]} {
      do_err "can't create $newCnt.lock"
   }
   puts stdout "created $newCnt.lock"
}
$strObj assign_Cstring $objName

if {[catch "$dirObj insert $strObj $newObj"]} {
   do_err "can't insert $objName in directory $dirPath ($errorInfo)"
}
cnt close $dirCnt

puts stdout "inserted $objName in directory $dirPath\n"

puts stdout "$SELF: finished"
exit 0
