/* =================================================================== */
/*       CLASS - IMPLEMENTATION                                    === */
/* =================================================================== */
/*                                                                     */
/* CLASS-NAME: Source_of_Error              Class_Id: ERR              */
/*                                                                     */
/* IMPLEMENTATION ISSUES:                                              */
/*                                                                     */
/* =================================================================== */
#define ERR_CONCEPT 1

/* =================================================================== */
/* Substitution of C-symbols :                                         */
#include "predef.h"

/* =================================================================== */
/* Trace switches:                                                     */
#include "trc_add.h"

/* =================================================================== */
// Export File of this concept
#include "Source_of_Error_obst.h"

   
sos_Id NOID = sos_Id::make (sos_Container::make (0), 0);


// ---- Buffer used to pass values to the application ---- //
static sos_Cstring copy_of_latest_error_message = 
   new char [ERR_MAX_ERR_MESSAGE_LENGTH];
static sos_Cstring err_unexpected_error = new char [30];

sos_Bool add_Source_of_Error::method_successfull()
   //
   // PURPOSE:
   // This function should be called after each execution of a method of
   // this class which may cause an error. If not errors may remain 
   // undetected or cause unexpected results.
   //
   // returns TRUE if the execution of the latest method was successfull.
   // Also signal_error(error_state(), latest_error_message()) is called.
   //
   // PRECONDITION:
   //      none
   //
   // POSTCONDITION:
   //      if (error != ERR_INCONSISTENT) clear_error();
{
  sos_Cstring state;

  T_PROC("add_Source_of_Error::method_successfull()");
  TT(serr_H, T_ENTER);
  sos_Bool result;

  int error = self.error_state();
  if (error != ERR_CONSISTENT)
      {
	  self.signal_error(error, self.latest_error_message());
	  if (error != ERR_INCONSISTENT) self.clear_error();
	  result = FALSE;
      }
  else
      {
	  result = TRUE;
	  TT(serr_H, TB(result));
      }
  TT(serr_H, T_LEAVE);
  return (result);
}


int add_Source_of_Error::error_state()

// PURPOSE:  
// returns an constant denoting the error_state of 
// this object
//
// PRECONDITION:                                           
//         None.                                          
//
// POSTCONDITION, ERRORS:                                  
//         None
{
   return(self.get_error());
}     


sos_Cstring add_Source_of_Error::latest_error_message()
//
// PURPOSE:                                                
// returns the error message flaged by the latest execution
// of a critical method on this object.
//
// PRECONDITION:                                           
//         None.                                          
//
// POSTCONDITION, ERRORS:                                  
//         None
{
   T_PROC("add_Source_of_Error::latest_error_message()");

   sos_Cstring tmp = self.get_latest_error_message().make_Cstring();
   strcpy(copy_of_latest_error_message, tmp);
   delete tmp;

   TT(serr_H, TS(copy_of_latest_error_message));
   return(copy_of_latest_error_message);
}     


sos_Bool add_Source_of_Error::ready_for_destruction()
//
// PURPOSE:                                                
// returns TRUE if this object can be destroyed correctly.
//
// PRECONDITION:                                           
//         None.                                          
//
// POSTCONDITION, ERRORS:                                  
//         None
{
   return(TRUE);
}     


Void add_Source_of_Error::destruction()
//
// PURPOSE:                                                
// returns TRUE if this object has been destroyed correctly.
//
// PRECONDITION:                                           
//         None.                                          
//
// POSTCONDITION, ERRORS:                                  
//         None
{
   T_PROC( "add_Source_of_Error::destruction()");
   TT(serr_H, T_ENTER);

   self.clear_error();
   if (self.ready_for_destruction())
     {
       self.destroy();
     }
   else
     self.signal_error (ERR_DELETION_NOT_PERMITTED, 
			self.error_message(ERR_DELETION_NOT_PERMITTED));
   TT(serr_H, T_LEAVE);
   return;
}     


sos_Bool add_Source_of_Error::ready_for_assignment(add_Source_of_Error source)
//
// PURPOSE:                                                
// returns TRUE if source can be assigned to this object correctly.
//
// PRECONDITION:                                           
//         None.                                          
//
// POSTCONDITION, ERRORS:                                  
//         None
{
   return(self.ready_for_destruction());
}     


Void add_Source_of_Error::assign_to(add_Source_of_Error destination)
//
// PURPOSE:                                                
// assigns this object to destination.
//
// PRECONDITION:                                           
//         None.                                          
//
// POSTCONDITION, ERRORS:                                  
//         None
{
   T_PROC
       ("add_Source_of_Error::assign_to(add_Source_of_Error destination)");
   TT(serr_H, T_ENTER);

   if(self.error_state() == ERR_INCONSISTENT)
      self.signal_error(ERR_INCONSISTENT,self.error_message(ERR_INCONSISTENT));
   else
       {
	   self.clear_error();
	   if (destination.ready_for_assignment(self))
	       destination.assign(self);
	   else
	       self.flag_error(ERR_ASSIGN_NOT_PERMITTED, 
			       self.error_message(ERR_ASSIGN_NOT_PERMITTED));
       }	   
   TT(serr_H, T_LEAVE);
   return;
}     


sos_Cstring add_Source_of_Error::description()

// PURPOSE:                                                
// returns a description of this object.
//
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  

{
   return(strnew("",""));
}     


/* void add_Source_of_Error::set_debug()
//                                                         
// PURPOSE:                                                
// set the global debug mode
//
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  

{
   T_PROC("add_Source_of_Error::clear_debug()");
   TT(serr_H, T_ENTER);
   int retvalue = putenv("ADD_DEBUG = ON");
   TT(serr_H, T_LEAVE);
   return;
}     


void add_Source_of_Error::clear_debug()
//                                                         
// PURPOSE:                                                
// clear the global debug mode
//
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  

{
   int retvalue; 

   T_PROC("add_Source_of_Error::clear_debug()");
   TT(serr_H, T_ENTER);

   retvalue = putenv("ADD_DEBUG = OFF");
   
   TT(serr_H, T_LEAVE);
   return;
}     


sos_Bool add_Source_of_Error::is_debug()
//                                                         
// PURPOSE:                                                
// returns TRUE if the debug mode is set 
// if ADD_DEBUG is not defined FALSE is returned        
//
// PRECONDITION:                                           
//                                                         
// POSTCONDITION, ERRORS:                                  

{
   sos_Bool retvalue; 
   sos_Cstring envi;

   T_PROC("add_Source_of_Error::is_debug()");
   TT(serr_H, T_ENTER);

   envi = getenv("ADD_DEBUG");
   if ( envi == "ON")  
      retvalue = TRUE;

   else
      retvalue = FALSE;

   TT(serr_H, T_LEAVE; TB(retvalue));
   return(retvalue);
}     


sos_Cstring add_Source_of_Error::debug_description()

// PURPOSE:                                                
// prints out this object in the debug mode 
// whether the debug mode was set before or not.
//                                                         
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  
//  error_state flagged by description

{   
   sos_Cstring state; 

   T_PROC("add_Source_of_Error::debug_description()");
   TT(serr_H, T_ENTER);

   if (add_Source_of_Error::is_debug() ) 
          state = self.description();
   else
     {
       add_Source_of_Error::set_debug();
       state = self.description();
       add_Source_of_Error::clear_debug();
     }
   TT(serr_H, T_LEAVE);
   return(state);
}     
           

sos_Cstring add_Source_of_Error::plain_description()
//                                                         
// PURPOSE:                                                
// prints out this object in the no_debug mode 
// whether the debug mode was set before or not.
//                                                         
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  
// error_state flagged by description

{
   sos_Cstring state; 

   T_PROC("add_Source_of_Error::plain_description()");
   TT(serr_H, T_ENTER);

   if (NOT add_Source_of_Error::is_debug() ) 
          state = self.description();
   else
   {
       add_Source_of_Error::clear_debug();
       state = self.description();
       add_Source_of_Error::set_debug();
    }
   TT(serr_H, T_LEAVE);
   return(state);
}
*/

sos_Cstring add_Source_of_Error::error_message(int err_value)

// PURPOSE:                                                
// returns the error message belonging to the error_state
//                                                         
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  
// If no message is found this method returns
// "!! unexpected error: err_value"
{
   sos_Cstring retvalue;

   T_PROC(" add_Source_of_Error::error_message(int err_value)");
   TT(serr_H, T_ENTER; TI(err_value));

   switch (err_value)
     {
     case ERR_CONSISTENT:
       retvalue = "this object is consistent.";
       break;
     case ERR_INCONSISTENT:
       retvalue = "this object is inconsistent !!";
       break;
     case ERR_FATAL_ERROR:
       retvalue = "a fatal error has occurred !!";
       break;
     case ERR_DELETION_NOT_PERMITTED:
       retvalue = "deletion not permitted \n";
       break;
     case ERR_ASSIGN_NOT_PERMITTED:
       retvalue = "assign_to not permitted \n";
       break;
     default:
       sprintf (err_unexpected_error, "!! unexpected error: %d", err_value);
       retvalue = err_unexpected_error;
    }

   TT(serr_H, T_LEAVE; TS(retvalue));
   return(retvalue);
}     


Void add_Source_of_Error::flag_error(int error_state, 
				     sos_Cstring latest_error_message)
//
// PURPOSE:
// used by a critical method in order to flag a recognized error.
// After the execution of this method error_state and latest_error_message 
// can be retrieved by the methods error_state() and latest_error_message().
//
// PRECONDITION:
//        None
//
// POSTCONDITION:
//        None
{
   T_PROC
     ("add_Source_of_Error::flag_error(int error_state, sos_Cstring latest_error_message)");
   TT(wsp_MISC, T_ENTER; TI(error_state));
   TT(wsp_MISC, TS(latest_error_message));

   self.set_error(error_state); 
   self.get_latest_error_message().assign_Cstring(latest_error_message);
   if (error_state == ERR_INCONSISTENT)
     self.signal_error(error_state, latest_error_message);

   TT(wsp_MISC, T_LEAVE);
   return;
}     


void add_Source_of_Error::clear_error()
//                                                         
// PURPOSE:                                                
// sets error to ERR_CONSISTENT in order to disable further
// reactions (e.g. by std_error_hdl) on an handled error.
//                                                         
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  
//         ERR_CONSISTENT  

{
   self.set_error(ERR_CONSISTENT); 
//   self.get_latest_error_message().assign_Cstring("");
   return;
}     


void add_Source_of_Error::signal_error(int error_state, 
				    sos_Cstring message)
//
// PURPOSE:
// signals the error_event to someone who might handle it.
//
// PRECONDITION:
//        None
//
// POSTCONDITION:
//        None
{
   T_PROC
     ("add_Source_of_Error::signal_error(int error_state, sos_Cstring message)");
   TT(serr_URGENT, T_ENTER; TI(error_state));
   TT(serr_URGENT, TS(message));

   // signal the error to the user:
//    err_raise (err_USE, message);
   cerr << "\n" << message;

   TT(serr_URGENT, T_LEAVE);
   return;
}

           
void add_Source_of_Error::local_initialize(add_Source_of_Error this_one)

// PURPOSE:                                                
// initializes error_state
//                                                         
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  
//           None

{
   T_PROC("add_Source_of_Error::local_initialize(add_Source_of_Error)");
   TT(serr_H, T_ENTER);

   this_one.set_error(ERR_CONSISTENT); 
   sos_String message = sos_String::create(this_one.container());
   this_one.set_latest_error_message(message);

   TT(serr_H, T_LEAVE);
   return;
}     


void add_Source_of_Error::local_assign(add_Source_of_Error destination,
				       sos_Object source_obj)

// PURPOSE:                                                
// initializes error_state
//                                                         
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  
//           None

{
   T_PROC("add_Source_of_Error::local_assign(add_Source_of_Error source, sos_Object dest)");
   TT(serr_H, T_ENTER);

   add_Source_of_Error source = add_Source_of_Error::make(source_obj);
   destination.set_error(source.get_error()); 
				
// 23.04.93 HUK: not allways necessary:
   if (source.get_error() == ERR_INCONSISTENT)
     destination.get_latest_error_message().assign(source.get_latest_error_message());

   TT(serr_H, T_LEAVE);
   return;
}     


void add_Source_of_Error::local_finalize(add_Source_of_Error this_one)

// PURPOSE:                                                
// clean up
//                                                         
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  
//           None

{
   T_PROC("add_Source_of_Error::local_finalize(add_Source_of_Error)");
   TT(serr_H, T_ENTER);

   this_one.get_latest_error_message().destroy();

   TT(serr_H, T_LEAVE);
   return;
}     


sos_Object sos_Id2Object(sos_Id id)

// PURPOSE:                                                
// returns the referenced object
//                                                         
// PRECONDITION:                                           
//          None.                                          
//                                                         
// POSTCONDITION, ERRORS:                                  
// returns NO_OBJECT if the container is not opened, or
// if the referenced object is not existing.

{
  T_PROC("sos_Id2Object()");
  TT(serr_H, T_ENTER);
  if (id == NOID)
    {
      TT(nsp_H, TXT("access to NOID"));
      TT(serr_H, T_LEAVE);
      return(NO_OBJECT);
    }

  sos_Container cnt = id.container();
  sos_Object obj;
  
  if ((cnt != TEMP_CONTAINER) AND
      ((cnt.status() == UNAVAILABLE) OR
       (cnt.status() == DESTROYED)))
    {
      TT(0, TXT("access to an object in an unopened container"));
      TT(serr_H, T_LEAVE);
      return(NO_OBJECT);
    }

  err_block
    {
      sos_Typed_id tid = sos_Typed_id::make(id);
      if (cnt.object_exists(id.offset(), SOS_ID_SIZE) == NOT_EXISTING)
//      if (obst_ID_valid(tid) == NOT_EXISTING)
	  {
	    TT(0, TXT(form("<%d,%d> accesses a non existing object", 
			   id.container(), id.offset())));
	    TT(serr_H, T_LEAVE);
	    return(NO_OBJECT);
	  }
      TT(nsp_H, 
	 if (obst_ID_valid(tid) == NOT_EXISTING)
	 {
	   cerr << form("obst_ID_valid: <%d,%d> accesses a non existing object\n", 
			id.container(), id.offset());
	   TXT(form("obst_ID_valid(tid): <%d,%d> accesses a non existing object", 
		    id.container(), id.offset()));
	 }
	 );
      obj = sos_Object::make(tid); 
    }
  err_exception
    {
      TT(0, TXT("access to object failed"));
      obj = NO_OBJECT;
    }
  err_block_end;
  
  TT(serr_H, T_LEAVE; TOBJ(obj));
  
  return(obj);
}

