/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
%token stf_Name_tok
%token stf_String_tok

%token class_tok
%token derive_tok
%token error_tok
%token grammar_tok
%token lexical_tok
%token symbol_tok
%token with_tok

%token l_br_tok
%token r_br_tok
%token l_brc_tok
%token r_brc_tok
%token l_par_tok
%token r_par_tok
%token colon_tok
%token comma_tok
%token div_tok
%token mult_tok
%token plus_tok
%token rshift_tok
%token semicolon_tok

%type <String>			stf_Name_tok
%type <String>			stf_Name
%type <String>			stf_String_tok
%type <Imports>			stf_Imports
%type <Imports>			sos_Schema_module_List
%type <Schema_module>		sos_Schema_module
%type <Symbol_rule>		stf_Symbol_name
%type <Symbol_rule>		stf_Lexical_symbol_name
%type <Schema_type>		sos_Type_name
%type <Derived_item_rule>	stf_Derived_from
%type <Component_list>		stf_Component_List
%type <Component>		stf_Component
%type <String_component>	stf_String_component
%type <Item_component>		stf_Item_component
%type <Comp_method>		sos_Comp_descr
%type <Item>			stf_Item
%type <Item_with_strings>	stf_Item_with_strings
%type <String_symbol_List>	stf_String_symbol_List
%type <String_symbol>		stf_String_symbol
%type <Symbol_item>		stf_Symbol_item
%type <General_list_item>	stf_General_list_item
%type <Non_empty_list_item>	stf_Non_empty_list_item
%type <Separator_list_item>	stf_Separator_list_item
%type <Optional_item>		stf_Optional_item
%type <Optional_strings>	stf_Optional_strings
%type <String>			stf_Conversion

%{

#define OBST_IMP_STRINGOP
#define OBST_IMP_MALLOC
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "smg.h"
#include "stf_err.h"
#include "stf_use.h"

typedef union {
   sos_String			String;
   stf_Imports			Imports;
   sos_Schema_module		Schema_module;
   stf_Rule_List		Rule_List;
   stf_Rule			Rule;
   stf_Symbol_rule		Symbol_rule;
   stf_Derived_item_rule	Derived_item_rule;
   sos_Type	                Schema_type;
   stf_Component_List		Component_list;
   stf_Component		Component;
   stf_String_component		String_component;
   stf_Item_component		Item_component;
   sos_Comp_descr		Comp_method;
   stf_Item			Item;
   stf_Item_with_strings	Item_with_strings;
   stf_String_symbol_List	String_symbol_List;
   stf_String_symbol		String_symbol;
   stf_Symbol_item		Symbol_item;
   stf_General_list_item	General_list_item;
   stf_Non_empty_list_item	Non_empty_list_item;
   stf_Separator_list_item	Separator_list_item;
   stf_Optional_item		Optional_item;
   stf_Optional_strings		Optional_strings;
} _YYSTYPE;
#define YYSTYPE _YYSTYPE

LOCAL sos_Container cnt;
LOCAL stf_Grammar gra;
LOCAL sos_Class_type ct;
LOCAL stf_Imports imp;
LOCAL stf_Rule_List rules;
LOCAL stf_Item_List aux_rules;
LOCAL stf_Structure_rule root_rule;
LOCAL sos_String_stf_Symbol_rule_Mapping symbol_table;
LOCAL sos_String_stf_String_symbol_Mapping string_symbols;

LOCAL void stf_error (err_class ec, err_msg s, char *where = 0);
LOCAL void stf_error (err_class ec, err_msg s, sos_String where);
LOCAL sos_Type lookup_schema_type (sos_String name);
LOCAL stf_Symbol_rule make_symbol_rule (sos_String name, sos_Type ct);
LOCAL sos_Comp_descr identify_component (sos_String name);
LOCAL void check_item (stf_Item, sos_Type, sos_String);
LOCAL void check_compatible (sos_Type, sos_Type, sos_String);
LOCAL void foldCstring (char *);

#include "stf_yacc.h"
#include "stf_lex.h"

void yyerror (char *s) { stf_error (err_USE, s);}
void yyecho () {if (stf_echo_flag) ECHO;}

%}

%%

/******************** Grammar, Imports, Rules *****************/

stf_Grammar :
   stf_Imports
   grammar_tok
   stf_Name
   l_brc_tok
   stf_Rule_List
   r_brc_tok
   {  gra = stf_Grammar::create (cnt);
      gra.set_imports ($1);
      gra.set_name ($3);
      gra.set_rules (rules);
      gra.set_aux_rules (aux_rules);
      gra.set_symbol_table (symbol_table);
      gra.set_string_symbols (string_symbols);
      gra.set_root_rule (root_rule);
   }
;

stf_Imports :
   {  $$ = stf_Imports::make (NO_OBJECT);
      imp = $$;
   }
|  with_tok
   sos_Schema_module_List
   semicolon_tok
   {  $$ = $2;
      imp = $$;
   }
;

sos_Schema_module_List :
   sos_Schema_module
   {  $$ = stf_Imports::create (cnt);
      if ($1 != NO_OBJECT)
          $$.insert (1, $1);
   }
|  sos_Schema_module
   comma_tok
   sos_Schema_module_List
   {  $$ = $3;
      if ($1 != NO_OBJECT)
	 $$.insert (1, $1);
   }
;

sos_Schema_module :
   stf_Name
   {  $$ = sos_Schema_module::lookup ($1);
      if ($$ == NO_OBJECT)
	 stf_error (err_USE, err_STF_NO_SCHEMA, $1);
   }
;

stf_Rule_List :
|  stf_Rule_List
   stf_Rule
;

/***************** Class Symbols, String symbols **************/

stf_Rule :
   symbol_tok
   sos_Type_name
   stf_Name
   semicolon_tok
   {  stf_Symbol_rule rl = stf_Symbol_rule::create (cnt);
      rl.set_schema_type ($2);
      rl.set_name ($3);
      rl.set_rules (stf_Structure_rule_List::create (cnt));
      rules.append (rl);
      symbol_table.insert ($3, rl);
   }
;

stf_Symbol_name :
   stf_Name
   {  $$ = symbol_table [$1];
      if ($$ == NO_OBJECT)
      {  sos_Type tp = lookup_schema_type ($1);
	 if (tp == NO_OBJECT)
	    stf_error (err_USE, err_STF_NO_SYMBOL, $1);
	 else
	    $$ = make_symbol_rule ($1, tp);
      }
   }
;

stf_Lexical_symbol_name :
   stf_Name
   {  $$ = symbol_table [$1];
      if ($$ == NO_OBJECT)
      {  sos_Type tp = lookup_schema_type ($1);
	 if (tp == NO_OBJECT)
	    tp = sos_Class_type::make (sos_String_type);
	 $$ = make_symbol_rule ($1, tp);
      }
   }
;

sos_Type_name :
   stf_Name
   {  $$ = lookup_schema_type ($1);
      if ($$ == NO_OBJECT)
	 stf_error (err_USE, err_STF_NO_TYPE, $1);
   }
;

stf_String_symbol :
   stf_String_tok
   {  $$ = string_symbols [$1];
      if ($$ == NO_OBJECT)
      {  string_symbols.insert ($1, $1);
	 $$ = $1;
      }
   }
;

/********** Lexical Rules, Derived Rules, Class Rules *********/

stf_Rule :
   lexical_tok
   stf_String_tok
   stf_Conversion
   colon_tok
   stf_Lexical_symbol_name
   semicolon_tok
   {  stf_Lexical_rule rl = stf_Lexical_rule::create (cnt);
      rl.set_symbol ($5);
      rl.set_pattern ($2);
      rl.set_conversion ($3);
      rules.append (rl);
      $5.get_rules().append(rl);
      check_compatible (sos_Type::make (sos_String_type),
			$5.get_schema_type(), $3);
   }
;

stf_Rule :
   derive_tok
   stf_Item_with_strings
   stf_Conversion
   colon_tok
   stf_Symbol_name
   semicolon_tok
   {  stf_Derived_item_rule rl = stf_Derived_item_rule::create (cnt);
      rl.set_symbol ($5);
      rl.set_item ($2);
      rl.set_conversion ($3);
      rules.append (rl);
      $5.get_rules().append(rl);
      check_item ($2, $5.get_schema_type(), $3);
      if (root_rule == NO_OBJECT) root_rule = rl;
   }
;

stf_Rule :
   derive_tok
   stf_String_symbol
   stf_Conversion
   colon_tok
   stf_Symbol_name
   semicolon_tok
   {  stf_Derived_string_rule rl = stf_Derived_string_rule::create (cnt);
      rl.set_symbol ($5);
      rl.set_string ($2);
      rl.set_conversion ($3);
      rules.append (rl);
      $5.get_rules().append(rl);
      check_compatible (sos_Type::make (sos_String_type),
			$5.get_schema_type(), $3);
      if (root_rule == NO_OBJECT) root_rule = rl;
   }
;

stf_Rule :
   error_tok
   colon_tok
   stf_Symbol_name
   semicolon_tok
   {  stf_Derived_string_rule rl = stf_Derived_string_rule::create (cnt);
      rl.set_symbol ($3);
      rl.set_string (sos_String::make (NO_OBJECT));
      rules.append (rl);
      $3.get_rules().append(rl);
      if (root_rule == NO_OBJECT) root_rule = rl;
   }
;

stf_Rule :
   class_tok
   stf_Symbol_name
   {  if ($2 != NO_OBJECT)
      {  sos_Type tp = $2.get_schema_type();
	 if (tp.has_type (sos_Class_type_type))
	    ct = sos_Class_type::make (tp);
	 else
	    stf_error (err_USE, err_STF_NO_CLASS, tp.get_name());
      }
   }
   stf_Derived_from
   l_brc_tok
   stf_Component_List
   r_brc_tok
   semicolon_tok
   {  stf_Class_rule rl = stf_Class_rule::create (cnt);
      rl.set_symbol ($2);
      rl.set_derived_from ($4);
      rl.set_components ($6);
      rules.append (rl);
      $2.get_rules().append(rl);
      ct = sos_Class_type::make (NO_OBJECT);
      if ($4 != NO_OBJECT)
      {  stf_Symbol_item si = stf_Symbol_item::create (cnt);
	 si.set_symbol ($2);
	 $4.set_item (si);
	 check_item (si, $4.get_symbol().get_schema_type(), $4.get_conversion());
      }
      else if (root_rule == NO_OBJECT) root_rule = rl;
   }
;

stf_Derived_from :
   {  $$ = stf_Derived_item_rule::make (NO_OBJECT); }
|  stf_Conversion
   colon_tok
   stf_Symbol_name
   {  $$ = stf_Derived_item_rule::create (cnt);
      $$.set_conversion ($1);
      $$.set_symbol ($3);
      rules.append ($$);
      $3.get_rules().append($$);
      if (root_rule == NO_OBJECT) root_rule = $$;
   }
;

stf_Component_List :
   {  $$ = stf_Component_List::create (cnt); }
|  stf_Component
   stf_Component_List
   {  $$ = $2;
      $$.insert (1, $1);
   }
;

stf_Component :
   stf_String_component
   {  $$ = $1; }
|  stf_Item_component
   {  $$ = $1; }
;

stf_String_component :
   stf_String_symbol
   {  $$ = stf_String_component::create (cnt);
      $$.set_string ($1);
   }
;

stf_Item_component :
   stf_Item
   stf_Conversion
   sos_Comp_descr
   {  $$ = stf_Item_component::create (cnt);
      $$.set_item ($1);
      $$.set_conversion ($2);
      $$.set_component ($3);
      check_item ($1, $3.get_get_method().get_result_type().make_type(), $2);
   }
;

sos_Comp_descr :
   stf_Name
   {  $$ = identify_component ($1); }
;

stf_Rule :
   error
;

/************************** Items *****************************/

stf_Item_with_strings :
   stf_String_symbol_List
   stf_Item
   stf_String_symbol_List
   {  $$ = stf_Item_with_strings::create (cnt);
      $$.set_before ($1);
      $$.set_item ($2);
      $$.set_after ($3);
   }
;

stf_String_symbol_List :
   {  $$ = stf_String_symbol_List::create (cnt); }
|  stf_String_symbol
   stf_String_symbol_List
   {  $$ = $2;
      $$.insert (1, $1);
   }
;

stf_Item :
   l_par_tok
   stf_Item_with_strings
   r_par_tok
   {  $$ = $2; }
;

stf_Item :
   stf_Symbol_item
   {  $$ = $1; }
;

stf_Symbol_item :
   stf_Symbol_name
   {  $$ = stf_Symbol_item::create (cnt);
      $$.set_symbol ($1);
   }
;

stf_Item :
   stf_Optional_item
   {  $$ = $1; }
;

stf_Optional_item :
   l_br_tok
   stf_Item_with_strings
   r_br_tok
   {  $$ = stf_Optional_item::create (cnt);
      $$.set_item ($2);
   }
;

stf_Item :
   stf_Optional_strings
   {  $$ = $1; }
;

stf_Optional_strings :
   l_br_tok
   stf_String_symbol_List
   r_br_tok
   {  $$ = stf_Optional_strings::create (cnt);
      $$.set_strings ($2);
   }
;

stf_Item :
   stf_General_list_item
   {  $$ = $1; }
;

stf_General_list_item :
   stf_Item
   mult_tok
   {  $$ = stf_General_list_item::create (cnt);
      $$.set_repeated ($1);
   }
;

stf_Item :
   stf_Non_empty_list_item
   {  $$ = $1; }
;

stf_Non_empty_list_item :
   stf_Item
   plus_tok
   {  $$ = stf_Non_empty_list_item::create (cnt);
      $$.set_repeated ($1);
   }
;

stf_Item :
   stf_Separator_list_item
   {  $$ = $1; }
;

stf_Separator_list_item :
   l_par_tok
   stf_Item_with_strings
   div_tok
   stf_String_symbol_List
   r_par_tok
   {  $$ = stf_Separator_list_item::create (cnt);
      $$.set_repeated ($2);
      $$.set_separator ($4);
   }
;

stf_Conversion :
   {  $$ = sos_String::make (NO_OBJECT); }
|  rshift_tok
   stf_Name
   {  $$ = $2; }
;

stf_Name :
   stf_Name_tok
|  class_tok
   {  $$ = smg_String ("class").make_String (cnt); }
|  derive_tok
   {  $$ = smg_String ("derive").make_String (cnt); }
|  error_tok
   {  $$ = smg_String ("error").make_String (cnt); }
|  grammar_tok
   {  $$ = smg_String ("grammar").make_String (cnt); }
|  lexical_tok
   {  $$ = smg_String ("lexical").make_String (cnt); }
|  symbol_tok
   {  $$ = smg_String ("symbol").make_String (cnt); }
|  with_tok
   {  $$ = smg_String ("with").make_String (cnt); }
;

%%

// ****************************************************************
// end of yacc grammar, function definitions follow
// ****************************************************************

LOCAL void stf_error (err_class ec, err_msg s, char *where /* = 0 */)
{
   smg_String msg = smg_String("line ") + yylineno;

   if (where)
      msg += smg_String(": ") + where;

   err_raise (ec, s, msg.make_Cstring(SMG_BORROW));
}

LOCAL void stf_error (err_class ec, err_msg s, sos_String where)
{  char *w = where.make_Cstring();
   stf_error (ec, s, w);
   delete w;
}

EXPORT int stf_echo_flag = 0;


// ****************************************************************

EXPORT stf_Grammar stf_compile (sos_Bool persistent_flag)
{  if (persistent_flag)
      {cnt = sos_Container::create ();}
   else 
      {cnt = TEMP_CONTAINER;}

   rules = stf_Rule_List::create (cnt, FALSE);
   aux_rules = stf_Item_List::create (cnt, FALSE);
   symbol_table = sos_String_stf_Symbol_rule_Mapping::create
                  (cnt, FALSE, TRUE);
   string_symbols =
      sos_String_stf_String_symbol_Mapping::create (cnt, FALSE, TRUE);
   root_rule = stf_Structure_rule::make (NO_OBJECT);

   yyparse();

   if (err_occurred (err_SYS) + err_occurred (err_USE) == 0)
   {  if (persistent_flag) cnt.commit();
      return gra;
   }
   else
   {  if (persistent_flag)
        {cnt.destroy(); cnt.close();
        }
      return stf_Grammar::make (NO_OBJECT);
   }
}

// ****************************************************************

LOCAL sos_Type lookup_schema_type (sos_String name)
{  sos_Type       result;
   sos_Type_descr temp;

   if (imp == NO_OBJECT || imp.card() == 0)
   {  static sos_String knlstr = sos_String::create (TEMP_CONTAINER, "knl");
      static sos_Schema_module knlmdl = sos_Schema_module::lookup (knlstr);

      result = knlmdl.lookup_type (name).make_type();
   }
   else
   {  agg_iterate (imp, sos_Schema_module i)
      {  temp = i.lookup_type (name, sos_IMP_NONE);
	 if (temp == NO_OBJECT) 
             result = sos_Type::make (NO_OBJECT);
         else 
             result = temp.make_type();
         if (result != NO_OBJECT)
	     break;
       }
      agg_iterate_end (imp, i);
   }
   return result;
}

// *****************************************************************

LOCAL stf_Symbol_rule make_symbol_rule (sos_String name, sos_Type ct)
{  stf_Symbol_rule sr = stf_Symbol_rule::create (cnt);
   sr.set_name (name);
   sr.set_schema_type (ct);
   sr.set_rules (stf_Structure_rule_List::create (cnt));
   symbol_table.insert (name, sr);
   return sr;
}

// *****************************************************************

LOCAL sos_Comp_descr identify_component (sos_String name)
{  sos_Comp_descr component;
   sos_Method  method_with_comp = ct.get_methods().lookup_comp(name, FALSE);

   if (method_with_comp == NO_OBJECT)
      { stf_error (err_USE, err_STF_NO_COMP, name);
        component = sos_Comp_descr::make(NO_OBJECT);
      }
   else
      component = method_with_comp.get_comp_descr();

   return component;
}

// *****************************************************************

LOCAL void check_item (stf_Item item, sos_Type tp, sos_String conversion)
{  if (item.has_type (stf_Item_with_strings_type))
   {  stf_Item_with_strings iws = stf_Item_with_strings::make (item);
      check_item (iws.get_item(), tp, conversion);
      item.set_schema_type (iws.get_item().get_schema_type());
   }
   else if (item.has_type (stf_Symbol_item_type))
   {  stf_Symbol_item si = stf_Symbol_item::make (item);
      sos_Type stp = si.get_symbol().get_schema_type();
      check_compatible (stp, tp, conversion);
      item.set_schema_type (stp);
   }
   else if (item.has_type (stf_Optional_item_type))
   {  stf_Optional_item oi = stf_Optional_item::make (item);
      check_item (oi.get_item(), tp, conversion);
      item.set_schema_type (oi.get_item().get_schema_type());
      aux_rules.append (oi);
   }
   else if (item.has_type (stf_Optional_strings_type))
   {  stf_Optional_strings os = stf_Optional_strings::make (item);
      item.set_schema_type (sos_Type::make (sos_Bool_type));
      aux_rules.append (os);
   }
   else // item.isa (stf_List_item_type)
   {  stf_List_item li = stf_List_item::make (item);
      sos_Bool error = TRUE;
      sos_Type elem_tp;
      if (tp.has_type (sos_Class_type_type))
      {  elem_tp = (sos_Class_type::make(tp))
	     .generic_param(sos_Class_type::make(sos_Object_List_type), 1);
	 if (elem_tp != NO_OBJECT)
	    error = FALSE;
      }
/*
      {  sos_Super_class_List scl =
	    sos_Class_type::make (tp).get_super_closure();
	 agg_iterate (scl, sos_Super_class sc)
	 {  if (sc.get_super_class().root_class()
				    .operator== (sos_Object_List_type))
	    {  sos_Generic_instantiation gi =
		  sc.get_super_class().get_generated_from();
	       if (gi == NO_OBJECT)
		  elem_tp = sos_Type::make (sos_Object_type);
	       else
		  elem_tp = gi.get_gen_params().get_nth(1).make_type();
	       error = FALSE;
	       break;
	    }
	 }
	 agg_iterate_end (scl, sc);
      }
*/
      if (error)
	 stf_error (err_USE, err_STF_NO_LIST_TYPE, tp.get_name());
      else
      {  check_item (li.get_repeated(), elem_tp, sos_String::make (NO_OBJECT));
	 li.set_elem_type (elem_tp);
      }
      item.set_schema_type (tp);
      aux_rules.append (li);
   }
}

// ********************************************************************

LOCAL void check_compatible (sos_Type stp, sos_Type tp, sos_String conversion)
{
  if (conversion != NO_OBJECT OR
       stp.equal(tp) OR //inserted 26-4-93 ;hack solution for bug
                        //error occured when stp=tp=sos_Bool
       stp.is_derived_from_some (tp.root()) OR
       (stp.is_derived_from (sos_String_type) AND
#ifdef ATT
	(tp.is_derived_from (sos_Named_type) OR tp.operator== (sos_Int_type))))
#else:	
        (tp.is_derived_from (sos_Named_type) OR tp == sos_Int_type)))
#endif
   {} // implicit or explicit conversion possible
   else
   {  smg_String s =
	 smg_String (stp.get_name()) + "," + tp.get_name();
      stf_error (err_USE, err_STF_INCOMPATIBLE_TYPES,
		 s.make_Cstring (SMG_BORROW));
   }
}

// ******************************************************************

LOCAL void foldCstring (char *c)
{  int l = strlen (c);
   int d = 1;
   for (int i = 1; i < l-1; i++)
   {  if (c[i] == '\\')
      {  d++;
	 i++;
	 if(c[i]=='n') c[i-d]='\n';
	 else if(c[i]=='t') c[i-d]='\t';
	 else c[i-d]=c[i];
      }
      else c[i-d]=c[i];
   }
   c[i-d] = 0;
}
