/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
/* ========================================================================= */
/* MODULE IMPLEMENTATION                                                     */
/* ========================================================================= */
/*                                                                           */
/* MODULE : syncronization (Main part)                                       */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* VERSION: none                                                             */
/* ========================================================================= */

#define OBST_IMP_STRINGOP
#define OBST_IMP_FORMATTED_IO
#define OBST_IMP_FILE
#define OBST_IMP_PROCESS
#define OBST_IMP_SOCKETS
#include "obst_stdinc.h"

#include "_obst_config.h"
 
#include "obst.h"
#include "obst_progstd.h"

#include "sync_obst.h"
#include "sync_decl.h"
#include "sync_err.h"
#include "sync_trc.h"

#include "trc_tsy.h"

#include "psm_util.h"


/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: Transaction                                                   */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

/* ================================================================ */
/* Method: Transaction::local_initialize()                          */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void Transaction::local_initialize (Transaction ta)

{
  T_PROC("Transaction::local_initialize");
  TT(tsy_H, T_ENTER);
  
  SyncCont       syct;                  // root_object of Sync_Container
  int            hpid;
  char*          env_path = obst_getenv(ENVVAR_CONTAINER);
  int            fd;
  char           path[MAXPATHLEN];
  struct utsname node;

  // get root_obj of SYNC_CONTAINER;
  syct = SyncCont::get_root();
  hpid = syct.get_HP();

  // create file to recognize process 
   
   // get path ENVVAR_CONTAINER/process to place the file
   if (env_path != 0)
     // 20 > length(process_id) + length('/process/')
     if (strlen(env_path) < MAXPATHLEN-20)
       { 
        strcpy(path, env_path);
        strcat(path,"/process/");
        sprintf(path + strlen(path),"%d",(int) hpid);
       }
      else err_raise(err_SYS, err_SYNC_PATH_TOO_LONG,ERR, FALSE);
    else err_raise(err_SYS, err_SYNC_NO_CONTAINER_PATH,ERR, FALSE); 

   // create file ( owner:rw, group:rw others:r )
   fd = ::open(path,O_RDWR | O_CREAT | O_EXCL,0664);
   if (fd == -1)
      err_raise(err_SYS, err_SYNC_PIDFILE_CREATE, ERR, FALSE);

   // lock the process-file (exclusive, testing)
   // -> flock(fd, LOCK_EX | LOCK_NB)
   if (!psm_lock_a_file(fd, WRITING, TESTING, psm_LOCK, FALSE))
     err_raise (err_SYS, err_SYNC_PIDFILE_LOCK, ERR, FALSE);

     
  // Create sets
  ta.set_openset  (sos_SOSet::create(SYNC_CONTAINER));
  ta.set_readset  (sos_SOSet::create(SYNC_CONTAINER));
  ta.set_writeset (sos_SOSet::create(SYNC_CONTAINER));

  ta.set_hostname (sos_String::create(SYNC_CONTAINER));
  
  // get hostname
  uname(&node);
  ta.get_hostname().assign_Cstring(node.nodename);

  // set variables
  ta.set_implicit     (FALSE);
  ta.set_hostproc_id  (hpid);
  ta.set_proc_fd      (fd);
  ta.set_user_id      (getuid());
  ta.set_status       (ACTIVE);
  ta.set_socket_port  (-1);
  ta.set_socket_seq_nr(0);
  TT(tsy_VL, TI(ta.get_hostproc_id()); TI(ta.get_user_id());
	     TI(ta.get_proc_fd()); TS(path);
             TS(ta.get_hostname().make_Cstring());
             TB(ta.get_implicit())); 
 

  // set timestamp
  ta.set_ts (syct.get_TS());
  TT(tsy_H, TI(ta.get_ts()));


  // put ta in mapping
  syct.insert_TA(ta);
  if (syct.get_error() != SYNC_OK)
    { ta.set_error(syct.get_error());
      TT(tsy_H, T_LEAVE);
      return; } 


  // end local_initialize
  ta.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: Transaction::local_finalize()                            */
/*                                                                  */
/* Error: SYNC_OK,                                                  */
/* ================================================================ */

void Transaction::local_finalize (Transaction ta)

{ 
  T_PROC("Transaction::local_finalize");
  TT(tsy_H, T_ENTER);
  TTA(ta);

  SyncCont      syct;                  // root_object of Sync_Container
  int           fd       = ta.get_proc_fd();
  char*         env_path = obst_getenv(ENVVAR_CONTAINER);
  char          path[MAXPATHLEN];
 

  // destroy sets
  ta.get_openset().destroy();
  ta.get_readset().destroy();
  ta.get_writeset().destroy();
  
 
  // get root_obj of SYNC_CONTAINER;
  syct = SyncCont::get_root();

  // remove ta from mapping
  syct.remove_TA(ta);
  if (syct.get_error() != SYNC_OK)
    { TT(tsy_H, T_LEAVE);
      return; } 

  
  // remove file, which was created in local_initialize
  if (ta.get_proc_fd() != -1)
    { // if the process, which started the transaction, is still running
      // ta.proc_fd is unequal -1. Because if the process is aborted,
      // ta.destroy is called from SyncCont::process_remove, where fd is
      // set to -1.
       // calculate path
       if (env_path != 0)
         // 20 > length(process_id) + length('/process/')
         if (strlen(env_path) < MAXPATHLEN-20)
           { strcpy(path, env_path);
             strcat(path,"/process/");
             sprintf(path + strlen(path),"%d", ta.get_hostproc_id());
           }
          else err_raise(err_SYS, err_SYNC_PATH_TOO_LONG,ERR, FALSE);
        else err_raise(err_SYS, err_SYNC_NO_CONTAINER_PATH,ERR, FALSE);

       // unlock,destroy, close  pid_file
       TT(tsy_VL, TI(fd); TS(path));
       // flock(fd, LOCK_UN | LOCK_NB);
       // Reading & Testing = dummy
       psm_lock_a_file(fd, READING, TESTING, psm_UNLOCK, FALSE); 
       unlink(path);
       ::close(fd);
    } 
  
  // end local_finalize
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: Transaction::is_accessed()                               */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

ta_Access_mode Transaction::is_accessed (SyncObj syob)
 
{ 
  T_PROC("Transaction::is_accessed");
  TT(tsy_H, T_ENTER);
  TSOBJ(syob);
  TTA(self);

  self.set_error(SYNC_OK);

  // search readset
  if (self.get_readset().is_element(syob))
    { TT(tsy_H, TXT("READ_AC"); T_LEAVE);
     return READ_AC; }


  // search writeset
  if (self.get_writeset().is_element(syob))
    { TT(tsy_H, TXT("WRITE_AC"); T_LEAVE);
      return WRITE_AC; }


  // container not in the sets
  TT(tsy_H, TXT("NO_AC"); T_LEAVE);
  return NO_AC;
}

/* ================================================================ */
/* Method: Transaction::is_released()                               */
/*                                                                  */
/* Error: SYNC_OK, SYNC_CONT_NO_FOUND                               */
/* ================================================================ */

sos_Bool Transaction::is_released (SyncObj syob)

{
  T_PROC("Transaction::is_released");
  TT(tsy_H, T_ENTER);
  TSOBJ(syob);
  TTA(self);

  // is syob in openset ?
  if (self.get_openset().is_element(syob))
    { // syob found 
      self.set_error(SYNC_OK);
      TT(tsy_H, TXT("FALSE"); T_LEAVE);
      return FALSE; }


  // does syncobj exists ?       
  if ((self.get_readset().is_element(syob)) ||
      (self.get_writeset().is_element(syob)))
    { // syncobj exists
      self.set_error(SYNC_OK); 
      TT(tsy_H, TXT("TRUE"); T_LEAVE);
      return TRUE;}
  
  
  // syob not in sets 
  self.set_error(SYNC_CONT_NO_FOUND);
  TT(tsy_H, TXT("SYNC_CONT_NO_FOUND"); T_LEAVE);
  return TRUE;
}

/* ================================================================ */
/* Method: Transaction::get_SO()                                    */
/*                                                                  */
/* Error: SYNC_OK, SYNC_WRONG_ACCESS                                */
/* ================================================================ */

void Transaction::get_SO (SyncObj syob, sos_Access_mode am)

{
  T_PROC("Transaction::get_SO");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TAM(am));
  TSOBJ(syob);
  TTA(self);

  sos_Bool         readfound;                // entry in readset ?
  sos_Bool         writefound;               // entry in writeset ?
  

  // controll parameter
  if ((am != READING) && (am != WRITING))
    { self.set_error(SYNC_WRONG_ACCESS);
      TT(tsy_H, TXT("SYNC_WRONG_ACCESS"); T_LEAVE); 
      return; }


  // search readset / writeset
  readfound  = self.get_readset().is_element(syob);  // readset
  writefound = self.get_writeset().is_element(syob); // writeset
  TT(tsy_VL, TB(readfound); TB(writefound));

  // READING / WRITING without previous access
  if ((!readfound) && (!writefound))
    { 
      // insert in sets
      if (am == READING)
        { TT(tsy_VL, TXT("insert in readset"));
          self.get_readset().insert(syob); }
       else
        { TT(tsy_VL, TXT("insert in writeset"));
          self.get_writeset().insert(syob); }


      // insert in openset
      TT(tsy_VL, TXT("insert in openset"));
      self.get_openset().insert(syob);


      // end it
      self.set_error(SYNC_OK);
      TT(tsy_H, T_LEAVE);
      return;
    }
    


  // READING changed to WRITING
  if ((readfound) && (am == WRITING))
     {
       // remove from readset
       TT(tsy_VL, TXT("remove from readset"));
       self.get_readset().remove(syob);

       // insert in writeset
       TT(tsy_VL, TXT("insert in writeset"));
       self.get_writeset().insert(syob);
     }
   

  // (re-) open container 
  if (!self.get_openset().is_element(syob))
    { TT(tsy_VL, TXT("insert in openset"));
      self.get_openset().insert(syob); }


  // end get_SO
  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: Transaction::release_SO()                                */
/*                                                                  */
/* Error: SYNC_OK, SYNC_CONT_NO_FOUND                               */
/* ================================================================ */

void Transaction::release_SO (SyncObj syob)

{
  T_PROC("Transaction::release_SO");
  TT(tsy_H, T_ENTER);
  TSOBJ(syob);
  TTA(self);

  // search openset 
  if (self.get_openset().is_element(syob))
    { 
      // remove found SyncObj
      TT(tsy_VL, TXT("remove from openset"));
      self.get_openset().remove(syob);
      self.set_error(SYNC_OK);
    }
   else
    { 
      // container not in sets
      self.set_error(SYNC_CONT_NO_FOUND);
      TT(tsy_H, TXT("SYNC_CONT_NO_FOUND"));
    }

  
  // end release_SO
  TT(tsy_H, T_LEAVE);
  return;
}
    


/**/
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: SyncObj (independent parts)                                   */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */


/* ================================================================ */
/* Method: SyncObj::local_initialize()                              */
/*                                                                  */
/* Error: SYNC_OK (syct.get_SO)                                     */
/* ================================================================ */

void SyncObj::local_initialize (SyncObj syob)

{
  // SyncCont        syct;

  T_PROC("SyncObj::local_initialize");
  TT(tsy_H, T_ENTER);

  // get root_obj of SYNC_CONTAINER;
  /* syct = SyncCont::get_root();

  // insert syob into mapping
  syct.get_SO (syob);
  if (syct.get_error() != SYNC_OK)
    { syob.set_error(syct.get_error());
      TT(tsy_H, T_LEAVE);
      return; } 
   $? */

  // end local_initialize
  syob.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: SyncObj::local_finalize()                                */
/*                                                                  */
/* Error: SYNC_OK, ??                                               */
/* ================================================================ */

void SyncObj::local_finalize (SyncObj syob)

{
  // SyncCont        syct;

  T_PROC("SyncObj::local_finalize");
  TT(tsy_H, T_ENTER);

  // get root_obj of SYNC_CONTAINER;
  /* syct = SyncCont::get_root();

  // remove syob from mapping
  syct.release_SO(syob);
  if (syct.get_error() != SYNC_OK)
    { syob.set_error(syct.get_error());
      TT(tsy_H, T_LEAVE);
      return; } 
  $? */

  // end local_initialize
  syob.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: SyncObj::get                                             */
/*                                                                  */
/* Error: SYNC_OK, SYNC_WRONG_ACCESS, SYNC_TA_ABORTED,              */
/*        SYNC_IMPLTA_ABORTED                                       */
/* ================================================================ */

sos_Bool SyncObj::get (sos_Access_mode am, sos_Sync_mode sm)

{ 
  T_PROC("SyncObj::get");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TAM(am); TSM(sm));
  TSOBJ(self);

  SyncCont         syct;
  Transaction      ta; 
  LockingTA        tplta;
  sos_HostProc_Id  hpid;
  sos_HostProc_Id  tapid;
  sos_Open_result  result;
  sos_Bool         access_delay = TRUE;
  sos_Bool         okay;
  sos_Bool         impl_firstcont = FALSE;
  ta_Access_mode   ct_am;

  // control parameter 
  if ((am != READING) && (am != WRITING))
    { self.set_error(SYNC_WRONG_ACCESS);
      TT(tsy_H, TXT("SYNC_WRONG_ACCESS"); T_LEAVE);
      return FALSE; }

  while (access_delay)
    {     
      access_delay = FALSE; 
       
      syct = SyncCont::get_root();

      // look for transaction 
      hpid = syct.get_HP();
      ta  = syct.get_ta_tab()[hpid];
      TT(tsy_VL, TI(hpid));
      TTA(ta);

      // is transaction intern aborted ?
      if (VALID(ta) && (ta.get_status() == ABORTED))
        { // nothing to do
          self.set_error(SYNC_OK);
          TT(tsy_H, TXT("nothing to do -> TRUE"); T_LEAVE);
          return TRUE; }
 

      if ((self.get_awaking()) &&
          (INVALID(ta) || (ta.get_type() != OPTIMISTIC)))
        { // another ta was waiting to access this SyncObj and to this ta a
          // message was sent that it can access the SyncObj. But as long
          // as awaking is set, the ta has not accessed the SyncObj, it has not
          // awake. The other ta, which has accessed the SyncObj before has 
          // probably released all locks, so that to keep evrything in order
          // we have to wait, till the other ta has accessed the Obj, 
          // under the assumption that the process was not aborted.
          
          if (!self.get_queue().is_empty())
            {    
              tapid = self.get_queue().get_nth(1).get_ta().get_hostproc_id();
              okay  = syct.process_check(tapid);
              if (!okay) 
                 syct.process_remove(tapid);
              else
              { // close Sync_Container and open it to give the awaking ta
                // the possibility to access the Sync_Container
                
                SYNC_CONTAINER.close();
      
                result=SYNC_CONTAINER.open(WRITING, WAITING);
                if (result != OPENED)
                   err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE);

                access_delay = TRUE;
              } 
            }
            else
              self.set_awaking(FALSE); 
        } 
    }  

  // if no transaction, create one implicit
  if INVALID(ta) 
    { TT(tsy_L, TXT("create implicit TA"));
      ta = LockingTA::create(SYNC_CONTAINER);
      if INVALID(ta) 
        { TT(tsy_H, TXT("SYNC_NO_CREATE_IMPLICIT_TA -> FALSE"); T_LEAVE);
          err_raise(err_SYS, err_SYNC_NO_CREATE_IMPLICIT_TA, ERR, FALSE); 
        }
      ta.set_implicit(TRUE);
      impl_firstcont = TRUE;
      TTA(ta);
    }
 

  // container already in set of transaction ?
  ct_am = ta.is_accessed (self);
  TT(tsy_VL, TXT("CT:"); TTM(ct_am); TXT("AM:"); TAM(am));

  // change to higher access
  if ((ct_am == READ_AC) && (am == WRITING))
    {
      if (ta.get_type() == LOCKING)
        { okay = self.tpl_writelock(LockingTA::make(ta), sm);
          if (!okay)
            { TT(tsy_H, TXT("FALSE"); T_LEAVE); 
              return FALSE; } 
        }     

      // move from readset to writeset
      ta.get_SO (self, am);
      self.set_error(ta.get_error());
      TT(tsy_H, TB(self.get_error()==SYNC_OK); T_LEAVE);
      return sos_Bool(self.get_error() == SYNC_OK);
    }

      
  // first access to container by transaction
  if ((ct_am == NO_AC) && ((am == READING) || (am == WRITING)))
    { 
       if (am == READING) 
	 okay = self.tpl_readlock(LockingTA::make(ta), sm);
       else
	 okay = self.tpl_writelock(LockingTA::make(ta), sm); 


      // if read failed, return
      if (!okay) 
        { // implict transaction objects have to be destroyed if the 
          // transaction was intern aborted or if the implict transaction just
          // has been started at the beginning of this method and the SyncObj,
          // which should be opened TESTING, failed to open.
          if ((ta.get_implicit()) && 
             ((impl_firstcont) || (self.get_error() == SYNC_IMPLTA_ABORTED)))
            { TT(tsy_L, TXT("Implicit TA intern aborted -> destroy"));
              tplta = LockingTA::make(ta);
              tplta.destroy(); }
          TT(tsy_H, TXT("FALSE"); T_LEAVE);
          return FALSE; } 

      ta.get_SO (self, am);
      self.set_error(ta.get_error());
      TT(tsy_H, TB(self.get_error() == SYNC_OK); T_LEAVE);
      return sos_Bool(self.get_error() == SYNC_OK);       
    }      



  // the other cases: ct_am = READ_AC && am == READING
  //                  ct_am = WRITE_AC && (am == READING || am == WRITING)

  // reopen if already released
  ta.get_SO(self, am);
  if (ta.get_error() != SYNC_OK)
    { self.set_error(ta.get_error());
      TT(tsy_H, TXT("FALSE"); T_LEAVE);
      return FALSE; }   


  // access is already given
  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return TRUE;      
}   

/* ================================================================ */
/* Method: SyncObj::release()                                       */
/*                                                                  */
/* Error: SYNC_OK, SYNC_NO_GET, SYNC_TA_ABORTED (ts),               */
/* ================================================================ */

void SyncObj::release ()

{ 
  T_PROC("SyncObj::release");
  TT(tsy_H, T_ENTER);
  TSOBJ(self);

  SyncCont        syct;
  Transaction     ta;
  LockingTA       tplta; 
  sos_HostProc_Id hpid;
  sos_HostProc_Id tapid;
  sos_Open_result result;
  sos_Bool        access_delay = TRUE;
  sos_Bool        okay;
  ta_Access_mode  ct_am;


  while (access_delay)
    { 
      access_delay = FALSE;

      syct = SyncCont::get_root();

      // look for transaction 
      hpid = syct.get_HP();
      ta  = syct.get_ta_tab()[hpid];  
      TT(tsy_VL, TI(hpid));
      TTA(ta);

      // if no transaction, no container open
      if INVALID(ta) 
        { self.set_error(SYNC_NOT_GET);
          TT(tsy_H, TXT("SYNC_NOT_GET"); T_LEAVE);
          return; }


      // is transaction intern aborted ?
      if (ta.get_status() == ABORTED)
        { // nothing to do
          self.set_error(SYNC_OK);
          TT(tsy_H, TXT("intern aborted - nothing to do"); T_LEAVE);
          return; }


      if ((self.get_awaking()) &&
          (ta.get_type() != OPTIMISTIC))
        { // another ta was waiting to access this SyncObj and to this ta a
          // message was sent that it can access the SyncObj. But as long
          // as awaking is set, the ta has not accessed the SyncObj, it has not
         // awake. The other ta, which has accessed the SyncObj before has
          // probably released all locks, so that to keep evrything in order
          // we have to wait, till the other ta has accessed the Obj,
          // under the assumption that the process was not aborted.
         
          if (!self.get_queue().is_empty())
            {
              tapid = self.get_queue().get_nth(1).get_ta().get_hostproc_id();
              okay  = syct.process_check(tapid);
              if (!okay)
                 syct.process_remove(tapid);
              else
              { // close Sync_Container and open it to give the awaking ta
                // the possibility to access the Sync_Container
 
                SYNC_CONTAINER.close();
 
                result=SYNC_CONTAINER.open(WRITING, WAITING);
                if (result != OPENED)
                   err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE);
 
                access_delay = TRUE;
              }
            }
            else
              self.set_awaking(FALSE);
        }
    } 

  // get access mode
  ct_am = ta.is_accessed(self);
  TT(tsy_L, TXT("CT:"); TTM(ct_am));
  if (ct_am == NO_AC)
    { self.set_error(SYNC_NOT_GET);
      TT(tsy_H, TXT("SYNC_NOT_GET"); T_LEAVE);
      return; }

  // remove from openset
  ta.release_SO(self);
  if (ta.get_error() != SYNC_OK)
    { self.set_error(ta.get_error());
      TT(tsy_H, T_LEAVE);
      return; }


  // commit implicit transactions
  // !! beware of  _TA aborted implicitly somewhere before 
  TT(tsy_VL, TI(ta.get_openset().card()));
  if (ta.get_implicit())
    { if (ta.get_openset().is_empty())
	{ tplta = LockingTA::make(ta);
	  tplta.commit();
	  tplta.destroy();
	}
      else
      if (!ta.get_writeset().is_empty())
	{ sos_SOSet helpset = sos_SOSet::create(TEMP_CONTAINER);
          helpset += ta.get_openset();
          helpset *= ta.get_writeset();
          if (helpset.is_empty())
            { tplta = LockingTA::make(ta);
              tplta.commit();
            } 
        }
    }
  

  // end release
  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}  

