#ifndef _PSM_UTIL_H
#define _PSM_UTIL_H 1
/* -----------------------------------------------------------------------------
* Copyright 1992-1994 by Forschungszentrum Informatik(FZI)
*
* You can use and distribute this software under the terms of the license
* you should have received along with this software.
* If not or if you want additional information, write to
* Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
* D-76131 Karlsruhe, Germany.
* -----------------------------------------------------------------------------
*/
// ****************************************************************************
// Module psm                 Emil Sekerinski, Oliver Spatscheck, Walter Zimmer
//
// ****************************************************************************
//  persistent storage manager: header utility implementation 
// ****************************************************************************

// ****************************************************************************
// INCLUDE FILES
// ****************************************************************************
 
#include "psm_err.h"

// ================ obst files ================================================

#include "obst_progstd.h"

#define OBST_IMP_FILE
#include "obst_stdinc.h"

#include "trc_psm.h"
#include "obst_stats.h"

#ifdef MONITOR
#include "psm_mon.h"     /* mon_win, mon_create, mon_black, ... */
#endif                   /* ... mon_white, mon_close            */

// ****************************************************************************
// CONSTANT DATA DEFINITIONS
// ****************************************************************************


// The following extern constant definitions are implemented as #define ...
// ... because of problems with the initialization in dynamic link ...
// ... libraries.

#define psm_C      CTTABLE_SIZE
				 // container table size ... 
                                 // ...(max. number of open containers)
#define psm_UNUSED 0
				 // used for cnt_id's(value must be zero !)
#define psm_UNLOCK 1
				 // constants for funtion psm_lock_a_file()
#define psm_LOCK   0 
#define psm_UNDEF  -1
				 //  used for fileindex, bufferindex ... 
                                 //  ...(also for fd in open()) to test, if ... 
                                 //  ...object still exists or not(see list ...
                                 //  ... free blocks)

#define psm_SZ_OF_INT SOS_INT_SIZE
				 // if this is not valid => ...
                                 // ... psm must be rewritten(!)
#define psm_SW_OFFSET 0
				 // offset of switch in containerfile
#define psm_P         1024
                                 // cnt_pg size in bytes
                                 // ROOT_OFFSET in obst_globals.C depends on
				 // psm_P !
#define psm_MAX_FREEBLOCK_LENGTH   (psm_P-1*psm_SZ_OF_INT) 
                                 // maximum for the length ...
                                 // ... of a free block

#define psm_N         ((psm_P/psm_SZ_OF_INT-4)/2)
				 // =#pagetable entries per page and version


extern const err_msg psm_ERR;

// ****************************************************************************
// TYPEDEFS AND ENUMS 
// ****************************************************************************

enum psm_end_operation           { NOP, SQUEEZE, COMPRESS };
enum psm_precondition_for_lookup { NO_PERM, READ_PERM, WRITE_PERM };
typedef char psm_PAGE[psm_P];

// ****************************************************************************
// STRUCTURE DECLARATIONS
// ****************************************************************************

#ifndef NO_TT
struct psm_cnt_stat		// container statistics
{
    int pgs_written;		// primary + secondary pages written to disk
    int pgs_read;		// primary + secondary pages read from disk
    int pgs_in_buffer;		// primary pages currently in page buffer
    int pgs_modified;		// buffered pages which are modified
    int pgs_secondary;		// secondary pages on disk
    int pgs_logical;		// logical container size (primary pages)
    int pgs_shadowed;		// shadowed primary pages
    int pgs_free;		// free primary pages
    int mem_secondary;		// secondary data in memory [bytes]

    obst_histentry* hist_wr;	// size distribution of written blocks
    int		    hist_wr_sz;
    obst_histentry* hist_rd;	// size distribution of read blocks
    int		    hist_rd_sz;
};
#endif

struct psm_perf_ctrls		// parameters for container performance control
{
    sos_Bool do_autosqueeze;	// true if automatic squeezing is to be done
    double   min_occupation;	// threshold for automatic squeezing
};

struct psm_pg_info              // info for each page of the pagetable
{
    sos_Bool is_mod;            // true if filetable was modified
    sos_Bool cur_sel_sw;        // this two sw select the correct part of ...
    sos_Bool old_sel_sw;        // ... the pagetablepage(the correct bit is ...
};                              // ... selected by the pagetable switch) 
struct psm_cnt_info 
{
   sos_Int id;                  // 1<=id<2^32-1 or id==UNUSED
   sos_Container_status stat;   // container status: READABLE or WRITEABLE ...
                                // ... or CHECKEDOUT
   sos_Container_status stat_ta;// most restrictive status of container ... 
                                // ... during transaction
   sos_Bool is_cnt_open;        // is container logical open ?
   sos_Bool is_cnt_created;     // is container opened the first time ? ...
                                // ... stat_ta and ct_open are of use ... 
                                // ... only if syncactivated
   int fd;                      // file descriptor of container file
   int fd_lock;                 // file descriptor of lock file
   sos_Int sw;                  // upper/lower pagetable switch for current ... 
                                // ... version
   int ub_free_sz;              // upper bound(approximative) of the size ...
                                // ... of all free blocks, used for ...
                                // ... searching free blocks more efficiently
   sos_Int cnt_pgs;             // number of container pages of current ...
                                // ... version(> highest logical container... 
                                // ... page)
   int cnt_tbl_sz;              // size of the three subsequent arrays, ...
                                // ... must be multiple of N(>= cnt_pgs)
   int *buf_index;              // maps [0..cnt_pgs) to [0..B)+psm_UNDEF
   int *file_index;             // maps [0..cnt_pgs) to [0..file_pgs)+ ...
                                // ... psm_UNDEF (page table)
   sos_Bool *is_shadowed;       // maps [0..cnt_pgs) to bool ...
                                // ...(indicates, if shadow page exist)
   int file_pgs;                // file pages occupied by the current ...
                                // ... container file(upper bound for ...
                                // ... reference +1 from fileindex to ... 
                                // ... containerpage)
   int file_tbl_sz;             // size of freefilepage array(>= file_pgs)
   psm_pg_info *pg_tbl_info;    // maps [0..max number of pagetablepages) ...
                                // ... to pageinfo
   sos_Bool *free_file_pg;      // maps [0..file_pgs) to bool, i.e., ...
                                // ... indicates the free ...
                                // ... pages in the container file
   int free_file_pg_index;      // index to make search for next ...
                                // ... freefilepage more efficient; points ...
                                // ... to last found empty page(in the ...
                                // ... beginning => -1)
   int last_read_time;          // it is always set to the last read time ... 
                                // ... of the file when the status changes ...
                                // ...  to READING(open->entercontainer, ...
                                // ... access).
   psm_end_operation operation_at_end;
                                // any operation that should be done on ...
                                // ... the container when it is closed ...
                                // ... via write_close. It is only used ...
                                // ... when syncactivated = TRUE. Operations...
                                // ... are squeeze(autosqueeze), compress ...
                                // ... and nop. 
   psm_perf_ctrls* ctrls;	// container local performance controls
#ifndef NO_TT
   psm_cnt_stat stats;
#endif

#ifdef MONITOR
   mon_win win;
#endif
};

struct psm_buf_info             // info for each page in buf[0..B)
{
    sos_Int id;
    int pg;                     // backward reference to container id and ... 
                                // ... container page, used for page ...
                                // ... replacement: id==UNUSED || (id== ... 
                                // ... containertable[?].id && 0<=page< ...
                                // ... containertable[?].contpages)
    sos_Bool is_ref;            // used for page replacement(second chance ...
                                // ... algorithm)
    sos_Bool is_mod;            // indicates that page has to be written on ...
                                // ... commit
};

// ****************************************************************************
// EXTERNAL DATA REFERENCES 
// ****************************************************************************

extern psm_PAGE*     psm_buf;
extern psm_buf_info* psm_buf_tbl;
extern psm_cnt_info  psm_cnt_tbl[psm_C];
extern sos_Bool      psm_is_deadlock;
extern sos_Bool      psm_is_syncactivated;

#ifndef NO_TT
extern psm_cnt_stat  psm_cnt_stats_global;
#endif

// ****************************************************************************
// MACROS 
// ****************************************************************************

#define nothing 
				 // necessary for hp-ux

#define psm_return_if_id0(test_id, returnvalue) \
		if ((sos_Int)(test_id)==0) \
                       { TT(psm_H, T_LEAVE); return returnvalue;}
#define psm_return_if_destroyed(status, returnvalue) \
		if ((status)==DESTROYED) \
		   { TT(psm_H, T_LEAVE); return returnvalue;} 
#define psm_return_if_id0_no_trc(test_id, returnvalue) \
                if ((sos_Int)(test_id)==0) return returnvalue; 
#define psm_return_if_destroyed_no_trc(status, returnvalue) \
		if ((status)==DESTROYED) return returnvalue; 
#ifdef OBST_HAVE_JOYCE
#  define psm_return_if_psm_deadlock(returnvalue, trc) \
        if ((psm_is_syncactivated) && (psm_is_deadlock) && \
            (id != SYNC_CONTAINER)) \
           { TT((trc ? psm_H : psm_M), T_LEAVE); return returnvalue; }
#  define psm_return_if_deadlock(returnvalue, trc) \
        if ((psm_is_syncactivated) && sync_deadlock && (id != SYNC_CONTAINER)) \
           { TT((trc ? psm_H : psm_M), T_LEAVE); return returnvalue; }
#else
#  define psm_return_if_psm_deadlock(returnvalue, trc)
#  define psm_return_if_deadlock(returnvalue, trc)
#endif

#define psm_number_in_pg_tbl_info(cnt_pg) ((cnt_pg) / psm_N)
           // calculates the index of the pagetablepage in which the ...
           // ... containerpage is mapped

// ****************************************************************************
// FUNCTION PROTOTYPES
// ****************************************************************************

// ================= initalize functions ======================================

void _psm_initialize();
void _psm_final_initialize();

// ================= auxiliary functions ======================================

LOCAL inline int psm_min(int a, int b) 
{ return(a < b ? a : b); }

LOCAL inline int psm_max(int a, int b) 
{ return(a > b ? a : b); }

//-----------------------------------------------------------------------------

void* psm_extend_tbl(void* tab,
                     unsigned unit, unsigned sz, unsigned new_sz);

int psm_preempt_pg(sos_Int id, int cnt_pg);

int psm_find_free_file_pg(psm_cnt_info &ci);

// ================= freelist functions =======================================

void     psm_print_f_lists(psm_cnt_info &ci);
void     psm_insert_in_f_list(psm_cnt_info &ci, unsigned sz, sos_Offset offset);
unsigned psm_rm_first_from_f_list(psm_cnt_info &ci, unsigned sz);
void     psm_check_f_list(psm_cnt_info &ci, int start, int length);


// ================= psm functions ============================================

// ----------------- calculation functions ------------------------------------

unsigned psm_occupied_bytes(psm_cnt_info &ci);

LOCAL inline int psm_offrounded_pg_number(int sz)     // returns floor(size/P)
{ return(int)(sz / psm_P); }

LOCAL inline int psm_uprounded_pg_number(int sz)     // returns ceil(size/P)
{ return(int)((sz + psm_P - 1) / psm_P); }

LOCAL inline sos_Offset psm_four_upround(sos_Offset sz)
{ return(sos_Offset)(((sz+3) /4) *4); }	// returns the next multiple of 4

LOCAL inline  int psm_file_position(int file_pg) 
{   

   // The term filepage/N + 1 adjusts for the pages of the "page table" ...
   // ... scattered in  the container file.
   // pages(= filepage) +  pages for pagetable(= filepage/N)
   //                     +  rootoffset(1 = table of free blocks) 

   return(file_pg + file_pg/psm_N + 1) * psm_P; 
}

// ----------------- time functions -------------------------------------------

void psm_set_last_read_time(psm_cnt_info &ci);
void psm_set_last_read_time(psm_cnt_info &ci, int time);

// ----------------- path functions -------------------------------------------
 
char* psm_cnt_path(sos_Int id, int path_for_lockfile = FALSE) ;
char* psm_cnt_path_lock(int id);

// ----------------- lock and lock auxiliary functions ------------------------

void            psm_check_lock_method();
int             psm_downgrading(sos_Container_status stat, sos_Access_mode am);
sos_Open_result psm_lock(int fd, sos_Access_mode am, sos_Sync_mode sm,
                         sos_Container_status previous_stat, int id,
                         int &fd_lock);
int             psm_lock_a_file(int fd, sos_Access_mode am, sos_Sync_mode sm, 
                                int lock_mode = psm_LOCK, 
                                sos_Bool is_lockfile = FALSE);
#ifdef OBST_HAVE_JOYCE
int             psm_sync_lock_a_file(sos_Access_mode am, sos_Sync_mode sm,
                                     int id, int lock_mode = psm_LOCK);
#endif
int             psm_upgrading(sos_Container_status stat, sos_Access_mode am);

// ----------------- statistics functions -------------------------------------

#ifndef NO_TT
psm_cnt_stat* psm_get_stats (sos_Int id);
void	      psm_clear_stats (sos_Int id);
void	      psm_size_stats (psm_cnt_info &ci, sos_Int sz, sos_Bool reading);
char*         psm_print_stats (psm_cnt_stat*);
#endif

// ----------------- page and integer functions -------------------------------

inline
void psm_read_pg(psm_cnt_info &ci, int cnt_pg, int buf_pg)
{
   // reads one page from file to buffer 

   int file_pg = ci.file_index[cnt_pg];
   if (file_pg == psm_UNDEF) 
      err_raise(err_SYS, err_PSM_WRONG_OFFSET, psm_ERR, FALSE);

   lseek(ci.fd, psm_file_position(file_pg), SEEK_SET);
   read(ci.fd,(char *)psm_buf[buf_pg], psm_P);

TTN (psm_CNT_STATS,
	 ++ ci.stats.pgs_read;
	 ++ psm_cnt_stats_global.pgs_read;)
}

inline
void psm_write_pg(psm_cnt_info &ci, int cnt_pg, int buf_pg)
{
   // Writes one page from buffer to file. If this page isn't yet shadowed ... 
   // ... then a new free page must be found.

   int &file_pg = ci.file_index[cnt_pg];
   if (file_pg == psm_UNDEF)
      err_raise(err_SYS, err_PSM_WRITEPAGE, psm_ERR, FALSE);

   if (! ci.is_shadowed[cnt_pg])
   {   
      file_pg = psm_find_free_file_pg(ci);
      ci.pg_tbl_info[psm_number_in_pg_tbl_info(cnt_pg)].is_mod = TRUE;
      ci.is_shadowed[cnt_pg] = TRUE; 
   }
   lseek(ci.fd, psm_file_position(file_pg), SEEK_SET);
   write(ci.fd,(char *)psm_buf[buf_pg], psm_P);

TTN (psm_CNT_STATS,
	 ++ ci.stats.pgs_written;
	 ++ psm_cnt_stats_global.pgs_written;)
}

//-----------------------------------------------------------------------------

inline
int psm_lookup_pg(psm_cnt_info &ci, int cnt_pg) 
{
   // Returns index of buffer page of container ci's contpage.

   // If contpage is not already in buffer then  read it from the ... 
   // ... container file. It can be necessary to swap(preemptpage) ...
   // ... a bufferpage of another container. The bufferpage is marked as ...
   // ...  referenced according the second chance algorithm(this happens ... 
   // ... only here).

   if (cnt_pg >= ci.cnt_pgs) 
      err_raise(err_SYS, err_PSM_WRONG_OFFSET, psm_ERR, FALSE);

   int buf_pg;
   if ((buf_pg = ci.buf_index[cnt_pg]) == psm_UNDEF)
   {   
      buf_pg		   = psm_preempt_pg(ci.id, cnt_pg);
      ci.buf_index[cnt_pg] = buf_pg;
#ifdef MONITOR
      if (mon_activated)
	 mon_black(ci.win, cnt_pg);
#endif
      psm_read_pg(ci, cnt_pg, buf_pg); 
   }
   psm_buf_tbl[buf_pg].is_ref = TRUE;
   return buf_pg; 
}


inline
sos_Int psm_read_int(psm_cnt_info &ci, sos_Offset o)
{
   // PRE: o%P+SIZEOF_INT<=P, i.e. data does not cross page boundary

   return *(sos_Int*) &psm_buf[psm_lookup_pg(ci, o/psm_P)][o%psm_P]; 
}

inline
void psm_write_int(psm_cnt_info &ci, sos_Offset o, sos_Int data) 
{
   // PRE: o%P+SIZEOF_INT<=P, i.e. data does not cross page boundary

   int buf_pg = psm_lookup_pg(ci, o/psm_P);
   *(sos_Int*) &psm_buf[buf_pg][o%psm_P] = data;
   psm_buf_tbl[buf_pg].is_mod = TRUE;
}

inline
void psm_read_from_pg(psm_cnt_info &ci, int pg, int disp, int len, void *data)
{
   // PRE : data does not cross page boundary
   // err_assert(disp + len <= P, "PSM:psm_read_from_pg");  // efficient ?

   memcpy(data, &psm_buf[psm_lookup_pg(ci, pg)][disp], len); 
}

inline
void psm_write_to_pg(psm_cnt_info &ci, int pg, int disp, int len, void *data)
{
   // PRE : data does not cross page boundary
   // err_assert(disp + len <= P, "PSM:writetopage");

   int buf_pg = psm_lookup_pg(ci, pg);
   memcpy(&psm_buf[buf_pg][disp], data, len);
   psm_buf_tbl[buf_pg].is_mod = TRUE;
}

// ----------------- pagetable functions --------------------------------------
 
sos_Offset psm_alloc_pg(psm_cnt_info &ci, int n);
void       psm_dealloc_pg(psm_cnt_info &ci, int p, int n);

// ----------------- container functions --------------------------------------
 
sos_Bool            psm_is_equal_cnt(psm_cnt_info ci1, psm_cnt_info ci2);
sos_Existing_status psm_check_object_exists(int id, sos_Offset o, sos_Int size);
psm_cnt_info*       psm_cnt_tbl_position(sos_Int id);
void                psm_commit_cnt(psm_cnt_info &ci);
void                psm_compress_cnt(psm_cnt_info &ci);
sos_Bool            psm_is_cnt_open(sos_Int id);
void                psm_enter_cnt(sos_Int id, int fd, sos_Container_status 
                                  stat, int fd_lock,
                                  sos_Bool ct_create = FALSE);
psm_cnt_info        &psm_lookup_cnt(int id, 
                                    psm_precondition_for_lookup 
                                       precondition_stat,
                                    sos_Bool is_destroyed_permitted = FALSE,
                                    sos_Bool is_open_needed = FALSE);
int                 psm_is_mod_since_last_commit(psm_cnt_info &ci);
int                 psm_is_mod_since_last_read(psm_cnt_info &ci);
void                psm_rm_cnt(psm_cnt_info &ci,
                               int close_filedescriptors = TRUE);
void                psm_reset_cnt(psm_cnt_info &ci);
void                psm_squeeze_cnt(psm_cnt_info &ci);
sos_Bool            psm_is_squeeze_possible(psm_cnt_info&);
void                psm_updating(psm_cnt_info &ci);

#endif /* _PSM_UTIL_H */
