/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
// **************************************************************************
// Module prot                   13-Aug-1993	       Bernhard Schiefer (bs)
// **************************************************************************
// implements methods of all classes declared in schema "prot"
// **************************************************************************
 
#include "obst_progstd.h"
#include "obst.h"
#include "smg.h"
#include "trc_prot.h"

#include "prot_obst.h"
#include "prot_err.h"

extern sos_Container_set& prot_open_cnts();              // open containers

// -------------------------------------------------------------------------
// sos_SchemaModif
// -------------------------------------------------------------------------
// *************************************************************************
void sos_SchemaModif::local_initialize(sos_SchemaModif sm)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::local_initialize");
   TT(prot_M, T_ENTER);

   sos_Container    cnt = sm.container();

   sm.set_modifs         (sos_Type_descr_sos_TypeModif_Mapping::create(cnt));
   sm.set_change         (PROT_NONE);
   sm.set_added_imports  (sos_Import_Set::create(cnt));
   sm.set_deleted_imports(sos_Import_Set::create(cnt));

   TT(prot_M, T_LEAVE);
}

// *************************************************************************
void sos_SchemaModif::local_finalize(sos_SchemaModif sm)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::local_finalize");
   TT(prot_M, T_ENTER);

   sos_ModifHistory  mh = sos_ModifHistory::modif_history();
   sos_Schema_module sc = sm.get_modified_schema();
   sos_SchemaHistory sh = mh.get_history()[sc];

   sos_Version_sos_SchemaModif_Mapping  sm_map;
   sos_Version                          ver;
   sos_SchemaModif                      smod;
   sos_Version                          finver = 0;
   
   sos_Type_descr_sos_TypeModif_Mapping tm_map;
   sos_Type_descr                       td;
   sos_TypeModif                        tmod;


   // remove SchemaModif from SchemaHistory
   sm_map = sh.get_schema_modifs();

   agg_iterate_association(sm_map, ver, smod)
   {
      if (smod == sm)
      {  finver = ver;
         break;
      }
   } agg_iterate_association_end(sm_map, ver, smod); 

   if (finver != 0)
      sm_map.remove(finver);
   else
      err_raise(err_SYS, err_PROT_SCHEMAMODIF_NOT_KNOWN, prot_ERR, FALSE);

  
   // remove TypeModifs 
   tm_map = sm.get_modifs();

   agg_iterate_association(tm_map, td, tmod)
   { tmod.destroy();
   } agg_iterate_association_end(tm_map, td, tmod); 

   tm_map.destroy();


   // remove old_name
   if (VALID(sm.get_old_name()))
      sm.get_old_name().destroy();
 
   
   // remove imports
   sm.get_added_imports().destroy();
   sm.get_deleted_imports().destroy();
   // <af> was passiert mit Inhalt des Sets 

   TT(prot_M, T_LEAVE);
}

// *************************************************************************
void sos_SchemaModif::add_schema(sos_Schema_module sc)
// *************************************************************************
{ 
   T_PROC("sos_SchemaModif::add_schema");
   TT(prot_H, T_ENTER);

   sos_SchemaModif sm = sos_SchemaModif::schema_modif(sc);

   if (VALID(sm))
   {
      if (sm.get_change() == PROT_DEL)
      {  // schema was deleted before and added again
         sm.set_change(PROT_MOD);
      }
      else if (sm.get_change() != PROT_NONE)
      {  // schema has been added or modified during the version 
         err_raise(err_SYS, err_PROT_SCHEMA_ADDMOD, prot_ERR, FALSE);
      }
      else sm.set_change(PROT_ADD);
   }

   TT(prot_H, T_LEAVE);
}
 
   
// *************************************************************************
void sos_SchemaModif::del_schema(sos_Schema_module sc)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::del_schema");
   TT(prot_H, T_ENTER);

   sos_SchemaModif sm = sos_SchemaModif::schema_modif(sc);

   if (VALID(sm))
   {
      if (sm.get_change() == PROT_ADD)
      {  // schema was added during this version
         sm.destroy();
      }
      else sm.set_change(PROT_DEL); 
   }
   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void sos_SchemaModif::ren_schema(sos_String name)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::ren_schema");
   TT(prot_H, T_ENTER);

   sos_Container     cnt = self.container();
   sos_Schema_module sc  = self.get_modified_schema(); 

   if (INVALID(self.get_old_name()))
     if (VALID(sc.get_name()))
        self.set_old_name(sos_String::clone(sc.get_name(), cnt));
     else
        self.set_old_name(sos_String::create(cnt)); 
 
   self.set_change(PROT_MOD);

   TT(prot_H, T_LEAVE);
}

// *************************************************************************
sos_SchemaModif sos_SchemaModif::mod_schema(sos_Schema_module sc)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::mod_schema");
   TT(prot_H, T_ENTER);

   sos_SchemaModif sm = sos_SchemaModif::schema_modif(sc);

   if (VALID(sm))
   {  if ((sm.get_change() == PROT_ADD) || (sm.get_change() == PROT_DEL))
         sm = sos_SchemaModif::make(NO_OBJECT);
   }

   TT(prot_H, T_LEAVE);
   return sm;
}

// *************************************************************************
sos_SchemaModif sos_SchemaModif::schema_modif(sos_Schema_module sc)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::schema_modif");
   TT(prot_M, T_ENTER);

   static sos_ModifHistory   mh = sos_ModifHistory::modif_history();

   sos_SchemaHistory  sh;
   sos_SchemaModif    sm;
   sos_Container      s_cnt;
   sos_Container      m_cnt = mh.container();
   sos_Version        ver;
   sos_Container_set& cnt_set = prot_open_cnts();      

    
   // check whether protocol information is to be generated for this
   // schema
   if (mh.get_not_gen_prot().is_element(sc)) 
   {  m_cnt.close();
      TT(prot_M, T_LEAVE);
      return sos_SchemaModif::make(NO_OBJECT);
   }


   // search for SchemaHistory
   ver = mh.act_version();
   sh  = mh.get_history()[sc];
   
   if (INVALID(sh))
   {  s_cnt = sos_Container::create();
      sh    = sos_SchemaHistory::create(s_cnt);   

      m_cnt.open(WRITING, WAITING);
      mh.get_history().insert (sc, sh);        
      s_cnt.commit();
   } 
   else
      s_cnt = sh.container();

   m_cnt.close();
   s_cnt.open(WRITING, WAITING);
   cnt_set+=(s_cnt);


   // search for SchemaModif
   sm = sh.get_schema_modifs()[ver];
 
   if (INVALID(sm))
   {  sm = sos_SchemaModif::create(s_cnt, sc, ver);
      sh.get_schema_modifs().insert(ver, sm);
      sm.set_change(PROT_NONE);
   } 

   TT(prot_M, T_LEAVE);
   return sm;
} 
    
// *************************************************************************
void sos_SchemaModif::add_type(sos_Type_descr type)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::add_type");
   TT(prot_H, T_ENTER);

   sos_TypeModif tm = self.type_modif(type);

   if (tm.get_change() == PROT_DEL) 
   {  // if the type just has been deleted, it can be added again 
      // without any problems  
       
      tm.set_change(PROT_MOD);
 
      // the tm is not destroyed, because the type might have been
      // changed before the deletion
   }
   else if (tm.get_change() != PROT_NONE)
   { // Somehow the type is already known
     err_raise(err_SYS, err_PROT_TYPE_ADDMOD, prot_ERR, FALSE);
   }
   else tm.set_change(PROT_ADD);

   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void sos_SchemaModif::del_type(sos_Type_descr type)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::del_type");
   TT(prot_H, T_ENTER);

   sos_TypeModif tm = self.type_modif(type);

   if (tm.get_change() == PROT_ADD)
   {  // if the type just has been inserted, it can be removed again 
      // without any problems
      self.get_modifs().remove(tm.get_modified_type());
      tm.destroy(); 
   }
   else
   {  // the type can be deleted
      tm.set_change(PROT_DEL);

      // if type is Class_type, the offsets of the superclasses have
      // have to be saved, because the deletion might be undone, and 
      // in this case, superclass structure might already have been
      // changed
      if (type.has_type(sos_Class_type_type))
         sos_ClassModif::make(tm).save_offsets(TRUE); 
   }
   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void sos_SchemaModif::ren_type(sos_Type_descr type, sos_String name)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::ren_type");
   TT(prot_H, T_ENTER);

   sos_Container  cnt = self.container();
   sos_TypeModif  tm  = self.type_modif(type);
   sos_Cstring    cname;

   if (tm.get_change() == PROT_DEL)
   {  // no rename can be executed if the type is already deleted
      err_raise(err_SYS, err_PROT_RENTYPE_DEL, prot_ERR, FALSE);
   }
   else if (tm.get_change() != PROT_ADD)
   {  // change = PROT_NONE or PROT_MOD 
      // rename can be executed

      if (type.has_type(sos_Typedef_type_type)) 
         cname = sos_Typedef_type::make(type).get_name().make_Cstring();
      else
         cname = type.make_type().get_name().make_Cstring();
 
      if (INVALID(tm.get_old_name()))
         tm.set_old_name(sos_String::create(cnt,cname)); 
      // else
      //   tm.get_old_name().copy(name, cnt);
 
      delete cname; 
       
      tm.set_change(PROT_MOD);
   }
   TT(prot_H, T_LEAVE);
}
 
// *************************************************************************
sos_TypeModif sos_SchemaModif::mod_type(sos_Type_descr type)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::mod_type");
   TT(prot_H, T_ENTER);

   sos_TypeModif tm = self.type_modif(type);

   // changes are only registered if the type is neither deleted nor
   // added during this version, otherwise NO_OBJECT is returned
   if ((tm.get_change() != PROT_NONE) && (tm.get_change() != PROT_MOD)) 
      tm = sos_TypeModif::make(NO_OBJECT);

   TT(prot_H, T_LEAVE);
   return tm;
} 

// *************************************************************************
sos_TypeModif sos_SchemaModif::type_modif(sos_Type_descr type)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::type_modif");
   TT(prot_H, T_ENTER);

   sos_Container cnt = self.container();
   sos_Version   ver = self.get_ver();
   sos_TypeModif tm;
 
   tm = self.get_modifs()[type];
 
   if (INVALID(tm))
   {
      if (type.has_type(sos_Class_type_type))
      {  tm = sos_ClassModif::create(cnt, ver, type);
         sos_ClassModif::make(tm).install(self.get_modified_schema());
      }
      else
       if (type.has_type(sos_Enum_type_type))
         tm = sos_EnumModif::create(cnt, type);
      else
       if (type.has_type(sos_Extern_type_type))
         tm = sos_ExternModif::create(cnt, type);
      else
       if (type.has_type(sos_Typedef_type_type))
         tm = sos_TypedefModif::create(cnt, type);
      else
       if (type.has_type(sos_Union_type_type))
         tm = sos_UnionModif::create(cnt, type);
 
      tm.set_change(PROT_NONE);
 
      self.get_modifs().insert(type, tm);

      if (self.get_change() == PROT_NONE)
         // change is set to often, because it is also set, if only
         // the offset of the classes have been changed, but there is no
         // other way to handle it, because there is no backpointer from
         // the typemodifs to the schemamodif
         self.set_change(PROT_MOD);
   }
 
   TT(prot_H, T_LEAVE);
   return tm;
}

// *************************************************************************
void sos_SchemaModif::add_import(sos_Import import)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::add_import");
   TT(prot_H, T_ENTER);

   sos_Import_Set ip_set = self.get_deleted_imports();
   sos_Import     ip     = sos_Import::make(NO_OBJECT);
   sos_Import     rip;

   // search for deleted import, which is equal the one to added
   // equality is based on equality of the schema_modules
   agg_iterate(ip_set, rip)
   {
      if (rip.get_module() == import.get_module())
      {  ip = rip; 
         break;
      }
   } agg_iterate_end(ip_set, rip);

   if (VALID(ip))
     ip_set.remove(ip);
   else 
   {  self.get_added_imports().insert(import);
      self.set_change(PROT_MOD);
   }

   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void sos_SchemaModif::del_import(sos_Import import)
// *************************************************************************
{
   T_PROC("sos_SchemaModif::del_import");
   TT(prot_H, T_ENTER);

   sos_Import_Set ip_set = self.get_added_imports();
   sos_Import     ip     = sos_Import::make(NO_OBJECT);
   sos_Import     rip;
 
   // search for deleted import, which is equal the one to added
   // equality is based on equality of the schema_modules
   agg_iterate(ip_set, rip)
   {
      if (rip.get_module() == import.get_module())
      {  ip = rip;
         break;
      }
   } agg_iterate_end(ip_set, rip);
 
   if (VALID(ip))
     ip_set.remove(ip);
   else
   {  self.get_deleted_imports().insert(import);
      self.set_change(PROT_MOD);
   }

   TT(prot_H, T_LEAVE);
}
