#line 1 "./src/mta/mta_type.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST Projekt", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */

/* OBST LIBRARY MODULE */

#include "mta.h"


EXPORT sos_String mta_get_name (const sos_Type_descr& td)
{  T_PROC ("mta_get_name");
   TT (mta_M, T_ENTER);
   
   sos_String result;

   if (td.has_type (sos_Typedef_type_type))
      result = sos_Typedef_type::make (td).get_name();
   else
      result = td.make_type().get_name();

   TT (mta_M, T_LEAVE);
   return result;
}

EXPORT void mta_destroy (sos_Expr e)
{  T_PROC ("mta_destroy (sos_Expr)");
   TT (mta_M, T_ENTER);
   
   if (VALID (e))
   {
#ifdef OBST_HAVE_PROT
      if (e.has_type (sos_Identifier_type))
         sos_ModifHistory::trash(sos_Identifier::make(e).get_id());
      sos_ModifHistory::trash(e);
#else
      if (e.has_type (sos_Identifier_type))
         sos_Identifier::make(e).get_id().destroy();
      e.destroy();
#endif
   }

   TT (mta_M, T_LEAVE);
}
 
EXPORT void mta_destroy (sos_Super_class sc)
{  T_PROC ("mta_destroy (sos_Super_class)");
   TT (mta_M, T_ENTER);
   
   if (VALID (sc))
   {  MTA_DESTROY_LIST (sc.get_create_params());
#ifdef OBST_HAVE_PROT
      sos_ModifHistory::trash(sc);
#else
      sc.destroy();
#endif
   }

   TT (mta_M, T_LEAVE);
}

EXPORT void mta_destroy (sos_Super_class_List scl)
{  T_PROC ("mta_destroy (sos_Super_class_List)");
   TT (mta_M, T_ENTER);
   
   if (VALID (scl))
   {  agg_iterate (scl, sos_Super_class sc)
         mta_destroy (sc);
      agg_iterate_end (scl, sc)
#ifdef OBST_HAVE_PROT
      sos_ModifHistory::trash(scl);
#else
      scl.destroy();
#endif
   }

   TT (mta_M, T_LEAVE);
}

EXPORT void mta_destroy (sos_Comp_descr cm)
{  T_PROC ("mta_destroy (sos_Comp_descr)");
   TT (mta_M, T_ENTER);
   
   if (VALID (cm))
   {  sos_Expr e = cm.get_init_expr();
      mta_destroy (e);
#ifdef OBST_HAVE_PROT
      sos_ModifHistory::trash(cm.get_name());
#else
      cm.get_name().destroy();
#endif
      mta_destroy (cm.get_get_method());
      mta_destroy (cm.get_set_method());
   }

   TT (mta_M, T_LEAVE);
}

EXPORT void mta_destroy (sos_Comp_descr_List cl)
{  T_PROC ("mta_destroy (sos_Comp_descr_List)");
   TT (mta_M, T_ENTER);

   if (VALID (cl))
   {  agg_iterate (cl, sos_Comp_descr c)
	 mta_destroy (c);
      agg_iterate_end (cl, c)
#ifdef OBST_HAVE_PROT
      sos_ModifHistory::trash(cl);
#else
      cl.destroy();
#endif
   }

   TT (mta_M, T_LEAVE);
}
 
EXPORT void mta_destroy (sos_Gen_param gp)
{  T_PROC ("mta_destroy (sos_Gen_param)");
   TT (mta_M, T_ENTER);
   
   if (VALID (gp))
   {
#ifdef OBST_PROT_ALL
      sos_ModifHistory::trash(gp.get_name());
      sos_ModifHistory::trash(gp);
#else
      gp.get_name().destroy();
      gp.destroy();
#endif
   }

   TT (mta_M, T_LEAVE);
}
 
EXPORT void mta_destroy (sos_String s)
{  T_PROC ("mta_Destroy (sos_String)");
   TT (mta_M, T_ENTER);
   
   if VALID (s)
   {
#ifdef OBST_PROT_ALL
      sos_ModifHistory::trash(s);
#else
      s.destroy();
#endif
   }

   TT (mta_M, T_LEAVE);
}

EXPORT void mta_destroy (sos_Gen_param_List gpl)
{  T_PROC ("mta_destroy (sos_Gen_param_List)");
   TT (mta_M, T_ENTER);
   
   if (VALID (gpl))
   {  agg_iterate (gpl, sos_Gen_param gp)
         mta_destroy (gp);
      agg_iterate_end (gpl, gp);
#ifdef OBST_PROT_ALL
      sos_ModifHistory::trash(gpl);
#else
      gpl.destroy();
#endif
   }

   TT (mta_M, T_LEAVE);
}

LOCAL void mta_destroy (sos_Type_descr td)
{  T_PROC ("mta_destroy (sos_Type_descr)");
   TT (mta_M, T_ENTER);
   
   if (VALID(td))
      if (td.has_type (sos_Class_type_type))
      {  sos_Class_type ct = sos_Class_type::make (td);  
         mta_destroy (ct.get_name());
         mta_destroy (ct.get_create_params());
         mta_destroy (ct.get_super_classes());
#ifdef OBST_HAVE_PROT
         sos_ModifHistory::trash(ct.get_super_closure());
#else
         ct.get_super_closure().destroy();
#endif

         if (VALID (ct.get_friends()))
#ifdef OBST_PROT_ALL
            sos_ModifHistory::trash(ct.get_friends());
#else
            ct.get_friends().destroy();
#endif
         mta_destroy(ct.get_local_methods());

	    // all other methods in ct.get_methods are derived:
         sos_Method_table mt = ct.get_methods();
         if (VALID (mt))
         {  agg_iterate_association (mt, sos_String s, sos_Method_List ml)
#ifdef OBST_PROT_ALL
               sos_ModifHistory::trash(ml);
#else
               ml.destroy();
#endif
            agg_iterate_association_end (mt, s, ml)
#ifdef OBST_PROT_ALL
            sos_ModifHistory::trash(mt);
#else
            mt.destroy();
#endif
         }

	    // component methods were already destroyed
#ifdef OBST_HAVE_PROT
         sos_ModifHistory::trash(ct.get_components());
#else
         ct.get_components().destroy();
#endif
         mta_destroy (ct.get_formal_gen_params());
         sos_Type_descr_List agpl = ct.get_actual_gen_params();
         if VALID (agpl)
#ifdef OBST_PROT_ALL
             sos_ModifHistory::trash(agpl);
#else
             agpl.destroy();
#endif
#ifdef OBST_HAVE_PROT
         sos_ModifHistory::trash(ct);
#else
         ct.destroy();
#endif
      }
      else if (td.has_type (sos_Extern_type_type))
      {  
#ifdef OBST_HAVE_PROT
         sos_ModifHistory::trash(td);
#else
         td.destroy();
#endif
      }
      else if (td.has_type (sos_Enum_type_type))
      {  sos_Enum_type et = sos_Enum_type::make (td);
#ifdef OBST_PROT_ALL
         sos_ModifHistory::trash(et.get_literals());
#else
         et.get_literals().destroy();
#endif
      }
      else if (td.has_type (sos_Typedef_type_type))
      {  sos_Typedef_type tdt = sos_Typedef_type::make (td);
#ifdef OBST_PROT_ALL
         sos_ModifHistory::trash(tdt.get_name());
#else
         tdt.get_name().destroy();
#endif
      }
      else if (td.has_type (sos_Union_type_type))
      {  sos_Union_type ut = sos_Union_type::make (td);
#ifdef OBST_PROT_ALL
         sos_ModifHistory::trash(ut.get_name());
#else
         ut.get_name().destroy();
#endif
      }
      else
         mta_error (err_SYS, err_MTA_INVALID_METHOD);
   
   TT (mta_M, T_LEAVE);
}


EXPORT sos_Import_mode mta_uses (const sos_Typedef_type& tdt,
				 const sos_Type_descr&   td)
{  T_PROC ("mta_uses (sos_Typedef_type, sos_Type_descr)");
   TT (mta_M, T_ENTER);
   
   sos_Import_mode result = sos_IMP_NONE;
   if (tdt.make_type() == td)
      result = sos_IMP_TYPE;
   
   TT (mta_M, T_LEAVE);
   return result;
}

EXPORT sos_Import_mode mta_uses (const sos_Type_descr& td,
				 const sos_Type_descr& used_tp)
{  T_PROC ("mta_uses (sos_Type_descr, sos_Type_descr)");
   TT (mta_M, T_ENTER);
   
   sos_Import_mode result = sos_IMP_NONE;

   if (td != used_tp)
   {  if (td.has_type (sos_Class_type_type))
         result = mta_uses (sos_Class_type::make (td), used_tp);
      else if (td.has_type (sos_Typedef_type_type))
         result = mta_uses (sos_Typedef_type::make (td), used_tp);
   }

   TT (mta_M, T_LEAVE);
   return result;
} 
 
LOCAL sos_Schema_module mta_uses (const sos_Type_descr& td)
{  // get the schema which uses td
   // Therefore only schemas importing the schema, in which td is declared
   // have to be inspected.

   T_PROC ("mta_uses (sos_Type_descr)");
   TT (mta_M, T_ENTER);
   
   sos_Schema_module           td_decl    = sos_Schema_module::retrieve (
						      td.container());
   sos_Schema_module_Directory schema_dir = sos_Schema_module::schema_dir();
   sos_Schema_module           result     = sos_Schema_module::make (NO_OBJECT);
   agg_iterate_association (schema_dir, sos_String name, sos_Schema_module s)
      if (s == td_decl  OR  VALID(mta_get_import (s, td_decl)))
      {
	 if (mta_uses(s, td) != sos_IMP_NONE)
	 {  result = s;
	    break;
	 }
      }
   agg_iterate_association_end (schema_dir, name, s)

   TT (mta_M, T_LEAVE);
   return result;
} 

// **************************************************************************
void _sos_Type_descr::remove (const sos_Typed_id&_OBSThis)
// **************************************************************************
const{     // check usage of the type
   T_PROC ("sos_Type_descr::remove");
   TT (mta_H, T_ENTER);

   sos_Type_descr    td  = sos_Type_descr::make(_OBSThis,this);
   sos_Container     cnt = td.container();
   sos_Schema_module sm  = _sos_Schema_module::retrieve (cnt);
#ifdef OBST_PROT_ALL
   sos_SchemaModif   smod = sos_SchemaModif::mod_schema (sm);
#endif
   sos_Schema_module using_schema = mta_uses (td);

   if (VALID (using_schema))
   {  mta_error (err_SYS, err_MTA_TYPE_IN_USE,using_schema.get_name());
      return;
   }

#ifdef OBST_PROT_ALL
   if (VALID(smod))
      smod.del_type (td); // don't use make_type(), since td might be typedef!
#endif
   mta_open_for_writing (cnt);
   sm.get_type_table().remove (mta_get_name (td));

   sos_Type_descr_List types = sm.get_types();
   Index 	       i     = types.find (td);
   err_assert (i>=1,"sos_Type_descr:remove:internal error:not found in types");
   types.remove (i);
   mta_destroy (td);

   TT (mta_H, T_LEAVE);
} // *** sos_Type_descr::remove ***

void mta_replace_name (sos_Schema_module sm, 	   sos_Type_descr td, 
		       sos_String	 old_name, sos_String	  new_name)
{  
   T_PROC ("mta_replace_name");
   TT (mta_M, T_ENTER);

   mta_open_for_writing(sm.container());
   sm.get_type_table().remove (old_name);
   old_name.assign (new_name);
   sm.get_type_table().insert (old_name, td);

   TT (mta_M, T_LEAVE);
} // *** mta_replace_name

// **************************************************************************
void _sos_Type_descr::modify_name (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_String) name)
// **************************************************************************
const{  T_PROC ("sos_Type_descr::modify_name");
   TT (mta_H, T_ENTER);
   
   sos_Type_descr    td	      = sos_Type_descr::make(_OBSThis,this);
   sos_Container     cnt      = td.container();
   sos_Schema_module sm	      = _sos_Schema_module::retrieve (cnt);
   sos_String	     old_name = mta_get_name (td);

   if (td.has_type (sos_Class_type_type))
   {  sos_Class_type ct = sos_Class_type::make (td);
      if (VALID(ct.get_actual_gen_params()))
      {  mta_error (err_SYS, err_MTA_DONT_MODIFY_INSTANTIATION, 
		    MTA_GET_NAME_DECLARATION(ct));
	 return;
      }
   }
   // NOTE: Future extension concerning gen classes must check on
   //	    generic class here. If TRUE, all instantiation class names
   //	    must be adapted.
   if (!old_name.equal (name))
   {
#ifdef OBST_HAVE_PROT
      sos_SchemaModif smod = sos_SchemaModif::mod_schema (sm);
      if (VALID(smod))
	 smod.ren_type (td, name);
#endif
	 // check if the new name is valid
	 // investigate all importing schemas
      sos_Schema_module_Directory smd = _sos_Schema_module::schema_dir();
      agg_iterate_association (smd, sos_String n, sos_Schema_module import)
	 if (VALID(mta_get_import (import, sm)) OR (import == sm))
	 {  if (VALID (import.lookup_type (name)))
	       mta_error (err_SYS, err_MTA_NAME_IN_USE, import.get_name(),name);
	 }
      agg_iterate_association_end (smd, n, sm)

      mta_replace_name (sm, td, old_name, name);
   }

   TT (mta_H, T_LEAVE);
} // *** sos_Type_descr::modify_name ***


// **************************************************************************
sos_Schema_module _sos_Type_descr::defined_in(const sos_Typed_id&_OBSThis)
// **************************************************************************
const{  T_PROC ("sos_Type_descr::defined_in");
   TT (mta_H, T_ENTER);
   
   sos_Schema_module result = _sos_Schema_module::retrieve(_OBSThis.container());

   TT (mta_H, T_LEAVE);
   return result;
} // *** sos_Type_descr::is_defined_in ***

 

/* --------------------------------------------------------------------------
 * methods to modify scalar types
 * --------------------------------------------------------------------------
*/


LOCAL sos_Int mta_enum_size (sos_Int nr_literals)
{  T_PROC ("mta_enum_size");
   TT (mta_H, T_ENTER);

   sos_Int result;

   if (nr_literals <= 256)
      result = 1;
   else if (nr_literals <= 65536)
      result = 2;
   else
      result = 0;

   TT (mta_H, T_LEAVE);
   return result;
}

LOCAL sos_Bool mta_valid_type_name (const sos_Object& o,
				    const sos_String& name)
{  T_PROC ("mta_valid_type_name");
   TT (mta_H, T_ENTER);
   
   sos_Bool result = INVALID(sos_Schema_module::make(o).lookup_type(name)); 

   TT (mta_H, T_LEAVE);
   return result;
}

LOCAL inline smg_String mta_generate_type_name
			  (const sos_Schema_module& sm,
			   const smg_String&        basic_name)
{  return mta_generate_valid_name (basic_name,sos_Object(sm),
				   mta_valid_type_name);
}

 
LOCAL sos_Enum_type mta_lookup_literal (const sos_Schema_module& sm,
					const sos_String&	 l,
					sos_Bool		 with_imports)
// returns the Enum_type that defines the literal l
{  T_PROC ("mta_lookup_literal");
   TT (mta_H, T_ENTER);

   static sos_Object_sos_Object_Mapping cache
      		= sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);
 
   sos_Enum_type result = sos_Enum_type::make(NO_OBJECT);
   
   if (with_imports)
   {  sos_Import_List imports = sm.get_imports();
      agg_iterate (imports, sos_Import imp)
      {  sos_Enum_type imported_et = mta_lookup_literal (imp.get_module(),
							 l, FALSE);
         if (VALID (imported_et))
         {  imports.close_cursor (agg_current_cursor());
            result = imported_et;
	    break;
         }
      }
      agg_iterate_end (imports, imp)
   }

   else 
   {
      if (NOT cache.is_key (sm))
      {  sos_Object_sos_Object_Mapping sem =
	    sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER, FALSE, TRUE);
	 cache.insert (sm, sem);

	 sos_Type_descr_List types = sm.get_types();
	 agg_iterate (types, sos_Type_descr td)
	    if (td.has_type (sos_Enum_type_type))
	    {  sos_Enum_type   et	= sos_Enum_type::make (td);
	       sos_String_List literals = et.get_literals ();
	       agg_iterate (literals, sos_String lit)
		 sem.insert (lit, et); 
	       agg_iterate_end (literals, lit)
	    }
	 agg_iterate_end (types, td)
      }
      result = sos_Enum_type::make(
			   sos_Object_sos_Object_Mapping::make(cache[sm])[l]);
   }

   TT (mta_H, T_LEAVE);
   return result;
}



EXPORT sos_Enum_type mta_create_enum (const sos_Schema_module& sm)
// create an enumeration in the schema sm. The enumeration will get a valid
// name new_enum[_<number>]. There are no literals within the enumeration.
{  
   T_PROC ("mta_create_enum");
   TT (mta_H, T_ENTER);

   sos_Container cnt   = sm.container();
   mta_open_for_writing (cnt);

   smg_String    smg_name = mta_generate_type_name (sm, "new_enum");
   sos_Enum_type et	  = sos_Enum_type::create(cnt);

   et.set_name(smg_name.make_String(cnt));
   et.set_literals(sos_String_List::create(cnt));
   et.set_object_size (mta_enum_size (et.get_literals().card()));
   sm.get_type_table().insert (et.get_name(), et);
   sm.get_types().append (et);
 
   TT (mta_H, T_LEAVE);
   return et;
}

// ************************************************************************
void _sos_Enum_type::insert_literal (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_String) name)
// ************************************************************************
const{  T_PROC ("sos_Enum_type::insert_literal");
   TT (mta_H, T_ENTER);
   
   sos_Enum_type     et  = sos_Enum_type::make(_OBSThis,this);
   sos_Container     cnt = et.container();
   sos_Schema_module sm  = _sos_Schema_module::retrieve (cnt);
#ifdef OBST_PROT_ALL
   sos_EnumModif emod = sos_EnumModif::make(MTA_GET_TYPE_MOD(cnt, et));
#endif

   sos_String_List literals = et.get_literals();
   sos_Bool	   found    = FALSE;
   agg_iterate (literals, sos_String l)
      if (l.equal (name))
      {	 found = TRUE;
         break;
      }
   agg_iterate_end (literals, l)

   if (NOT found)
   {  sos_Int size        = mta_enum_size (et.get_literals().card());
      sos_Int object_size = mta_enum_size (size+1);
      sos_Int position;
      if (size == 0)
        err_raise (err_SYS, err_MTA_TOO_MANY_LITERALS,MTA_ERR, FALSE);

#ifdef OBST_PROT_ALL
      if (VALID(emod)) // generate protocol info if needed
	 position = emod.add_lit(name);
      else
#endif
	 position = literals.card()+1;

      mta_open_for_writing (cnt);
      sos_String lit = sos_String::copy(name,cnt);
      et.set_object_size (object_size);
      literals.insert(position, lit);
   }

   TT (mta_H, T_LEAVE);
} // *** sos_Enum_type::insert_literal ***
 

// ************************************************************************
void _sos_Enum_type::remove_literal (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_String) name)
// ************************************************************************
const{  T_PROC ("sos_Enum_type::remove_literal");
   TT (mta_H, T_ENTER);
   
   sos_Enum_type     et  = sos_Enum_type::make(_OBSThis,this);
   sos_Container     cnt = et.container();
   sos_Schema_module sm  = _sos_Schema_module::retrieve (cnt);
#ifdef OBST_PROT_ALL
   sos_EnumModif emod = sos_EnumModif::make(MTA_GET_TYPE_MOD(cnt, et));
#endif
 
   sos_String_List literals = et.get_literals();
   sos_Bool        found    = FALSE;
   for (sos_Cursor c = literals.open_cursor();
        literals.is_valid(c);
	literals.to_succ(c))
      if (literals.get(c).equal(name))
      {  found = TRUE;
         mta_open_for_writing (cnt);
#ifdef OBST_PROT_ALL
	 if (VALID(emod)) // generate protocol information if needed
	    emod.del_lit (name);
#endif
         sos_String s = literals.get(c);
         literals.remove_at(c);
#ifdef OBST_PROT_ALL
         sos_ModifHistory::trash(s);
#else
         s.destroy();
#endif
         et.set_object_size (mta_enum_size (literals.card()));
         break;
      }
   literals.close_cursor(c);
   if (NOT found)
     err_raise (err_SYS, err_MTA_NOT_FOUND, MTA_ERR,FALSE);
   
   TT (mta_H, T_LEAVE);
} // *** sos_Enum_type::remove_literal ***


EXPORT sos_Extern_type mta_create_extern (const sos_Schema_module& sm)
{  
   T_PROC ("mta_create_extern");
   TT (mta_H, T_ENTER);

   sos_Container cnt = sm.container();
   mta_open_for_writing (cnt);

   smg_String smg_name = mta_generate_type_name
				(sm, "new_extern").make_String (cnt);

   sos_Extern_type et = sos_Extern_type::create (cnt);
   et.set_name (smg_name.make_String (cnt));
   et.set_object_size (SOS_ID_SIZE);
 
   sm.get_type_table().insert (et.get_name(), et);
   sm.get_types().append (et);

   TT (mta_H, T_LEAVE);
   return et;
} // *** mta_create_extern ***

// ************************************************************************
void _sos_Extern_type::modify_size (const sos_Typed_id&_OBSThis,sos_Int size)
// ************************************************************************
const{  
   T_PROC ("sos_Extern_type::modify_size");
   TT (mta_H, T_ENTER);
   
   if (size > SOS_ID_SIZE)
      err_raise (err_SYS, err_MTA_SIZE_TOO_BIG,MTA_ERR, FALSE);

   sos_Extern_type et  = sos_Extern_type::make(_OBSThis,this);
   sos_Container   cnt = et.container();
#ifdef OBST_HAVE_PROT
   sos_ExternModif emod = sos_ExternModif::make (MTA_GET_TYPE_MOD(cnt, et));
#endif
   mta_open_for_writing (_OBSThis.container());
#ifdef OBST_HAVE_PROT
   if (VALID(emod)) // generate protocol info if needed
      emod.mod_size();
#endif
   et.set_object_size (size);

   // Now determine all sos_Class_types using this extern type. Their
   // offsets and sizes must be updated
   sos_Class_type_List using_classes = MTA_USING_CLASSES(et);
   agg_iterate (using_classes, sos_Class_type ct)
      mta_set_offsets_and_size (ct);
   agg_iterate_end (using_classes, ct);

   TT (mta_H, T_LEAVE);
} // *** sos_Extern_type::modify_size ***


// ************************************************************************
sos_Union_type mta_create_union (const sos_Schema_module& sm)
// ************************************************************************
{  T_PROC ("mta_create_union");
   TT (mta_H, T_ENTER);
   
   sos_Container cnt = sm.container();
   mta_open_for_writing (cnt);

   smg_String     smg_name = mta_generate_type_name (sm, "new_union");
   sos_Union_type ut	   = sos_Union_type::create(cnt);

   ut.set_united (sos_Type_List::create (cnt));
   ut.set_name(smg_name.make_String(cnt));

   sm.get_type_table().insert (ut.get_name(), ut);
   sm.get_types().append (ut);
 
 // object_size wird nicht gesetzt ??? (bs)

   TT (mta_H, T_LEAVE);
   return ut;
}

// ************************************************************************
void _sos_Union_type::insert_type(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Type_descr) td)
// ************************************************************************
const{  T_PROC ("sos_Union_type::insert_type");
   TT (mta_H, T_ENTER);
   
   sos_Union_type ut  = sos_Union_type::make(_OBSThis,this);
   sos_Container  cnt = ut.container();
   mta_open_for_writing (cnt);

   if (td.has_type (sos_Class_type_type))
   {  sos_Class_type ct = sos_Class_type::make (td);
      if (ct.is_generic_class())
	 mta_error (err_SYS, err_MTA_CLASS_EXPECTED);
   }
 
   sos_Type tp = td.make_type();
   if (ut.get_united().find(tp) == 0)
   {
#ifdef OBST_HAVE_PROT
      sos_UnionModif umod = sos_UnionModif::make (MTA_GET_TYPE_MOD (cnt, ut));
      if (VALID (umod))
         umod.add_type (tp);
#endif
      ut.get_united().append (tp);
   }
 
   TT (mta_H, T_LEAVE);
} // *** sos_Union_type::insert ***

// ************************************************************************
void _sos_Union_type::remove_type (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Type_descr) td)
// ************************************************************************
const{  T_PROC ("sos_Union_type::remove_type");
   TT (mta_H, T_ENTER);
   
   sos_Union_type ut  = sos_Union_type::make(_OBSThis,this);
   sos_Container  cnt = ut.container();
   mta_open_for_writing (cnt);

   if (NOT mta_remove_from_list (ut.get_united(), td.make_type()))
      mta_error(err_SYS,err_MTA_NOT_FOUND, mta_get_name_declaration(td));

#ifdef OBST_HAVE_PROT
   sos_UnionModif umod = sos_UnionModif::make (MTA_GET_TYPE_MOD (cnt, ut));
   if (VALID (umod))
      umod.del_type (td.make_type());
#endif
   
   TT (mta_H, T_LEAVE);
} // *** sos_Union_type::remove_type ***


EXPORT sos_Typedef_type mta_create_typedef (const sos_Schema_module& sm)
{  T_PROC ("mta_create_typedef");
   TT (mta_H, T_ENTER);
   
   sos_Container cnt = sm.container();
   mta_open_for_writing (cnt);
   sos_String name = mta_generate_type_name (sm,
					     "new_typedef").make_String (cnt);
 
   sos_Typedef_type td = sos_Typedef_type::create(cnt);
   td.set_name (name);
   td.set_named_type (sos_Type::make(void_type));
 
   sm.get_type_table().insert (name, td);
   sm.get_types().append (td);
  
   TT (mta_H, T_LEAVE);
   return td;
}

// ***************************************************************************
void _sos_Typedef_type::modify (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Type_descr) named_type)
// ***************************************************************************
// WARNING: Since OBST3-3 types which uses a typedef declaration won't use it
// in the meta-information, because the typedef substitution was resolved
// by the compiler. So modifying of a typedeftype will not modify
// dependent types, because of no information which types uses the typedef
const{  
   T_PROC ("sos_Typedef_type::modify");
   TT (mta_H, T_ENTER);

   sos_Typedef_type  td  = sos_Typedef_type::make(_OBSThis,this);
   sos_Container     cnt = td.container();
   sos_Schema_module sm  = _sos_Schema_module::retrieve (cnt);
 
   if (named_type.operator!=(td.get_named_type()))
   {  if (named_type.make_type().is_derived_from_some (td.make_type()))
         mta_error (err_SYS, err_MTA_CIRCLED_DECL,
		    mta_get_name_declaration(named_type));
   }
 
#ifdef OBST_HAVE_PROT
   sos_TypedefModif tdmod = sos_TypedefModif::make(MTA_GET_TYPE_MOD(cnt, td));
   if (VALID (tdmod))
      tdmod.mod_type ();
#endif
   mta_open_for_writing (cnt);
   td.set_named_type (named_type.make_type());

   TT (mta_H, T_LEAVE);
}
