/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST Projekt", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */

/* OBST LIBRARY MODULE */

#include "mta.h"

#define OBST_IMP_FORMATTED_IO
#define OBST_IMP_STRINGOP
#define OBST_IMP_ERROR
#define OBST_IMP_FILE
#include "obst_stdinc.h"

// Macros for smg_String's usable in the context of parsing type names
#define is_comma(s)     ((s).equal (","))
#define is_abr_left(s)  ((s).equal ("<"))
#define is_abr_right(s) ((s).equal (">"))

LOCAL sos_Type_descr mta_get_td_from_name (const sos_String& name,
					   char**	     n = NULL);


LOCAL inline sos_Schema_module mta_schema_lookup (sos_String name)
{  sos_Schema_module_Directory smd = sos_Schema_module::schema_dir();
   return smd[name];
}


EXPORT sos_Import mta_get_import (const sos_Schema_module& sm,
				  const sos_Schema_module& import)
// gets the corresponding sos_Import object to the passed sos_Schema_module
{  
   T_PROC ("mta_get_import");
   TT (mta_H, T_ENTER);
   
   sos_Import_List imports = sm.get_imports();
   sos_Import	   result  = sos_Import::make (NO_OBJECT);
   agg_iterate (imports, sos_Import imp)
      if (imp.get_module().operator==(import))
      {  result = imp;
	 break;
      }
   agg_iterate_end (imports, imp)

   TT (mta_H, T_LEAVE);
   return result;
}


LOCAL sos_Bool mta_import_needed_by_schema (const sos_Import&	     imp,
					    const sos_Schema_module& sm)
{
   // Checks, whether schema sm uses a type defined in imp

   T_PROC ("mta_import_needed_by_schema");
   TT (mta_M, T_ENTER);

   sos_Type_descr_List imported_types = imp.get_module().get_types();
   sos_Type_descr_List types	      = sm.get_types();
   sos_Bool            result	      = FALSE;

   agg_iterate (types, sos_Type_descr type);
      agg_iterate (imported_types, sos_Type_descr td)
	 sos_Import_mode tmpim = mta_uses (type, td);
	 if (tmpim != sos_IMP_NONE)
	 {
	    result = TRUE;
	    break;
	 }
      agg_iterate_end (imported_types, td)
      if (result)
	 break;
   agg_iterate_end (types, type)

   TT (mta_M, T_LEAVE);
   return result;
}


// **************************************************************************
void sos_Schema_module::update_import(sos_Schema_module imported_sm)
// **************************************************************************
// check if the import imported_sm of sm is used. 
// That means, check if there is a type in sm which 
// uses a type defined in imported_sm.
// If this is not the case, remove the import imported_module
// Otherwise correct the import mode
{  
   T_PROC ("sos_Schema_module::update");
   TT (mta_H, T_ENTER);

   sos_Schema_module sm	     = self;
   sos_Import_List   imports = sm.get_imports();
   sos_Import	     import  = mta_get_import (sm, imported_sm);
#ifdef OBST_PROT_ALL
   sos_SchemaModif smod = sos_SchemaModif::mod_schema(sm);
#endif
   if INVALID (import)
      err_raise (err_SYS, err_MTA_INVALID_IMPORT, MTA_ERR, FALSE);

   sos_Bool remove = import.get_is_direct(); // only remove, if direct!
   // if remove == TRUE then imported_sm is a leaf import, 
   // that means that there is no other import of sm that 
   // imports imported_sm implicitely

   // iterate over all types of sm to check whether 
   // they use a type of imported_sm
   // iterate even if remove = FALSE because the work of updating
   // the import mode remains
   sos_Type_descr_List imported_types = imported_sm.get_types();
   sos_Type_descr_List types	      = sm.get_types();
   sos_Import_mode     im	      = sos_IMP_NONE;

   agg_iterate (types, sos_Type_descr type);
      agg_iterate (imported_types, sos_Type_descr td)
	    sos_Import_mode tmpim = mta_uses (type, td);
	    if (tmpim != sos_IMP_NONE)
	    {  remove = FALSE;

		  // sos_Import_mode is sorted in priority order
	       if (int(tmpim) > int(im))
		  im = tmpim;
	    }
      agg_iterate_end (imported_types, td)
   agg_iterate_end (types, type)

   if (remove)
   {
      mta_open_for_writing (sm.container());
      mta_remove_from_list (imports, import);
#ifdef OBST_PROT_ALL
      if (VALID(smod))
	 smod.del_import(import);
      sos_ModifHistory::trash(import);
#else
      import.destroy();
#endif
   }
   else
   {     // update import mode, but only if it changed
      if (im != import.get_mode())
      {
	 mta_open_for_writing (sm.container());
	 import.set_mode (im);
      }
   }
   TT (mta_H, T_LEAVE);
}  // *** sos_Schema_module::update_import **

// **************************************************************************
void sos_Schema_module::update_imports()
// **************************************************************************
{
   T_PROC ("sos_Schema_module::update_imports");
   TT (mta_H, T_ENTER);

   sos_Import_List il = self.get_imports();
   agg_iterate (il, sos_Import i)
      self.update_import (i.get_module());
   agg_iterate_end (il, i)

   TT (mta_H, T_LEAVE);
}

// **************************************************************************
void sos_Schema_module::generate (sos_String host_language)
// **************************************************************************
// Calls the C++ generator genCC via the shellscript obst-gen
// Assert that no container is opened for writing. Container that 
// are opened for reading are closed implicitely.
{ 
   T_PROC ("sos_Schema_module::generate");
   TT (mta_H, T_ENTER);

   sos_Schema_module sm = self;

   smg_String cmd = smg_String("obst-gen ") + sm.get_name() + " ";

   if VALID (host_language)
      cmd += smg_String("-L ") + host_language;

      // check that all schema containers are closed
      // other containers are in the resposibility of the user
   sos_Schema_module_Directory smd = sos_Schema_module::schema_dir();
   agg_iterate_association(smd, sos_String n, sos_Schema_module sm)
   {  sos_Container	   cnt	  = sm.container();
      sos_Container_status status = cnt.status();
      if (status == READABLE)
	 cnt.close();
      else if (status == WRITEABLE)
      {  
	 mta_error (err_SYS, err_MTA_CONTAINER_YET_OPEN, sm.get_name());
	 return;
      }
   }
   agg_iterate_association_end(smd,n,sm);
      // All schemacontainer are closed

   if (system (cmd.make_Cstring(SMG_BORROW)))
      err_raise (err_SYS, err_MTA_GENERATOR_COULD_NOT_CALLED, MTA_ERR, FALSE);
   mta_generate_schema_file (sm);

   TT (mta_H, T_LEAVE);
} // *** sos_Schema_module::generate ***


LOCAL sos_Bool valid_schema_name(const sos_Object&, const sos_String& name)
{  return INVALID (mta_schema_lookup (name));
}


LOCAL sos_Schema_module mta_get_knl_module ()
{
   T_PROC ("mta_get_knl_module");
   TT (mta_M, T_ENTER);

   static sos_Schema_module knl;
   static sos_Bool	    needs_init = TRUE;

   if (needs_init)
   {
      sos_String knl_str = smg_String("knl").make_String(TEMP_CONTAINER);
      knl = mta_schema_lookup (knl_str);
      knl_str.destroy();
#ifndef BOOT
      err_assert (VALID(knl),"sos_Schema_module::create_schema - no kernel");
      // During boot knl must not exist, if knl is the compiled schema
#endif
      if (VALID(knl))
	 needs_init = FALSE;
   }

   TT (mta_M, T_LEAVE);
   return knl;
}


// **************************************************************************
sos_Schema_module sos_Schema_module::create_schema (sos_String name,
						    sos_Bool   overwrite)
// **************************************************************************
// creates a new sos_Schema_module with the default import knl
{  T_PROC ("sos_Schema_module::create_schema")
   TT (mta_H, T_ENTER; TB(overwrite));

   if (VALID (name))
   {  
      sos_Schema_module old_schema = mta_schema_lookup(name);
      if (VALID(old_schema))
	 if (overwrite)
	    old_schema.remove (TRUE);
	 else
	 {  err_raise (err_SYS, err_MTA_NAME_IN_USE, MTA_ERR, FALSE);
	    return sos_Schema_module::make (NO_OBJECT);
	 }
   }

   sos_Container new_cnt = sos_Container::create();
   mta_open_for_writing (new_cnt);

   // NOTE: sos_Schema_module object must be the first to create in the
   // new container in order to become the container's root object!
   sos_Schema_module new_sm = sos_Schema_module::create(new_cnt);

   sos_String new_name;  
   if VALID (name)
      new_name = sos_String::clone (name, new_cnt);
   else
      new_name = mta_generate_valid_name ("new_schema",
					  NO_OBJECT, valid_schema_name)
		    .make_String(new_cnt);
   new_sm.set_name (new_name);

   // Note, that imports are created with list_cursor=TRUE, since
   // constructing the transitive closure is easier with remove_at.
   new_sm.set_imports    	  (sos_Import_List::create(new_cnt, TRUE));
   new_sm.set_types      	  (sos_Type_descr_List::create(new_cnt));
   new_sm.set_type_table 	  (sos_Type_table::create(new_cnt,FALSE,TRUE));
   new_sm.set_impls      	  (sos_Schema_impl_List::make(NO_OBJECT));
   new_sm.set_has_external_import (FALSE);
   new_sm.set_has_external_types  (FALSE);

   sos_Schema_module knl = mta_get_knl_module ();
#ifdef BOOT
   if (VALID(knl))
   {
#endif
   sos_Import knl_import = sos_Import::create (new_cnt);
   new_sm.get_imports().append (knl_import);
   knl_import.set_module    (knl);
   knl_import.set_mode      (sos_IMP_NONE);
   knl_import.set_is_direct (FALSE);
#ifdef BOOT
   }
#endif

   sos_Schema_module_Directory smd     = sos_Schema_module::schema_dir();
   sos_Container               smd_cnt = smd.container();
   mta_open_for_writing (smd_cnt);
   smd.insert (new_name, new_sm);
#ifdef OBST_PROT_ALL
   sos_SchemaModif::add_schema(new_sm);
#endif

   TT (mta_H, T_LEAVE);
   return new_sm;
} // *** sos_Schema_module::create_schema() ***


EXPORT sos_Bool mta_is_bootschema (const sos_String& name)
{
   sos_Bool   is_bootschema = FALSE;
   smg_String c (name);
   for (char** snm = _obst_bootschemas;  *snm;  ++ snm)
      if (c.equal (*snm))
      {  is_bootschema = TRUE;
	 break;
      }
   return is_bootschema;
}

// **************************************************************************
void sos_Schema_module::remove (sos_Bool overwrite /*= FALSE*/)
// **************************************************************************
// destroyes a schema if it is not imported (or unused imported) from 
// another schema
{  
   T_PROC ("sos_Schema_module::remove");
   TT (mta_H, T_ENTER);

   sos_Schema_module sm = self;
#ifndef BOOT
   if (mta_is_bootschema(sm.get_name()))
      mta_error (err_SYS, err_MTA_RESERVED_SCHEMA_NAME, sm.get_name());
#endif
#ifdef OBST_PROT_ALL
   sos_SchemaModif::del_schema(sm);
#endif
      // check wheather schema is imported from another
   sos_Schema_module_Directory smd = sos_Schema_module::schema_dir();
   sos_Schema_module_Set importing_schemas
      			    = sos_Schema_module_Set::create(TEMP_CONTAINER);

   agg_iterate_association(smd,sos_String n,sos_Schema_module another_sm)
      if (VALID(mta_get_import (another_sm,sm)))
      {  if (NOT overwrite)
	 {  mta_error (err_SYS, err_MTA_SCHEMA_IN_USE,
		       sm.get_name(), another_sm.get_name());
	    return;
	 }
	 else
	    importing_schemas.insert (another_sm);
      }
   agg_iterate_association_end(smd,n,another_sm);

   if (sm.get_types().card() != 0
       AND NOT overwrite) 	     // check whether schema is empty
   {  mta_error (err_SYS, err_MTA_SCHEMA_NOT_EMPTY,sm.get_name());
      return;
   }
      // remove eventually importing schemas 
   if (importing_schemas.card() AND (overwrite))
   {  
      mta_open_for_writing (smd.container());
      agg_iterate (importing_schemas, sos_Schema_module loc_sm)
	 smd.remove (loc_sm.get_name());
#ifdef OBST_PROT_ALL
	 sos_SchemaModif::del_schema(loc_sm);
#endif
	 mta_open_for_writing (loc_sm.container());
	 loc_sm.container().destroy();
      agg_iterate_end (importing_schemas, loc_sm)
      importing_schemas.destroy();
   }

      // remove schema from root_directory
   mta_open_for_writing (smd.container());
   smd.remove (self.get_name());

      // delete all types defined in self by destroying its container
   mta_open_for_writing (self.container());
#ifdef OBST_PROT_ALL
   sos_ModifHistory::trash(self); // trashing a sos_Schema_module trashes
				  // corresponding container instead!
#else
   self.container().destroy();
#endif
   TT (mta_H, T_LEAVE);
} // *** sos_Schema_module::remove ***


// **************************************************************************
void sos_Schema_module::modify_name (sos_String new_name, sos_Bool overwrite)
// **************************************************************************
// modifies a schemas name 
// if overwrite == TRUE, then remove an maybe existing schema which has the
// passed name
{
   T_PROC ("sos_Schema_module::modify_name");
   TT (mta_H, T_ENTER);
   
   sos_Schema_module sm       = self;
#ifdef OBST_PROT_ALL
   sos_SchemaModif   smod     = sos_SchemaModif::mod_schema (sm);
#endif
   sos_String        old_name = sm.get_name();
   sos_Container     cnt      = sm.container();

#ifndef BOOT
   if (mta_is_bootschema(old_name) OR mta_is_bootschema(new_name))
      err_raise (err_SYS, err_MTA_RESERVED_SCHEMA_NAME,MTA_ERR,FALSE);
#else
   if (NOT mta_is_bootschema(new_name))
      err_raise (err_SYS, err_MTA_NO_RESERVED_SCHEMA_NAME,MTA_ERR,FALSE);
#endif

#ifdef OBST_PROT_ALL
   if (VALID(smod))
      smod.ren_schema (new_name);
#endif
   sos_Schema_module old_with_new_name = mta_schema_lookup(new_name);

   if VALID(old_with_new_name)
      if (overwrite)
	 old_with_new_name.remove(TRUE);
      else
	 mta_error (err_SYS, err_MTA_NAME_IN_USE, new_name);

   sos_Schema_module_Directory smd = sos_Schema_module::schema_dir();
   mta_open_for_writing (cnt);
   mta_open_for_writing (smd.container());

   sos_String name = sos_String::clone(new_name, cnt);
   smd.remove (old_name);    // destroys the old name object
   smd.insert (name,sm);
   sm.set_name (name);
   
   TT (mta_H, T_LEAVE);
} // *** sos_Schema_module::modify_name ***


LOCAL void mta_insert_import (const sos_Schema_module &sm,
			      const sos_Schema_module &new_import,
			      sos_Bool		      is_direct)
// append the passed import and nothing else
{
   T_PROC ("mta_insert_import");
   TT (mta_H, T_ENTER);

   sos_Import old_import = mta_get_import (sm, new_import);

   if (INVALID(old_import))
   {   if (VALID (mta_get_import (new_import, sm)))
	  mta_error (err_SYS, err_MTA_CIRCLED_IMPORT,new_import.get_name());

	 // check ambiguous types:
      sos_Type_descr_List import_types = new_import.get_types();
      agg_iterate (import_types, sos_Type_descr td)
	 if (VALID(sm.lookup_type (mta_get_name(td))))
	     mta_error (err_SYS, err_MTA_AMBIGUOUS_TYPES,
			mta_get_name_declaration(td));
      agg_iterate_end (import_types, td)

	 // arrived here, all should be alright:
      sos_Container cnt = sm.container();
#ifdef OBST_PROT_ALL
      sos_SchemaModif smod = sos_SchemaModif::mod_schema (sm);
#endif
      mta_open_for_writing(cnt);
      sos_Import imp = sos_Import::create (cnt);
      imp.set_module (new_import);
      imp.set_mode (sos_IMP_NONE);
      imp.set_is_direct (is_direct);
#ifdef OBST_PROT_ALL
      if (VALID(smod))
	 smod.add_import(imp);
#endif
      sm.get_imports().append(imp);
   }
   else // set is_direct, if not already set:
   if (is_direct AND NOT old_import.get_is_direct())
   {
      sos_Container cnt = sm.container();
      mta_open_for_writing(cnt);
      old_import.set_is_direct(TRUE);
   }

   TT (mta_H, T_LEAVE);
}


LOCAL void mta_insert_import_with_transitives (
				const sos_Schema_module &sm,
				const sos_Schema_module &new_import,
				sos_Bool		as_direct)
{
   T_PROC ("mta_insert_import_with_transitives");
   TT (mta_M, T_ENTER);

   mta_insert_import (sm, new_import, as_direct);

      // append all transitive (via new_import)
      // imported sos_Schema_modules
   sos_Import_List new_import_imports = new_import.get_imports();
   agg_iterate (new_import_imports, sos_Import i)
      // insert transitive imports with is_direct=FALSE
      mta_insert_import (sm, i.get_module(), FALSE);
   agg_iterate_end (new_import_imports, i)

   TT (mta_M, T_LEAVE);
}


LOCAL sos_Bool mta_imp_closure (const sos_Schema_module &sm)
{
   /* This function preserves all direct imports in sm's import list
      and from them determines the transitive closure.
      If an import would be removed from the list, of which sm uses a type,
      then FALSE is returned indicating the error, the import-list remains
      unchanged. If no use-conflict occurred, TRUE is returned.
      Since there need not be an order on the import list, the algorithm
      looks like this:
      o deep-copy the old import list and replace the old one in sm with
	the copy.
      o remove all indirect imports from the copied list during copying it
	(NOTE: remove knl here, but append it at the end of the list later)
      o Start at the end of the list an go backwards
      o For each import on this way (all of them are direct) insert the
	transitive imports, if they are not already in the list (use
	mta_insert_import for this purpose).
      o For each import in the old list but not in the new list check, if
	types from these imports are used in sm.
      o If so, destroy new list and place the old one back into sm; return
	FALSE.
      o Otherwise replace the old list by the new one. Preserve the old
	list, since it may be needed on higher level to reset the changes
	when performing the recursive recalculation of import closures
	(see also mta_remove_import_recursively).
   */

   T_PROC ("mta_imp_closure");
   TT (mta_H, T_ENTER);

   sos_Container     cnt    = sm.container();
   mta_open_for_writing (cnt);

   sos_Import        knl_import;
   sos_Schema_module knl    = mta_get_knl_module();
   sos_Import_List   old_il = sm.get_imports();
   sos_Import_List   new_il = sos_Import_List::create(cnt, TRUE);
   sos_Cursor        c      = new_il.open_cursor();
   sos_Bool          result = TRUE;

   agg_iterate(old_il, sos_Import i)
      if (i.get_module() == knl) // remember the knl-Import to be able to
	 knl_import = i;         // append it to the new list later.
      if (i.get_is_direct())
	 new_il.append(i);
   agg_iterate_end(old_il, i);
   new_il.append(knl_import);
   sm.set_imports (new_il);

   for (new_il.to_last(c), new_il.to_pred(c);
	new_il.is_valid(c);
	new_il.to_pred(c))
   {
      // Now for each direct import append transitive imports to new_il:
      sos_Import_List trans_il = new_il.get (c).get_module().get_imports();
      agg_iterate (trans_il, sos_Import trans_i)
	 mta_insert_import (sm, trans_i.get_module(), FALSE);
      agg_iterate_end (trans_il, trans_i)
   }
   new_il.close_cursor(c);

   // Now perform the use-check on all imports in the old list not contained
   // in the new list:
   agg_iterate(old_il, sos_Import i)
      if (NOT new_il.find (i) AND mta_import_needed_by_schema (i, sm))
      {
	 result = FALSE;
	 break;
      }
   agg_iterate_end(old_il, i)
   // Now destroy destroy new_il and reset imports to old_il if a use-error
   // occurred
   if (NOT result)
   {
      sm.set_imports (old_il);
      new_il.destroy(); // needs not be trashed, never visible in schema module.
   }

   TT (mta_H, T_LEAVE);
   return result;
}


// **************************************************************************
void sos_Schema_module::insert_import (sos_Schema_module new_import)
// **************************************************************************
// insert an import and all transitive imports. Importing schemas of self
// will also get the new import
{
   T_PROC ("sos_Schema_module::insert_import");
   TT (mta_H, T_ENTER);

   sos_Schema_module sm = self;

   // insert new_import as direct import (as_direct = TRUE), except if
   // new_import is the knl-schema. In this case insert it with
   // direct=FALSE, since the knl-import is indirect by definition!
   mta_insert_import_with_transitives (sm, new_import, 
				       new_import != mta_get_knl_module());

      // iterate all schemas and append the new import to
      // all importing schemas as indirect import (as_direct=FALSE)
   sos_Schema_module_Directory smd = sos_Schema_module::schema_dir();
   agg_iterate_association (smd, sos_String n, sos_Schema_module tmpsm)
      if VALID(mta_get_import (tmpsm, sm))
	 mta_insert_import_with_transitives (tmpsm, new_import, FALSE);
   agg_iterate_association_end (smd, n, tmpsm)

   TT (mta_H, T_LEAVE);
} // *** sos_Schema_module::insert_import ***


LOCAL sos_Bool mta_remove_import_recursively (const sos_Schema_module& sm,
					      const sos_Import&	       imp)
{
   /* This function removes the import imp from sm's import list and
      recalculates its transitive import closure, if this
      causes no use-error (i.e. no import disappearing from sm's import-
      list containes a type, that sm needs).
      For all schemas importing sm, the transitive import closure is also
      recalculated. If any of these recalculations causes a use error, all
      changes are reset.
   */

   T_PROC ("mta_remove_import_recursively");
   TT (mta_M, T_ENTER);

   // The following mapping maps sos_Schema_modules to their old
   // import lists (backups). Note, that hence the old import lists must
   // not be destroyed in mta_imp_closure() !
   sos_Object_sos_Object_Mapping imp_backup =
	sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);
   sos_Schema_module_Directory sd     = sos_Schema_module::schema_dir();
   sos_Container               cnt    = sm.container();
   sos_Bool                    result = TRUE;

   sos_Int imp_pos = sm.get_imports().find (imp);
   err_assert (imp_pos, "mta_remove_import_recursively: internal error");
   imp_backup.insert (sm, sm.get_imports());
   mta_open_for_writing (cnt);
   sm.get_imports().remove(imp_pos);
   result = mta_imp_closure (sm);
   if (result) // No use-error for sm?
   {
      agg_iterate_association (sd, sos_String name, sos_Schema_module importing)
	if (VALID (mta_get_import (importing, sm)))
	 {  // module importing the modified module was found, recalculate
	    // the transitive import closure of the importing module
	    imp_backup.insert (importing, importing.get_imports());
	    if (NOT mta_imp_closure (importing)) // Use-Error ?
	    {  // Reset changes already made to importing modules' import
	       // lists and to sm itself
	       mta_error (err_WNG, err_MTA_SCHEMA_IN_USE, importing.get_name());
	       result = FALSE;
	       break;
	    }
	 }
      agg_iterate_association_end (sd, name, importing)
   }
   else
      mta_error (err_WNG, err_MTA_SCHEMA_IN_USE, sm.get_name());

   if (result)  // No error occurred, so objects that are no longer needed
   {            // can now be trashed.
      agg_iterate_association (imp_backup, sos_Object a, sos_Object b)
#ifdef OBST_PROT_ALL
	 sos_ModifHistory::trash (b); // trash (not detroy), because old
				      // import lists were visible before.
#else
         b.destroy();
#endif
      agg_iterate_association_end (imp_backup, a, b);
   }
   else         // Use-error: Changes must be reset!
   {
      agg_iterate_association (imp_backup, sos_Object a, sos_Object b)
	 sos_Schema_module sub_sm = sos_Schema_module::make(a);
	 sos_Import_List   sub_il = sos_Import_List::make(b);
	 // NOTE: In case the use-error occurred for sub_sm (which can
	 //       also be sm), then the import-list is still the old one.
	 //	  In that case don't destroy it, just keep it the same.
	 if (sub_sm.get_imports() != sub_il)
	 {
	    sub_sm.get_imports().destroy();
	    sub_sm.set_imports(sub_il);
	 }
      agg_iterate_association_end (imp_backup, a, b)
      // NOTE: The following must be done after restoring the import
      // lists, since sm also may have got reset its import list!
      sm.get_imports().insert (imp_pos, imp);
   }
   imp_backup.destroy(); // destroy temporary mapping

   TT (mta_M, T_LEAVE);
   return result;
}


// **************************************************************************
void sos_Schema_module::remove_import (sos_Schema_module import_to_delete)
// **************************************************************************
// an Import is removed if it is not used, and it is direct, where the first
// means, the import is removed if no type from self uses any
// type from import_to_delete.
// Note, that the protocol information is generated in update_import and
// not here, since removal doesn't take place in every case.
{
   T_PROC ("sos_Schema_module::remove_import");
   TT (mta_H, T_ENTER);

   sos_Import_List il = self.get_imports();
   sos_Import      imp;
   sos_Bool        found = FALSE;
   sos_Cursor      c  = il.open_cursor();

   for (il.to_first(c);  il.is_valid(c);  il.to_succ(c))
   {  // Now find the import to delete:
      imp = il.get(c);
      if (imp.get_module() == import_to_delete)
      {
	 found = TRUE;
	 break;
      }
   }
   if (NOT found)  // import to delete not found in import list!
      mta_error(err_SYS, err_MTA_IMPORT_NOT_FOUND, import_to_delete.get_name());

   else            // import was found, delete it, recalc. transitive closure
   {
      if (NOT imp.get_is_direct())
	 // Schemas imported only transitively cannot be removed
	 mta_error (err_SYS, err_MTA_TRANS_IMP_REMOVE,
			import_to_delete.get_name());
      mta_remove_import_recursively (self, imp);
   }

   TT (mta_H, T_LEAVE);
} // *** sos_Schema_module::remove_import***


// **************************************************************************
void sos_Schema_module::modify_with_extern (sos_Bool yep)
// **************************************************************************
// The flag has_external_import is used to 
// indicate the inclusion of <schema>_ext.h
{  
   T_PROC ("sos_Schema_module::modify_with_extern");
   TT (mta_H, T_ENTER);

   sos_Schema_module sm = self;
   mta_open_for_writing (sm.container());
   sm.set_has_external_import(yep);

   TT (mta_H, T_LEAVE);
} // *** sos_Schema_module::modify_with_extern ***


EXPORT sos_Import_mode mta_uses (const sos_Schema_module& sm,
				 const sos_Type_descr&    td)
// check if there is a type in sm which uses td. 
{  
   T_PROC ("mta_uses");
   TT (mta_H, T_ENTER);

   sos_Import_mode result = sos_IMP_NONE;

      // is sm the same schema as td's schema or 
      // imports td's schema sm ?
   if (   sm.container() == td.container()
       OR VALID(mta_get_import (sm,
				sos_Schema_module::retrieve(td.container()))))
   {
      sos_Type_descr_List types = sm.get_types();
      agg_iterate (types, sos_Type_descr t)
	 sos_Import_mode tmp = mta_uses (t, td);
	 if (tmp != sos_IMP_NONE)
	 {  result = tmp;
	    break;
	 }
      agg_iterate_end (types, t)
   }

   TT (mta_H, T_LEAVE);
   return result;
}

// **************************************************************************
sos_Type_descr sos_Schema_module::create_type (sos_Type tp, sos_String name)
// **************************************************************************
{  
   T_PROC ("sos_Schema_module::create_type");
   TT (mta_H, T_ENTER);

   sos_Schema_module sm     = self;
   sos_Type_descr    result = sos_Type_descr::make (NO_OBJECT);
#ifdef OBST_PROT_ALL
   sos_SchemaModif   smod   = sos_SchemaModif::mod_schema (sm);
#endif

   if (tp == sos_Extern_type_type)
      result = mta_create_extern (sm);
   else if (tp == sos_Class_type_type)
      result = mta_create_class (sm);
   else if (tp == sos_Typedef_type_type)
      result = mta_create_typedef (sm);
   else if (tp == sos_Union_type_type)
      result = mta_create_union (sm);
   else if (tp == sos_Enum_type_type)
      result = mta_create_enum (sm);
   else
      err_raise (err_SYS, err_MTA_INVALID_METHOD, MTA_ERR, FALSE);

#ifdef OBST_PROT_ALL
   if (VALID(smod))
      smod.add_type (result);
#endif
   err_block 
      if VALID (name)
	 result.modify_name (name);
   err_exception
      result.remove();
      result = sos_Type_descr::make (NO_OBJECT);
      mta_repeat_last_error();
   err_block_end

   TT (mta_H, T_LEAVE);
   return result;
} // *** sos_Schema_module::create_type ***


// **************************************************************************
EXPORT sos_Type_descr mta_lookup_type (const sos_String& s)
// **************************************************************************
{
   // scans all schemas for the named type. The first one found is returned.
   T_PROC ("mta_lookup_type");
   TT (mta_M, T_ENTER);
   sos_Type_descr result;

   sos_Schema_module_Directory sd = sos_Schema_module::schema_dir();
   agg_iterate_association (sd, sos_String name, sos_Schema_module sm)
      result = sm.get_type_table() [s];
      if (VALID(result))
	 break;
   agg_iterate_association_end (sd, name, sm)
   
   TT (mta_M, T_LEAVE);
   return result;
}


LOCAL smg_String& mta_next_token_from_string (char **s)
{
   /* This function will first skip all leading whitespaces. Then an
      identifier, a comma or an angle bracket will be read and returned.
      Identifiers are read up to a whitespace, a comma or an angle bracket.
      The passed pointer is made to point on the next non-whitespace character
      following the parsed token.
      The resulting smg_String is managed. It can/must thus be deleted.
      Furthermore the smg_String object is allocated. It must be deleted, too.
   */

   T_PROC ("mta_next_token_from_string");
   TT (mta_L, T_ENTER);
   smg_String *result = new smg_String;
   char tmp [2];
   tmp[1] = '\0';

   while (**s AND strchr (" \t\n", **s))
      *s++; // skip leading whitespaces but not trailing '\0'
   if (strchr (",<>", **s))  // in case of comma or bracket make this the result
   {
      *tmp = *(*s)++;
      *result += tmp; // makes result to be managed
   }
   else
   {
      *result += *s; // manages a copy of the string starting at the first
		     // non-whitespace character
      // Now find identifier end and place a '\0' behind it in the result
      // (to make changes take effect for result, use SMG_BORROW).
      char *id = result->make_Cstring (SMG_BORROW);
      while (!strchr (" \t\n,<>", *id))  // as long as identifier end not found
      {
	 id++;
	 (*s)++; // adapt passed pointer in order to make it point to the
      }          // next token afterwards
      *id = '\0';
   }
   while (**s AND strchr (" \t\n", **s))
      (*s)++; // skip trailing whitespaces, but not trailing '\0'

   TT (mta_L, T_LEAVE);
   return *result;
}

LOCAL sos_Type_descr mta_gen_inst_from_name (const sos_String& name,
					     char**	       n,
					     sos_Type_descr    gen_template)
{
   /* This function tries to retrieve a generic instantiation from the
      passed generic class (it is checked here to be one) and the
      actual generic parameters provided in angle brackets separated by
      commas. The first angle bracket has already been read, so *n should
      point to the name of the first actual generic parameter.
      If an error occurres (e.g. a syntactical one), NO_OBJECT is returned.
      Otherwise the generic instantiation class type is the result.
   */
   T_PROC ("mta_gen_inst_from_name");
   TT (mta_M, T_ENTER);

   sos_Type_descr      result = sos_Type_descr::make (NO_OBJECT);
   sos_Type_descr_List agpl = sos_Type_descr_List::create (TEMP_CONTAINER);
   sos_Type_descr      gp;
   smg_String 	       *next_token = (smg_String*) NULL;

   if (INVALID(gen_template) OR  NOT gen_template.has_type(sos_Class_type_type))
      mta_error (err_SYS, err_MTA_CLASS_EXPECTED);

   sos_Class_type gen_class = sos_Class_type::make (gen_template);
   if (NOT gen_class.is_generic_class())
      mta_error (err_SYS, err_MTA_GENERIC_CLASS_EXPECTED, gen_class.get_name());
   
   // Now we can be sure, that gen_class is really a generic class.
   // So now parse one type name recursively:
   do
   {
      gp = mta_get_td_from_name (name, n);
      if (INVALID(gp))
	 result = gp; // error in mta_get_td_from_name, no result can be found
      else
      {
	 agpl.append (gp);
	 if (next_token)       // discard previously read token
	    delete next_token;
	 next_token = &mta_next_token_from_string (n);
      }
   } while (is_comma (*next_token));
   if (NOT is_abr_right (*next_token))
      err_raise (err_SYS, err_MTA_SYNTAX);
   delete next_token;

   // Now make instantiation from generic class and actual parameter list.
   // Since only existing instantiations have to be found, it is sufficient
   // to determine the instantiation name and look through the type tables
   // with the so-constucted name.
   result = mta_lookup_type (mta_instantiation_name(gen_class.get_name(),agpl));

   TT (mta_M, T_LEAVE);
   return result;
}

LOCAL sos_Type_descr mta_get_td_from_name (const sos_String& name,
					   char**	     n /* = NULL */)
{
   T_PROC ("mta_get_td_from_name");
   TT (mta_M, T_ENTER);

   sos_Type_descr result = sos_Type_descr::make (NO_OBJECT);
   smg_String     *next_token;
   sos_String	  tmp_string;
   char		  *name_as_Cstring;  // only assigned on 1st recursion level!!!
   sos_Bool	  created_n = FALSE;
   
   if (!n)  // First recursion level, create n!
   {
      created_n = TRUE;
      name_as_Cstring = name.make_Cstring();
      n = &name_as_Cstring;
   }

   next_token = &mta_next_token_from_string (n);
   tmp_string = next_token->make_String (TEMP_CONTAINER);
   result     = mta_lookup_type (tmp_string);
   tmp_string.destroy();
   delete next_token;
   if (**n) // Not at end of string, look what comes next
   {
      if (**n == '<')  // read generic instantiation
      {
	 next_token = &mta_next_token_from_string (n); // skip the '<'
	 result     = mta_gen_inst_from_name (name, n, result);
	 delete next_token;
      }
      // Now we have seen a type name followed by something which is
      // not '<'. If it is ',' or '>' it is OK in case we are in a recursion,
      // otherwise it is an error.
      else if (created_n OR (**n != ',' AND **n != '>'))
	 mta_error (err_SYS, err_MTA_INVALID_TYPE_NAME, name); 
   }

   if (created_n) // If this incarnation created the Cstring, delete it
   {		  // Further, since this is the 1st recursion level, 
		  // check, if end of string is reached!
      if (**n)
	 mta_error (err_SYS, err_MTA_INVALID_TYPE_NAME, name);
      delete name_as_Cstring;
   }

   TT (mta_M, T_LEAVE);
   return result;
}


// **************************************************************************
sos_Type_descr sos_Schema_module::type_from_name (sos_String name)
// **************************************************************************
{
   T_PROC ("sos_Schema_module::type_from_name")
   TT (mta_H, T_ENTER);

   sos_Type_descr result = mta_get_td_from_name (name);

   TT (mta_H, T_LEAVE);
   return result;
}
