#line 1 "./src/mta/mta_out.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST Projekt", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
*/
/* OBST LIBRARY MODULE */

#define OBST_IMP_STREAM
#define OBST_IMP_TIME
#include "obst_stdinc.h"

#include "mta.h"
#define _AGG_MAX_LINE_LENGTH 60


LOCAL smg_String mta_space (int i)
{  T_PROC ("mta_space");
   TT (mta_L, T_ENTER);
   
   smg_String s;
   while (i-- > 0)
      s += " ";

   TT (mta_L, T_LEAVE);
   return s;
}

LOCAL sos_Bool mta_skip_line (const sos_String_List& sl, 
			      sos_Int                indentation,
			      sos_Int                prob_add,
			      smg_String&            s)
{  
   T_PROC ("mta_skip_line");
   TT (mta_M, T_ENTER);

   if (s.length() + prob_add > _AGG_MAX_LINE_LENGTH)
   {  sl.append (s.make_String (TEMP_CONTAINER));
      s = mta_space (indentation);
      return TRUE;
   }
   TT (mta_M, T_LEAVE);
   return FALSE;
}  

LOCAL inline void mta_skip_line (const sos_String_List& sl,
				 smg_String&     	s)
{   mta_skip_line (sl, 0, _AGG_MAX_LINE_LENGTH+1, s);
}

LOCAL smg_String mta_make_smg_String (sos_Method_kind mk)
{  T_PROC ("mta_make_smg_String (sos_Method_kind)");
   TT (mta_L, T_ENTER);
   
   smg_String s;
   switch (mk)
   {  case sos_PRIVATE  : s = "private";  break;
      case sos_PUBLIC   : s = "public";   break;
      case sos_PROTECTED: s = "protected";break;
   }

   TT (mta_L, T_LEAVE);
   return s;
}

LOCAL smg_String mta_make_smg_String (sos_Method_kind get,
				      sos_Method_kind set)
{
   T_PROC ("mta_make_smg_String (sos_Method_kind, sos_Method_kind)");
   TT (mta_M, T_ENTER);

   smg_String s = mta_make_smg_String (get);
   if (get != set)
      s += smg_String (", ") + mta_make_smg_String(set) + " set";

   TT (mta_M, T_LEAVE);
   return s;
}

LOCAL smg_String mta_make_smg_String (const sos_Expr& e)
{  T_PROC ("mta_make_smg_String (sos_Expr)");
   TT (mta_M, T_ENTER);
   
   smg_String s;
   if (VALID(e))
      if (e.has_type (sos_Identifier_type))
	 s = smg_String(sos_Identifier::make (e).get_id());
      else
         s = smg_String (sos_Int_expr::make (e).get_value());

   TT (mta_M, T_LEAVE);
   return s;
}

LOCAL smg_String mta_new_method_kind (sos_Bool	       first,
				      sos_Bool&	       new_mk,
				      sos_Method_kind& old_get_mk,
				      sos_Method_kind& old_set_mk,
				      sos_Method_kind  new_get_mk,
				      sos_Method_kind  new_set_mk)
// used as local function to mta_class_declaration. A string
// containing a method kind in the schema syntax is created if it`s
// the first method kind in a class declaration (first = TRUE) or the
// kind differs from the old kind. Furthermore the parameter old_g/set_mk
// are set to new_g/set_mk
{  T_PROC ("mta_new_Method_kind");
   TT (mta_H, T_ENTER);
   
   smg_String s;
   if (new_mk = (sos_Bool) (   first
                 	    OR new_set_mk != old_set_mk
			    OR new_get_mk != old_get_mk))
   {  s += mta_make_smg_String (new_get_mk, new_set_mk) + ":";
   }
   old_get_mk = new_get_mk;
   old_set_mk = new_set_mk;

   TT (mta_H, T_LEAVE);
   return s;
}

LOCAL sos_String_List mta_method_declaration (const sos_Method& m)
// creates a string containing the declaration of a method. with_kind = TRUE
// add the method kind to the string (private, protected, public)
{  
   T_PROC ("mta_method_declaration");
   TT (mta_H, T_ENTER);
   
   sos_String_List result = sos_String_List::create (TEMP_CONTAINER);
   smg_String s = mta_make_smg_String (m.get_kind()) + ":";

   mta_skip_line (result, s);
   s += "   ";
   sos_Int indentation = 3;

   if (m.get_is_abstract())
      s += "abstract ";
   if (m.get_is_static())
      s += "static ";

   s += smg_String(mta_get_name_declaration (m.get_result_type())) + " ";
   indentation = s.length()+3;

   if (m.get_is_operator())
      s += "operator";

   s += smg_String(m.get_name());

   sos_Bool	  first = TRUE;
   sos_Param_List pl    = m.get_params();
   if (VALID (pl))
   {  agg_iterate(m.get_params(),sos_Param p)
      {  
	 sos_String ptypename = mta_get_name_declaration(p.get_type());
         smg_String pstr(ptypename);
	 ptypename.destroy();

         if (p.get_is_ref())
            pstr += "&";
         if VALID (p.get_name())
            pstr += smg_String(" ") + p.get_name();

         sos_Expr expr = p.get_default_expr();
         if VALID (expr)
            pstr += smg_String(" = ") + mta_make_smg_String (expr);

	 if (first)
         {  first = FALSE;
	    mta_skip_line (result, indentation, 2+pstr.length(), s);
            s += " (";
	    indentation = s.length();
	 }
         else
            s += ", ";
  
         mta_skip_line (result, indentation, pstr.length(),s);
	 s += pstr;
      }
      agg_iterate_end(pl, p);
   } 
   else
      s += "(";

   s += ");";
   mta_skip_line (result, s);
 
   TT (mta_H, T_LEAVE);
   return result;
}

LOCAL sos_String_List mta_comp_declaration (const sos_Comp_descr& cd)
// results in a string of the form
// [<get_kind>[ <set_kind> set]:<comp_type> <comp_name> [= <comp_init>]
{  
   T_PROC ("mta_comp_declaration");
   TT (mta_H, T_ENTER);

   sos_String_List result = sos_String_List::create (TEMP_CONTAINER);

   sos_Method get_method = cd.get_get_method();
   sos_Method set_method = cd.get_set_method();

   smg_String s = mta_make_smg_String (get_method.get_kind(),
				       set_method.get_kind())
      		  + ": ";
   mta_skip_line (result, s);
   s += "   ";

   if (cd.get_is_local())
      s += "local ";
   sos_String result_decl = mta_get_name_declaration(
				   get_method.get_result_type());
   s += smg_String(result_decl) + " ";
   sos_Int indentation = s.length()+3;
   result_decl.destroy();

   s += cd.get_name();

   sos_Expr expr = cd.get_init_expr();
   if (VALID (expr))
   {  smg_String init_str = smg_String(" = ") + mta_make_smg_String (expr);

      mta_skip_line (result, indentation, init_str.length(), s);
      s += init_str;
   }
   s += ";";
   mta_skip_line (result, s);

   TT (mta_H, T_LEAVE);
   return result;
}


LOCAL sos_String mta_genparamname_declaration (const sos_Gen_param& gp)
{  T_PROC ("mta_genparamname_declaration");
   TT (mta_M, T_ENTER);
   
   sos_String result = sos_String::create (TEMP_CONTAINER);
   result.assign (gp.get_name());
   sos_Type_descr st = gp.get_super_type();
   if (VALID (st))
   {  sos_String ss = mta_get_name_declaration(st);
      smg_String s  = smg_String(" ") + ss;

      result += s.make_String (TEMP_CONTAINER);
      ss.destroy();
   }

   TT (mta_M, T_LEAVE);
   return result;
} 

LOCAL sos_String mta_genericname_declaration
				(const sos_Generic_instantiation& gi)
{  
   T_PROC ("mta_genericname_declaration");
   TT (mta_M, T_ENTER);

   smg_String s = smg_String(gi.get_gen_class().get_name()) + "<";

   sos_Bool first = TRUE;
   sos_Type_descr_List tdl = gi.get_act_gen_params();
   agg_iterate (tdl, sos_Type_descr td)
      if (first)
         first = FALSE;
      else
	 s += ", ";
      s += mta_get_name_declaration(td);
   agg_iterate_end (tdl, tdl)
   s += ">";
   
   TT (mta_M, T_LEAVE);
   return s.make_String (TEMP_CONTAINER);
}


LOCAL sos_String_List mta_class_declaration(const sos_Class_type& ct)
// creates a list of strings containing the schema-like
// declaration of a class
{  T_PROC ("mta_class_declaration");
   TT (mta_H, T_ENTER);
   
   sos_String_List result = sos_String_List::create (TEMP_CONTAINER);
   sos_Bool first;
 
      // name
   smg_String s;
   if (ct.get_is_abstract())
      s = "abstract ";
   s += "class ";
   sos_Int indentation = s.length();

   sos_String class_name= MTA_GET_NAME_DECLARATION(ct);
   s += class_name;
 
     // create parameter
   sos_Param_List pl = ct.get_create_params();
   if (VALID (pl))
   {  
      first = TRUE;
      agg_iterate (pl, sos_Param p)
      {     // build string of sos_Param p
	 smg_String pstr(p.get_type().make_type().get_name());

	 sos_String pname = p.get_name();
         if (VALID (pname))
            pstr += smg_String(" ") + pname;

         if (p.get_is_ref())
            pstr += "&";

         sos_Expr e = p.get_default_expr();
	 if (VALID (e))
            pstr += smg_String(" = ") + mta_make_smg_String (e);
	 
	 if (first)
         {  first = FALSE;
	    mta_skip_line (result, indentation, 1+pstr.length(), s);
	    s += "(";
	    indentation = s.length();
	 }
         else
            s += ", ";

         mta_skip_line (result, indentation, pstr.length(), s); 
         s += pstr;
      }
      agg_iterate_end (pl, p)
      s += ")";
   }

   indentation=0;
      // super_classes
   sos_Super_class_List scl = ct.get_super_classes();
   if (VALID (scl))
   {  first = TRUE;
      agg_iterate (scl, sos_Super_class sc)
      {  
	 sos_Bool create_par_skipped = FALSE;
         sos_Type_descr super_class = sc.get_super_class();
         if (super_class.make_type() != sos_Object_type)
         {  
            sos_Type_descr sctd   = sc.get_super_class();
            sos_Expr_List  scpl   = sc.get_create_params();
            sos_String     scname = mta_get_name_declaration(sctd);
	    smg_String     scstr(scname);
            if (first)
            {  first = FALSE;
	       
               sos_Int crparadd = 0;
	       if (VALID (scpl))
               {  crparadd += 2;
		  agg_iterate (scpl, sos_Expr e)
	             crparadd += mta_make_smg_String(e).length();
                  agg_iterate_end (scpl, e)
               }
               mta_skip_line(result, indentation, 1+scstr.length()+crparadd, s);
               s += ":  ";
               indentation = 6; // eventual skipped create parameter
            }
            else
	    {  s += ", ";
	       if (create_par_skipped)
	       {  mta_skip_line (result, s);
                  create_par_skipped = TRUE;
               }
	    }

	    mta_skip_line (result, indentation, scstr.length(), s);     
            s += scstr; 
            scname.destroy();

            if (VALID(scpl))
            {  
               sos_Bool crfirst = TRUE;
               agg_iterate (scpl, sos_Expr e)
               {  smg_String crstr(mta_make_smg_String (e));
		  if (crfirst)
                  {  crfirst = FALSE;
		     if (mta_skip_line(result,indentation, crstr.length()+1, s))
			create_par_skipped = TRUE;
		     s += "(";
		     indentation = s.length();
		  }
                  else
                     s += ", ";
                  if (mta_skip_line (result, indentation, crstr.length(),s))
		     create_par_skipped = TRUE;
                  s += crstr;
               }
               agg_iterate_end (scpl, e)
               s += ")";
            }
	    indentation = 3;
         }
      }
      agg_iterate_end (scl, sc)
   }

   mta_skip_line (result, s);
   indentation = 3;
   s += "{  ";
 
      // friends
   sos_Bool something_inside = FALSE;
   sos_Type_List friends = ct.get_friends();
   if (VALID (friends))
      if (friends.card() != 0)
      {  first = TRUE;
	 agg_iterate (friends, sos_Type n)
         {  smg_String 	   friend_str;
	    sos_Class_type fct = sos_Class_type::make(n);
	    if (VALID(fct.get_generic_class()))
	       friend_str += fct.get_generic_class().get_name();
	    else
	       friend_str += fct.get_name();

	    if (first)
	    {  first = FALSE;
	       mta_skip_line (result, indentation, friend_str.length()+13, s);
	       s += "friend class ";
	       something_inside = TRUE;
	       indentation = s.length();
	    }
	    else
	       s += ", ";

	    mta_skip_line (result, indentation, friend_str.length(), s);
	    s += friend_str;
         }
         agg_iterate_end (friends, n)
         s += ";";
	 mta_skip_line (result, s);
	 s += "   ";
      }
 
      // local_methods
   sos_Method_List lml = ct.get_local_methods();
   sos_Method_kind old_set_kind, old_get_kind, new_get_kind;
   sos_Method_kind new_set_kind = sos_PRIVATE;
   sos_Bool	   new_mk;

   first = TRUE;
   indentation = 3;
   agg_iterate (lml, sos_Method m)
   {  if (    INVALID(m.get_comp_descr())
          AND m.get_defined_in() == ct
          AND NOT m.get_is_generated())
      {  
	 something_inside = TRUE;
	 new_get_kind = m.get_kind();
	 s += mta_new_method_kind (first, new_mk,
				   old_get_kind, old_set_kind,
				   new_get_kind, new_get_kind);
         if (new_mk)
         {  mta_skip_line (result, s);
	    s += "   ";
         }
         sos_String_List method_decl = mta_method_declaration(m);
         for (; method_decl.card() > 1;  method_decl.remove(2))
	 {  s += method_decl.get_nth(2);
            mta_skip_line (result, s);
	    s += "   ";
         }
         MTA_DESTROY_LIST (method_decl);
         first = FALSE;
      }
   }
   agg_iterate_end (lml, m);

      // components
   first = TRUE;
   sos_Comp_descr_List comps = ct.get_components();
   agg_iterate (comps, sos_Comp_descr cd)
   {  
      sos_Method get_method = cd.get_get_method();
        // don't display predefined components (type_id)
      if (get_method.get_defined_in() == ct)
      {  
	 something_inside = TRUE;
         sos_Method set_method = cd.get_set_method();
         new_get_kind = get_method.get_kind();
         new_set_kind = set_method.get_kind();
         s += mta_new_method_kind (first, new_mk,
				   old_get_kind, old_set_kind,
				   new_get_kind, new_set_kind);
         if (new_mk)
	 {  mta_skip_line (result, s);
	    s += "   ";
         }
         sos_String_List comp_decl = mta_comp_declaration(cd);
         for (; comp_decl.card() > 1;  comp_decl.remove(2))
	 {  s += comp_decl.get_nth(2);
            mta_skip_line (result, s);
	    s += "   ";
         }
	 MTA_DESTROY_LIST (comp_decl);
	 first = FALSE;
      }
   }
   agg_iterate_end (comps, cd)

      // remove indentation (spaces)
   if (something_inside)
      s = "";

   s += smg_String("}; // *** class ") + class_name + " ***";
   mta_skip_line (result, s);

   TT (mta_H, T_LEAVE);
   return result;
}

LOCAL sos_String_List mta_classname_declaration(const sos_Class_type& ct,
						sos_Int		      pos,
						sos_Bool 	      skip_line)
// creates a list with the name including actual or formal parameters
{  T_PROC ("mta_classname_declaration (sos_Class_type, sos_Int, sos_Bool)");
   TT (mta_M, T_ENTER);
   
   sos_String_List result = sos_String_List::create (TEMP_CONTAINER);
   smg_String s;
   sos_Bool first = TRUE;

   if (VALID(ct.get_generic_class()))
      s = ct.get_generic_class().get_name();
   else
      s = ct.get_name();
   sos_Int indentation = s.length()+pos;

      // generic parameter
   sos_Gen_param_List gpl = ct.get_formal_gen_params();
   if (VALID (gpl))
   {  agg_iterate (gpl, sos_Gen_param gp)
      {  smg_String gpstr(gp.get_name());
         sos_Type t = gp.make_type();
         if (t != sos_Object_type)
            gpstr += smg_String(":") + gp.make_type().get_name();

	 if (first)
         {  first = FALSE;
	    if (skip_line)
	       mta_skip_line (result, indentation, gpstr.length()+1,s);
            s += "<";
	    indentation = s.length()+pos;
         }
         else
            s += ", ";

	 if (skip_line)
	    mta_skip_line (result, indentation, gpstr.length()+1,s);
         s += gpstr;
      }
      agg_iterate_end (gpl, gp)
      s += ">";
   }
   else
   {  sos_Type_descr_List agpl = ct.get_actual_gen_params ();
      if (VALID (agpl))
      {  agg_iterate (agpl, sos_Type_descr tn)
         {  smg_String tnstr(mta_get_name_declaration(tn));

            if (first)
            {  first = FALSE;
               if (skip_line)
                  mta_skip_line (result, indentation, tnstr.length()+1, s);
               s += "<";
               indentation = s.length()+pos;
            }
            else
               s += ", ";
            if (skip_line)
               mta_skip_line (result, indentation, tnstr.length()+1,s); 
            s += tnstr;
         }
         agg_iterate_end (agpl, tn);
         s += ">";
      }
   }
   mta_skip_line (result, s);

   TT (mta_M, T_LEAVE);
   return result;
}

LOCAL sos_String mta_classname_declaration (const sos_Class_type& ct)
// creates a list with the name including actual or formal parameters
{  
   T_PROC ("mta_classname_declaration (sos_Class_type)");
   TT (mta_M, T_ENTER);

   sos_String_List sl = mta_classname_declaration (ct, 0 ,FALSE);
      // sl contains exactly one sos_String 
   sos_String result = sl.get_nth(1);
   sl.destroy();

   TT (mta_M, T_LEAVE);
   return result;
}

 
LOCAL sos_String_List mta_enum_declaration (const sos_Enum_type& et)
// creates a string with the schema declaration of an enumeration:
//     enum <name> {<literal1>, <literal2>, ... ,<literaln>};
{   
   T_PROC ("mta_enum_declaration");
   TT (mta_H, T_ENTER);

   sos_Container     cnt    = et.container();
   sos_Schema_module sm	    = sos_Schema_module::retrieve (cnt);
   sos_String_List   result = sos_String_List::create (TEMP_CONTAINER);
 
   smg_String s = smg_String("enum ") + et.get_name() + " {";
   sos_Int indentation = s.length();

   sos_String_List literals = et.get_literals();
   sos_Bool first = TRUE;
   agg_iterate(literals,sos_String n)
   {  if (first)
         first = FALSE;
      else
         s += ", ";
      mta_skip_line (result, indentation, n.get_length(),s);
      s += smg_String(n);
   }
   agg_iterate_end(literals,n)
   s+= "};";
   mta_skip_line (result, s);

   TT (mta_H, T_LEAVE);
   return result;
}

LOCAL sos_String_List mta_extern_declaration (const sos_Extern_type& et)
{
   T_PROC ("mta_extern_declaration");
   TT (mta_H, T_ENTER);

   sos_String_List result = sos_String_List::create (TEMP_CONTAINER);
   smg_String s = smg_String("extern ") + et.get_name()
      		  + " (" + et.get_object_size() + ");";
   result.append (s.make_String (TEMP_CONTAINER));
 
   TT (mta_H, T_LEAVE);
   return result;
}
 
LOCAL sos_String_List mta_union_declaration (const sos_Union_type& ut)
{  
   T_PROC ("mta_union_declaration");
   TT (mta_H, T_ENTER);

   sos_Container   cnt	      = ut.container();
   sos_String_List result     = sos_String_List::create (TEMP_CONTAINER);
   sos_String      union_name = MTA_GET_NAME (ut);

   smg_String s = smg_String("union ") + union_name;
   mta_skip_line (result, s);


   s += "{";

   sos_Type_List united = ut.get_united();
   sos_Bool first = TRUE;
   agg_iterate (united, sos_Type t)
   {  s += "  ";
      if (first)
         first = FALSE;
      else
         s += " ";
      sos_String tmp_sos_str = mta_get_name_declaration(
					   sos_Type_descr::make (t));
      s += tmp_sos_str;
      tmp_sos_str.destroy();
      s += ";";
      mta_skip_line (result, s);
   }
   agg_iterate_end (united, t)
   s += smg_String("}; // *** union ") + smg_String (union_name) + " ***";
   mta_skip_line (result, s);

   TT (mta_H, T_LEAVE);
   return result;
}

LOCAL sos_String_List mta_typedef_declaration (const sos_Typedef_type& td)
{  T_PROC ("mta_typedef_declaration");
   TT (mta_H, T_ENTER);
   
   sos_String_List result = sos_String_List::create (TEMP_CONTAINER);
   sos_Container cnt = td.container();
 
   sos_String tmp_sos_str = mta_get_name_declaration(
				   sos_Type_descr::make(td.get_named_type()));
   smg_String s = smg_String("typedef ") + tmp_sos_str
      					 + " " + td.get_name() + ";";
   tmp_sos_str.destroy();
   result.append (s.make_String (TEMP_CONTAINER)); 

   TT (mta_H, T_LEAVE);
   return result;
}
 
// **************************************************************************
EXPORT sos_String_List mta_get_declaration(const sos_Type_descr& td)
// **************************************************************************
{  T_PROC ("mta_get_declaration");
   TT (mta_H, T_ENTER);
   
   sos_String_List result;
   if (td.has_type(sos_Enum_type_type))
      result = mta_enum_declaration (sos_Enum_type::make (td));
   else if (td.has_type (sos_Class_type_type))
      result = mta_class_declaration (sos_Class_type::make(td));
   else if (td.has_type(sos_Extern_type_type))
      result = mta_extern_declaration (sos_Extern_type::make (td));
   else if (td.has_type(sos_Union_type_type))
      result = mta_union_declaration (sos_Union_type::make(td));
   else if (td.has_type(sos_Typedef_type_type))
      result = mta_typedef_declaration (sos_Typedef_type::make (td));
   else if (td.has_type(sos_Typedef_type_type))
      result = mta_typedef_declaration (sos_Typedef_type::make (td));
   else 
   {  smg_String errmsg(":get_declaration");
      errmsg += td.make_type().get_name();
      mta_error (err_SYS,
		 err_MTA_INVALID_METHOD, errmsg.make_Cstring(SMG_BORROW));
   }
      
   TT (mta_H, T_LEAVE);
   return result;
} // ** mta_get_declaration ***

// **************************************************************************
EXPORT sos_String mta_get_name_declaration(const sos_Type_descr& td)
// **************************************************************************
{  T_PROC ("mta_get_name_declaration");
   TT (mta_H, T_ENTER);
   
   sos_String result;

   if (td.has_type(sos_Class_type_type))
      result = mta_classname_declaration (sos_Class_type::make (td));
   else if (td.has_type(sos_Gen_param_type))
      result = mta_genparamname_declaration (sos_Gen_param::make (td));
   else if (td.has_type(sos_Generic_instantiation_type))
      result = mta_genericname_declaration (
				sos_Generic_instantiation::make (td));
   else
   {
      result = sos_String::create (TEMP_CONTAINER);
      result.assign (td.make_type().get_name());
   }

   TT (mta_H, T_LEAVE);
   return result;
} // ** mta_get_declaration ***

// **************************************************************************
EXPORT void mta_schema_declaration (ostream& out, const sos_Schema_module& sm)
// **************************************************************************
{
   sos_String_List result = sos_String_List::create (TEMP_CONTAINER);
   smg_String	   s;

   sos_Import_List imports = sm.get_imports();
   sos_Bool first = TRUE;
   agg_iterate (imports, sos_Import imp)
      if (first)
      {  first = FALSE;
         out << "with ";
      }
      else
         out << ", ";
      out << imp.get_module().get_name();
   agg_iterate_end (imports, imp)
   if (imports.card() != 0)
      out << ";\n";

   if (sm.get_has_external_import())            // with extern
      out << "with extern;\n";
   out << "\nschema " << sm.get_name() << "\n" // schemas' name
                                               // schema bracket:
       << "{\n";

   sos_Type_descr_List types = sm.get_types(); // schema types:
   agg_iterate (types, sos_Type_descr td)
          // Don't display generic instantiations
       sos_Bool instantiation = FALSE;
       if (td.has_type (sos_Class_type_type))
          instantiation = sos_Class_type::make(td).is_instantiation();

       if (NOT instantiation)
       {  sos_String_List tmp_sos_str_list = mta_get_declaration(td);

          agg_iterate (tmp_sos_str_list, sos_String ss)
             out << "   " << ss << "\n";           // indent the types
          agg_iterate_end (tmp_sos_str_list, ss)

          MTA_DESTROY_LIST(tmp_sos_str_list);
          out << "\n";
       }
    agg_iterate_end (types, td);

    timeval tp;
    gettimeofday(&tp, NULL);
    char* timestr = ctime(&tp.tv_sec);
                                                  // schema bracket:
    out << "} // *** schema " << sm.get_name() << " ***\n"
	<< "  // by OBST" << obst_version << " at " << timestr;

} // *** mta_schema_declaration ***


// **************************************************************************
EXPORT void mta_generate_schema_file (const sos_Schema_module& sm)
// **************************************************************************
{
   // generate <schema>.obst.new

   smg_String filename = smg_String(sm.get_name()) + ".obst.new";

   filebuf schemafilebuf;
   schemafilebuf.open (filename.make_Cstring(SMG_BORROW), output);
   ostream schemafile(&schemafilebuf);

   mta_schema_declaration(schemafile,sm);
}
