/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST Projekt", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */

#include "obst_progstd.h"
#include "mta_use.h"
#include "mta.h"
#include "mta_err.h"
#include "smg.h"
#define OBST_IMP_FORMATTED_IO
#include "obst_stdinc.h"

LOCAL  sos_Access_mode   mta_open_mode;	// initialized in mta_open_for_writing

/*
EXPORT sos_Container_set mta_impl_opened_cnt;
	     // defined in obst_globals.C because sos_Container_set has *tors.
*/


// **************************************************************************
EXPORT void mta_open_status (const sos_Access_mode& am)
// **************************************************************************
// specifies the mode in which the implicit opened 
// container are opened.
{  T_PROC ("mta_open_status");
   TT (mta_H, T_ENTER);

   mta_open_mode = am;

   TT (mta_H, T_LEAVE);
} 

// **************************************************************************
EXPORT void mta_open_for_writing (const sos_Container& cnt)
// **************************************************************************
{  T_PROC ("mta_open_for_writing")
   TT (mta_H, T_ENTER);

   static int initialized /* = 0 */;
   if (initialized == 0)
   {  mta_open_mode = WRITING;
      initialized   = 1;
   }

   if (mta_open_mode == READING)
   {  smg_String errmsg(":");
      if (cnt == TEMP_CONTAINER)
	 errmsg += "TEMP_CONTAINER";
      else if (cnt == ROOT_CONTAINER)
	 errmsg += "ROOT_CONTAINER";
      else
      {  sos_Object root = cnt.root_object();
	 if (root.has_type (sos_Schema_module_type))
	 {
	    errmsg += "schema container";
	    errmsg += smg_String(sos_Schema_module::make (root).get_name())
		   +  ":" + (sos_Int)cnt;
	 }
	 else
	 {
	    char cid_string [100];
	    sprintf (cid_string, "%d", (sos_Int) cnt);
	    errmsg += cid_string;
	 }
      }
      mta_error (err_SYS,
		 err_MTA_UNOPENED_CONTAINER, errmsg.make_Cstring(SMG_BORROW));
   }
   if (mta_open_mode == WRITING  OR  mta_open_mode == CHECKOUT)
   {  if (cnt != TEMP_CONTAINER)
      {  if (cnt.status() != WRITEABLE)
            cnt.open (mta_open_mode, WAITING);
         mta_impl_opened_cnt += cnt;

	 TT (mta_L, TS ("Container appended to mta_impl_opened_cnt: "));
	 TT (mta_L, TU ((sos_Int) cnt));
      }   
   }
   TT (mta_H, T_LEAVE);
}

// **************************************************************************
EXPORT void mta_reset()
// **************************************************************************
// resets all container that are used from schema-modifying commands,
// that means that all modifies exists no more
{  T_PROC ("mta_reset");
   TT (mta_H, T_ENTER);

   cnt_iterate (mta_impl_opened_cnt, sos_Container cnt)
      cnt.reset();
   cnt_iterate_end (mta_impl_opened_cnt, cnt)

   TT (mta_H, T_LEAVE);
}


// **************************************************************************
EXPORT void mta_commit()
// **************************************************************************
// commits all container that are used from schema-modifying commands.
{  T_PROC ("mta_commit");
   TT (mta_H, T_ENTER);

   cnt_iterate (mta_impl_opened_cnt, sos_Container cnt)
   {   
       TT (mta_L, TS ("Closing container #"));
       TT (mta_L, TU ((sos_Int) cnt));

       cnt.close();
   } cnt_iterate_end (mta_impl_opened_cnt, cnt);

   mta_impl_opened_cnt.clear();

   TT (mta_H, T_LEAVE);
}


// **************************************************************************
EXPORT smg_String mta_generate_valid_name
                       (const smg_String& basic_name,
                        const sos_Object& o,
                        sos_Bool _valid(const sos_Object&, const sos_String&))
// **************************************************************************
// Searches for a valid name of the form basic_name_<number>
{  T_PROC ("mta_generate_valid_name");
   TT (mta_H, T_ENTER);
   
   sos_String name  = basic_name.make_String(TEMP_CONTAINER);
   smg_String name_ = smg_String(basic_name) + "_";

   for (sos_Int no = 1;  NOT _valid(o, name);  no++)
   {  smg_String name_smg = name_ + no;
      name.assign_Cstring(name_smg.make_Cstring(SMG_BORROW));
   }
   smg_String result(name);
   name.destroy();

   TT (mta_H, T_LEAVE);
   return result;
} 
 
// **************************************************************************
EXPORT void mta_destroy_list (const sos_Object_List& sl)
// **************************************************************************
{  T_PROC ("mta_destroy_list");
   TT (mta_H, T_ENTER);
   
   if (VALID (sl))
   {  agg_iterate (sl, sos_Object o)
	 o.destroy();
      agg_iterate_end (sl, o);
      sl.destroy();
   }

   TT (mta_H, T_LEAVE);
}

// **************************************************************************
EXPORT sos_Bool internal_remove_from_list
			(const sos_Object_List& l,
                         const sos_Object&      o)
// **************************************************************************
{  T_PROC ("internal_remove_from_list");
   TT (mta_H, T_ENTER);
   
   sos_Int  ind   = l.find(o);
   sos_Bool found = (sos_Bool) (ind>0);
   if (found)
      l.remove (ind);
                 // True, if o could be removed from l

   TT (mta_H, T_LEAVE);
   return found;
}
