/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// OBST gsh
// **************************************************************************

#include "graph_ut.h"
#include "obst_trc.h"

#include "obst.h"
#include "mta_use.h"
#include "dir_use.h"
#include "cci_use.h"

// for PSM statistics in GshApplication::display_root
#include "_obst_config.h"
#include "psm_util.h"

// ---------------------------------------------------------------------------
// ************************  Classes  ****************************************
// ---------------------------------------------------------------------------

struct object_menu_comp
{  sos_Bool   is_method;
   sos_Object o;	// isa sos_Method, if is_method
};

struct what_to_show
{  sos_Bool cmp, met;
   int operator== (what_to_show &s) {return cmp==s.cmp AND met == s.met;}
   int operator!= (what_to_show &s) {return cmp!=s.cmp OR met != s.met;}
};

class GshKnownObject : public KnownObject
{ 
  friend class GshItemList;

   object_menu_comp *omc;
  public:
   what_to_show     show;

  protected:
   void item_selected (int);
  public:
   GshKnownObject() { Application::enter_ptr (this, (KnownObject*)this); }
   ~GshKnownObject();

   static void 		  display (sos_Object, WdgtPosition, what_to_show);
   static GshKnownObject* make (KnownObject*);

#ifndef OBST_CAN_OVERLOAD_INHERITED
   void display (sos_Bool b, WdgtPosition w) { KnownObject::display (b, w); }
#endif
  private:
   static void display_class_instance (sos_Object, what_to_show);
   static void display_component (sos_Object, sos_Method, what_to_show);
};

class GshKnownObjectList : public KnownObjectList
{
  public:
   GshKnownObjectList() 
   		   { Application::enter_ptr (this, (KnownObjectList*)this); }

   void  insert (GshKnownObject*);

   static GshKnownObjectList* make (KnownObjectList*);
};

class GshItemList
{ friend class KnownObject;

  class ObjectItem
  { public:
     object_menu_comp omc;
     char             *descr;
  };
 
  private:
   ObjectItem *items;
   int        size;
   int        nr_items;

   static int compare_items (_qsort_const_ void*, _qsort_const_ void*);
  public:
   GshItemList() { items = NULL; nr_items = 0; size = 0; }
   
   void	append (sos_Object, sos_Bool, char*);
   void make_KnownObject (GshKnownObject*);  
};

class GshObjectCmdMenu : public ObjectCmdMenu
{ 
  enum GshObjectCmd { ocmd_FORMAT, ocmd_CMP, ocmd_MET, ocmd_CLOSE };

  public:
   void install();

  protected:
   void do_it (int); 
};

class MethodDialog
{
  public:
   void              activate (sos_Object, sos_Method, int);
   static sos_Bool   convert (char*, sos_Type, sos_Object&);
   static smg_String convert (sos_Object);
   static sos_Bool   exec_method (sos_Method, sos_Object, sos_Object_Array, 
				  sos_Object&);
};

class GshMenuSystem : public MenuSystem
{  enum MenuID     { menu_FILE, menu_VIEW, menu_CONTAINER };
   enum FileItemID { file_STD_ROOT, file_INIT_ROOT, file_OBJECT, file_QUIT };
   enum ViewItemID { view_CMP, view_MET, view_CLOSE_ALL };
   enum CntItemID  { cnt_OPEN_READ, cnt_OPEN_WRITE, cnt_COMMIT, cnt_RESET,
		     cnt_CLOSE, cnt_CREATE, cnt_DESTROY,
		     cnt_RESET_ALL, cnt_COMMIT_ALL };

  protected:
   void set_view_list (what_to_show);
   void do_it (int, int);
};
  
class GshApplication : public Application
{
 public:
   what_to_show current_show;

   GshApplication() { current_show.cmp = TRUE;
		      current_show.met = FALSE;
		      Application::enter_ptr (this, (Application*)this); }

   void display_root (sos_Bool);
   void init (int, char**);

   static GshApplication* make();
};


// ---------------------------------------------------------------------------
// ************************  Variables  **************************************
// ---------------------------------------------------------------------------

static GshItemList 	   the_item_list;
static MethodDialog        the_method_dialog;

// ---------------------------------------------------------------------------
// *********************  class: GshKnownObject  *****************************
// ---------------------------------------------------------------------------

GshKnownObject* GshKnownObject::make (KnownObject* baseptr)
{  
   return (GshKnownObject*)Application::lookup_ptr (baseptr);
}

GshKnownObject::~GshKnownObject()
{
   for (int i=0; i<list_length; i++)
      delete descr_list[i];
   delete omc;
}

void GshKnownObject::display (sos_Object o, WdgtPosition pos, 
			      what_to_show show)
{
   if INVALID(o) return;

   // determine new window position

   GshKnownObject *kobj = GshKnownObject::make (
			     the_known_objects->known_object (o));

   if (kobj && kobj->show == show)     // object is already displayed
      kobj->display (TRUE, pos);       //  --> move to new position and raise

   else                                // construct new object widget
   { 
      static int object_nr = 0;
      int        object_key;
      sos_Type   tp = o.type();
      if (kobj)
      {  object_key = kobj->key;
	 the_known_objects->remove (kobj);
         delete kobj;
      }
      else
	 object_key = ++object_nr;

      smg_String descr = smg_String("#") + object_key;
      if (tp.has_type (sos_Class_type_type))
	 descr += smg_String("<") + int(o.container()) + "," + o.offset() 
	       + ">";
      the_item_list.append (NO_OBJECT, FALSE, 
			    descr.make_Cstring (SMG_TRANSFER));
      
      descr  = smg_String("type=") + tp.get_name();
      the_item_list.append (tp, FALSE, descr.make_Cstring (SMG_TRANSFER));

      if (tp.has_type (sos_Class_type_type))
	 display_class_instance (o, show);
      else
      {  descr = smg_String("*value*=") + MethodDialog::convert (o);
	 the_item_list.append (NO_OBJECT, FALSE, 
			       descr.make_Cstring (SMG_TRANSFER));
      }
      GshKnownObject *kobj = new GshKnownObject;
      kobj->obj = o;
      kobj->show = show;
      kobj->key = object_key;
      the_item_list.make_KnownObject (kobj);

      kobj->display (FALSE, pos);

      GshKnownObjectList::make (the_known_objects)->insert (kobj);
   }
}

void GshKnownObject::display_component (sos_Object o, sos_Method m, 
					what_to_show show)
{  sos_Method rm = m.get_generated_from();
   if INVALID(rm)
      rm = m;

   if (rm.get_kind() == sos_PUBLIC)
   {  if (VALID(rm.get_comp_descr()) AND
	  NOT rm.get_is_set())
      {  if (show.cmp)
	 {  sos_Object val;
	    smg_String descr = smg_String(rm.get_name()) + "=";
	    sos_Bool   is_ok;
            
	    if (is_ok = MethodDialog::exec_method
 	                   (rm, o, sos_Object_Array::make (NO_OBJECT), val))
	       descr += MethodDialog::convert (val);
	    else
	       descr += "???";

	    the_item_list.append (val, FALSE, 
				  descr.make_Cstring (SMG_TRANSFER));
	 }
      }
      else
      {  if (show.met)
	 {  smg_String descr = smg_String(rm.get_name()) + "(";
	    sos_Param_List pl = rm.get_params();
	    int i = INVALID(pl) ? 0 : pl.card();
	    while (i--)
	      descr += "x";
	    descr += ")";
	    the_item_list.append (rm, TRUE, descr.make_Cstring (SMG_TRANSFER));
	 }
      }
   }
}

void GshKnownObject::display_class_instance (sos_Object o, what_to_show show)
{  sos_Class_type   ct = sos_Class_type::make (o.type());

   sos_Method_table mt = ct.get_methods();
   agg_iterate_association (mt, sos_String name, sos_Method_List ml)
      agg_iterate (ml, sos_Method m)
         if (NOT m.get_is_static ())
	    display_component (o, m, show);
      agg_iterate_end (ml, m);
   agg_iterate_association_end (mt, name, ml);

   if (ct == sos_Class_type_type && show.met)
   {  mt = sos_Class_type::make (o).get_methods();
      agg_iterate_association (mt, sos_String name, sos_Method_List ml)
	 agg_iterate (ml, sos_Method m)
	    if (m.get_is_static ())
	       display_component (o, m, show);
         agg_iterate_end (ml, m);
      agg_iterate_association_end (mt, name, ml);
   }
   if (show.cmp)
   {  if(o.is_some (sos_String_type))
      {  smg_String descr = smg_String("*value*=\"")
	                    + sos_String::make (o) + "\"";
	 the_item_list.append (NO_OBJECT, FALSE, 
			       descr.make_Cstring (SMG_TRANSFER));
      }
      else if (o.is_some (sos_Object_Collection_type))
      {  sos_Object_Collection coll = sos_Object_Collection::make (o);
	 int no = 1;
	 agg_iterate (coll, sos_Object co)
	 {  smg_String descr = "get_elem[";
            if (no < 10)
	       descr += " ";
	    descr += smg_String(no) + "]=" + MethodDialog::convert (co);
	    the_item_list.append (co, FALSE, 
				  descr.make_Cstring (SMG_TRANSFER));
	    no++;
	 }
	 agg_iterate_end (coll, co);
      }
      else if (o.is_some (sos_Object_sos_Object_Association_type))
      {  sos_Object_sos_Object_Association as =
	    sos_Object_sos_Object_Association::make (o);
	 int no = 1;
	 agg_iterate_association (as, sos_Object role1, sos_Object role2)
	 {  smg_String no_str = smg_String(no < 10 ? " " : "") + no;
	    smg_String descr = smg_String("get_role1[") + no_str + "]="
	                       + MethodDialog::convert (role1);
	    the_item_list.append (role1, FALSE, 
				  descr.make_Cstring (SMG_TRANSFER));

	    descr = smg_String("get_role2[") + no_str + "]="
	            + MethodDialog::convert (role2);
	    the_item_list.append (role2, FALSE, 
				  descr.make_Cstring (SMG_TRANSFER));
	    no++;
	 }
	 agg_iterate_association_end (as, role1, role2);
      }
   }
}

void GshKnownObject::item_selected (int item)
{
   static char *items[] = { "format", NULL, NULL, "close", NULL };

   if (item == 0)
   {
      items [1] = (show.cmp) ? "hide components"
	                     : "show components";
      items [2] = (show.met) ? "hide methods"
	                     : "show methods";

      the_object_menu->activate (this, items);
   }
   else if (omc[item].is_method)
      the_method_dialog.activate (obj, sos_Method::make (omc[item].o), key);
   else
      display (omc[item].o, WdgtPosition::next_to (wdgt), 
               GshApplication::make()->current_show);
}


// ---------------------------------------------------------------------------
// *******************  class: GshKnownObjectList  ***************************
// ---------------------------------------------------------------------------

GshKnownObjectList* GshKnownObjectList::make (KnownObjectList* baseptr)
{
   return (GshKnownObjectList*)Application::lookup_ptr (baseptr);
}

void GshKnownObjectList::insert (GshKnownObject *kobj)
{  KnownObjectList::insert (new GshKnownObjectList, kobj);
}

// ---------------------------------------------------------------------------
// *************************  class: GshItemList  ****************************
// ---------------------------------------------------------------------------

void GshItemList::append (sos_Object o, sos_Bool is_method, char *s)
{
   if (nr_items EQ size)
   {  if (size)
	 items = (ObjectItem*)REALLOC ((void*)items,
				       (size_t)sizeof(ObjectItem)*(size +=10));
      else
	 items = new ObjectItem [size = 10];
   }
   items [nr_items  ].descr         = s;
   items [nr_items  ].omc.is_method = is_method;
   items [nr_items++].omc.o         = o;
}

int GshItemList::compare_items (_qsort_const_ void *i1, _qsort_const_ void *i2)
{  return strcmp (((ObjectItem*)i1)->descr, ((ObjectItem*)i2)->descr);
}

void GshItemList::make_KnownObject(GshKnownObject *kobj)
{  qsort (&items[2], nr_items-2, sizeof(ObjectItem),
	  (qsort_cmpfct_t*)compare_items);

   kobj->list_length = nr_items;
   kobj->descr_list  = new char* [nr_items];
   kobj->omc         = new object_menu_comp [nr_items];
   while (--nr_items >= 0)
   {  kobj->descr_list[nr_items] = items[nr_items].descr;
      kobj->omc[nr_items]        = items[nr_items].omc;
   }
   nr_items = 0;
}

// ---------------------------------------------------------------------------
// *********************  class: GshObjectCmdMenu  ***************************
// ---------------------------------------------------------------------------

void GshObjectCmdMenu::do_it (int current_cmd)
{
   GshKnownObject *gkobj = GshKnownObject::make(kobj);
   sos_Object     obj    = gkobj->obj;
   Widget         w      = the_known_objects->widget (obj);
   
   if (!w)
      return;

   WdgtPosition   pos    = WdgtPosition::at (w);
   what_to_show   show   = gkobj->show;

   switch (current_cmd)
   {  case ocmd_FORMAT:
      {  break;
      }
      case ocmd_CMP:
      {  show.cmp = (sos_Bool)(NOT show.cmp);
	 gkobj->display (obj, pos, show);
	 break;
      }
      case ocmd_MET:
      {  show.met = (sos_Bool)(NOT show.met);
	 gkobj->display (obj, pos, show);
	 break;
      }
      case ocmd_CLOSE:
      {  the_known_objects->remove (gkobj);
         delete gkobj;
	 break;
      }
      default:
      {  the_info_line.display ("INTERNAL ERROR: unknown command");
      }
   }
}

void GshObjectCmdMenu::install()
{
   static char *items[] = { "format", "hide components",
			    "show methods" , "close", NULL };
   ObjectCmdMenu::install (items);
}

// ---------------------------------------------------------------------------
// ***********************  class: MethodDialog  *****************************
// ---------------------------------------------------------------------------

smg_String MethodDialog::convert (sos_Object o)
{  smg_String descr;
   if INVALID(o)
      return smg_String("<>");
 
   sos_Type tp = o.type();

   if (tp.has_type (sos_Class_type_type))
   {  if (o.isa (sos_Named_type))
      {  sos_String str = sos_Named::make (o).get_name();
	 if VALID(str)
            descr = str.make_Cstring();
	 descr += "...";
      }
      else if (o.isa (sos_String_type))
         descr = smg_String("\"") + sos_String::make (o) + "\"";
      else
	 descr = "...";
   }
   else if (tp.has_type (sos_Enum_type_type))
   {  int i = _obst_Int_from_enum (o, tp);
      descr = sos_Enum_type::make (tp).get_literals().get_nth (i+1);
   }
   else if (tp.has_type (sos_Extern_type_type))
      descr = cci_Schema_impl::string_from_extern_object (o);
   else
      descr = "???";

   return descr;
}

sos_Bool MethodDialog::convert (char* str, sos_Type tp, sos_Object &ob)
{  sos_Bool error = FALSE;
   sos_String sstr = sos_String::create (TEMP_CONTAINER, str);

   int parsed, strlength = strlen(str);
   int idx;

   sscanf (str, "#%d%n", &idx, &parsed);
   if (parsed EQ strlength)
   {  GshKnownObject *gkobj = GshKnownObject::make(
				 the_known_objects->known_object (idx));
      if (gkobj)
	 ob = gkobj->obj;
      else
	 error = TRUE;
   }
   else if (tp.has_type (sos_Enum_type_type))
   {  sos_String_List ll = sos_Enum_type::make (tp).get_literals();
      int lnr = 0;
      error = TRUE;
      agg_iterate (ll, sos_String l)
 	 if (l.equal (sstr))
	 {  ob = _obst_enum_from_sos_Int (lnr, tp);
	    error = FALSE;
	    break;
	 }
	 lnr++;
       agg_iterate_end (ll, l);
   }
   else if (tp.has_type (sos_Extern_type_type))
   {  sos_Object ext = cci_Schema_impl::extern_object_from_string (tp, sstr);
      if INVALID(ext)
	 error = TRUE;
      else
	 ob = ext;
   }
   else
   {  int ct, os;

      sscanf (str, "<%d,%d>%n", &ct, &os, &parsed);
      if (parsed EQ strlength)
	 ob = Application::make_object (sos_Container::make (ct), os);
      else
	 error = TRUE;
   }
   sstr.destroy();
   return error;
}

static void dummy_exec_method (sos_Method &m, sos_Object &o, 
			       sos_Object_Array &p, sos_Object &result)
{  result = m.execute (o, p);
}

sos_Bool MethodDialog::exec_method (sos_Method m, sos_Object o, 
				    sos_Object_Array p, sos_Object &result)
{  err_block
      dummy_exec_method (m, o, p, result);
   err_exception
      result = NO_OBJECT;
      the_info_line.display_OBST_error();
      return FALSE;
   err_block_end

   return TRUE;
}

void MethodDialog::activate (sos_Object obj, sos_Method met, int object_nr)
{
   sos_Param_List pl  = met.get_params();
   VarTextDialog *vtd = VarTextDialog::alloc_dialog();
   vtd->install();

   smg_String txt = smg_String("#") + object_nr + "."
                    + met.get_name() + "(...)->"
		    + met.get_result_type().make_type().get_name();

   vtd->prepare_start (txt.make_Cstring (SMG_BORROW));
 
   int i = 0;
   if VALID(pl)
      agg_iterate (pl, sos_Param p)
      {  txt = smg_String(p.get_type().make_type().get_name()) + " ";

	 sos_String pn = p.get_name();
	 if VALID(pn)
	    txt += pn;
     
	 vtd->prepare_add (txt.make_Cstring (SMG_BORROW));
	 i++;
      }
      agg_iterate_end (pl, p);

   sos_Bool activate_done = vtd->activate(FALSE);
                   /* activate returns TRUE if the 'do'-button is pressed */

   if (activate_done)
   {  sos_Object_Array objar;
      sos_Bool	       error = FALSE;
      
      if INVALID(pl)
         objar = sos_Object_Array::make (NO_OBJECT);
      else
      {  objar = sos_Object_Array::create(TEMP_CONTAINER, 0,
					  		  vtd->get_parnr()-1);
	 i     = 0;
	 agg_iterate (pl, sos_Param p)
	    sos_Object ob;
	    sos_Type   tp    = p.get_type().make_type();
	    char       *text = vtd->get_text(i);
	    
	    error = MethodDialog::convert (text, tp, ob);
	    
	    if (error)
	    {  smg_String info = smg_String("parameter ") + (i+1) + " invalid";
	       the_info_line.display (info.make_Cstring (SMG_BORROW));
	       break;
	    }
	    else
	       objar.set_nth (i, ob);
	    
	    i++;
	 agg_iterate_end (pl, p);
      }
      if (NOT error)
      {  sos_Object result;
	 sos_Bool success = MethodDialog::exec_method (met,obj,objar,result);
	 if (success) 
            GshKnownObject::display (result, 
				     WdgtPosition::at (vtd->get_main()),
				     GshApplication::make()->current_show);
	 else
  	    the_info_line.display_OBST_error();
      }
      if VALID(objar)
	 objar.destroy();
   }
   vtd->finish();
}


// ---------------------------------------------------------------------------
// **********************  class: GshMenuSystem  *****************************
// ---------------------------------------------------------------------------

void GshMenuSystem::do_it (int menu, int item)
{  GshApplication *gappl = GshApplication::make();
   switch (menu)
   {  case GshMenuSystem::menu_FILE:
      {  FileItemID file_item = (FileItemID)item;
	 switch (file_item)
	 {  case file_STD_ROOT:
	    {  the_application->display_root (TRUE);
	       break;
	    }
            case file_INIT_ROOT:
	    {  the_application->display_root (FALSE);
	       break;
	    }
	    case file_OBJECT:
	    {  the_container_dialog->prepare_start ("Object: ");
	       the_container_dialog->prepare_add ("container");
	       the_container_dialog->prepare_add ("offset");
	       if (the_container_dialog->activate(TRUE))
		  if (the_container_dialog->get_text(0))
		  {  int cnt, os;
		     cnt = atoi (the_container_dialog->get_text(0));
		     if (strlen (the_container_dialog->get_text(1)))
		        os = atoi (the_container_dialog->get_text(1));
		     else
		        os = ROOT_OFFSET;

		     err_block
		     {  sos_Object o = Application::make_object (
					      sos_Container::make (cnt), os);
		        GshKnownObject::display (o, WdgtPosition::root(), 
         			                 gappl->current_show);
		     }
		     err_exception
		     {  smg_String msg = smg_String ("object <") + cnt + "," 
                                       + os + "> does not exist";
			the_info_line.display (msg.make_Cstring (SMG_BORROW));
		     }
		     err_block_end
		  }  
	       break;
	    }
	    case file_QUIT:
	    {  the_application->commit_and_exit();
	       break;
	    }
	 }
	 break;
      }
      case GshMenuSystem::menu_VIEW:
      {  ViewItemID view_item = (ViewItemID)item;
	 switch (view_item)
	 {  case view_CMP:
	    {  gappl->current_show.cmp 
                          = (sos_Bool)(NOT gappl->current_show.cmp);
	       set_view_list(gappl->current_show);
	       break;
	    }
	    case view_MET:
	    {  gappl->current_show.met 
        	          = (sos_Bool)(NOT gappl->current_show.met);
	       set_view_list(gappl->current_show);
	       break;
	    }
	    case view_CLOSE_ALL:
	    {  the_known_objects->remove_all();
	       break;
	    }
	 }
	 break;
      }
      case GshMenuSystem::menu_CONTAINER:
      {  CntItemID cnt_item = (CntItemID)item;
	 if (cnt_item == cnt_CREATE)
	 {  GshKnownObject::display
	       (make_sos_Container_object (sos_Container::create()),
		WdgtPosition::root(), 
		gappl->current_show);
	 }
	 else if (cnt_item == cnt_RESET_ALL)
	    the_application->reset();
	 else if (cnt_item == cnt_COMMIT_ALL)
	    the_application->commit();
	 else
	 {  the_text_dialog->prepare_start ("Container: ");
	    the_text_dialog->prepare_add ("");
	    if (the_text_dialog->activate(TRUE))
  	       if (the_text_dialog->get_text(0))
	       {  err_block
	             sos_Container cnt =
		       sos_Container::make(atoi(the_text_dialog->get_text(0)));
		     switch (cnt_item)
		     {  case cnt_OPEN_READ:
		        {  cnt.open (READING, WAITING);	break; 
			}
		        case cnt_OPEN_WRITE:
		        {  cnt.open (WRITING, WAITING); break;
			}
			case cnt_COMMIT:
			{  cnt.commit ();   		break;
			}
			case cnt_RESET:
			{  cnt.reset ();   		break;
			}
			case cnt_CLOSE:
			{  cnt.close ();   		break;
			}
			case cnt_DESTROY:
			{  cnt.destroy ();   		break;
			}
			case cnt_CREATE:
			case cnt_RESET_ALL:
			case cnt_COMMIT_ALL:
			   break;
		     }   
	          err_exception
	             the_info_line.display_OBST_error();
	          err_block_end
	       }
	    break;
	 }
      }
   }
}

void GshMenuSystem::set_view_list (what_to_show cs)
{
   static char* items[] = { NULL, NULL, "close all", NULL };
   
   items [view_CMP] = (cs.cmp) ? "hide components"
                               : "show components";
   items [view_MET] = (cs.met) ? "hide methods"
                               : "show methods";

   change_list (menu_VIEW, items);
}

// ---------------------------------------------------------------------------
// **********************  class: GshApplication  ****************************
// ---------------------------------------------------------------------------

GshApplication* GshApplication::make ()
{ 
   return (GshApplication*)Application::lookup_ptr(the_application);
}

void GshApplication::init (int argc, char* argv[])
{
   standard_root = ROOT_CONTAINER.root_object();

   if (argc > 3)
   {  err_raise (err_USE, err_GSH_USAGE, NULL, FALSE);
      exit (1);
   }
   else if (argc > 1)
   {  sos_Container ct = sos_Container::make (atoi (argv [1]));
      sos_Offset    os = (argc > 2) ? atoi (argv [2])
	                            : ROOT_OFFSET;
      if (ct == 0 OR os == 0)
      {  err_raise (err_USE, err_GSH_USAGE, NULL, FALSE);
	 exit (1);
      }
      err_block
         initial_root = Application::make_object (ct, os);
      err_exception
         err_raise (err_USE, err_GTOOLS_INVALID_ROOT, NULL, FALSE);
	 exit (1);
      err_block_end
   }
   else
      initial_root = standard_root;
}

void GshApplication::display_root(sos_Bool std_root)
{
   GshKnownObject::display ((std_root) ? standard_root : initial_root,
			    WdgtPosition::root(), current_show);

TTN (psm_CNT_STATS,
   char* txt = psm_print_stats (NULL);
   cout << "\n" << txt << "------------------\n";
   delete txt);
}


// ---------------------------------------------------------------------------
// *****************************  Main  **************************************
// ---------------------------------------------------------------------------

int gsh_main (int argc, char *argv[])
{  T_FILE ("gsh.out");
   T_REDEF(NULL);

   err_block
      the_application      = new GshApplication;
      the_known_objects    = new GshKnownObjectList;
      the_object_menu      = new GshObjectCmdMenu;
      the_menu_system      = new GshMenuSystem;

      static char* headers[]    = {"File", "View", "Container"};
      static char* file_items[] = {"standard root", "initial root",
				   "object", "quit", NULL};
      static char* view_items[] = {"hide components", "show methods",
				   "close all", NULL};
      static char* cont_items[] = {"open reading", "open writing", "commit",
                                   "reset", "close", "create", "destroy", 
				   "reset all", "commit all",
                                   NULL};
      static char** menulists[]  = {file_items, view_items, cont_items};

      the_application->install (argc, argv, "Gsh", 3, headers, menulists);
   err_exception
      cerr << "*** ERROR <gsh::main> : uncaught OBST error\n\t"
	   << err_last_raised() << " (" << err_last_origin() << ")\n";
   err_block_end
      
   return 1;
}
