/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// Module genCC_code                                      Michael Gravenhorst
//
// **************************************************************************
// OBST interface generator / C++ code generation
// **************************************************************************
//
// tracing conventions: see trc_genCC.h

#define OBST_IMP_STDCONST
#define OBST_IMP_SORT_SEARCH
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "smg.h"
#include "genCC_err.h"
#include "trc_genCC.h"
#include "mta_use.h"

#include "genCC.h"

// --------  string constants  ---------------------------------------------

char* nm_local_equal  = "local_equal";
char* nm_total_equal  = "total_equal";
char* nm_local_assign = "local_assign";
char* nm_total_assign = "total_assign";
char* nm_local_hashv  = "local_hash_value";
char* nm_total_hashv  = "total_hash_value";
char* nm_local_dtor   = "local_finalize";
char* nm_total_dtor   = "total_finalize";
char* nm_local_ctor   = "local_initialize";
char* nm_total_ctor   = "total_initialize";

// --------  global variables  ----------------------------------------------

LOCAL sos_Expr   no_expr;


//-------- forward declarations ---------------------------------------------

LOCAL smg_String code_type_conversion
		    (sos_Type_descr, sos_Type_descr, smg_String);


//======== code generation ==================================================

//-------- global-initialization --------------------------------------------

EXPORT void code_initialize ()
{  T_PROC ("code_initialize")
   TT (genCC_VL, T_ENTER);

   no_expr	   = sos_Expr::make(NO_OBJECT);
  
   TT (genCC_VL, T_LEAVE);
}

//-------- class initialization ---------------------------------------------

LOCAL sos_Class_type class_type;
LOCAL sos_Param_List create_params;
LOCAL smg_String     nm_class;
LOCAL smg_String     nm_root;
LOCAL smg_String     impl_nm_class;
LOCAL sos_Bool	     class_is_instantiation;
SPLIT_CREATE_SHORTCUT(
LOCAL sos_Bool	     split_create_copy;
)
ABSTRACT_SHORTCUT(
LOCAL sos_Bool	     class_is_abstract;
)

EXPORT void code_class_start (sos_Class_type ct,
                              sos_Param_List pl)
{  T_PROC ("code_class_start")
   TT (genCC_VL, T_ENTER);

   class_type             = ct;
   create_params          = pl;
   nm_class	          = ct.get_name();
   nm_root	          = ct.get_root_class().get_name();
   impl_nm_class          = smg_String(_ID) + nm_root;
   class_is_instantiation = ct.is_instantiation();
   split_create_copy 	  = SPLIT_CREATE_SHORTCUT(class_is_instantiation);
   class_is_abstract      = ct.get_is_abstract();
  
   TT (genCC_VL, T_LEAVE);
}

//-------- methods: initialization, utilities --------------------------------

LOCAL sos_Method     code_method;
LOCAL sos_Method     code_patch_method;
LOCAL sos_Type       code_result_type;
LOCAL sos_Param_List code_params;

EXPORT void code_method_start (sos_Method     meth,
                               sos_Type       res_type,
			       sos_Param_List params)
{  T_PROC ("code_method_start")
   TT (genCC_VL, T_ENTER);

   code_result_type = res_type;
   code_method      = meth;
   code_params	    = params;

  TT (genCC_VL, T_LEAVE);
}

EXPORT void code_init_patch (sos_Method pm)
{  T_PROC ("code_init_patch")
   TT (genCC_VL, T_ENTER);

   code_patch_method = pm;

   TT (genCC_VL, T_LEAVE);
}
 
//--------

typedef int bsearch_cmp_fct (const void *, const void *);

struct mdata
{  char*	 name;
   Predef_Method encoding;
   int		 param_card;
   sos_Bool	 check_if_generated;
};


LOCAL int mdata_cmp_fct (mdata* md1, mdata* md2)
{  return strcmp (md1->name, md2->name);
}

EXPORT Predef_Method code_whatis (smg_String methname,
				  int pcard, sos_Bool generated)
{  T_PROC ("code_whatis")
   TT (genCC_VL, T_ENTER);

   static
   mdata _md,	       // Table must be lexicographically sorted!
	 mdtable[] = { {"!=",		    OP_NOT_IDENTICAL,  1, FALSE},
		       {"==",		    OP_IDENTICAL,      1, FALSE},
		       {"assign", 	    ASSIGN,	       1, TRUE },
		       {"clone", 	    CLONE,	       2, TRUE },
		       {"compare_ids",	    COMPARE_IDS,       1, FALSE},
		       {"container", 	    CONTAINER,	       0, FALSE},
		       {"copy", 	    COPY,	       2, TRUE },
		       {"create", 	    CREATE,	      -1, TRUE },
		       {"destroy", 	    DESTROY,	       0, TRUE },
		       {"equal", 	    EQUAL,	       2, TRUE },
		       {"hash_value", 	    HASH_VALUE,	       0, TRUE },
		       {"identical",	    IDENTICAL,	       1, FALSE},
		       {"local_assign",     LOCAL_ASSIGN,     -1, TRUE },
		       {"local_equal",      LOCAL_EQUAL,      -1, TRUE },
		       {"local_hash_value", LOCAL_HASH_VALUE, -1, TRUE },
		       {"offset", 	    OFFSET, 	       0, FALSE},
		     };
   _md.name = methname.make_Cstring (SMG_BORROW);

   mdata* mdptr = (mdata*)bsearch ((char*)&_md, (char*)mdtable,
				   sizeof(mdtable)/sizeof(mdata),
				   sizeof(mdata),
				   (bsearch_cmp_fct*)mdata_cmp_fct);
   Predef_Method result
      = (   NOT mdptr
       	 OR mdptr->check_if_generated AND NOT generated
	 OR mdptr->param_card >= 0    AND pcard != mdptr->param_card)
      	    ? NOT_PREDEF
	    : mdptr->encoding;

   TT (genCC_VL, T_LEAVE);
   return result;
}

//-------- methods: generation of additional methods -------------------------

EXPORT void code_additional_methods ()
{  T_PROC ("code_additional_methods")
   TT (genCC_M, T_ENTER);

   // "_copy_comps" and "_create_comps" allow public access in order for
   // "split_create_copy" to work.

   sos_Method no_method      = sos_Method::make(NO_OBJECT);

   code_method_start (no_method,
		      sos_Type::make(NO_OBJECT),
		      sos_Param_List::make(NO_OBJECT));

   if (back_method_start (no_method, FIXED_FCT("make"), nm_class,
			  TRUE, FALSE, FALSE, TRUE, FALSE,
                          TRUE, FALSE, sos_PUBLIC, MAKE_HANDLE))
   {  back_param_end();
      back_method_end();
   }
   if (back_method_start (no_method, FIXED_FCT("make"), nm_class,
			  TRUE, FALSE, FALSE, TRUE, FALSE,
                          TRUE, FALSE, sos_PUBLIC, MAKE_CP_HANDLE,1))
   {  back_param ("*_y", impl_nm_class, FALSE, FALSE, no_expr);
      back_param_end();
      back_method_end();
   }
   if (back_method_start (no_method, FIXED_FCT("make"), nm_class,
			  TRUE, FALSE, FALSE, TRUE, FALSE,
                          TRUE, FALSE, sos_PUBLIC, MAKE_SC_HANDLE,1))
   {  back_param ("_x", nm_ROOT, FALSE, FALSE, no_expr, FALSE);
      back_param_end();
      back_method_end();    
   }
   if (    SPLIT_CREATE_SHORTCUT(split_create_copy)
       AND ABSTRACT_SHORTCUT(NOT class_is_abstract))
      if (back_method_start (no_method, FIXED_FCT("_copy"),
			     FIXED_TYPE("void"), TRUE, FALSE, FALSE, TRUE, 
                             FALSE, TRUE, FALSE, sos_PUBLIC, _COPY,2))
      {  back_param("_x", nm_class, FALSE, FALSE, no_expr, FALSE);
	 back_param("_y", nm_class, FALSE, FALSE, no_expr, FALSE);
	 back_param_end();
	 back_method_end();
      }
   if (class_type.get_has_init_comps())
      if (back_method_start (no_method, FIXED_FCT("_copy_comps"),
			     FIXED_TYPE("void"), TRUE, FALSE, FALSE, 
                             TRUE, FALSE, TRUE, FALSE,
			     SPLIT_CREATE_SHORTCUT(sos_PUBLIC), _COPY_COMPS,2))
      {  back_param("_x", nm_class, FALSE, FALSE, no_expr, FALSE);
	 back_param("_y", nm_class, FALSE, FALSE, no_expr, FALSE);
	 back_param_end();
	 back_method_end();
      }

   code_method_start (no_method,
		      sos_Object_type,
		      sos_Param_List::make(NO_OBJECT));

   if (back_method_start (no_method, FIXED_FCT("_clone"),
			  FIXED_TYPE("sos_Object"), TRUE, FALSE,
			  ABSTRACT_SHORTCUT(class_is_abstract), 
                          FALSE, FALSE, TRUE, FALSE, sos_PUBLIC, _CLONE,1))
   {  
      back_param("_ct", "sos_Container", FALSE, FALSE, no_expr);
      back_param_end();
      back_method_end();
   }

   code_method_start (no_method, sos_Type::make(NO_OBJECT), create_params);

   if (    SPLIT_CREATE_SHORTCUT(split_create_copy)
       AND ABSTRACT_SHORTCUT(NOT class_is_abstract))
      if (back_method_start (no_method, FIXED_FCT("_create"), 
			     FIXED_TYPE("void"), TRUE, FALSE, FALSE, TRUE, 
                             FALSE, TRUE, FALSE, sos_PUBLIC, _CREATE,1))
      {  back_param ("_x", nm_class, FALSE, FALSE, no_expr, FALSE);
	 code_param_list();
	 back_param_end();
	 back_method_end();
      }
   if (class_type.get_has_init_comps())
      if (back_method_start (no_method, FIXED_FCT("_create_comps"),
			     FIXED_TYPE("void"), TRUE, FALSE, FALSE, 
                             TRUE, FALSE, TRUE, FALSE,
			     SPLIT_CREATE_SHORTCUT(sos_PUBLIC),
			     _CREATE_COMPS,1))
      {  back_param("_x", nm_class, FALSE, FALSE, no_expr, FALSE);
	 code_param_list();
	 back_param_end();
	 back_method_end();
      }
   TT (genCC_M, T_LEAVE);
}

//-------- code generation: abstract methods --------------------------------

EXPORT smg_String code_abstract_method ()
{  T_PROC ("code_abstract_method")
   TT (genCC_L, T_ENTER);

   sos_Bool has_result = (sos_Bool)(
			    ! nm_result_type.equal(FIXED_TYPE("void")));

   smg_String result = smg_String("err_raise_abstract_method (\"") 
          	     + nm_class + "::" + nm_method + "\");\n";
   if (has_result)
   {  result += smg_String("return ");
      if (code_result_type.is_scalar())
         result += smg_String("(") + nm_result_type + ") 0;\n";
      else if (sos_OBJECT_SHORTCUT(
	       nm_result_type.equal("sos_Object")))
	 result += "NO_OBJECT;\n";
      else
         result += nm_result_type + "::make(NO_OBJECT);\n";
   }
   TT (genCC_L, T_LEAVE);
   return result;
}

//-------- code generation: component access methods ------------------------

EXPORT smg_String code_comp_method ()
{  T_PROC ("code_comp_method");
   TT (genCC_L, T_ENTER);

   sos_Bool   is_set    = code_method.get_is_set();
   sos_Type   comp_tp   = (is_set)
			     ? code_params.get_nth(1).get_type().make_type()
			     : code_result_type;
   sos_Bool   is_scalar = comp_tp.is_scalar ();
   sos_Bool   is_local  = code_method.get_comp_descr().get_is_local();
   sos_Int    offset    = code_method.get_comp_descr().get_offset();
   sos_Int    size;
   smg_String cpp_tp;

   if (is_scalar)
   {  size   = comp_tp.get_object_size();
      cpp_tp = comp_tp.get_name();
   }
   else if (is_local)
   {  size   = SOS_OFFSET_SIZE;
      cpp_tp = FIXED_TYPE("sos_Offset");
   }
   else
   {  size   = SOS_TYPED_ID_SIZE;
      cpp_tp = FIXED_TYPE("sos_Typed_id");
   }

   smg_String result     = smg_String ("union{")
			   + cpp_tp + " _x;char _c[" + size + "];}_u;";
   smg_String cntop_args = smg_String(THIS_param)
			   + OBJ_offset + "+_offset_" + nm_class + "()";
   if (offset > 0)
      cntop_args += smg_String("+") + offset;
   cntop_args += smg_String(",") + size + ",_u._c)" + NL;

   if (is_set)
   {  result += "\n";
      if (is_scalar)
         result += smg_String ("bcopy_from_") + cpp_tp
	    	   + FIXED_TYPE("((void*)&_p0,_u._c)") + NL;
      else if (is_local)
      	 result += cpp_tp + " _x=" + cvt_to_LOCAL + "(_p0," + THIS_param 
                +  OBJ_container + ")" + NL
	        +  "bcopy_from_" + cpp_tp + "(&_x,_u._c)" + NL;
      else
         result += cpp_tp + " _x=_p0" + OBJ_typed_id + NL
	        +  "bcopy_from_" + cpp_tp + "(&_x,_u._c)" + NL;
      result += smg_String(THIS_param) + OBJ_container + ".write(" 
	     +  cntop_args;
   }
   else
   {  result += cpp_tp + " _a;\n" + THIS_param 
             +  OBJ_container + ".read(" + THIS_param + OBJ_offset 
             +  "+_offset_" + nm_class + "()";
      if (offset > 0)
	 result += smg_String ("+") + offset;
      result += smg_String (",") + size + ",_u._c)" + NL
	     +  "bcopy_to_" + cpp_tp + "(&_a,_u._c)" + NL;
      if (is_scalar)
	 result += smg_String("return _a") + NL; 
      else if (is_local)
	 result += smg_String ("return ") + nm_result_type
	        +  "::make(" + cvt_from_LOCAL + "(_a,"
	        +  THIS_param + OBJ_container + "))" + NL;
      else
      {  result += smg_String ("return ") + nm_result_type;
      if (NOT code_result_type.has_type (sos_Union_type_type))
         result += "::make";
          result += smg_String("(_a)") + NL;
      }
   }
   TT (genCC_L, T_LEAVE);

   return result;
}

// -------- code generation: universal + interface methods -------------------

LOCAL smg_String code_convert_args (sos_Param_List intf_pl,
				    sos_Param_List impl_pl,
				    smg_String&    prefix,
				    smg_String&    suffix)
{  T_PROC ("code_convert_args")
   TT (genCC_L, T_ENTER);

   static sos_Type_descr object_td = sos_Type_descr::make(
					_obst_knl_type (_THE_OBJECT_TYPE));
   smg_String result;

   if VALID(impl_pl)
   {  sos_Bool   univ_intf = INVALID(intf_pl);
      sos_Cursor intf_cursor = sos_Cursor::make (NO_OBJECT);
      smg_String pname, idx_str, conversion, last_temp_tp;

      int param_idx = 0;
      agg_iterate (impl_pl, sos_Param impl_p)
         sos_Bool is_refparam  = impl_p.get_is_ref();

	 if (param_idx > 0)
            result += ",";
         else if (NOT univ_intf)
	    intf_cursor = intf_pl.open_cursor();

         idx_str = param_idx++;

	 sos_Type_descr impl_td = impl_p.get_type(),
			intf_td;
         if (univ_intf)
	 {  pname   = smg_String("_p[") + idx_str + "]";
            intf_td = object_td;
	 }
         else
         {  sos_Param intf_p = intf_pl.get (intf_cursor);
	    intf_pl.to_succ (intf_cursor);

	    intf_td = intf_p.get_type();
	    pname   = intf_p.get_name();
            if (pname.length() == 0)
               pname = smg_String("_p") + idx_str;
	 }
  	 conversion = code_type_conversion (intf_td, impl_td, pname);

         if (NOT (is_refparam AND univ_intf)  AND  conversion.equal(pname))
            result += conversion;
	 else
	 {  smg_String temp_nm = smg_String("_t") + idx_str;
	    smg_String temp_tp = impl_td.make_type().get_name();

	    if (last_temp_tp.equal(temp_tp))
	       prefix += ",";
	    else
	    {  if (last_temp_tp.length())
		  prefix += NL;
	       prefix += (last_temp_tp = temp_tp) + " ";
	    }
	    prefix += temp_nm + "=" +  conversion;
	    result += temp_nm;
	    if (is_refparam)
	    {  if (univ_intf)
		  suffix += smg_String("_p.set_nth(") + idx_str + ",";
	       else
		  suffix += pname + "=";
               suffix += code_type_conversion (impl_td, intf_td, temp_nm)
		      +  (univ_intf    ? ")" : "")
		      +  NL;
	    }
	 }
      agg_iterate_end (impl_pl, impl_p);

      if (prefix.length())
	 prefix += NL;

      if (VALID (intf_cursor) AND NOT univ_intf)
	 intf_pl.close_cursor (intf_cursor);
   }
   TT (genCC_L, T_LEAVE);
   return result;
}

LOCAL smg_String code_call_impl (sos_Method intf_m,
				 sos_Method impl_m,
				 smg_String nm_impl_m,
				 smg_String nm_impl_class,
                                 sos_Bool   called_in_intf)
{  T_PROC ("code_call_impl")
   TT (genCC_L, T_ENTER);
  
   smg_String     result, args, prefix, suffix;
   sos_Bool       bracketed_call = FALSE;
   sos_Bool	  univ_intf      = INVALID(intf_m);
   sos_Param_List impl_pl	 = impl_m.get_params();

   if (OBSThis_param AND NOT univ_intf)
      args += ((called_in_intf) ? THIS_comp : THIS_param);

   if VALID(impl_pl)
   {  if (OBSThis_param AND NOT univ_intf)
         args += ",";
      args   += code_convert_args ((univ_intf ? sos_Param_List::make(NO_OBJECT)
					      : intf_m.get_params()),
				   impl_pl, prefix, suffix);
      result += prefix;
   }
   smg_String call;
   sos_Bool   do_deref = FALSE;
   sos_Bool   call_on_type = is_static;

   if(called_in_intf)
   {  if (is_static)
         call = nm_impl_class;
      else if (NOT univ_intf)
      {  call     = IMPL_comp;
         do_deref = TRUE;
      }
      else if (nm_impl_class.equal (FIXED_TYPE("sos_Object")))
         call = "_x";
      else
         call = nm_impl_class + "::make(_x)";
   }
   else 
   {
      call_on_type = TRUE;
      call         = nm_impl_class;
   }
      
  

   call = back_call_method (call, nm_impl_m,
                            call_on_type, do_deref, FALSE, args);
   if (is_void)
      result += call;
   else
   {  static sos_Type_descr object_td = sos_Type_descr::make (
					   _obst_knl_type (_THE_OBJECT_TYPE));

      sos_Type_descr intf_result_td = (univ_intf) ? object_td
					          : intf_m.get_result_type();
      if (suffix.length())
	 result += smg_String(intf_result_td.make_type().get_name()) + " _r=";
      else
	 result += "return ";

//      if (called_in_intf)
         result += code_type_conversion(impl_m.get_result_type(), 
                                        intf_result_td, call);
//      else
//         result += call;
   }
   result += smg_String(NL) + suffix;

   if (is_void)
   {  if (univ_intf)
	 result += smg_String("return NO_OBJECT") + NL;
   }
   else if (suffix.length())
      result += smg_String("return _r") + NL;

   TT (genCC_L, T_LEAVE);
   return result;
}

EXPORT smg_String code_univ_body ()
{  T_PROC ("code_univ_body")
   TT (genCC_L, T_ENTER);

   smg_String result = code_call_impl (sos_Method::make(NO_OBJECT),
				       code_method,
				       nm_method,
				       nm_class,
                                       TRUE);
   TT (genCC_L, T_LEAVE);
   return result;
}

EXPORT smg_String code_method_call_impl (smg_String impl_mname)
{  T_PROC ("code_method_call_impl")
   TT (genCC_L, T_ENTER);

   sos_Method  method_impl = code_method.get_generated_from();

   if (INVALID(method_impl)
       OR method_impl.get_defined_in().is_generic_class())
      method_impl = code_method;

   smg_String result = code_call_impl (code_method,
				       method_impl,
				       impl_mname,
				       impl_nm_class,
                                       TRUE);
   TT (genCC_L, T_LEAVE);
   return result;
}

EXPORT smg_String code_method_patch (smg_String    impl_mname,
                                     Predef_Method which_patch)
{  T_PROC ("code_generic_patch")
   TT (genCC_L, T_ENTER);

   smg_String result;
   
   if (which_patch == GENERIC_PATCH)
      result = code_call_impl (code_method,
                               code_patch_method,
                               impl_mname,
                               impl_nm_class,
                               FALSE);
   else if (which_patch == OVERLOADED_INHERITED)
   {
      sos_Class_type sc  = code_method.get_defined_in().get_root_class();
      result = code_call_impl (code_method,
                               code_patch_method,
                               impl_mname,
                               smg_String(_ID) + sc.get_name(),
                               FALSE);
   }

   TT (genCC_L, T_LEAVE);
   return result;
}

//-------- code generation: predefined methods -------------------------------

#define HAS_LOCAL_ASSIGN	0x001
#define HAS_TOTAL_ASSIGN	0x002
#define HAS_LOCAL_EQUAL		0x004
#define HAS_TOTAL_EQUAL		0x008
#define HAS_LOCAL_HASHVAL	0x010
#define HAS_TOTAL_HASHVAL	0x020
#define HAS_LOCAL_CTOR          0x040
#define HAS_TOTAL_CTOR          0x080
#define HAS_LOCAL_DTOR          0x100
#define HAS_TOTAL_DTOR          0x200

LOCAL void local_total_closure (sos_Super_class_List       	   scl,
				sos_Object_sos_Object_Mapping_Hash map)
{  T_PROC ("local_total_closure")
   TT (genCC_VL, T_ENTER);

   static sos_String
          nm_lassign = sos_String::create(TEMP_CONTAINER, nm_local_assign),
	  nm_tassign = sos_String::create(TEMP_CONTAINER, nm_total_assign),
	  nm_lequal  = sos_String::create(TEMP_CONTAINER, nm_local_equal),
	  nm_tequal  = sos_String::create(TEMP_CONTAINER, nm_total_equal),
	  nm_lhashv  = sos_String::create(TEMP_CONTAINER, nm_local_hashv),
	  nm_thashv  = sos_String::create(TEMP_CONTAINER, nm_total_hashv),
	  nm_lctor  = sos_String::create(TEMP_CONTAINER, nm_local_ctor),
	  nm_tctor  = sos_String::create(TEMP_CONTAINER, nm_total_ctor),
	  nm_ldtor  = sos_String::create(TEMP_CONTAINER, nm_local_dtor),
	  nm_tdtor  = sos_String::create(TEMP_CONTAINER, nm_total_dtor);

   agg_iterate (scl, sos_Super_class sc)
      sos_Class_type sctp = sos_Class_type::make (sc.make_type());

      if (sctp == _obst_knl_type(_THE_OBJECT_TYPE)) 
      {        // The methods of sos_Object always return the neutral element.
         sos_OBJECT_SHORTCUT(	        	    
	    map.insert ((sos_Object)sc, make_sos_Int_object(0));
	    continue);				    
      } 

      sos_Method_table sctbl      = sctp.get_methods();
      int 	       sc_key     = 0,
      	  	       delete_key = 0;

      if VALID(sctbl[nm_tassign])
      {  sc_key     |= HAS_TOTAL_ASSIGN;
	 delete_key |= HAS_TOTAL_ASSIGN | HAS_LOCAL_ASSIGN;
      }
      else if VALID(sctbl[nm_lassign])
	 sc_key     |= HAS_LOCAL_ASSIGN;

      if VALID(sctbl[nm_tequal])
      {  sc_key     |= HAS_TOTAL_EQUAL;
	 delete_key |= HAS_TOTAL_EQUAL | HAS_LOCAL_EQUAL;
      }
      else if VALID(sctbl[nm_lequal])
	 sc_key     |= HAS_LOCAL_EQUAL;

      if VALID(sctbl[nm_thashv])
      {  sc_key     |= HAS_TOTAL_HASHVAL;
	 delete_key |= HAS_TOTAL_HASHVAL | HAS_LOCAL_HASHVAL;
      }
      else if VALID(sctbl[nm_lhashv])
	 sc_key     |= HAS_LOCAL_HASHVAL;

      if VALID(sctbl[nm_tctor])
      {  sc_key     |= HAS_TOTAL_CTOR;
	 delete_key |= HAS_TOTAL_CTOR | HAS_LOCAL_CTOR;
      }
      else if VALID(sctbl[nm_lctor])
	 sc_key     |= HAS_LOCAL_CTOR;

      if VALID(sctbl[nm_tdtor])
      {  sc_key     |= HAS_TOTAL_DTOR;
	 delete_key |= HAS_TOTAL_DTOR | HAS_LOCAL_DTOR;
      }
      else if VALID(sctbl[nm_ldtor])
	 sc_key     |= HAS_LOCAL_DTOR;

      if (delete_key)
      {  sos_Cursor c = scl.duplicate(agg_current_cursor());
	 while (scl.to_pred(c))
	 {  sos_Super_class _sc = sos_Super_class::make(scl.get(c));
	    if (sctp.is_derived_from (_sc.make_type()))
	    {  int val = make_sos_Int(map[(sos_Object)_sc]);
	       if (val & delete_key)
	          map.insert ((sos_Object)_sc, 
                              make_sos_Int_object(val & ~delete_key));
	    }
	 }
	 scl.close_cursor (c);
      }
      map.insert ((sos_Object)sc, make_sos_Int_object(sc_key));
   agg_iterate_end (scl, sc);

   TT (genCC_VL, T_LEAVE);
}

LOCAL smg_String code_cast_args (smg_String nm_target, smg_String nm_param)
{  T_PROC ("code_cast_args")
   TT (genCC_VL, T_ENTER);

   smg_String result;

#if EXPLICIT_TYPECAST_BUG   
   result = smg_String("OBST_") + "CAST(" + nm_target + "," + nm_param + ")";
#else
   result = smg_String(nm_param);
#endif

   TT (genCC_VL, T_LEAVE);
   return result;
}

LOCAL smg_String code_get_impl_ptr (smg_String from_tp)
{  T_PROC ("code_get_impl_ptr")
   TT (genCC_L, T_ENTER);

   smg_String result = smg_String("(") + impl_nm_class + "*)" 
                       + get_IMPL_OBJ + impl_nm_class + "_type,"
		       + from_tp + ")";

   TT (genCC_L, T_LEAVE);
   return result;
}

// This refers to a bug in the AT&T 2.0, 2.1 front ends which dump a core if
// a static method of a super class is called in the generated methods which
// use the function CALL_BASE_STATIC introduced below.
// The call is then transfered into a macro application which hides if the
// bug actually occurs. We do not directly generate the code which will
// circumvent the bug since this might entail a significant penalty.

// (The macro name is split to account for old style preprocessors
//  which replace even in string literals.)

// The parameters of _copy(_comps) and the first one of _create(_comps), the 
// types appearing in the `decls` argument of CALL_BASE_STATIC and
// the implementation of the operator methods are defined const. (graven 12.7.)
 
#if CALL_STATIC_OF_BASE_BUG
smg_String CALL_BASE_STATIC (smg_String cnm,  smg_String mnm,
			     smg_String args, smg_String decls)
{  smg_String result = smg_String("OBST_") + "CALLSTATIC("
	    	       + cnm + "::" + mnm + ",(" + args + "),(" + decls + "))";
   return result;
}
#else
#   define CALL_BASE_STATIC(cnm,mnm,args,decls) \
	   back_call_method (cnm, mnm, TRUE, FALSE, FALSE, args)
#endif

EXPORT smg_String code_generate_body (Predef_Method actual_method)
{  T_PROC ("code_generate_body")
   TT (genCC_L, T_ENTER);

   static smg_String 			     _nm_class;
   static sos_Super_class_List		     super_closure;
   static sos_Object_sos_Object_Mapping_Hash local_total_map
	     = sos_Object_sos_Object_Mapping_Hash::create (TEMP_CONTAINER);

   if (NOT _nm_class.equal (nm_class))
   {  _nm_class     = nm_class;
      super_closure = class_type.get_super_closure();
      local_total_closure (class_type.get_super_closure(), local_total_map);
   }

   smg_String   result;
     
   switch(actual_method)
   {  case CREATE :
      {  result += nm_class + " _x=" 
                +  back_call_method (nm_class, FIXED_FCT("make"),
				     TRUE, FALSE, FALSE,
	               back_call_method (FIXED_TYPE("sos_Typed_id"),
					 FIXED_FCT("allocate"),
					 TRUE, FALSE, FALSE,
                                         smg_String(_ID) + nm_class
					  + "_type,_ct,"
					  + class_type.get_object_size())) +NL;
	 if (SPLIT_CREATE_SHORTCUT(split_create_copy))
	 {  

	    smg_String args = code_cast_args(nm_root, "_x");

            if VALID(create_params)
            {  agg_iterate (create_params, sos_Param cp)
	       args += smg_String(", ") + cp.get_name();
	       agg_iterate_end (create_params, cp);
	    }
	    result += back_call_method (impl_nm_class, FIXED_FCT("_create"),
					TRUE, FALSE, FALSE, args) + NL;
	 }
	 else
	    result += code_generate_body (_CREATE);
	 result += smg_String("return _x") + NL;
      }
      break;

      case _CREATE :
      {  smg_String rslt, ctors;
         agg_iterate (super_closure, sos_Super_class sc)
         {  sos_Class_type sct   = sos_Class_type::make(sc.make_type());
	    smg_String     sc_nm = sct.root().get_name();

            if (sct.get_has_init_comps())
            {  sos_Expr_List cel = sc.get_create_params();
#if CALL_STATIC_OF_BASE_BUG
	       smg_String decls = smg_String("const ") + sc_nm + "&";
#endif
               smg_String exprs;
	       if VALID(cel)
	       {  sos_Param_List cpl = sct.get_create_params();
		  int _i;
		  agg_iterate_double (cel, sos_Expr expr, cpl, sos_Param p, _i)
		     smg_String tpnm = p.get_type().make_type().get_name();
#if CALL_STATIC_OF_BASE_BUG
		     decls += smg_String(",") + "const " + tpnm; 
#endif
		     exprs += code_expr (expr, ",", tpnm);

                  agg_iterate_double_end (cel, expr, cpl, p, _i);
	       }

               rslt += CALL_BASE_STATIC (smg_String(_ID) + sc_nm,
                                         FIXED_FCT("_create_comps"),
                                         code_cast_args(sc_nm, "_x") + exprs,
					 decls) + NL;
            }
	    int   key    = make_sos_Int (local_total_map[(sos_Object)sc]);
	    char* methnm = (key & HAS_LOCAL_CTOR)
			      ? nm_local_ctor
	       		      : (key & HAS_TOTAL_CTOR) ? nm_total_ctor
			      				 : NULL;
	    if (methnm) 
            {
               smg_String args;
               if (sc_nm.equal(nm_class))
                  args = "_x";
               else 
                  args = code_cast_args (sc_nm, "_x");
               ctors += CALL_BASE_STATIC (smg_String(_ID) + sc_nm,
                                          methnm, args, 
                                          smg_String("OBST_") + "PARDECL("
                                          + sc_nm + ")") + NL;
            }
         }
         agg_iterate_end (super_closure, sc);
	 rslt += ctors;
	 if (rslt.length())
	    result += smg_String("OBST_CREATE_") + "START\n"
	           +  rslt + "OBST_CREATE_" + "END\n";
      }
      break;

      case _CREATE_COMPS :
      {  sos_Method_List ml = class_type.get_local_methods();
         agg_iterate (ml, sos_Method cm)
         {  
            if (VALID(cm.get_comp_descr()) && cm.get_is_set())
            {  sos_Expr e = cm.get_comp_descr().get_init_expr();
               if VALID(e)
               {  result += smg_String("_x.") + cm.get_name() + "("
		         +  code_expr (e, "", cm.get_params().get_nth(1)
                                                .get_type().make_type()
                                                .get_name())
			 +  ")" + NL;
	       }
	    }
         }
         agg_iterate_end (ml, m);
      }
      break;

      case COPY :
      {  result += nm_class + " _x="
                +  back_call_method (nm_class, FIXED_FCT("make"),
				     TRUE, FALSE, FALSE,
	               back_call_method (FIXED_TYPE("sos_Typed_id"),
					 FIXED_FCT("allocate"),
                                         TRUE, FALSE, FALSE,
                                         smg_String(_ID) + nm_class
					  + "_type,_ct,"
					  + class_type.get_object_size())) 
                                          + NL;

         if (SPLIT_CREATE_SHORTCUT(split_create_copy))
	    result += back_call_method (impl_nm_class, FIXED_FCT("_copy"),
					TRUE, FALSE, FALSE, 
                                        code_cast_args(nm_root, "_x") + ","
                                         + code_cast_args(nm_root, "_y")) + NL;
	 else
	    result += code_generate_body (_COPY);

         result += smg_String(FIXED_FCT("_x.assign(_y)")) + NL
		+  "return _x" + NL;
      }
      break;
   
      case _COPY :
      {  smg_String rslt, ctors;
         agg_iterate (super_closure, sos_Super_class sc)
         {  sos_Class_type sct   = sos_Class_type::make(sc.make_type());
	    smg_String     sc_nm = sct.make_type().root().get_name();
            if (sct.get_has_init_comps())
               result += CALL_BASE_STATIC(smg_String(_ID) + sc_nm,
                                          FIXED_FCT("_copy_comps"),
                                          code_cast_args(sc_nm, "_x") + ","
                                           + code_cast_args(sc_nm, "_y"),
				          smg_String("const ")+ sc_nm + "&,"
                                                   + "const " + sc_nm + "&")
		      +  NL;

	    int   key    = make_sos_Int (local_total_map[(sos_Object)sc]);
	    char* methnm = (key & HAS_LOCAL_CTOR)
			      ? nm_local_ctor
	       		      : (key & HAS_TOTAL_CTOR) ? nm_total_ctor
			      			       : NULL;
	    if (methnm) 
            {
               smg_String args;
               if (sc_nm.equal(nm_class))
                  args = "_x";
               else 
                  args = code_cast_args (sc_nm, "_x");
               ctors += CALL_BASE_STATIC (smg_String(_ID) + sc_nm,
                                          methnm, args, 
                                          smg_String("OBST_") + "PARDECL("
                                          + sc_nm + ")") + NL;
            }
         }
         agg_iterate_end (super_closure, sc);
         rslt += ctors;
	 if (rslt.length())
	    result += smg_String("OBST_COPY_") + "START\n"
	           +  rslt + "OBST_COPY_" + "END\n";
      }
      break;

      case _COPY_COMPS :
      {  sos_Method_List ml = class_type.get_local_methods();
         agg_iterate (ml, sos_Method cm)
         {  if (VALID(cm.get_comp_descr()) && cm.get_is_set())
	    {  sos_Expr e = cm.get_comp_descr().get_init_expr();
	       if VALID(e)
	       {  char* mn = cm.get_name().make_Cstring();
	          result += smg_String("_x.") + mn + "(_y.";
	          mn[0] = 'g';
	          result += smg_String(mn) + "())" + NL;
	          delete mn;
	       }
	    }
         }
         agg_iterate_end (ml, m);
      }
      break;       

      case CLONE :
      {  result += smg_String("return ")
                +  back_call_method (nm_class, FIXED_FCT("make"),
				     TRUE, FALSE, FALSE,
                       back_call_method ("_y", FIXED_FCT("_clone"),
					 FALSE, FALSE, TRUE,
                                         FIXED_FCT("_y._self,_ct")))
		+  NL;
      }
      break;

      case _CLONE :
      {  result += smg_String("return ") 
                +  code_cast_args ("sos_Object",
                                   back_call_method (nm_class, 
                                                     FIXED_FCT("copy"),
				                     TRUE, FALSE, FALSE,
				                     nm_class + make_THIS 
                                                      + ",_ct"))
		+ NL;
      }
      break;

      case DESTROY :
      {  agg_iterate_reverse (super_closure, sos_Super_class sc)
         {  sos_Class_type sct = sos_Class_type::make(sc.make_type());


	    int   key    = make_sos_Int (local_total_map[(sos_Object)sc]);
	    char* methnm = (key & HAS_LOCAL_DTOR)
			      ? nm_local_dtor
	       		      : (key & HAS_TOTAL_DTOR) ? nm_total_dtor
			      			       : NULL;
	    if (methnm)
            {  smg_String stn = sct.root().get_name();
               result += back_call_method (smg_String(_ID) + stn, 
                                           methnm,
                                           TRUE, FALSE, FALSE,
					   stn + make_THIS) + NL;
            }
         }
         agg_iterate_reverse_end (super_closure, sc);

         result += smg_String(THIS_param) + OBJ_container 
		+  FIXED_FCT(".deallocate(") + THIS_param 
                +  OBJ_offset + "," + class_type.get_object_size() + ")" + NL;
      }
      break;

      case ASSIGN :
      {  sos_Bool first = TRUE;
	 agg_iterate (super_closure, sos_Super_class sc)
	    int   key    = make_sos_Int (local_total_map[(sos_Object)sc]);
	    char* methnm = (key & HAS_LOCAL_ASSIGN)
			      ? nm_local_assign
	       		      : (key & HAS_TOTAL_ASSIGN) ? nm_total_assign
			      				 : NULL;
	    if (methnm)
	    {  
               if (first)
	       {  first = FALSE;
		  result += nm_class + " _x=" + nm_class + make_THIS + NL;
	       }
               sos_Class_type sct   = sos_Class_type::make(sc.make_type());
	       smg_String     sc_nm = sct.root().get_name();

               smg_String args;
               if (sc_nm.equal(nm_class))
                  args = "_x,_y";
               else 
                  args = code_cast_args (sc_nm, "_x") + ",_y";
	       result += back_call_method
			    (smg_String(_ID)
			        + sos_Class_type::make (sc.make_type())
			             .get_root_class().get_name(),
			     methnm, TRUE, FALSE, FALSE, args)
		      +  NL;
	    }
	 agg_iterate_end (super_closure, sc);
      }
      break;

      case LOCAL_ASSIGN :
      {  if (class_type.get_local_size() > 0)
	 {  result += impl_nm_class + "*_yp="
	    	   +  code_get_impl_ptr (smg_String("_y") + OBJ_type_ID)
	           +  NL
		   +  "_y" + OBJ_container
			   + FIXED_FCT(".copy(_y") + OBJ_offset
			   + FIXED_FCT("+_yp->_offset_" + nm_class) + "(),"
	           	   + class_type.get_local_size()
			   + ",_x" + OBJ_container
	           	   + ",_x" + OBJ_offset
	           	   + "+"
			   + back_call_method("_x",
					      smg_String("_offset_")+ nm_class,
					      FALSE, FALSE, TRUE, "")
			   + ")"
		   +  NL;
         }
      }
      break;

      case EQUAL :
      {  result += smg_String("if(_eq==EQ_STRONG){if(");

	 if (class_is_instantiation)
	    result += smg_String(THIS_param) + ".get_tp()";
	 else
	    result += smg_String(_ID) + nm_class + "_type";

	 result += smg_String("!=_y") + OBJ_type_ID + ")return FALSE;}\n"
		+  "else if(!_y.isa(";

	 if (class_is_instantiation)
	    result += smg_String("type(") + THIS_param + ")";
	 else
	    result += nm_class + "_type";

	 result += "))return FALSE;\n";
		 
	 sos_Bool first = TRUE;
	 agg_iterate (super_closure, sos_Super_class sc)
	    int   key    = make_sos_Int (local_total_map[(sos_Object)sc]);
	    char* methnm = (key & HAS_LOCAL_EQUAL)
			      ? nm_local_equal
	       		      : (key & HAS_TOTAL_EQUAL) ? nm_total_equal
			      				: NULL;
	    if (methnm)
	    {  if (first)
	       {  first = FALSE;
		  result += nm_class + " _x=" + nm_class + make_THIS + NL;
	       }
               sos_Class_type sct   = sos_Class_type::make(sc.make_type());
	       smg_String     sc_nm = sct.root().get_name();

               smg_String args;
               if (sc_nm.equal(nm_class))
                  args = "_x,_y,_eq";
               else 
                  args = code_cast_args (sc_nm, "_x") + ",_y,_eq";
	       result += smg_String("if(!")
		      +  back_call_method
			    (smg_String(_ID)
			        + sos_Class_type::make (sc.make_type())
			             .get_root_class().get_name(),
			     methnm, TRUE, FALSE, FALSE, args)
		      +  ")return FALSE" + NL;
	    }
	 agg_iterate_end (super_closure, sc);

         result += smg_String("return TRUE") + NL;
      }
      break;

      case LOCAL_EQUAL :
      {  if (class_type.get_local_size() == 0)
	    result += smg_String("return TRUE") + NL; 
	 else
	    result += impl_nm_class + "*_yp="
	    	   +  code_get_impl_ptr (smg_String("_y") + OBJ_type_ID)
	           +  NL
		   +  "return _x" + OBJ_container + FIXED_FCT(".equal(")
		      + "_x" + OBJ_offset
                        + "+"
			+ back_call_method ("_x",
					    smg_String("_offset_") + nm_class,
					    FALSE, FALSE, TRUE, "")
		      + "," + class_type.get_local_size()
		      + ",_y" + OBJ_container
		      + ",_y" + OBJ_offset
		      + FIXED_FCT("+_yp->_offset_" + nm_class) + "())"
		   +  NL;
      }
      break;
 
      case HASH_VALUE :
      {  sos_Bool first = TRUE;
	 agg_iterate (super_closure, sos_Super_class sc)
	    int   key    = make_sos_Int (local_total_map[(sos_Object)sc]);
	    char* methnm = (key & HAS_LOCAL_HASHVAL)
			      ? nm_local_hashv
	       		      : (key & HAS_TOTAL_HASHVAL) ? nm_total_hashv
			      				  : NULL;
	    if (methnm)
	    {  if (first)
	       {  first = FALSE;
		  result += nm_class + " _x=" + nm_class + make_THIS + NL
		         +  "sos_Int _r=";
	       }
	       else
		  result += "_r^=";

               sos_Class_type sct   = sos_Class_type::make(sc.make_type());
	       smg_String     sc_nm = sct.root().get_name();

               smg_String args;
               if (sc_nm.equal(nm_class))
                  args = "_x";
               else 
                  args = code_cast_args (sc_nm, "_x");
	       result += back_call_method
			    (smg_String(_ID)
			        + sos_Class_type::make (sc.make_type())
			             .get_root_class().get_name(),
			     methnm, TRUE, FALSE, FALSE, args)
		      +  NL;
	    }
	 agg_iterate_end (super_closure, sc);

         result += smg_String(first ? "return 0" : "return _r") + NL;
      }
      break;
      
      case LOCAL_HASH_VALUE :
      {  if (class_type.get_local_size() == 0)
	    result += smg_String("return 0") + NL;
	 else
	   result += smg_String("return _x") + OBJ_container
		  +  FIXED_FCT(".hash_value(_x") + OBJ_offset
		  +  "+" + back_call_method("_x",
					    smg_String("_offset_") + nm_class,
					    FALSE, FALSE, TRUE, "")
		  +  "," + class_type.get_local_size() + ")" + NL;
      }
      break;

      case MAKE_HANDLE :
      {  result += smg_String("   ") + nm_class + " _x" + "; "
                +  "_x." + THIS_comp + "=" + THIS_param + "; "
	        +  "_x." + IMPL_comp + "="
		+  code_get_impl_ptr (smg_String(THIS_param) + ".get_tp()")
		+  "; " 
                +  "return _x" + NL;
      }
      break;

      case MAKE_CP_HANDLE :
      {  result += smg_String("   ") + nm_class + " _x;" 
                +  "_x." + THIS_comp + "=" + THIS_param 
	        +  ";_x." + IMPL_comp + "=_y;return _x" + NL;
      }
      break;

      case MAKE_SC_HANDLE :
      {  result += smg_String ("   return ") + nm_class + "::make(_x" 
                +  OBJ_typed_id + ")" + NL;
      }
      break;
 
      default : ;
   }
   TT (genCC_L, T_LEAVE);
   return result;
}

//-------- argument lists, type conversions, expressions  ---------------------

LOCAL smg_String code_type_conversion (sos_Type_descr from_td,
		          	       sos_Type_descr target_td,
				       smg_String     expr)
{  T_PROC ("code_type_conversion")
   TT (genCC_L, T_ENTER);

   sos_Type from_tp   = from_td.make_type();
   sos_Type target_tp = target_td.make_type();
   smg_String result;

   if (from_tp.equal ((sos_Object)target_tp))
      result = expr;
   else
   {  if (target_tp.is_scalar())
         result += smg_String ("::make_") + target_tp.get_name()
		+  "(" + expr + ")";

      else if (from_tp.is_scalar())
         result += smg_String ("::make_") + from_tp.get_name() + "_object("
	    	+  expr + ")";

      else if (target_tp.has_type (sos_Union_type_type))
      {  if (from_tp.equal ((sos_Object)sos_Object_type))
	    result += smg_String(target_tp.get_name())
	           +  "((" + expr + ")._typed_id())";
         else
	    result += smg_String("(") + expr + ")";
      }
      else if (   from_tp.root() == target_tp
	       OR from_tp.is_derived_from (target_tp))
	 result += smg_String("(") + target_tp.get_name() + ")(" + expr + ")";

      else
         result += smg_String(target_tp.get_name()) + "::make(" + expr + ")";
   }
   TT (genCC_L, T_LEAVE);
   return result;
}

EXPORT void code_param_list ()
{  T_PROC ("code_param_list")
   TT (genCC_M, T_ENTER);

   if VALID(code_params)
   {  agg_iterate (code_params, sos_Param p)
         sos_Type   pt  = p.get_type().make_type();
         smg_String ptn = pt.get_name();
         sos_Expr   e   = p.get_default_expr();
         
	 back_param (p.get_name(),
		     ptn,
		     p.get_is_ref(),
		     pt.is_scalar(),
		     e);	    //GCC BUG: p.get_default_expr() --> error
      agg_iterate_end (code_params, p);
   }
   TT (genCC_M, T_LEAVE);
}

EXPORT smg_String code_expr (sos_Expr   expr,
			     smg_String pref, smg_String target_tp /* = "" */)
{  T_PROC ("code_expr")
   TT (genCC_L, T_ENTER);

   smg_String result;
   if VALID(expr)
   {  result += pref;

      if (expr.has_type (sos_Identifier_type))
      {  smg_String ident = sos_Identifier::make(expr).get_id();
	 if (    target_tp.length()
	     AND ident.equal ("NO_OBJECT")
	     AND NOT target_tp.equal(FIXED_TYPE("sos_Object")))
#if COMPLICATED_ARGUMENT_BUG
	    result += smg_String("OBST_") + "MAKE_ARG(" + target_tp
	           +  ",_" + target_tp + "_tmp,NO_OBJECT)";
#else
	    result += target_tp + "::make(NO_OBJECT)";
#endif
	 else
	    result += ident;
      }
      else
         result += sos_Int_expr::make(expr).get_value();
   }
   TT (genCC_L, T_LEAVE);
   return result;
}
