/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// Module dmp_main                                           Juergen Uhl (ju)
//                                                            Jochen Alt (ja)
//                                                     Bernhard Schiefer (bs)
// **************************************************************************
// OBST dump tool
// **************************************************************************

#define OBST_IMP_STDCONST
#define OBST_IMP_STRINGOP
#define OBST_IMP_PROCEXEC
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "obst_trc.h"
#include "obst.h"
#include "smg.h"
#include "dmp_err.h"
#include "knl_use.h"
#include "mta_use.h"
#include "dir_use.h"
#include "cci_use.h"

LOCAL sos_Object_Set dumped_containers;
LOCAL sos_Object_Set dumped_objects;
LOCAL sos_Bool       only_one_container;
LOCAL sos_Bool       only_container;
LOCAL sos_Bool       only_oids;
LOCAL sos_Bool       short_output;
LOCAL sos_Container  root_container;

LOCAL void dump_object (sos_Object o,
			sos_Int    level,
                        sos_Bool   as_component = FALSE);

LOCAL sos_Object get_component (sos_Object o,
				sos_Int    os,
				sos_Type   tp,
				sos_Bool   is_local)
{  sos_Typed_id tpid;

   if (tp.is_scalar ())
   {  sos_Int sz = tp.get_object_size();
      sos_Id  id = sos_Id::make (sos_Container::make (0), 0);
      o.container().read (o.offset() + os, sz, &id);
      tpid = sos_Typed_id::make (id, tp._self_id());
   }
   else if (is_local)
   {  union { sos_Char c[SOS_OFFSET_SIZE]; sos_Offset _dummy; } u;

      o.container().read (o.offset() + os, SOS_OFFSET_SIZE, u.c);
      sos_Offset os1;
      bcopy_to_sos_Offset(&os1, u.c);
      tpid = _obst_make_local_typed_id (os1, o.container());
   }
   else
   {  union { sos_Char c[SOS_TYPED_ID_SIZE]; sos_Typed_id _dummy; } u;

      o.container().read (o.offset() + os, SOS_TYPED_ID_SIZE, u.c);
      bcopy_to_sos_Typed_id(&tpid, u.c);
   }

   return sos_Object::make (tpid);
}

LOCAL void dump_immediate_components (sos_Object     o,
				      sos_Class_type ct,
				      sos_Int        os,
				      sos_Int        level,
				      sos_Bool       as_component,
				      sos_Bool       &is_first)
{
   sos_Comp_descr_List ml = ct.get_components();
   agg_iterate (ml, sos_Comp_descr m)
   {  sos_Method cm  = m.get_get_method();
      sos_Type   ctp = cm.get_result_type().make_type();
      if (NOT (only_container AND ctp.is_scalar()))
      {     // dump only local components, components of higher classes
            // are dumped from calls of dump_immediate_components with
	    // these super classes (ja)
				 // ATT2.0 QUEERNESS: explicit operator
         if (cm.get_defined_in ().operator==(ct))
         {  sos_Object co = get_component (o, os + m.get_offset(),
					   ctp, m.get_is_local());
	    if (as_component)
	    {  if (is_first)
	          is_first = FALSE;
	       else
	          cout << ",";
	       if (NOT short_output)
	       {  char *cn = cm.get_name().make_Cstring();
		  cout << (cn + 4) << "=";	 // +4 skips "get_"
	          delete cn;
	       }
	    }
	    dump_object (co, (as_component?level:level-1), as_component);
         }
      }
   }
   agg_iterate_end (ml, m);

   if (     as_component	//ATT2.0 QUEERNESS: explicit operator
        AND ct.operator==(sos_String_type)
        AND NOT short_output )
   {  if (NOT is_first)
	 cout << ",";
      cout << "*value*=\"" << sos_String::make(o) << "\"";
   }				//ATT2.0 QUEERNESS: explicit operator
   else if (ct.root().operator==(sos_Object_Collection_type))
   {  if (as_component)
      {  if (is_first)
	    is_first = FALSE;
	 else
	    cout << ",";
	 cout << (short_output ? "{" : "*value*={");
      }
      sos_Bool		    is_first_elem = TRUE;
      sos_Object_Collection coll	  = sos_Object_Collection::make (o);
      agg_iterate (coll, sos_Object co)
      {  if (as_component)
	 {  if (is_first_elem)
	       is_first_elem = FALSE;
	    else
	       cout << ",";
	 }
	 dump_object (co, (as_component?level:level-1), as_component);
      }
      agg_iterate_end (coll, co);
      if (as_component)
	 cout << "}";
   }				//ATT2.0 QUEERNESS: explicit operator
   else if (ct.root().operator==(sos_Object_sos_Object_Association_type))
   {  if (as_component)
      {  if  (is_first)
	    is_first = FALSE;
	 else
	    cout << ",";
	 cout << "*value*={";
      }
      sos_Bool				is_first_elem = TRUE;
      sos_Object_sos_Object_Association as =
	 sos_Object_sos_Object_Association::make (o);
      agg_iterate_association (as, sos_Object role1, sos_Object role2)
      {  if (as_component)
	 {  if (is_first_elem)
	       is_first_elem = FALSE;
	    else
	       cout << ",";
	    cout << "<";
	 }
	 dump_object (role1, (as_component?level:level-1), as_component);
         if (as_component)
	    cout << ",";
	 dump_object (role2, (as_component?level:level-1), as_component);
         if (as_component)
	    cout << ">";
      }
      agg_iterate_association_end (as, role1, role2);
      if (as_component)
	 cout << "}";
   }
}

LOCAL void dump_class_instance (sos_Object     o,
				sos_Class_type ct,
				sos_Int        level,
				sos_Bool       as_component)
{  sos_Super_class_List scl
      = sos_Class_type::make (ct.root()).get_super_closure();

   sos_Bool is_first = TRUE;
   agg_iterate (scl, sos_Super_class sc)
   {  sos_Class_type sct = sos_Class_type::make(
					sc.get_super_class().make_type());
      sos_Int os = sc.get_offset();
      dump_immediate_components (o, sct, os, level, as_component, is_first);
   }
   agg_iterate_end (scl, sc)
}

LOCAL void dump_container (sos_Container ct)
{  sos_Object oct = make_sos_Container_object (ct);
   if (dumped_containers.is_element (oct))
      return;
   dumped_containers.insert (oct);
   cout << (sos_Int)ct << "\n";
}

LOCAL void dump_object (sos_Object o,
			sos_Int    level,
			sos_Bool   as_component /* = FALSE */)
{  if (level == 0)
      return;

   if (INVALID(o))
   {  if (as_component)
	 cout << "<>";
      return;
   }
   if (    NOT as_component
       AND only_one_container
       AND root_container != o.container())
      return;

   sos_Type tp = o.type();

   if (as_component)
   {  err_block
      {  if (tp.has_type (sos_Class_type_type))
	    cout << "<" << (sos_Int)o.container()
		 << "," << (int)o.offset() << ">";

	 else if (tp.has_type (sos_Enum_type_type) AND NOT only_oids)
	 {  sos_Int i  = _obst_Int_from_enum (o, tp);
	    cout << sos_Enum_type::make (tp).get_literals().get_nth(i+1) << ":";
	 }
	 else if (tp.has_type (sos_Extern_type_type) AND NOT only_oids)
	 {  cout << cci_Schema_impl::string_from_extern_object (o);
	 }
	 else if (NOT only_oids)
	    cout << "???";
      }
      err_exception
      {  cout << "???";
      }
      err_block_end
   }
   else if (tp.has_type (sos_Class_type_type))
   {  if (dumped_objects.is_element (o))
	 return;
      dumped_objects.insert (o);

      if (only_container)
	 dump_container (o.container());
      else
      {  err_block
	 {  cout << "<"   << (sos_Int)o.container()
		 << ","   << (int)o.offset()
		 << ">:[" << tp.get_name() << "(" << o.size() << "):";
	    dump_class_instance (o, sos_Class_type::make (tp), level, TRUE);
	    cout << "]\n";
	 }
	 err_exception
	 {  cout << "???\n";
	 }
	 err_block_end
      }
      dump_class_instance (o, sos_Class_type::make (tp), level, FALSE);
   }
}

#ifdef PROFILE
extern "C" void monitor(...);
#endif

EXPORT int dmp_main (int argc, char *argv[])
{  T_FILE ("dmp.out");
   T_REDEF(NULL);

   sos_Object o;
   sos_Int level=-1;
   sos_Int i=1;

   only_one_container = FALSE;
   only_container     = FALSE;
   only_oids          = FALSE;
   short_output       = FALSE;

   while (argc > i AND argv[i][0] == '-')
   {  if (argc > i+1 AND streql (argv [i], "-l"))
      {  level = atoi (argv [i+1]);
	 if (level==0)
	 {  err_raise (err_USE, err_DMP_USAGE, NULL, FALSE);
	    exit (-1);
	 }
	 i+=2;
      }
      else if (streql (argv [i], "-c"))
      {  only_container = TRUE;
	 i++;
      }
      else if (streql (argv [i], "-o"))
      {  only_one_container = TRUE;
	 i++;
      }
      else if (streql (argv [i], "-s"))    // (bs) 20.07.93
      {  short_output = TRUE;
	 i++;
      }
      else if (streql (argv [i], "-i"))    // (bs) 20.07.93
      {  short_output = TRUE;
	 only_oids    = TRUE;
	 i++;
      }
      else
      {  err_raise (err_USE, err_DMP_USAGE, NULL, FALSE);
	 exit (-1);
      }
   }

   if (argc > i+2)
   {  err_raise (err_USE, err_DMP_USAGE, NULL, FALSE);
      exit (-1);
   }
   else if (argc > i)
   {  sos_Container ct = sos_Container::make (atoi (argv [i]));
      sos_Offset    os;

      if (argc > i+1)
	 os = atoi (argv [i+1]);
      else
	 os = ROOT_OFFSET;

      if (ct == 0 OR os == 0)
      {  err_raise (err_USE, err_DMP_USAGE, NULL, FALSE);
	 exit (-1);
      }
      err_block
      {  o = sos_Object::make
		(sos_Typed_id::make
		   (sos_Id::make (sos_Container::make (ct), os)));
      }
      err_exception
      {  err_raise (err_USE, err_DMP_INVALID_ROOT, NULL, FALSE);
	 exit (-1);
      }
      err_block_end
   }
   else
      o = sos_Object_Directory::root();

   sos_Container dct = sos_Container::create();
   dumped_objects    = sos_Object_Set::create (dct, FALSE);
   dumped_containers = sos_Object_Set::create (dct, FALSE);
   root_container    = o.container();

   dump_object (o, level);
   cout << "Ready: " << dumped_objects.card() << " objects dumped\n";
   dct.destroy();
   dct.close();

#ifdef PROFILE
   monitor (0);
#endif

   return 0;
}
