/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-7500 Karlsruhe 1, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// Module cfe_type                                           Juergen Uhl (ju)
//
// **************************************************************************
// OBST schema compiler front end (cfe)
// **************************************************************************
//
// tracing conventions: see trc_cfe.h

#include "cfe.h"

// LOCAL


EXPORT void cfe_set_offsets_and_size (sos_Class_type ct, sos_Int os)
{
   T_PROC ("cfe_set_offsets_and_size")
   TT (cfe_L, T_ENTER);

   ct.set_local_size (os);

   sos_Int              offset = 0;
   sos_Super_class_List scl    = ct.get_super_closure();

   agg_iterate (scl, sos_Super_class sc)
   {  sc.set_offset(offset);
      offset += sos_Class_type::make(sc.get_super_class()).get_local_size();
   }
   agg_iterate_end (scl, sc);
   ct.set_object_size (offset);

   TT (cfe_L, T_LEAVE);
}


EXPORT void cfe_check_type (sos_String t_name)
// Checks multiple declarations of type names
{
   T_PROC ("cfe_check_type")
   TT (cfe_M, T_ENTER);

   sos_Type_descr t = cfe_schema.lookup_type (t_name);

   if VALID(t)
      cfe_error (err_USE, err_CFE_MULTIPLE_TYPE_DECL, t_name);

   TT (cfe_M, T_LEAVE)
}


EXPORT sos_Class_type cfe_make_class_type (sos_String name,
					   sos_Type_table type_tab,
					   sos_Type_descr_List type_list)
// Checks multiple declarations of type names, make forwards to real types
{
   T_PROC ("cfe_make_class_type")
   TT (cfe_H, T_ENTER);

   sos_Class_type result;
   sos_Type_descr t = cfe_schema.lookup_type (name);

   if INVALID(t) 				// not found, then create
   {  result = sos_Class_type::create(cfe_cnt);
      result.set_name(name);
      // Initialize fgpl to show that no temporay list must be destroyed.
      result.set_formal_gen_params (sos_Gen_param_List::make(NO_OBJECT));
      result.set_methods(sos_Method_table::make(NO_OBJECT));
   }
   else if (t.has_type (sos_Class_type_type)) 	// Found? Must be forward!
   {  sos_Class_type fwc = sos_Class_type::make(t);
      if VALID(fwc.get_methods()) // class not forward ?
      {  cfe_error (err_USE, err_CFE_MULTIPLE_TYPE_DECL, name);
	 result = sos_Class_type::create(cfe_cnt); // to allow further compile
      }
      else // Class was forward. Keep class forward
      {    // and remove it from type table and type list, since
	   // it will be entered during its declaration.
	 result = fwc;
         type_tab.remove(name);
	 type_list.remove(type_list.find(fwc));
      }
   }
   else 					  // found, but not forward!
      cfe_error (err_USE, err_CFE_MULTIPLE_TYPE_DECL, name);

   TT (cfe_H, T_LEAVE);
   return result;
}


EXPORT void cfe_check_no_generic (sos_Type_descr td)
{
   T_PROC ("cfe_check_no_generic")
   TT (cfe_M, T_ENTER);

   if (td.has_type (sos_Class_type_type))
   {  sos_Class_type ct = sos_Class_type::make (td);
      if VALID(ct.get_formal_gen_params())
	 cfe_error (err_USE, err_CFE_NO_GENERIC_TYPE, ct.get_name());
   }

   TT (cfe_M, T_LEAVE);
}


EXPORT sos_Type_descr cfe_lookup_type_name(sos_Class_type ct, sos_String name,
					   sos_Import_mode imp_mode)
{  T_PROC ("cfe_lookup_type_name")
   TT (cfe_H, T_ENTER);

   sos_Type_descr result = sos_Type_descr::make(NO_OBJECT);

   if VALID(ct)
   {  sos_Gen_param_List gpl = ct.get_formal_gen_params();
      if VALID(gpl)
      {  sos_Gen_param gp = cfe_lookup_gen_params (name, gpl);
	 if VALID(gp)
	    result = gp;
      }
   }

   if INVALID(result)
   {
      // Now search type in the type_table(s).
      result = cfe_schema.lookup_type (name, imp_mode);

      if (result.identical (NO_OBJECT)) // type was not found
      {  sos_String n = sos_String::copy (name, cfe_cnt);
	 // enter as forward type with no actual generic parameters
	 result = cfe_create_and_enter_forward (
		    n, sos_Type_descr_List::make(NO_OBJECT));
      }
   }
   TT (cfe_H, T_LEAVE);
   return result;
}


EXPORT sos_Int cfe_enum_size (sos_Int nr_literals)
{  T_PROC ("cfe_enum_size");
   TT (cfe_H, T_ENTER);
   
   if (nr_literals <= 256)
      return 1;
   else if (nr_literals <= 65536)
      return 2;
   else
      cfe_error (err_USE, err_CFE_TOO_MANY_LITERALS);

   TT (cfe_H, T_LEAVE);
   return 0;
}


LOCAL sos_Class_type cfe_create_forward_gen_class
			       		(sos_String	     name,
					 sos_Type_descr_List act_gen_params)
{  T_PROC ("cfe_create_forward_gen_class")
   TT (cfe_H, T_ENTER);

   sos_Class_type result;
   
   /* If the generic class "name" is not found in any type table, it
      is created. A temporary formal generic parameter list is established.
      This must contain as much elements as the actual list, but all must 
      have no superclass limitation (sos_Object). This list must be
      located in the current class' container, since formal_gen_params
      is declared local. Thus, this list must be destroyed when replaced
      by the final list.  Further the created class is inserted into
      the type list and type table. */

   result= sos_Class_type::make(cfe_schema.lookup_type(name,sos_IMP_GENERIC));
   if INVALID(result) // generic class not found, so create it
   {
      static sos_Gen_param gp   = sos_Gen_param::create (TEMP_CONTAINER);
      static char needs_init	= TRUE;

      if (needs_init)
      {
	 gp.set_name (sos_String::create (TEMP_CONTAINER, "temporary"));
	 gp.set_super_type (sos_Type_descr::make (NO_OBJECT));
         needs_init = FALSE;
      }

      sos_Gen_param_List fgpl = sos_Gen_param_List::create (cfe_cnt);

      result = sos_Class_type::create (cfe_cnt);
      for (int i = act_gen_params.card(); i >= 1; i--)
	 fgpl.append (gp);
      result.set_formal_gen_params (fgpl);
      result.set_actual_gen_params (sos_Type_descr_List::make (NO_OBJECT));
      result.set_name (sos_String::copy (name, cfe_cnt));
      result.set_generic_class (result);
      result.set_methods (sos_Method_table::make (NO_OBJECT));

      cfe_type_tab.insert (name, result);
      cfe_types.append (result);
   }
   TT (cfe_H, T_LEAVE);
   return result;
}

EXPORT sos_Generic_instantiation cfe_get_instantiation
				   	(sos_Class_type	     gencl,
				    	 sos_Class_type      instcl,
      				    	 sos_Type_descr_List agpl)
{  T_PROC ("cfe_get_instantiation")
   TT (cfe_L, T_ENTER);

   static sos_Object_sos_Object_Mapping_Hash gil_table
      	     = sos_Object_sos_Object_Mapping_Hash::create (TEMP_CONTAINER);

   sos_Generic_instantiation gi;
   sos_Bool do_create = TRUE;

   sos_Object_List gil = sos_Object_List::make (gil_table[(sos_Object)instcl]);

   if INVALID(gil)
      gil_table.insert (instcl, gil= sos_Object_List::create(TEMP_CONTAINER));
   else
      agg_iterate (gil, sos_Object giobj)
         gi = sos_Generic_instantiation::make (giobj);
	 sos_Type_descr_List gipl = gi.get_act_gen_params();

   	 if (gipl == agpl OR gipl.equal (agpl)) // '==' accounts for NO_OBJECT
	 {  do_create = FALSE;
	    break;
	 }
      agg_iterate_end (gil, obj);

   if (do_create)
   {  gi = sos_Generic_instantiation::create (cfe_cnt);
      gi.set_gen_class (gencl);
      gi.set_instantiation (instcl);
      gi.set_act_gen_params (agpl);
      gil.insert (1, gi);
   }
   TT (cfe_L, T_LEAVE);
   return gi;
}

EXPORT sos_Type_descr cfe_create_and_enter_forward
					(sos_String	     name,
					 sos_Type_descr_List act_gen_params)
{
   /* This function handles a forward occurrence of a class type either
      as normal forward use or as a forward use of a generic class.
      The class type is created, the methods component is set to NO_OBJECT
      in order to indicate the class' forward property. The actual generic
      parameter list is entered into the class type object. If this list
      contains formal generic parameters, an sos_Generic_instantiation
      object is created and filled in.
      If a forward generic instantiation was the first of that generic class,
      the sos_Class_type object for the generic class is created by the
      function cfe_create_forward_gen_class and inserted
      as the first element in the corresponding forward list.
      When entering the object, a check is performed, whether the name of
      the class was previously used forward. If so, the actual generic
      parameter list of the former use is checked to have the same length
      as the now provided one.
      All created class types are entered in the type table and the type list.

      act_gen_params is destroyed, if not stored somewhere.
   */

   T_PROC ("cfe_create_and_enter_forward")
   TT (cfe_H, T_ENTER);

   sos_Class_type gen_class;
   sos_Bool	  destroy_params = TRUE;

   // get the right class type list
   sos_Class_type_List fctl = cfe_forward_tab [name];
   if INVALID(fctl)
   {
      fctl = sos_Class_type_List::create(TEMP_CONTAINER);
      cfe_forward_tab.insert (name, fctl);
      if VALID(act_gen_params) // first instantiation of that class
      { 
	 gen_class = cfe_create_forward_gen_class (name, act_gen_params);
	 fctl.append (gen_class);       // insert as first element

         if (act_gen_params == gen_class.get_actual_gen_params())
	    destroy_params = FALSE;
      }
   }
   else // check, if generic parameter lists conform
   {
      gen_class = fctl.get_nth(1);
      if INVALID(act_gen_params)
      {
	 if VALID(gen_class.get_formal_gen_params())
	 {
	    cfe_error (err_USE, err_CFE_GEN_PARAM_MISMATCH, name);
	    TT (cfe_H, T_LEAVE);
	    return ERROR_TYPE;
         }
      }
      else // was generic class, count parameters
      {
	 if (INVALID(gen_class.get_formal_gen_params())
	     ||  gen_class.get_formal_gen_params().card()
	         != act_gen_params.card())
         {
            cfe_error (err_USE, err_CFE_GEN_PARAM_MISMATCH, name);
            TT (cfe_H, T_LEAVE);
            return ERROR_TYPE;
         }
      }
   }
   sos_Type_descr result;
   sos_Class_type fwc;

   if INVALID(act_gen_params)
   { // non-generic class, create class type and enter it
      fwc = sos_Class_type::create (cfe_cnt);
      fwc.set_methods (sos_Method_table::make(NO_OBJECT)); // forward !!!
      fwc.set_formal_gen_params (sos_Gen_param_List::make(NO_OBJECT));
      fwc.set_name (name);
      fwc.set_actual_gen_params (act_gen_params);
      cfe_type_tab.insert (fwc.get_name(), fwc);
      cfe_types.append (fwc);
      result = fwc;
   }
   else if (cfe_contains_formal_gen_params (act_gen_params))
   {  // formal parameters as actual, sos_Generic_instantiation required!
      // Not yed allow deletion of act_gen_params list, since it may be needed
      // for the sos_Generic_instantiation object.
      fwc = cfe_create_inst_class_type(name, act_gen_params,FALSE, cfe_types);
      sos_Generic_instantiation gi
		     = cfe_get_instantiation (gen_class, fwc, act_gen_params);
      if (gi.get_act_gen_params() == act_gen_params)
         destroy_params = FALSE;

      result = gi;
   }
   else // actual types completely resolved, sos_Class_type needed
      result =
      fwc    = cfe_create_inst_class_type(name,act_gen_params,TRUE,cfe_types);

   fctl.find_or_append (fwc);

   if (    destroy_params
       AND VALID(act_gen_params)
       AND fwc.get_actual_gen_params() != act_gen_params)
      act_gen_params.destroy();

   TT (cfe_H, T_LEAVE);
   return result;
}


#ifdef CFE_DEBUG

EXPORT void cfe_print_typedescr (sos_Type_descr td)
{  if (td.has_type(sos_Class_type_type))
      cout << "Class[" << sos_Named::make(td).get_name() << "]";
   else if (td.has_type(sos_Generic_instantiation_type))
   {  sos_Generic_instantiation gi = sos_Generic_instantiation::make(td);
      cout << "Generic_inst[";
      cfe_print_typedescr(gi.get_gen_class());
      cout << "-->";
      cfe_print_typedescr(gi.get_instantiation());
      cout << "{";
      sos_Type_descr_List tdl = gi.get_act_gen_params();
      agg_iterate (tdl, sos_Type_descr td)
	 cfe_print_typedescr(td);
      agg_iterate_end (tdl, td);
   }
   else if (td.has_type(sos_Super_class_type))
   {  cout << "Super_class[";
      cfe_print_typedescr(sos_Super_class::make(td).get_super_class());
      cout << "]";
   }
   else if (td.has_type(sos_Gen_param_type))
   {  sos_Gen_param gp = sos_Gen_param::make(td);
      cout << "Gen_param[" << gp.get_name() << "-";
      if INVALID(gp.get_super_type())
	 cout << "NO_OBJECT";
      else
         cfe_print_typedescr(gp.get_super_type());
      cout << "]";
   }
   else
      cout << "???[" << td.make_type().get_name() << "]";
}

#endif
