#line 1 "./src/agg/List.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */

#include "agg.h"

// **************************************************************************
// Module List              20/10/92                  Jochen Alt (ja)
//                                   modified 19.2.94 Bernhard Schiefer (bs)
//				     read_node replaced by read_prev_next
//				     and read_node_entry
// **************************************************************************
// implements methods of classes: List
// **************************************************************************

// Die Knoten werden direkt ueber psm abgespeichert, um bei einheitlicher
// Typinformation bzw. einheitlichem Container viiiiel Platz zu sparen. 
// Da eventuell der Container und der Typ nicht abgespeichert werden muss,
// kann dadurch pro Knoten 12 Byte gespart werden. 
// 
// Es wird fuer den Eintrag ein default Typ und ein default
// Container abgespeichert, beide werden beim Eintragen des ersten Elementes 
// gesetzt. Entspricht ein folgender Eintrag einem solchen Defaultwert,
// wird nur das unbedingt notwendige gespeichert. In der Standardanwendung,
// in der der Eintrag immer denselben Typ hat und im selben Container liegt,
// braucht ein Knoten dann 16 Bytes 
// 
// 
// Ein Knoten wird in folgendem Format abgespeichert:
//   sos_Offset : Verweis auf Vorgaengerknoten
//   sos_Offset : Verweis auf Nachfolgerknoten
//   sos_Int    : Typinformation des Knotens und Groesse des Knotens
//      evt. sos_Typed_id Typinformation des Eintrages
//      evt. sos_Container Container des Eintrages
//   sos_Offset : Offset des Eintrages
// 
//
// Der Typ eines Knotens wird durch die folgenden Konstanten identifiziert,
// die die Wertigkeit des Bits angeben, welches bestimmt, ob ein jeweiliges
// Datum abgespeichert wurde. Beispiel: Ist in einem Knoten
// der Typ abgespeichert, weil er dem Standard-typ nicht entspricht, 
// so ist das ENTRY_TYPE_DIFF.te Bit gesetzt.
// Die Groesse des Knotens ist im 2.Byte abgespeichert (Koennte man sich
// aus dem Type bequem berechnen, den Platz im sos_Int kriegt man jedoch 
// geschenkt).

const sos_Offset NO_OFFSET=0;
#define ENTRY_CNT_DIFF 1
#define ENTRY_TYPE_DIFF 2
// Die Groesse befindet sich im 2.ten Byte der Typinformation eines Knotens
#define SIZE_POS 256  

// Erzeuge den Typ Node_type mit NODE_TYPE_SIZE und den entsprechenden bcopies
#define Node_type sos_Int
#define NODE_TYPE_SIZE SOS_INT_SIZE
#define bcopy_from_Node_type bcopy_from_sos_Int
#define bcopy_to_Node_type bcopy_to_sos_Int


LOCAL const int PREV_OFFSET		= 0;
LOCAL const int NEXT_OFFSET		= SOS_OFFSET_SIZE;
LOCAL const int NODE_TYPE_OFFSET	= 2*SOS_OFFSET_SIZE;
LOCAL const int ENTRY_OFFSET		= 2*SOS_OFFSET_SIZE + NODE_TYPE_SIZE;
LOCAL const int MAX_NODE_SIZE		= ENTRY_OFFSET + 
			  			(SOS_CONTAINER_SIZE + 
			   			 SOS_OFFSET_SIZE +
			   			 SOS_ID_SIZE);

// **************************************************************************
void _sos_Object_List::change_list_cursor (const sos_Typed_id&_OBSThis,sos_Bool new_list_cursor)
// **************************************************************************
const{  set_list_cursor (_OBSThis,new_list_cursor);
}

inline LOCAL void write_offset (sos_Offset offset, sos_Container cnt,
                                sos_Offset contents)
{
   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;
 
   bcopy_from_sos_Offset (&contents, &u.mem);
 
   cnt.write (offset, SOS_OFFSET_SIZE, &u.mem);
}


// **************************************************************************
void _sos_Object_List::set_next (const sos_Typed_id&_OBSThis,sos_Offset offset, sos_Offset next)
// **************************************************************************
// Set the field with the successor to 'next'
const{
   if (offset == NO_OFFSET)
      set_first (_OBSThis,next);
   else
      write_offset (offset+NEXT_OFFSET, _OBSThis.container(), next);
}

// **************************************************************************
void _sos_Object_List::set_prev (const sos_Typed_id&_OBSThis,sos_Offset offset, sos_Offset prev)
// **************************************************************************
const{
   if (offset == NO_OFFSET)
      set_last (_OBSThis,prev);
   else 
      write_offset (offset+PREV_OFFSET, _OBSThis.container(), prev);
}
 
 

// ************************************************************************** 
sos_Offset _sos_Object_List::create_node (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) entry, 
					 sos_Offset prev, sos_Offset next)
// ************************************************************************** 
// create a node with contents (entry, prev, next) and return the offset
const{  
   T_PROC("List::create_node")
   TT(agg_L, T_ENTER; TOBJ(entry); TI(prev); TI(next))

   sos_Container cnt        = _OBSThis.container();
   sos_Container entry_cnt  = entry.container();
   sos_Id        entry_type = entry._type_id();
   sos_Id        def_type;
   sos_Container def_cnt;

   if (get_cardinality(_OBSThis) == 0)
   {  set_default_type (_OBSThis,def_type = entry_type);
      set_default_container (_OBSThis,def_cnt = entry_cnt);
   }
   else
   {
      def_type = get_default_type(_OBSThis);
      def_cnt  = get_default_container(_OBSThis);
   }
   
   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;

   char* ptr = u.mem + ENTRY_OFFSET;
   Node_type node_type = 0;

   if (entry_type != def_type)
   {  bcopy_from_sos_Id (&entry_type, ptr);
      ptr += SOS_ID_SIZE;
      node_type += ENTRY_TYPE_DIFF;
   }
   if (entry_cnt != def_cnt)
   {  bcopy_from_sos_Container (&entry_cnt, ptr);
      ptr += SOS_CONTAINER_SIZE;
      node_type += ENTRY_CNT_DIFF;
   }
   sos_Offset entry_offset = entry.offset();
   bcopy_from_sos_Offset (&entry_offset, ptr);
   ptr += SOS_OFFSET_SIZE;

   sos_Int size = ptr - u.mem;
   node_type += size*SIZE_POS;

   ptr = u.mem + PREV_OFFSET;
   bcopy_from_sos_Offset (&prev, ptr);
   ptr = u.mem + NEXT_OFFSET;
   bcopy_from_sos_Offset (&next, ptr);
   ptr = u.mem + NODE_TYPE_OFFSET;
   bcopy_from_Node_type (&node_type, ptr);

   sos_Offset offset = cnt.allocate (size);
   cnt.write (offset, size, &u);

   set_size (_OBSThis,get_size(_OBSThis) + size);
   set_cardinality (_OBSThis,get_cardinality(_OBSThis) + 1);
   
   set_prev (_OBSThis,next, offset);
   set_next (_OBSThis,prev, offset);

   TT(agg_L, T_LEAVE; TI(offset); TI(size))
   return offset;
}


inline LOCAL void get_size_and_node_type (
		sos_Offset    offset,
		sos_Container cnt,
		Node_type     &node_type,
		sos_Int       &size)
// 19.2.94 (bs)
{
   TT(agg_L, TI(offset); TI(int(cnt)))

   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;

   cnt.read (offset + NODE_TYPE_OFFSET, NODE_TYPE_SIZE, &u);

   bcopy_to_Node_type (&node_type, u.mem);
   size = node_type/SIZE_POS;

   TT(agg_L, TI(size))
}

// ************************************************************************** 
void _sos_Object_List::remove_node(const sos_Typed_id&_OBSThis,sos_Offset offset, sos_Offset pred,
                                  sos_Offset next)
// ************************************************************************** 
// remove a node at position // The size is determined inside !
const{  T_PROC("List:sos_Object_List::remove_node")
   TT(agg_L, T_ENTER; TI(offset); TI(pred); TI(next))

   sos_Container cnt = _OBSThis.container();
   Node_type node_type;
   sos_Int size;
   get_size_and_node_type (offset, cnt, node_type, size);

   cnt.deallocate (offset, size);
   set_size (_OBSThis,get_size(_OBSThis) - size);
   set_cardinality  (_OBSThis,get_cardinality(_OBSThis) - 1);

   set_next (_OBSThis,pred, next);
   set_prev (_OBSThis,next, pred);

   TT (agg_L, T_LEAVE)
}

LOCAL sos_Object read_node_entry (
		     sos_Offset    offset,
                     sos_Container cnt,
                     const sos_Id&       def_type,
                     sos_Container def_cnt)
// read a node entry at position offset
// 19.2.94 (bs)
{ 
   T_PROC("List:read_node_entry")
   TT(agg_L, T_ENTER; TI(offset); TI((int)cnt); TI((int)def_cnt))

   Node_type node_type;
   int size;
   get_size_and_node_type (offset, cnt, node_type, size);

   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;
   char* ptr = u.mem;

   cnt.read (offset+ENTRY_OFFSET,
	     size-ENTRY_OFFSET, &u);

   sos_Id entry_type = def_type;
   if (node_type BITAND ENTRY_TYPE_DIFF)
   {  bcopy_to_sos_Id (&entry_type, ptr);  
      ptr += SOS_ID_SIZE;
   }
   sos_Container entry_cnt = def_cnt;
   if (node_type BITAND ENTRY_CNT_DIFF)
   {  bcopy_to_sos_Container (&entry_cnt, ptr);  
      ptr += SOS_CONTAINER_SIZE;
   }
   sos_Offset entry_offset;
   bcopy_to_sos_Offset (&entry_offset, ptr);
 
   sos_Typed_id entry_tpid = 
      sos_Typed_id::make(sos_Id::make(entry_cnt,entry_offset),entry_type);
   sos_Object entry = sos_Object::make (entry_tpid);

   TT (agg_L, T_LEAVE; TOBJ(entry))
   return entry;
}

LOCAL void read_prev_next (
		     sos_Offset    offset,
                     sos_Container cnt,
                     sos_Offset    &prev,
                     sos_Offset    &next)
// read prev and next of a node at position offset
// 19.2.94 (bs)
{
   T_PROC("List:read_prev_next")
   TT(agg_L, T_ENTER; TI(offset); TI((int)cnt))

   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;
   char* ptr = u.mem;

   cnt.read (offset, 2*SOS_OFFSET_SIZE, &u);

   bcopy_to_sos_Offset (&prev, ptr);
   ptr += SOS_OFFSET_SIZE;
   bcopy_to_sos_Offset (&next, ptr);

   TT (agg_L, T_LEAVE; TI(prev); TI(next))
}

// ************************************************************************** 
void _sos_Object_List::local_initialize (OBST_PARDECL(sos_Object_List) l)
// ************************************************************************** 
{ 
   T_PROC("List::local_initialize");
   TT(agg_H, T_ENTER; TOBJ(l));

   l.set_cardinality (0);
   l.set_size (0);
   l.set_first (NO_OFFSET);
   l.set_last (NO_OFFSET);

   TT(agg_H,T_LEAVE)
}


// ************************************************************************** 
void _sos_Object_List::local_finalize (OBST_PARDECL(sos_Object_List) l)
// ************************************************************************** 
{  T_PROC("List::local_finalize");
   TT (agg_H, T_ENTER; TOBJ(l));

   l.clear();

   TT (agg_H, T_LEAVE)
}


// ************************************************************************** 
void _sos_Object_List::append (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) o)
// ************************************************************************** 
const{  T_PROC("List::append");
   TT(agg_H, T_ENTER; TOBJ(o));

   create_node (_OBSThis,o, get_last(_OBSThis), NO_OFFSET);

   TT(agg_H,T_LEAVE)   
}

// ************************************************************************** 
void _sos_Object_List::insert (const sos_Typed_id&_OBSThis,sos_Int pos,OBST_PARDECL( sos_Object) o)
// ************************************************************************** 
const{  T_PROC("List::insert");
   TT(agg_H, T_ENTER; TI(pos); TOBJ(o))

   sos_Int crd = get_cardinality(_OBSThis);

   err_assert ((0 <= pos) AND (pos <= crd+1),
               "List::insert position invalid");

      // pos == 0 forces the list to append the entity
   if (pos == 0)
      pos = crd + 1;

   sos_Container cnt = _OBSThis.container();
   sos_Offset offset = NO_OFFSET;

   if (pos <=crd/2)
   {
      sos_Offset prev = NO_OFFSET;
      sos_Offset next = get_first(_OBSThis);
      for (sos_Int i = 1;(i<pos) AND (next != NO_OFFSET); i++)
      {  
         offset = next;
         read_prev_next (offset, cnt, prev, next);
      }
      create_node (_OBSThis,o, offset, next);
   }
   else
   {
      sos_Offset next   = NO_OFFSET;
      sos_Offset prev   = get_last(_OBSThis);
      for (sos_Int i = crd;(i>=pos) AND (prev != NO_OFFSET); i--)
      {
         offset = prev;
         read_prev_next (offset, cnt, prev, next);
      }
      create_node (_OBSThis,o, prev, offset);
   }

   TT(agg_H,T_LEAVE)   
}

// ************************************************************************** 
void _sos_Object_List::remove (const sos_Typed_id&_OBSThis,sos_Int pos)
// ************************************************************************** 
const{  T_PROC("List::remove");
   TT(agg_H, T_ENTER; TI(pos))

   sos_Int crd = get_cardinality(_OBSThis);

   err_assert ((0  <  pos) AND (pos  <=  crd),
               "List::remove pos invalid");

   sos_Container cnt = _OBSThis.container();
   sos_Offset offset = NO_OFFSET;

   if (pos <=crd/2)
   {
      sos_Offset prev = NO_OFFSET;
      sos_Offset next = get_first(_OBSThis);
 
      for (sos_Int i = 0;(i<pos); i++)
      {
         offset = next;
         read_prev_next (offset, cnt, prev, next);
      }
      remove_node (_OBSThis,offset, prev, next);
   }
   else
   {  sos_Offset prev = get_last(_OBSThis);
      sos_Offset next = NO_OFFSET;
  
      for (sos_Int i = crd;(i>=pos); i--)
      {  
         offset = prev;
         read_prev_next (offset, cnt, prev, next);
      }
      remove_node (_OBSThis,offset, prev, next);
   }
 
   TT(agg_H,T_LEAVE)
}

// ************************************************************************** 
sos_Object _sos_Object_List::get_nth(const sos_Typed_id&_OBSThis,sos_Int pos)
// ************************************************************************** 
const{  T_PROC("List::get_nth")
   TT(agg_H, T_ENTER; TI(pos))

   err_assert ((0 < pos) AND (pos <= get_cardinality(_OBSThis)),
	       "List::get_nth position invalid");

   sos_Offset prev   = NO_OFFSET;
   sos_Offset next   = get_first(_OBSThis);;
   sos_Offset offset = NO_OFFSET;
   sos_Object entry  = NO_OBJECT;

   sos_Container cnt = _OBSThis.container();

   for (sos_Int i = 0;(i<pos) AND (next != NO_OFFSET); i++)
   {  
      offset = next;
      read_prev_next (offset, cnt, prev, next);
   }
   entry = read_node_entry (offset, cnt,
                            get_default_type(_OBSThis), get_default_container(_OBSThis));
   TT(agg_H,T_LEAVE)
   return entry;
}

// ************************************************************************** 
void _sos_Object_List::set_nth (const sos_Typed_id&_OBSThis,Index pos,OBST_PARDECL( sos_Object) o)
// ************************************************************************** 
const{  T_PROC("List::set_nth")
   TT(agg_H, T_ENTER; TI(pos))

   err_assert ((0 < pos) AND (pos <= get_cardinality(_OBSThis)),
	       "List::set_nth position invalid");

   sos_Cursor c = open_cursor(_OBSThis);
   move_cursor (_OBSThis,c, pos);
   set (_OBSThis,c,o);
   close_cursor(_OBSThis,c);

   TT(agg_H,T_LEAVE)
}

// ************************************************************************** 
Index _sos_Object_List::find(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) o)
// ************************************************************************** 
const{  T_PROC("List::find(sos_Object)");
   TT(agg_H, T_ENTER; TOBJ(o));

   Index result = 0; // assume not found
   sos_Bool boe = get_based_on_equal(_OBSThis);
 
   sos_Cursor c = open_cursor(_OBSThis);
   for (sos_Bool valid = is_valid(_OBSThis,c); valid; valid = to_succ(_OBSThis,c))
   {
      sos_Object x = get(_OBSThis,c);
      if (agg_same_entity (o, x, boe, EQ_WEAK))
      {  result = current_pos(_OBSThis,c);
         break;
      }
   }
   close_cursor(_OBSThis,c);

   TT(agg_H,T_LEAVE)

   return result;
}


// ************************************************************************** 
sos_Object _sos_Object_List::find_or_append (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) o)
// ************************************************************************** 
const{  T_PROC ("List::find_or_append");
   TT( agg_H, T_ENTER; TOBJ(o));
   
   Index      i = find(_OBSThis,o);
   sos_Object result;
   if (i == 0) // not found, then append
   {  append(_OBSThis,o);
      result = o;
   }
   else
      result = get_nth(_OBSThis,i);
  

   TT(agg_H,T_LEAVE)
   return result;
}

// **************************************************************************
void _sos_Object_List::__plus_assign (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object_List) alist)
// **************************************************************************
const{
   T_PROC ("List::operator+=");
   TT (agg_H, T_ENTER; TOBJ(alist))
 
   agg_iterate (alist, sos_Object o)
      append (_OBSThis,o);
   agg_iterate_end(alist,o)
 
   TT (agg_H, T_LEAVE)
} // ** operator+= **
 

#define valid_list_cursor(node) (sos_Bool)(node.get_offset()!=NO_OFFSET)

// LOCAL inline sos_Bool valid_list_cursor (sos_List_node node)
// {  return (sos_Bool)(node.get_offset() != NO_OFFSET); }

// **************************************************************************
sos_Bool _sos_Object_List::is_valid (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// **************************************************************************
const{
   sos_List_node node = sos_List_node::make (c.get_current());
   return valid_list_cursor (node);
}

// **************************************************************************
Index _sos_Object_List::current_pos (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// **************************************************************************
const{
   T_PROC ("List::current_pos");
   TT (agg_H, T_ENTER)
 
   Index result = 0;

   sos_List_node node = sos_List_node::make(c.get_current());
   sos_Offset offset  = node.get_offset();
 
   err_assert (valid_list_cursor(node) AND c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::current_pos cursor invalid");

   sos_Container cnt = _OBSThis.container();

   for (;offset != NO_OFFSET;result++)
   {  
      sos_Offset prev, next;
      read_prev_next (offset, cnt, prev, next);
      offset = prev;
   }

   TT (agg_H, T_LEAVE; TI (result))
   return result;
} 

// **************************************************************************
sos_Bool _sos_Object_List::move_cursor (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c, Index pos)
// **************************************************************************
const{
   T_PROC ("List::move_cursor")
   TT (agg_H, T_ENTER; TI (pos))

   sos_Int  crd = get_cardinality(_OBSThis);
   sos_Bool result = FALSE;

   err_assert ((0 < pos) AND (pos <= crd),
	       "List::move_cursor position invalid");

   if (pos <= crd/2)
   {  result = to_first (_OBSThis,c);
      if (pos > 1)
         result = to_succ (_OBSThis,c,pos-1);
   }
   else
   {  result = to_last(_OBSThis,c);
      if (pos < crd)
	 result = to_pred (_OBSThis,c, crd-pos);
   }

   TT (agg_H, T_LEAVE; TB(result))
   return result;
}
 
 
// ************************************************************************** 
void _sos_Object_List::insert_before (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c,OBST_PARDECL( sos_Object) o)
// ************************************************************************** 
const{  T_PROC("List::insert_before");
   TT(agg_H, T_ENTER; TOBJ(c))

   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::insert_before cursor invalid");

   sos_Offset    offset = node.get_offset();
   sos_Offset prev, next;
   read_prev_next (offset, _OBSThis.container(), prev, next);
   node.set_prev (create_node (_OBSThis,o, prev, offset));
   node.set_entry (o);

   TT(agg_H, T_LEAVE)
}

// ************************************************************************** 
void _sos_Object_List::insert_after (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c,OBST_PARDECL( sos_Object) o)
// ************************************************************************** 
const{  T_PROC("List::insert_after")
   TT(agg_H, T_ENTER; TOBJ(c))
 
   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::insert_after cursor invalid");

   sos_Offset offset = node.get_offset();
   sos_Offset prev, next;
   read_prev_next (offset, _OBSThis.container(), prev, next);
   node.set_next  (create_node (_OBSThis,o, offset, next));
   node.set_entry (o);

   TT(agg_H, T_LEAVE)
}

// ************************************************************************** 
void _sos_Object_List::local_assign (OBST_PARDECL(sos_Object_List) x,OBST_PARDECL(
                                    sos_Object)      o)
// ************************************************************************** 
{  T_PROC("List::local_assign");

   sos_Object_List y = sos_Object_List::make (o);
   x.clear();
   x.operator+= (y);

   TT(agg_H,T_LEAVE) 
}

// ************************************************************************** 
sos_Bool _sos_Object_List::local_equal (OBST_PARDECL(sos_Object_List) x,OBST_PARDECL(
                                       sos_Object)      o,
                                       sos_Eq_kind     eq_kind) 
// ************************************************************************** 
{  sos_Bool result;
 
   sos_Object_List y = sos_Object_List::make (o);
   if (x.get_cardinality() != y.get_cardinality())
      result = FALSE;
   else
   {  result = TRUE;
      sos_Bool based_on_equal = x.get_based_on_equal();
      int  comp = 0;
      agg_iterate_double (x, sos_Object x_e, y, sos_Object y_e, comp)
      {  result = agg_same_entity (x_e, y_e, based_on_equal, eq_kind);
	 if (NOT result) break;
      }
      agg_iterate_double_end (x, x_e, y, y_e, comp);
      result = (sos_Bool)(result AND comp == 0);
   }
   
   return result;
} // ** List::local_equal **

// ************************************************************************** 
sos_Int _sos_Object_List::local_hash_value (OBST_PARDECL(sos_Object_List) x)
// ************************************************************************** 
{
   T_PROC ("List::local_hash_value");
   TT (agg_H, T_ENTER );
 
   sos_Int result;
 
   if (x.is_empty())
      result = 0;
   else
      result = x.get_nth (1).hash_value();
 
   TT (agg_H, T_LEAVE; TB(result));
 
   return result;
}

// **************************************************************************
sos_Object _sos_Object_List::get (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// **************************************************************************
const{
   T_PROC ("List::get (sos_Cursor)");
   TT (agg_H, T_ENTER)
 
   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::get(sos_Cursor) cursor invalid");

   sos_Object entry = NO_OBJECT;

   if (NOT node.get_list_cursor())
      entry = node.get_entry();
   else
   {  sos_Offset offset = node.get_offset();
      sos_Offset prev, next;
      entry = read_node_entry (offset, _OBSThis.container(),
			       get_default_type(_OBSThis), get_default_container(_OBSThis));
   }

   TT (agg_H, T_LEAVE; TOBJ(entry))
   return entry;
}

// **************************************************************************
void _sos_Object_List::set (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c,OBST_PARDECL( sos_Object) o)
// **************************************************************************
const{
   T_PROC ("List::set")
   TT (agg_H, T_ENTER; TOBJ(o))

   sos_List_node node = sos_List_node::make (c.get_current());
 
   err_assert (valid_list_cursor(node) AND c.defined_for(sos_Object_List::make(_OBSThis,this)),
	       "List::set cursor invalid");

   sos_Offset offset = node.get_offset();
   sos_Offset prev, next;

   read_prev_next (offset, _OBSThis.container(), prev, next);
   remove_node (_OBSThis,offset, prev, next);   // das ist ja schauderhaft !

   offset = create_node (_OBSThis,o, prev, next);

   node.set_offset (offset);
   node.set_prev   (prev);
   node.set_next   (next);
   node.set_entry  (o);
 
   TT (agg_H, T_LEAVE)
}

// ************************************************************************** 
void _sos_Object_List::remove_at(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// ************************************************************************** 
const{  T_PROC("List::remove_at")
   TT(agg_H, T_ENTER)

   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::remove_at cursor invalid");

   sos_Container cnt  = _OBSThis.container();
   sos_Offset  offset = node.get_offset();
   sos_Offset  next,prev;

   read_prev_next (offset, cnt, prev, next);
   remove_node (_OBSThis,offset,prev, next);

   if (next == NO_OFFSET)
      node.set_offset (NO_OFFSET);
   else
   {  offset = next;
      read_prev_next (offset, cnt, prev, next);
      sos_Object entry = read_node_entry (offset, cnt, get_default_type(_OBSThis),
                                          get_default_container(_OBSThis));
      node.set_offset (offset);
      node.set_prev (prev);
      node.set_next (next);
      node.set_entry (entry);
   }

   TT(agg_H, T_LEAVE)
} 
 

// ************************************************************************** 
sos_Int _sos_Object_List::size(const sos_Typed_id&_OBSThis)
// ************************************************************************** 
const{  return get_size(_OBSThis) +_sos_Object::size(_OBSThis);
}

// **************************************************************************
void _sos_Object_List::clear(const sos_Typed_id&_OBSThis)
// **************************************************************************
const{  T_PROC("List::clear")
   TT(agg_H, T_ENTER)
 
   for (;get_cardinality(_OBSThis) > 0;)
      remove (_OBSThis,1);
 
   TT(agg_H, T_LEAVE)
}

// **************************************************************************
sos_Int _sos_Object_List::card (const sos_Typed_id&_OBSThis)
// **************************************************************************
const{  T_PROC ("List::card")
   TT (agg_H, T_ENTER)

   sos_Int crd = get_cardinality(_OBSThis);

   TT (agg_H, T_LEAVE; TI(crd));
   return crd;
}

// ************************************************************************** 
sos_Cursor _sos_Object_List::open_cursor(const sos_Typed_id&_OBSThis,sos_Container cnt)
// ************************************************************************** 
const{  T_PROC ("List::open_cursor")
   TT( agg_H, T_ENTER)

   sos_Cursor c = sos_Cursor::create (cnt, sos_Object_List::make(_OBSThis,this));
   sos_List_node node = sos_List_node::create (cnt, get_list_cursor(_OBSThis));
   c.set_current (node);
   to_first (_OBSThis,c);

   TT(agg_H, T_LEAVE)
   return c;
}

// ************************************************************************** 
void _sos_Object_List::close_cursor(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// ************************************************************************** 
const{  T_PROC ("List::close_cursor")
   TT( agg_H, T_ENTER; TOBJ(c))

   err_assert (c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::close_cursor cursor not legal for this List");

   c.get_current().destroy();
   c.destroy();

   TT(agg_H, T_LEAVE)
}

// ************************************************************************** 
sos_Cursor _sos_Object_List::duplicate(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c, 
				      sos_Container cnt /*= TEMP_CONTAINER */)
// ************************************************************************** 
const{  T_PROC ("List::duplicate")
   TT( agg_H, T_ENTER; TOBJ(c))

   err_assert (c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::duplicate cursor not legal for this List");

   sos_Cursor new_c = open_cursor (_OBSThis,cnt);
   new_c.get_current().assign (c.get_current());

   TT(agg_H, T_LEAVE)
   return new_c;
}


// ************************************************************************** 
sos_Bool _sos_Object_List::to_first(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// ************************************************************************** 
const{
   err_assert (c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::to_first cursor not legal for this List");

   sos_Object entry;
   sos_Offset offset = get_first(_OBSThis);
   sos_Offset prev, next;
   sos_Container cnt = _OBSThis.container();

   if (offset != NO_OFFSET)
   {
      read_prev_next (offset, cnt, prev, next);
      entry = read_node_entry (offset, cnt,
                               get_default_type(_OBSThis),
                               get_default_container(_OBSThis));
   }
   sos_List_node node = sos_List_node::make (c.get_current());

   node.set_offset (offset);
   node.set_prev (prev);
   node.set_next (next);
   node.set_entry (entry);

   return valid_list_cursor (node);
}

// ************************************************************************** 
sos_Bool _sos_Object_List::to_last(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// ************************************************************************** 
const{
   err_assert (c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::to_last cursor not legal for this List");

   sos_Object entry;
   sos_Offset offset = get_last(_OBSThis);
   sos_Offset prev, next;
   sos_Container cnt = _OBSThis.container();

   if (offset != NO_OFFSET)
   {
      read_prev_next (offset, cnt, prev, next);
      entry = read_node_entry (offset, cnt,
			       get_default_type(_OBSThis),
                               get_default_container(_OBSThis));
   }

   sos_List_node node = sos_List_node::make (c.get_current());

   node.set_offset (offset);
   node.set_prev (prev);
   node.set_next (next);
   node.set_entry (entry);

   return valid_list_cursor (node);
}

// ************************************************************************** 
sos_Bool _sos_Object_List::to_succ (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c, Index steps) 
// ************************************************************************** 
const{  T_PROC ("List::to_succ")
   TT( agg_H, T_ENTER; TI(steps))

   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::to_succ cursor invalid");

   if (steps>0)
   {
      sos_Container cnt	= _OBSThis.container();
      sos_Object entry  = NO_OBJECT;
      sos_Offset offset = NO_OFFSET;
      sos_Offset prev, next;

      if (node.get_list_cursor())
      {  next = node.get_offset();
	 steps++;
      }
      else
         next = node.get_next();
   
      for (; (steps > 0) AND (next != NO_OFFSET); steps--)
      {
         offset = next;
         read_prev_next (offset, cnt, prev, next);
      }

      if (offset != NO_OFFSET)
      {
         entry = read_node_entry (offset, cnt,
				  get_default_type(_OBSThis),
				  get_default_container(_OBSThis));
         if (steps != 0)
	    offset = NO_OFFSET;
      }
   
      node.set_offset (offset);
      node.set_next   (next);
      node.set_prev   (prev);
      node.set_entry  (entry);
   }
   else
      if (steps < 0)
	 to_pred (_OBSThis,c, -steps);

   TT(agg_H, T_LEAVE; TB(valid_list_cursor (node)))
   return valid_list_cursor (node);
}

// ************************************************************************** 
sos_Bool _sos_Object_List::to_pred (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c, Index steps)
// ************************************************************************** 
const{  T_PROC ("List::to_pred")
   TT( agg_H, T_ENTER; TI(steps))

   sos_List_node node = sos_List_node::make (c.get_current());

   err_assert (valid_list_cursor(node) AND c.defined_for(sos_Object_List::make(_OBSThis,this)),
               "List::to_pred cursor invalid");

   if (steps > 0)
   { 
      sos_Container cnt	= _OBSThis.container();
      sos_Object entry  = NO_OBJECT;
      sos_Offset offset = NO_OFFSET;
      sos_Offset prev, next;

      if (node.get_list_cursor())
      {  prev = node.get_offset();
	 steps++;
      }
      else
         prev = node.get_prev();
   
      for (; (steps > 0) AND (prev != NO_OFFSET); steps--)
      {  
         offset = prev;
         read_prev_next (offset, cnt, prev, next);
      }

      if (offset != NO_OFFSET)
      {
         entry = read_node_entry (offset, cnt,
				  get_default_type(_OBSThis),
				  get_default_container(_OBSThis));
         if (steps != 0)
	    offset = NO_OFFSET;
      }

      node.set_offset (offset);
      node.set_next   (next);
      node.set_prev   (prev);
      node.set_entry  (entry);
  }
  else 
     if (steps<0)
        to_succ (_OBSThis,c, -steps);

   TT(agg_H, T_LEAVE; TB(valid_list_cursor (node)))
   return valid_list_cursor (node);
}
