;;; --------------------------------------------------------------------------
;;; MH support
;;; Adjusted for Epoch 4.0
;;; 
(require 'mh-e)
(mh-find-path)				;set up MH paths

;; Force this variable to be bound to something, in case someone else uses
;; this code. This var is used in functions in amc.el.
(defvar inhibit-rename-screen nil)

(define-key mh-letter-mode-map "\C-c\C-a"
  (definteractive
    (save-excursion
      (goto-char (point-max))
      (if (/= (current-column) 0) (insert "\n"))
      (insert "____\n")
      (mh-insert-signature)
)))

(defvar mh-screen nil)
(defun mh-set-up () (interactive)
  (if (and (screenp mh-screen) (get-screen-id mh-screen))
    (mapraised-screen mh-screen)
    (setq mh-screen
      (create-screen "*scratch*"
	'((geometry . "80x45-0-160") (icon-name . "MH-Mailer")
	  (title . "MH-Mailer") (font . "ncenr10")
  ))))
  (or
    (memq mh-screen inhibit-rename-screen)
    (setq inhibit-rename-screen (cons mh-screen inhibit-rename-screen))
  )
  (on-map-do mh-screen 'mh-screen-mapper)
)

(defun mh-screen-mapper (s)
  (on-map-do s 'mh-screen-mapper)
  (select-screen s)
  (mh-inc-folder)
)

(defun mh-motion-style ()
  (make-variable-buffer-local 'motion::style)
  (setq motion::style (make-style))
  (let ((s motion::style))
    (set-style-underline s "red")
  )
  (make-variable-buffer-local 'mode:*hilight-style*)
  (setq mode:*hilight-style* mh-mode-hilight-style)
)

;; Style to use for headers of mail letters.  Display in red if color, else
;; using reverse video for mono systems
(defvar mh-header-style
  (let ((s (make-style)))
    (when (< 2 (number-of-colors))
      (set-style-foreground s "red")
      (set-style-background s (background))
)))

(defvar mh-mode-hilight-style
  (let ((s (make-style)))
    (set-style-background s (foreground))
    (set-style-font s "ncenb12")
    (set-style-foreground s "blue")
))

(defvar mh-mode-headers
  (list
    (cons "^To:" mh-header-style)
    (cons "^From:" mh-header-style)
    (cons "^[cC]c:" mh-header-style)
    (cons "^Subject:" mh-header-style)
    (cons "^Date:" mh-header-style)
    (cons "^Return-Path:" mh-header-style)
    (cons "^Newsgroups:" mh-header-style)
    (cons "^Reply-To:" mh-header-style)
  )
  "*A list of (re . style) pairs for marking headers in the MH buffer"
)

;; Add an RE to auto-mode-list corresponding to files for mh mode
(setq mh-letter-mode-hook 'do-mh-buttons) ; For drafts
(let ((re (concat mh-user-path "[A-Z---a-z]+/[0-9]+$")))
  (setq auto-mode-alist
    (cons
      (cons re 'mh-mode)
      (delete-if
	(eval (` '(lambda (x) (equal (, re) (car x)))))
	auto-mode-alist
))))

(defun do-mh-buttons ()
  (dolist (h mh-mode-headers)
    (let
      (
	(re (car h))			        ;the RE to find
	(style (or (cdr h) mh-header-style))	;the style to use
      )
      (goto-char (point-min))
      (while (re-search-forward re nil t)
	(end-of-line)
	(add-button (match-end 0) (point) style)
  )))
  (auto-fill-mode 1)
  (goto-char (point-max))
)

(defun mh-mode()
  "Switch into mh-mode"
  (setq major-mode 'mh-mode)
  (setq mode-name "MH")
  (do-mh-buttons)
  (run-hooks 'mh-mode-hooks)
)

;; Redefinition of this fcn to add calls to do buttons and set motion style.
(defun mh-show (&optional msg)
  "Show MESSAGE (default: displayed message).
Forces a two-window display with the folder window on top (size
mh-summary-height) and the show buffer below it."
  (interactive)
  (if (not msg)
      (setq msg (mh-get-msg-num t)))
  (setq mh-showing t)
  (mh-set-mode-name "mh-e show")
  (if (not (eql (next-window (minibuffer-window)) (selected-window)))
      (delete-other-windows))		; force ourself to the top window
  (let ((folder mh-current-folder))
    (mh-show-message-in-other-window)
    (mh-display-msg msg folder))
  (do-mh-buttons)
  (mh-motion-style)
  (other-window -1)
  (shrink-window (- (window-height) mh-summary-height))
  (mh-recenter nil)
  (if (not (memq msg mh-seen-list)) (mh-push msg mh-seen-list)))

(defun mh-make-folder-mode-line (&optional annotation)
  ;; Set the fields of the mode line for a folder buffer.
  ;; The optional ANNOTATION string is displayed after the folder's name.
  (make-variable-buffer-local 'mode:*hilight-style*)
  (setq mode:*hilight-style* mh-mode-hilight-style)
  (save-excursion
    (mh-first-msg)
    (setq mh-first-msg-num (mh-get-msg-num nil))
    (mh-last-msg)
    (setq mh-last-msg-num (mh-get-msg-num nil))
    (let ((lines (count-lines (point-min) (point-max))))
      (setq mode-line-buffer-identification
            (list (format "{%%b%s}  %d msg%s"
                          (if annotation (format "/%s" annotation) "")
                          lines
                          (if (zerop lines)
                              "s"
                              (if (> lines 1)
                                  (format "s (%d-%d)" mh-first-msg-num
                                          mh-last-msg-num)
                                  (format " (%d)" mh-first-msg-num)))))))))

(defun mh-scan-folder (folder range)
  ;; Scan the FOLDER over the RANGE.  Return in the folder's buffer.
  (cond ((null (get-buffer folder))
         (mh-make-folder folder))
        (t
         (mh-process-or-undo-commands folder)
         (switch-to-buffer folder)))
  (make-variable-buffer-local 'mode:*hilight-style*)
  (setq mode:*hilight-style* mh-mode-hilight-style)    
  (mh-regenerate-headers range)
  (mh-when (zerop (buffer-size))
    (if (equal range "all")
        (message "Folder %s is empty" folder)
        (message "No messages in %s, range %s" folder range))
    (sit-for 5))
  (mh-goto-cur-msg))

(defun mh-inc-folder (&optional maildrop-name)
  "Inc(orporate) new mail into +inbox.
Optional prefix argument specifies an alternate maildrop from the default.
If this is given, mail is incorporated into the current folder, rather
than +inbox.  Run mh-inc-folder-hook after incorporating new mail."
  (interactive (list (if current-prefix-arg
                         (expand-file-name
                          (read-file-name "inc mail from file: "
                                          mh-user-path)))))
  (let ((config (current-window-configuration)))
    (if (not maildrop-name)
        (cond ((not (get-buffer "+inbox"))
               (mh-make-folder "+inbox")
               (setq mh-previous-window-config config))
              ((not (eq (current-buffer) (get-buffer "+inbox")))
               (switch-to-buffer "+inbox")
               (setq mh-previous-window-config config)))))
  (make-variable-buffer-local 'mode:*hilight-style*)
  (setq mode:*hilight-style* mh-mode-hilight-style)  
  (mh-get-new-mail maildrop-name)
  (run-hooks 'mh-inc-folder-hook))

