#! /bin/sh

TEMPDIR=/tmp/pspp-tst-$$
mkdir -p $TEMPDIR
trap 'cd /; rm -rf $TEMPDIR' 0

# ensure that top_builddir  are absolute
if [ -z "$top_builddir" ] ; then top_builddir=. ; fi
if [ -z "$top_srcdir" ] ; then top_srcdir=. ; fi
top_builddir=`cd $top_builddir; pwd`
PSPP=$top_builddir/src/ui/terminal/pspp

# ensure that top_srcdir is absolute
top_srcdir=`cd $top_srcdir; pwd`

STAT_CONFIG_PATH=$top_srcdir/config
export STAT_CONFIG_PATH

fail()
{
    echo $activity
    echo FAILED
    exit 1;
}


no_result()
{
    echo $activity
    echo NO RESULT;
    exit 2;
}

pass()
{
    exit 0;
}

cd $TEMPDIR

activity="write program to generate PSPP syntax and data"
cat > date-in.pl <<'EOF'
#! /usr/bin/perl

use strict;
use warnings;

our $next = 1;

my @formats = (['date', 'd-m-y'],
	       ['adate', 'm-d-y'],
	       ['edate', 'd-m-y'],
	       ['jdate', 'j'],
	       ['sdate', 'y-m-d'],
	       ['qyr', 'qQy'],
	       ['moyr', 'm-y'],
	       ['wkyr', 'wWy'],
	       ['datetime', 'd-m-y +H:M', 'd-m-y +H:M:S']);

my @dates = (#yyyy  mm  dd  jjj  HH  MM  SS
	     [1648,  6, 10, 162,  0,  0,  0],
	     [1680,  6, 30, 182,  4, 50, 38],
	     [1716,  7, 24, 206, 12, 31, 35],
	     [1768,  6, 19, 171, 12, 47, 53],
	     [1819,  8,  2, 214,  1, 26,  0],
	     [1839,  3, 27,  86, 20, 58, 11],
	     [1903,  4, 19, 109,  7, 36,  5],
	     [1929,  8, 25, 237, 15, 43, 49],
	     [1941,  9, 29, 272,  4, 25,  9],
	     [1943,  4, 19, 109,  6, 49, 27],
	     [1943, 10,  7, 280,  2, 57, 52],
	     [1992,  3, 17,  77, 16, 45, 44],
	     [1996,  2, 25,  56, 21, 30, 57],
	     [1941,  9, 29, 272,  4, 25,  9],
	     [1943,  4, 19, 109,  6, 49, 27],
	     [1943, 10,  7, 280,  2, 57, 52],
	     [1992,  3, 17,  77, 16, 45, 44],
	     [1996,  2, 25,  56, 21, 30, 57],
	     [2038, 11, 10, 314, 22, 30,  4],
	     [2094,  7, 18, 199,  1, 56, 51]);

open (SYNTAX, '>', 'date-in.pspp') or die "date-in.pspp: create: $!\n";
print SYNTAX "SET EPOCH 1930.\n";
for my $format (@formats) {
    my ($name) = @$format;
    print SYNTAX "DATA LIST file='$name.data'/$name 1-40 ($name).\n";
    print SYNTAX "PRINT OUTFILE='$name.out'/$name (F16.2).\n";
    print SYNTAX "EXECUTE.\n";
}
close (SYNTAX);

for my $format (@formats) {
    my ($fmt_name, @templates) = @$format;
    my ($fn) = "$fmt_name.data";
    open (DATA, '>', $fn) or die "$fn: create: $!\n";
    select DATA;
    for my $template (@templates) {
	for my $date (@dates) {
	    print_date_with_template ($date, $template) for 1...10;
	}
    }
    close (DATA);
}

sub print_date_with_template {
    my ($date, $template) = @_;
    my ($year, $month, $day, $julian, $hour, $minute, $second) = @$date;
    my ($quarter) = int (($month - 1) / 3) + 1;
    my ($week) = int (($julian - 1) / 7) + 1;
    my (@year_types) = ('full');
    push (@year_types, '2digit') if $year >= 1930 && $year < 2030;
    for my $c (split ('', $template)) {
	if ($c eq 'd') {
	    printf (+pick ('%d', '%02d'), $day);
	} elsif ($c eq 'm') {
	    my ($type) = pick ('arabic', 'roman', 'abbrev', 'full');
	    if ($type eq 'arabic') {
		printf (+pick ('%d', '%02d'), $month);
	    } elsif ($type eq 'roman') {
		my ($mmm) = ('i', 'ii', 'iii',
			     'iv', 'v', 'vi',
			     'vii', 'viii', 'ix',
			     'x', 'xi', 'xii')[$month - 1];
		print_rand_case ($mmm);
	    } elsif ($type eq 'abbrev') {
		my ($mmm) = qw (jan feb mar apr may jun
                                jul aug sep oct nov dec)[$month - 1];
		print_rand_case ($mmm);
	    } elsif ($type eq 'full') {
		my ($mmm) = qw (january february march
				april may june
				july august september
				october november december)[$month - 1];
		print_rand_case ($mmm);
	    } else {
		die;
	    }
	} elsif ($c eq 'y') {
	    my ($type) = pick (@year_types);
	    if ($type eq '2digit') {
		printf (+pick ('%d', '%02d'), $year % 100);
	    } elsif ($type eq 'full') {
		print $year;
	    } else {
		die;
	    }
	} elsif ($c eq 'j') {
	    my ($type) = pick (@year_types);
	    if ($type eq '2digit') {
		printf ("%02d%03d", $year % 100, $julian);
	    } elsif ($type eq 'full') {
		printf ("%04d%03d", $year, $julian);
	    } else {
		die;
	    }
	} elsif ($c eq 'q') {
	    print $quarter;
	} elsif ($c eq 'w') {
	    print $week;
	} elsif ($c eq 'H') {
	    printf (+pick ('%d', '%02d'), $hour);
	} elsif ($c eq 'M') {
	    printf (+pick ('%d', '%02d'), $minute);
	} elsif ($c eq 'S') {
	    printf (+pick ('%d', '%02d'), $second);
	} elsif ($c eq '-') {
	    print +pick (' ', '-', '.', ',', '/');
	} elsif ($c eq ':') {
	    print +pick (' ', ':');
	} elsif ($c eq ' ') {
	    print ' ';
	} elsif ($c eq 'Q') {
	    maybe_print_space ();
	    print_rand_case ('q');
	    maybe_print_space ();
	} elsif ($c eq 'W') {
	    maybe_print_space ();
	    print_rand_case ('wk');
	    maybe_print_space ();
	} elsif ($c eq '+') {
	    print +pick ('', '-', '+');
	} else {
	    die;
	}
    }
    print "\n";
}

sub print_rand_case {
    my ($s) = @_;
    my ($case) = pick (qw (uc lc tc));
    if ($case eq 'uc') {
	print uc ($s);
    } elsif ($case eq 'lc') {
	print lc ($s);
    } elsif ($case eq 'tc') {
	print ucfirst ($s);
    } else {
	die;
    }
}

sub maybe_print_space {
   print +pick ('', ' ');
}

sub pick {
   return $_[int (my_rand ($#_ + 1))];
}

sub my_rand {
    my ($modulo) = @_;
    $next = ($next * 1103515245 + 12345) % (2**32);
    return int ($next / 65536) % $modulo;
}
EOF
if [ $? -ne 0 ] ; then no_result ; fi

activity="generate PSPP syntax and data"
$PERL date-in.pl
if [ $? -ne 0 ] ; then no_result ; fi

activity="run program"
$SUPERVISOR $PSPP --testing-mode date-in.pspp
if [ $? -ne 0 ] ; then no_result ; fi

activity="compare adate.out output"
diff -u adate.out - <<EOF
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
EOF
if [ $? -ne 0 ] ; then fail ; fi

activity="compare date.out output"
diff -u date.out - <<EOF
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
EOF
if [ $? -ne 0 ] ; then fail ; fi

activity="compare datetime.out output"
diff -u datetime.out - <<EOF
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    3083547000.00
    3083547000.00
    3083547000.00
    3083512200.00
    3083512200.00
    3083547000.00
    3083547000.00
    3083547000.00
    3083547000.00
    3083512200.00
    4221545340.00
    4221635460.00
    4221635460.00
    4221635460.00
    4221635460.00
    4221635460.00
    4221635460.00
    4221635460.00
    4221545340.00
    4221545340.00
    5859607620.00
    5859607620.00
    5859515580.00
    5859607620.00
    5859515580.00
    5859607620.00
    5859607620.00
    5859607620.00
    5859607620.00
    5859607620.00
    7472644440.00
    7472654760.00
    7472644440.00
    7472654760.00
    7472654760.00
    7472654760.00
    7472654760.00
    7472654760.00
    7472644440.00
    7472654760.00
    8092666920.00
    8092817880.00
    8092817880.00
    8092817880.00
    8092817880.00
    8092817880.00
    8092666920.00
    8092817880.00
    8092666920.00
    8092817880.00
   10114356960.00
   10114356960.00
   10114302240.00
   10114302240.00
   10114356960.00
   10114356960.00
   10114356960.00
   10114356960.00
   10114302240.00
   10114356960.00
   10945986180.00
   10945986180.00
   10945986180.00
   10945986180.00
   10945986180.00
   10945986180.00
   10945986180.00
   10945986180.00
   10945873020.00
   10945873020.00
   11327628900.00
   11327628900.00
   11327660700.00
   11327660700.00
   11327660700.00
   11327628900.00
   11327660700.00
   11327660700.00
   11327660700.00
   11327660700.00
   11376658140.00
   11376609060.00
   11376658140.00
   11376609060.00
   11376658140.00
   11376658140.00
   11376609060.00
   11376609060.00
   11376658140.00
   11376658140.00
   11391397380.00
   11391418620.00
   11391397380.00
   11391397380.00
   11391418620.00
   11391397380.00
   11391418620.00
   11391418620.00
   11391397380.00
   11391397380.00
   12920229900.00
   12920109300.00
   12920109300.00
   12920229900.00
   12920229900.00
   12920109300.00
   12920229900.00
   12920229900.00
   12920109300.00
   12920229900.00
   13044663000.00
   13044508200.00
   13044663000.00
   13044663000.00
   13044663000.00
   13044508200.00
   13044508200.00
   13044663000.00
   13044663000.00
   13044508200.00
   11327660700.00
   11327660700.00
   11327660700.00
   11327660700.00
   11327660700.00
   11327628900.00
   11327628900.00
   11327660700.00
   11327628900.00
   11327660700.00
   11376609060.00
   11376658140.00
   11376658140.00
   11376658140.00
   11376609060.00
   11376609060.00
   11376658140.00
   11376658140.00
   11376658140.00
   11376658140.00
   11391418620.00
   11391418620.00
   11391397380.00
   11391418620.00
   11391418620.00
   11391418620.00
   11391418620.00
   11391418620.00
   11391418620.00
   11391397380.00
   12920109300.00
   12920229900.00
   12920229900.00
   12920109300.00
   12920109300.00
   12920229900.00
   12920109300.00
   12920229900.00
   12920229900.00
   12920109300.00
   13044508200.00
   13044663000.00
   13044663000.00
   13044663000.00
   13044663000.00
   13044508200.00
   13044508200.00
   13044663000.00
   13044508200.00
   13044508200.00
   14392420200.00
   14392420200.00
   14392420200.00
   14392420200.00
   14392258200.00
   14392420200.00
   14392258200.00
   14392258200.00
   14392258200.00
   14392420200.00
   16149635760.00
   16149621840.00
   16149635760.00
   16149635760.00
   16149635760.00
   16149635760.00
   16149635760.00
   16149635760.00
   16149635760.00
   16149621840.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    3083547038.00
    3083547038.00
    3083512162.00
    3083512162.00
    3083547038.00
    3083512162.00
    3083547038.00
    3083547038.00
    3083547038.00
    3083547038.00
    4221545305.00
    4221635495.00
    4221635495.00
    4221545305.00
    4221545305.00
    4221635495.00
    4221545305.00
    4221635495.00
    4221635495.00
    4221635495.00
    5859515527.00
    5859515527.00
    5859607673.00
    5859607673.00
    5859515527.00
    5859607673.00
    5859515527.00
    5859607673.00
    5859607673.00
    5859607673.00
    7472644440.00
    7472644440.00
    7472654760.00
    7472654760.00
    7472654760.00
    7472654760.00
    7472654760.00
    7472644440.00
    7472654760.00
    7472654760.00
    8092817891.00
    8092817891.00
    8092666909.00
    8092817891.00
    8092666909.00
    8092817891.00
    8092666909.00
    8092666909.00
    8092666909.00
    8092666909.00
   10114302235.00
   10114356965.00
   10114356965.00
   10114302235.00
   10114302235.00
   10114356965.00
   10114356965.00
   10114302235.00
   10114302235.00
   10114356965.00
   10945986229.00
   10945986229.00
   10945986229.00
   10945986229.00
   10945986229.00
   10945986229.00
   10945986229.00
   10945986229.00
   10945872971.00
   10945872971.00
   11327628891.00
   11327660709.00
   11327660709.00
   11327628891.00
   11327660709.00
   11327628891.00
   11327628891.00
   11327660709.00
   11327628891.00
   11327628891.00
   11376658167.00
   11376658167.00
   11376658167.00
   11376658167.00
   11376658167.00
   11376658167.00
   11376609033.00
   11376658167.00
   11376658167.00
   11376658167.00
   11391397328.00
   11391397328.00
   11391418672.00
   11391418672.00
   11391418672.00
   11391397328.00
   11391418672.00
   11391397328.00
   11391418672.00
   11391397328.00
   12920229944.00
   12920229944.00
   12920109256.00
   12920109256.00
   12920229944.00
   12920229944.00
   12920229944.00
   12920109256.00
   12920229944.00
   12920229944.00
   13044508143.00
   13044663057.00
   13044663057.00
   13044508143.00
   13044663057.00
   13044663057.00
   13044663057.00
   13044508143.00
   13044508143.00
   13044508143.00
   11327660709.00
   11327660709.00
   11327660709.00
   11327660709.00
   11327628891.00
   11327628891.00
   11327660709.00
   11327660709.00
   11327660709.00
   11327660709.00
   11376609033.00
   11376609033.00
   11376658167.00
   11376658167.00
   11376658167.00
   11376609033.00
   11376658167.00
   11376658167.00
   11376609033.00
   11376609033.00
   11391418672.00
   11391397328.00
   11391397328.00
   11391397328.00
   11391418672.00
   11391397328.00
   11391397328.00
   11391397328.00
   11391418672.00
   11391418672.00
   12920109256.00
   12920229944.00
   12920229944.00
   12920109256.00
   12920109256.00
   12920229944.00
   12920229944.00
   12920109256.00
   12920109256.00
   12920109256.00
   13044508143.00
   13044508143.00
   13044663057.00
   13044663057.00
   13044663057.00
   13044663057.00
   13044663057.00
   13044663057.00
   13044508143.00
   13044508143.00
   14392420204.00
   14392258196.00
   14392258196.00
   14392420204.00
   14392420204.00
   14392258196.00
   14392420204.00
   14392420204.00
   14392420204.00
   14392258196.00
   16149635811.00
   16149635811.00
   16149635811.00
   16149621789.00
   16149621789.00
   16149635811.00
   16149635811.00
   16149635811.00
   16149621789.00
   16149635811.00
EOF
if [ $? -ne 0 ] ; then fail ; fi

activity="compare edate.out output"
diff -u edate.out - <<EOF
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
EOF
if [ $? -ne 0 ] ; then fail ; fi

activity="compare jdate.out output"
diff -u jdate.out - <<EOF
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
EOF
if [ $? -ne 0 ] ; then fail ; fi

activity="compare moyr.out output"
diff -u moyr.out - <<EOF
    2071180800.00
    2071180800.00
    2071180800.00
    2071180800.00
    2071180800.00
    2071180800.00
    2071180800.00
    2071180800.00
    2071180800.00
    2071180800.00
    3081024000.00
    3081024000.00
    3081024000.00
    3081024000.00
    3081024000.00
    3081024000.00
    3081024000.00
    3081024000.00
    3081024000.00
    3081024000.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    5858006400.00
    5858006400.00
    5858006400.00
    5858006400.00
    5858006400.00
    5858006400.00
    5858006400.00
    5858006400.00
    5858006400.00
    5858006400.00
    7472563200.00
    7472563200.00
    7472563200.00
    7472563200.00
    7472563200.00
    7472563200.00
    7472563200.00
    7472563200.00
    7472563200.00
    7472563200.00
    8090496000.00
    8090496000.00
    8090496000.00
    8090496000.00
    8090496000.00
    8090496000.00
    8090496000.00
    8090496000.00
    8090496000.00
    8090496000.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10943856000.00
   10943856000.00
   10943856000.00
   10943856000.00
   10943856000.00
   10943856000.00
   10943856000.00
   10943856000.00
   10943856000.00
   10943856000.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11325225600.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   12918787200.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   13042512000.00
   14391561600.00
   14391561600.00
   14391561600.00
   14391561600.00
   14391561600.00
   14391561600.00
   14391561600.00
   14391561600.00
   14391561600.00
   14391561600.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
EOF
if [ $? -ne 0 ] ; then fail ; fi

activity="compare qyr.out output"
diff -u qyr.out - <<EOF
    2065910400.00
    2065910400.00
    2065910400.00
    2065910400.00
    2065910400.00
    2065910400.00
    2065910400.00
    2065910400.00
    2065910400.00
    2065910400.00
    3075753600.00
    3075753600.00
    3075753600.00
    3075753600.00
    3075753600.00
    3075753600.00
    3075753600.00
    3075753600.00
    3075753600.00
    3075753600.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    4219603200.00
    5852736000.00
    5852736000.00
    5852736000.00
    5852736000.00
    5852736000.00
    5852736000.00
    5852736000.00
    5852736000.00
    5852736000.00
    5852736000.00
    7469884800.00
    7469884800.00
    7469884800.00
    7469884800.00
    7469884800.00
    7469884800.00
    7469884800.00
    7469884800.00
    7469884800.00
    7469884800.00
    8085398400.00
    8085398400.00
    8085398400.00
    8085398400.00
    8085398400.00
    8085398400.00
    8085398400.00
    8085398400.00
    8085398400.00
    8085398400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10112774400.00
   10941177600.00
   10941177600.00
   10941177600.00
   10941177600.00
   10941177600.00
   10941177600.00
   10941177600.00
   10941177600.00
   10941177600.00
   10941177600.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11319868800.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11375078400.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   12913603200.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   13039833600.00
   14388883200.00
   14388883200.00
   14388883200.00
   14388883200.00
   14388883200.00
   14388883200.00
   14388883200.00
   14388883200.00
   14388883200.00
   14388883200.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
   16148160000.00
EOF
if [ $? -ne 0 ] ; then fail ; fi

activity="compare sdate.out output"
diff -u sdate.out - <<EOF
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    3083529600.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    4221590400.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    5859561600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    7472649600.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
    8092742400.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10114329600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   10945929600.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11327644800.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11376633600.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   11391408000.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   12920169600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   13044585600.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   14392339200.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
   16149628800.00
EOF
if [ $? -ne 0 ] ; then fail ; fi

activity="compare wkyr.out output"
diff -u wkyr.out - <<EOF
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    2071958400.00
    3083011200.00
    3083011200.00
    3083011200.00
    3083011200.00
    3083011200.00
    3083011200.00
    3083011200.00
    3083011200.00
    3083011200.00
    3083011200.00
    4221417600.00
    4221417600.00
    4221417600.00
    4221417600.00
    4221417600.00
    4221417600.00
    4221417600.00
    4221417600.00
    4221417600.00
    4221417600.00
    5859388800.00
    5859388800.00
    5859388800.00
    5859388800.00
    5859388800.00
    5859388800.00
    5859388800.00
    5859388800.00
    5859388800.00
    5859388800.00
    7472390400.00
    7472390400.00
    7472390400.00
    7472390400.00
    7472390400.00
    7472390400.00
    7472390400.00
    7472390400.00
    7472390400.00
    7472390400.00
    8092656000.00
    8092656000.00
    8092656000.00
    8092656000.00
    8092656000.00
    8092656000.00
    8092656000.00
    8092656000.00
    8092656000.00
    8092656000.00
   10114070400.00
   10114070400.00
   10114070400.00
   10114070400.00
   10114070400.00
   10114070400.00
   10114070400.00
   10114070400.00
   10114070400.00
   10114070400.00
   10945497600.00
   10945497600.00
   10945497600.00
   10945497600.00
   10945497600.00
   10945497600.00
   10945497600.00
   10945497600.00
   10945497600.00
   10945497600.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11327212800.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11376374400.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   11390889600.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   12919651200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   13044067200.00
   14391907200.00
   14391907200.00
   14391907200.00
   14391907200.00
   14391907200.00
   14391907200.00
   14391907200.00
   14391907200.00
   14391907200.00
   14391907200.00
   16149456000.00
   16149456000.00
   16149456000.00
   16149456000.00
   16149456000.00
   16149456000.00
   16149456000.00
   16149456000.00
   16149456000.00
   16149456000.00
EOF
if [ $? -ne 0 ] ; then fail ; fi

pass
