/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU Affero General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

   You should have received a copy of the GNU Affero General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file include/taler/taler-exchange/aml_legitimizations_get.h
 * @brief C interface for the GET /aml/$OFFICER_PUB/legitimizations endpoint
 * @author Christian Grothoff
 */
#ifndef _TALER_EXCHANGE__AML_LEGITIMIZATIONS_GET_H
#define _TALER_EXCHANGE__AML_LEGITIMIZATIONS_GET_H

#include <taler/taler-exchange/common.h>

/**
 * Possible options we can set for the GET legitimizations request.
 */
enum TALER_EXCHANGE_AmlLegitimizationsGetOption
{
  /**
   * End of list of options.
   */
  TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_END = 0,

  /**
   * Return at most N values, default is -20 to return
   * the last 20 entries before start. Negative values
   * to return before limit, positive to return after limit.
   */
  TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_LIMIT,

  /**
   * Row number threshold, defaults to INT64_MAX, namely
   * the biggest row id possible in the database.
   */
  TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_OFFSET,

  /**
   * Filter by account using a normalized payto URI hash.
   */
  TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_H_PAYTO,

  /**
   * If set to #TALER_EXCHANGE_YNA_YES, only return active
   * results, #TALER_EXCHANGE_YNA_NO, only return inactive
   * results, #TALER_EXCHANGE_YNA_ALL, to return all
   * decisions. Default is all.
   */
  TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_ACTIVE

};


/**
 * Possible options we can set for the GET legitimizations request.
 */
struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue
{

  /**
   * Type of the option being set.
   */
  enum TALER_EXCHANGE_AmlLegitimizationsGetOption option;

  /**
   * Specific option value.
   */
  union
  {

    /**
     * Value of if @e option is TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_LIMIT.
     */
    int64_t limit;

    /**
     * Value of if @e option is TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_OFFSET.
     * Note that in practice the maximum value is INT64_MAX, even though
     * this value is unsigned.
     */
    uint64_t offset;

    /**
     * Value of if @e option is TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_H_PAYTO.
     */
    const struct TALER_NormalizedPaytoHashP *h_payto;

    /**
     * Value of if @e option is TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_ACTIVE.
     */
    enum TALER_EXCHANGE_YesNoAll active;

  } details;

};


/**
 * Handle for an operation to GET /aml/$OFFICER_PUB/legitimizations.
 */
struct TALER_EXCHANGE_AmlLegitimizationsGetHandle;


/**
 * Set up GET /aml/$OPUB/legitimizations operation.
 * Note that you must explicitly start the operation after
 * possibly setting options.
 *
 * @param ctx the context
 * @param url base URL of the exchange
 * @param officer_priv private key of the officer
 * @return handle to operation
 */
struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *
TALER_EXCHANGE_aml_legitimizations_get_create (
  struct GNUNET_CURL_Context *ctx,
  const char *url,
  const struct TALER_AmlOfficerPrivateKeyP *officer_priv);


/**
 * Terminate the list of the options.
 *
 * @return the terminating object of struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue
 */
#define TALER_EXCHANGE_aml_legitimizations_get_option_end_()           \
        (const struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue) \
        {                                                              \
          .option = TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_END  \
        }

/**
 * Set limit @a l on the number of results to return.
 *
 * @param l limit on the number of results to return
 * @return representation of the option as a struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue
 */
#define TALER_EXCHANGE_aml_legitimizations_get_option_limit(l)          \
        (const struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue)   \
        {                                                                \
          .option = TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_LIMIT, \
          .details.limit = (l)                                           \
        }


/**
 * Set row offset from which to return results.
 *
 * @param o offset to use
 * @return representation of the option as a struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue
 */
#define TALER_EXCHANGE_aml_legitimizations_get_option_offset(o)          \
        (const struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue)    \
        {                                                                 \
          .option = TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_OFFSET, \
          .details.offset = (o)                                           \
        }


/**
 * Set filter on which account to filter legitimization measures by.
 *
 * @param p normalized payto URI hash of the account to filter by
 * @return representation of the option as a struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue
 */
#define TALER_EXCHANGE_aml_legitimizations_get_option_h_payto(p)          \
        (const struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue)     \
        {                                                                  \
          .option = TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_H_PAYTO, \
          .details.h_payto = (p)                                           \
        }

/**
 * Set filter on active (or inactive) results.
 *
 * @param a activity filter to use
 * @return representation of the option as a struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue
 */
#define TALER_EXCHANGE_aml_legitimizations_get_option_active(a)          \
        (const struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue)    \
        {                                                                 \
          .option = TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_ACTIVE, \
          .details.active = (a)                                           \
        }


/**
 * Set the requested options for the operation.
 *
 * If any option fail other options may be or may be not applied.
 *
 * @param algh the request to set the options for
 * @param num_options length of the @a options array
 * @param options array of options, each option must be created
 *            by helpers TALER_EXCHANGE_aml_legitimizations_get_option_NAME(VALUE)
 * @return #GNUNET_OK on success,
 *         #GNUNET_NO on failure,
 *         #GNUNET_SYSERR on internal error
 */
enum GNUNET_GenericReturnValue
TALER_EXCHANGE_aml_legitimizations_get_set_options_ (
  struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *algh,
  unsigned int num_options,
  const struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue *options);


/**
 * Set the requested options for the operation.
 *
 * If any option fail other options may be or may be not applied.
 *
 * It should be used with helpers that creates required options, for example:
 *
 * TALER_EXCHANGE_aml_legitimizations_get_set_options (
 *   algh,
 *   TALER_EXCHANGE_aml_legitimizations_get_option_h_payto_(&h_payto));
 *
 * @param algh the request to set the options for
 * @param ... the list of the options, each option must be created
 *            by helpers TALER_EXCHANGE_aml_legitimizations_get_option_NAME(VALUE)
 * @return #GNUNET_OK on success,
 *         #GNUNET_NO on failure,
 *         #GNUNET_SYSERR on internal error
 */
#define TALER_EXCHANGE_aml_legitimizations_get_set_options(algh,...)              \
        TALER_EXCHANGE_aml_legitimizations_get_set_options_ (                     \
          algh,                                                                   \
          TALER_EXCHANGE_COMMON_OPTIONS_ARRAY_MAX_SIZE,                           \
          ((const struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue[])       \
           {__VA_ARGS__, TALER_EXCHANGE_aml_legitimizations_get_option_end_ () }  \
          ))


/**
 * Entry in the set of legitimization measures that are returned
 * by the server in a single request.
 */
struct TALER_EXCHANGE_AmlLegitimizationsGetMeasureDetails
{
  /**
   * Hash of the normalized payto:// URI of the account the
   * measure applies to.
   */
  struct TALER_NormalizedPaytoHashP h_payto;

  /**
   * Row ID of the measure in the exchange database.
   */
  uint64_t rowid;

  /**
   * When was the measure started / triggered?
   */
  struct GNUNET_TIME_Timestamp start_time;

  /**
   * Object with the legitimization measures that are to be applied.
   */
  const json_t *measures;

  /**
   * Was this measure finished by the customer? (or obsoleted
   * by a subsequent other measure taken)?
   */
  bool is_finished;
};

/**
 * Information returned from the exchange for a
 * GET /aml/$OFFICER_PUB/legitimizations request.
 */
struct TALER_EXCHANGE_AmlLegitimizationsGetResult
{
  /**
   * HTTP response data
   */
  struct TALER_EXCHANGE_HttpResponse hr;

  /**
   * Details depending on the HTTP status code.
   */
  union
  {

    /**
     * Details on #MHD_HTTP_OK.
     */
    struct
    {
      /**
       * Length of the @e measures array.
       */
      size_t measures_length;

      /**
       * Legitimization measures.
       */
      const struct TALER_EXCHANGE_AmlLegitimizationsGetMeasureDetails *measures;

    } ok;

  } details;
};


#ifndef TALER_EXCHANGE__AML_LEGITIMIZATIONS_GET_RESULT_CLOSURE
/**
 * Type of the closure used by
 * the #TALER_EXCHANGE_AmlLegitimizationsGetCallback.
 */
#define TALER_EXCHANGE__AML_LEGITIMIZATIONS_GET_RESULT_CLOSURE void
#endif

/**
 * Type of the function that receives the result of a
 * GET /aml/$OFFICER_PUB/legitimizations request.
 *
 * @param cls closure
 * @param result result returned by the HTTP server
 */
typedef void
(*TALER_EXCHANGE_AmlLegitimizationsGetCallback)(
  TALER_EXCHANGE__AML_LEGITIMIZATIONS_GET_RESULT_CLOSURE *cls,
  const struct TALER_EXCHANGE_AmlLegitimizationsGetResult *result);


/**
 * Start GET /aml/$OPUB/legitimizations operation.
 *
 * @param[in,out] algh operation to start
 * @param cb function to call with the exchange's result
 * @param cb_cls closure for @a cb
 * @return status code
 */
enum TALER_EXCHANGE_AmlLegitimizationsGetStartError
{
  /**
   * Success.
   */
  TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_START_OK = 0,
  /**
   * Only allowed to be started once.
   */
  TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_START_E_AGAIN = 1,
  /**
   * Internal logic failure.
   */
  TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_START_E_INTERNAL = 2,
}
TALER_EXCHANGE_aml_legitimizations_get_start (
  struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *algh,
  TALER_EXCHANGE_AmlLegitimizationsGetCallback cb,
  TALER_EXCHANGE__AML_LEGITIMIZATIONS_GET_RESULT_CLOSURE * cb_cls);


/**
 * Cancel GET /aml/$OPUB/legitimizations operation.
 *
 * @param[in] algh operation to cancel
 */
void
TALER_EXCHANGE_aml_legitimizations_get_cancel (
  struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *algh);


#endif
