/*
  t-filecopy.c

  Author: Sami Lehtinen <sjl@ssh.fi>

  Copyright (C) 1999 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Tests for sshfilecopy.c .
  
 */

#include "sshincludes.h"
#include "sshfilecopy.h"
#include "sshgetopt.h"

#define SSH_DEBUG_MODULE "TestSshFileCopy"

void test_debug(const char *msg, void *context)
{
  Boolean verbose = *(Boolean *)context;

  if (verbose)
    fprintf(stderr, "%s\n", msg);
}

void test_warning(const char *msg, void *context)
{
  Boolean verbose = *(Boolean *)context;

  if (verbose)
    fprintf(stderr, "warning: %s\n", msg);
}

int main(int argc, char **argv)
{
  Boolean verbose = FALSE;
  char *test_string;
  char *temp_string, *temp_string2;
  int ret = 0;
  
  while (1)
    {
      int option;
      ssh_opterr = 0;
      ssh_optallowplus = 1;
      
      option = ssh_getopt(argc, argv, "d:v", NULL);
      
      if (option == -1)
        break;

      switch (option)
        {
        case 'd':
          ssh_debug_set_global_level(atoi(ssh_optarg));
          verbose = TRUE;
          break;
        case 'v':
          ssh_debug_set_global_level(2);
          verbose = TRUE;
          break;
        }  
    }

  ssh_debug_register_callbacks(NULL, test_warning, test_debug, &verbose);

  fprintf(stderr, "Running test for sshfilecopy, use -v for verbose "
                  "output, and -d <level> to set debug level.\n");

  SSH_DEBUG(0, ("Running tests for ssh_file_copy_get_basedir()..."));

  test_string = "ssh/ssh-2*12/foobar";

  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\" "   \
                "and temp_string2 should be \"%s\"", test_string,       \
                "ssh", "ssh-2*12/foobar"));
  

  temp_string = ssh_file_copy_get_basedir(test_string, &temp_string2);

  if (strcmp(temp_string, "ssh") != 0 ||
      strcmp(temp_string2, "ssh-2*12/foobar"))
    ret = 1;

  ssh_xfree(temp_string);
  ssh_xfree(temp_string2);

  if (ret != 0)
    return ret;
  
  SSH_DEBUG(0, ("Passed."));

  test_string = "ssh*/ssh-2*12/foobar";

  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\" "   \
                "and temp_string2 should be \"%s\"", test_string,       \
                "", "ssh*/ssh-2*12/foobar"));
  

  temp_string = ssh_file_copy_get_basedir(test_string, &temp_string2);

  if (strcmp(temp_string, "") != 0 ||
      strcmp(temp_string2, "ssh*/ssh-2*12/foobar"))
    ret = 1;

  ssh_xfree(temp_string);
  ssh_xfree(temp_string2);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/ssh*/ssh-2*12/foobar";

  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\" "   \
                "and temp_string2 should be \"%s\"", test_string,       \
                "/", "ssh*/ssh-2*12/foobar"));
  

  temp_string = ssh_file_copy_get_basedir(test_string, &temp_string2);

  if (strcmp(temp_string, "/") != 0 ||
      strcmp(temp_string2, "ssh*/ssh-2*12/foobar"))
    ret = 1;

  ssh_xfree(temp_string);
  ssh_xfree(temp_string2);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/";

  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\" "   \
                "and temp_string2 should be \"%s\"", test_string,       \
                "/", "."));
  

  temp_string = ssh_file_copy_get_basedir(test_string, &temp_string2);

  if (strcmp(temp_string, "/") != 0 ||
      strcmp(temp_string2, "."))
    ret = 1;

  ssh_xfree(temp_string);
  ssh_xfree(temp_string2);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/*";

  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\" "   \
                "and temp_string2 should be \"%s\"", test_string,       \
                "/", "*"));
  

  temp_string = ssh_file_copy_get_basedir(test_string, &temp_string2);

  if (strcmp(temp_string, "/") != 0 ||
      strcmp(temp_string2, "*"))
    ret = 1;

  ssh_xfree(temp_string);
  ssh_xfree(temp_string2);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/w*k/s*/s?*/bi*.fi??";

  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\" "   \
                "and temp_string2 should be \"%s\"", test_string,       \
                "/", "w*k/s*/s?*/bi*.fi??"));
  

  temp_string = ssh_file_copy_get_basedir(test_string, &temp_string2);

  if (strcmp(temp_string, "/") != 0 ||
      strcmp(temp_string2, "w*k/s*/s?*/bi*.fi??"))
    ret = 1;

  ssh_xfree(temp_string);
  ssh_xfree(temp_string2);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  SSH_DEBUG(0, ("Running tests for "                                    \
                "ssh_file_copy_strip_escapes_before_slashes()..."));

  test_string = "\\/\\\\//fo\\o";
  temp_string2 = "/\\\\//fo\\o";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_escapes_before_slashes(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }
  
  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "\\/\\\\/\\\\\\/fo\\o";
  temp_string2 = "/\\\\/\\\\/fo\\o";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_escapes_before_slashes(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }
  
  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  SSH_DEBUG(0, ("Running tests for ssh_file_copy_strip_extra_slashes()..."));

  test_string = "";
  temp_string2 = "";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_extra_slashes(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }
  
  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "///foo";
  temp_string2 = "/foo";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_extra_slashes(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }
  
  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));
  
  test_string = "//////////foo/////bar/////....////dsfgd";
  temp_string2 = "/foo/bar/..../dsfgd";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_extra_slashes(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }
  
  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  SSH_DEBUG(0, ("Running tests for ssh_file_copy_strip_dot_dots()..."));

  test_string = "/../foo";
  temp_string2 = "/../foo";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }
  
  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/../";
  temp_string2 = "/../";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }
  
  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/..";
  temp_string2 = "/..";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }
  
  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/foo/../sufee";
  temp_string2 = "/sufee";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }

  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/foo/../";
  temp_string2 = "/";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }

  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/foo/../munaa/../kukkuu/huhhei/..";
  temp_string2 = "/kukkuu/";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }

  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/foo/../munaa/../kukkuu/huhhei/../";
  temp_string2 = "/kukkuu/";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }

  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/foo/../munaa/../kukkuu/huhhei/../munaa/../kukkuu/huhhei";
  temp_string2 = "/kukkuu/kukkuu/huhhei";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }

  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "../munaa/../kukkuu/huhhei/../munaa/../kukkuu/huhhei";
  temp_string2 = "../kukkuu/kukkuu/huhhei";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }

  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "../../sufee";
  temp_string2 = "../../sufee";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }

  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "CVS";
  temp_string2 = "CVS";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }

  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "./CVS";
  temp_string2 = "./CVS";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }

  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  test_string = "/hohaa/./CVS";
  temp_string2 = "/hohaa/./CVS";
  
  SSH_DEBUG(1, ("test_string = \"%s\", temp_string should be \"%s\"."   \
                ,test_string, temp_string2));

  temp_string = ssh_file_copy_strip_dot_dots(test_string);

  if (strcmp(temp_string, temp_string2))
    {
      SSH_DEBUG(2, ("temp_string = \"%s\"", temp_string));
      ret = 1;
    }

  ssh_xfree(temp_string);

  if (ret != 0)
    return ret;

  SSH_DEBUG(0, ("Passed."));

  SSH_DEBUG(0, ("Passed all tests."));
  
  return 0;
}
