#define _POSIX_SOURCE
/*
 * This is a mess.  HP-UX ANSI C won't define pid_t in <sys/types.h> unless
 * _POSIX_SOURCE is defined.  Other machines may refuse to define it.  Pick
 * what's appropriate.  <utmp.h> uses pid_t.  On __SOLARIS__, <utmpx.h>
 * uses (struct timeval) which won't be defined unless !_POSIX_SOURCE.
 */
#if defined(__SOLARIS__) /* { */
#undef _POSIX_SOURCE	/* allows struct timeval to be defined */
#undef _POSIX_C_SOURCE
#include <sys/types.h>	/* gid_t pid_t uid_t clock_t */
#include <time.h>	/* time (also needed for "utmp.h") */
#define _POSIX_SOURCE
#define _POSIX_C_SOURCE
#else  /* } { */
#include <sys/types.h>	/* gid_t pid_t uid_t clock_t */
#include <time.h>	/* time (also needed for "utmp.h") */
#endif /* } */

#include <sys/times.h>	/* struct tms */
			/* mips fails to define prototype for times() unless */
			/* __POSIX is defined.  Strange.  */
#include <unistd.h> 	/* chdir close dup execve fork */
			/* getlogin getpid getppid getpgrp */
			/* setsid setgid setpgid setuid ctermid */
			/* _SC_NGROUPS_MAX */
#include <limits.h>	/* NGROUPS_MAX */
#include <ctype.h>	/* must be before stdlib.h for BSDI's BSD 386 */
#include <stdlib.h>	/* atexit exit free malloc */
#include <stdio.h>	/* (fopen fclose fgets scanf printf etc.) */
#include <string.h>	/* memcmp memcpy memset */
#include <errno.h>
#include <sys/stat.h>	/* umask flags */
#include <fcntl.h>
#include <termios.h>
#include <signal.h>
#include <pwd.h>
#include <sys/wait.h>	/* WNOHANG */
#include "tty.h"
#include "auth.h"
#include "socket.h"
#include "log.h"
#include "deslogin.h"
#include "posignal.h"

#if defined(sun)
#define BSD
#endif
#if defined(mips)
#define BSD
#endif

/*
 * Begin customizable area
 */

/*
 * These environment variables may be overwridden in the makefile
 */
#if defined(BSD)		/* sun dec mips */
#if !defined(USERPATH)
#define USERPATH	":/usr/ucb:/bin:/usr/bin"
#endif
#if !defined(SYSTEMPATH)
#define SYSTEMPATH	"/etc:/usr/ucb:/bin:/usr/bin"
#endif
#endif
#if defined(hpux)		/* hpux only */
#define DEFAULT_MAIL	"/usr/mail/"
#endif

#if !defined(USER_ENV)	/* Additional envt variable to store userName */
#define USER_ENV	"USER"		/* BSD (sun dec mips) xdm (not hpux) */
#endif
#if !defined(USERPATH)
#define USERPATH	":/bin:/usr/bin"
#endif
#if !defined(SYSTEMPATH)
#define SYSTEMPATH	"/etc:/bin:/usr/bin"
#endif
#if !defined(DEFAULT_TERM)
#define DEFAULT_TERM	"network"
#endif

#if !defined(USER_FILE)
#define USER_FILE	"/usr/local/etc/deslogind.users"
#endif

#if !defined(NOLOGIN_FILE)
#define NOLOGIN_FILE	"/etc/nologin"
#endif
#if !defined(PWDLOG_FILE)
#define PWDLOG_FILE	"/usr/adm/dlpasswdd.log"
#endif

#define SLEEPTIME 1;

/*
 * End of customizable area
 */
#define PROGNAME_SIZE	255
#define USERNAME_SIZE	17	/* assumption based on rshd man page */

#define MAX_HOSTNAME	255

extern int optind;
extern char *optarg;

#if defined(__STDC__) || defined(__cplusplus)
extern int askTty(char *, char *, unsigned, int);
#else
extern int askTty();
#endif

extern unsigned long defaultKey[64];

static char ident[] = 
" @(#) chpwdd.c\n";
static char RCSid[] = 
" @(#) chpwdd.c\n";

static int use_ndbm = 0; /* Added by Federico Audano */

char mainProgName[PROGNAME_SIZE];

char *progName, *argcName;
int  debug   = 0;
int  verbose = 0;

unsigned long inBytes = 0;
unsigned long outBytes = 0;

void setProgName() {
   pid_t pid = getpid();				/* can't fail */
   sprintf(mainProgName, "%s[%lu]", argcName, (unsigned long) pid);
   progName = mainProgName;
}

void sigHandler(sig)
   int sig;
{
   switch (sig) {
   case SIGHUP:
   case SIGTERM:
   case SIGINT:
      log("%s: Terminated by signal %d\n", progName, sig);
      exit(1);			/* make sure cleanup occurs on interrupt */
      break;
   case SIGCHLD: 			/* child died */
      if (debug) {
	 log("%s: (main) SIGCHLD\n", progName);
      }
      break;
   default:
      break;
   }
}

char userName[USERNAME_SIZE];   	/* login name of user */
char rhostName[MAX_HOSTNAME];		/* remote login host */

/*
 * Memory allocation with error check and abort
 */
void *allocate(size)
   register unsigned size;
{
   register void *res = (void *) malloc(size);

   if (res == (void *) 0) {
      log("%s: allocate of %u bytes failed\n", progName, size);
      exit(1);
   }
   return res;
}

/*
 * Create a copy of the input string. (0 if out of memory)
 */
char *newString(chp)
   register char *chp;
{
   register unsigned len = 0;
   register char *res;

   if (chp != (char *) 0) {
      len = strlen(chp);
   }

   res = (char *) allocate(len+1);
   if (res != (char *) 0) {
      memcpy(res, chp, len);
      res[len] = '\0';
   }
   return res;
}

/*
 * Get and validate the userfile cipher key.  Returns the binary key or -1.
 *
 * If noUserKey is 1, then userKey is 0, and the userFile is not encrypted.  
 * Otherwise, it's expected to have been encrypted with the cipher program.
 */
keyType getUserFileKey(userFile, userPhrase, noUserKey) 
   char *userFile;
   char *userPhrase;
   int noUserKey;
{
   int res;
   char passPhrase[PHRASE_SIZE];
   keyType userKey = 0;

   if (noUserKey && userPhrase) {
      log("%s: can't use -n option if UserFile cipher key given\n", progName);
      return (keyType) -1;
   }
   if (!noUserKey) {
      if (userPhrase == (char *) 0) {	/* user for passphrase if none given */
	 res = askTty("UserFile cipher key: ", passPhrase, PHRASE_SIZE-1, 0);
	 if (res < 0) {
	    log("%s: couldn't get key\n", progName);
	    return (keyType) -1;
	 }
	 userPhrase = passPhrase;
      }
      if (userPhrase == (char *) -1) {
	 userKey = (keyType) &defaultKey[0];	/* use compile-time key */ 
      } else {
	 userKey = mkKey(userPhrase);
	 memset(userPhrase, '\0', strlen(userPhrase));	/* possibly optarg */
	 if (userKey == (keyType) 0) {
	    log("%s: couldn't make key\n", progName);
	    return (keyType) -1;
	 }
      }
   }
   /*
    * Make userfile exists and the correct key was specified
    */

   if (!use_ndbm)
    res = getUserPhrase(userFile, passPhrase, PHRASE_SIZE, "#", userKey);

   if (res == -1) {
      log("%s: cannot open \"%s\" for user database\n", progName, userFile);
      return (keyType) -1;
   }
   if (res == -2) {
      log("%s: incorrect decryption key for user database \"%s\"\n", 
	 progName, userFile);
      return (keyType) -1;
   }
   return userKey;
}

int handleSignals() {
   register char *chp;

   /*
    * To make sure we cleanup utmp with atexit if we're aborted
    */
   chp = (char *) posignal(SIGINT, sigHandler);
   if (chp == (char *) SIG_ERR) {
      log("%s: sigaction SIGINT failed--%s\n", progName, ERRMSG);
      return -1;
   }
   /*
    * Sent by kill command with no arguments
    */
   chp = (char *) posignal(SIGTERM, sigHandler); 
   if (chp == (char *) SIG_ERR) {
      log("%s: sigaction SIGTERM failed--%s\n", progName, ERRMSG);
      return -1;
   }
   /*
    * Controlling terminal hangup, or death of controlling process
    */
   chp = (char *) posignal(SIGHUP, sigHandler); 
   if (chp == (char *) SIG_ERR) {
      log("%s: sigaction SIGHUP failed--%s\n", progName, ERRMSG);
      return -1;
   }
   return 0;
}

/*
 * Became a deamon
 */
pid_t becomeDaemon() {
   pid_t pid = getpid();

   if (debug < 2) {	/* deamon: disassociate from controlling terminal */
      pid = fork();			/* make pgid != pid for setsid() */
      if (pid < 0) {
	 log("%s: unable to fork as daemon--%s\n", progName, ERRMSG);
	 return -1;
      }
      if (pid > 0) {			/* the parent is finished */
	 exit(1);
      }
      /* child */

      setProgName(); 		
      pid = setsid();		/* break control terminal affiliation */
      if (pid == -1) {
	 log("%s: setsid failed--%s\n", progName, ERRMSG);
	 return -1;
      }
      /*
       * We may still have a problem here.  When the child dies, we will 
       * create a zombie which will have to be waited for by the parent!
       * Since we're a session leader, presumably that parent is now init.
       */
   }
   return pid;
}

/*
 * Negotiate protocol with remote site.  
 * Return the selected protocol or 0 for failure.
 */
char *serverHandshake(nfd, rhostName, rport, timeout)
   int nfd;
   char *rhostName;
   int rport;
   unsigned timeout;
{
   int len, res, count;
   static char protover[VERS_SIZE];

   len = strlen(PROTOCOL_VERS);
   res = write(nfd, PROTOCOL_VERS, len+1);
   if (debug) {
      log("%s: server protocol \"%-.*s\"\n", progName, len, PROTOCOL_VERS);
   }
   if (res < 0) {
      log("%s: write (%d, \"%-.*s\", %d) failed--%s\n", 
	 progName, nfd, len, PROTOCOL_VERS, len+1, ERRMSG);
      return 0;
   }
   count = getString(nfd, protover, VERS_SIZE-1, timeout);
   if (count == 0) {
      log("%s: NOPROTO %s:%d\n", progName, rhostName, rport);
      return 0;
   }
   if (debug) {
      log("%s: client protocol \"%-.*s\"\n", progName, VERS_SIZE-1, protover);
   }
   if (protover[0] != PROTOCOL_VERS[0]) {
      log("%s: BADPROTO(%-.*s) %s:%d\n",
       progName, count, protover, rhostName, rport);
      return 0;
   }
   return &protover[0];
}

int main(argc, argv)
   int argc;
   char *argv[];
{
   int ch, res, count, noUserKey = 0; 
   char *chp, *protocol;
   char *path = USERPATH, *userFile = USER_FILE, *logName = PWDLOG_FILE;
   char ruser[USERNAME_SIZE], passPhrase[PHRASE_SIZE];
   char *userPhrase = (char *) 0;
   unsigned port = 0, rport = 0;
   unsigned connTimeout  = 1000 * INACTIVITY_TIMEOUT;
   unsigned loginTimeout = 1000 * LOGIN_TIMEOUT;	/* login timer */
   unsigned bufSize = 128;			/* connection bufsize */
   int      nfd = -1,  stype;
   pid_t    pid, pid_res;
   keyType  key = (keyType) 0, userKey;
   char old[PHRASE_SIZE],new[PHRASE_SIZE],temp[PHRASE_SIZE],buff[PHRASE_SIZE];

   argcName = *argv;
   if ((chp = strrchr(argcName, '/')) != (char *) 0) argcName = chp + 1;
   progName = argcName;

   stype = issocket(0);
   if (stype > 0) {		/* if we're invoked from inetd */
      res = openLog(logName);
      if (res < 0) {
	 _exit(1);
      }
   }

   while ((ch = getopt(argc, argv, "k:l:t:f:p:Ddvnc")) != EOF) switch (ch) {
   case 'c': 				   /* user compile'd userFile key */
      userPhrase = (char *) -1;
      break;
   case 'k': 				   /* user file cipher key */
      userPhrase = optarg;
      break;
   case 'n': 				   /* no user file cipher key */
      noUserKey++;
      break;
   case 'l':
      logName = optarg;
      break;
   case 't':					/* passphrase timeout */
      count = sscanf(optarg, "%u", &loginTimeout);	/* in seconds */
      if (count != 1) goto usage;
      loginTimeout *= 1000;
      break;
   case 'f':					/* set userfile */
      userFile = optarg;
      break;
   case 'p':					/* bind to port # */
      count = sscanf(optarg, "%u", &port);
      if (count != 1) goto usage;
      break;
   case 'D':                             /* New option Federico Audano */
      use_ndbm = 1;                     /* use NDBM functions for user*/
      break;				/* database		      */
   case 'v':
      verbose++;
      break;
   case 'd':
      debug++;
      break;
   default:
usage:	log(
"usage: %s  [-dn] [-t loginSecs ] [-l logfile ] [-f userfile ] [-k userfileCipherKey ] [-p port ]\n", 
progName);
      return 1;
   }
   argc -= optind;
   argv += optind;

   if (debug > 1) {
      log("%s: issocket(0) returned %d\n", progName, stype);
   }
   userKey = getUserFileKey(userFile, userPhrase, noUserKey);
   if (userKey == (keyType) -1) {
      return 1;
   }

   /*
    * Open log *after* options parsed and some simple error checking,
    * and *before* closing fd's 0,1, and 2.
    */
   res = openLog(logName);		/* leaves existing log if failed */
   if (res < 0) { 
      log("%s: couldn't open logfile \"%s\"--%s\n", progName, logName, ERRMSG);
      destroyKey(&userKey);
      return 1;
   }
   setProgName();
   res = handleSignals();		/* handle quit attempts */
   if (res < 0) {
      destroyKey(&userKey);
      return 1;
   }
   /*
    * The remote host name is not trustworthy for several reasons.
    * We don't use it (or the remote IP address) for authentication.
    * Relying on the Domain Name System, or the remote IP address
    * is not something I'm willing to trust.  Here, we use it strictly
    * for logging, and informational purposes (who, RHOSTNAME, etc).
    */
   if (getRemoteAddr(0, rhostName, sizeof rhostName, &rport) == 0) {
      nfd = 0; 			/* From inetd.  0 and 1 is our socket. */
      close(2);		/* inetd leaves it open */
   } else {			/* Interactive invocation */
      if (port == 0) {
	 port = getServicePort(PW_SERVICE_NAME);
	 if ((int) port < 0) {
	    port = DLPASSWD_PORT;
	    log(
	 "%s: WARNING--no \"%s\" service; using port %u\n", 
	       progName, PW_SERVICE_NAME, port);
	 }
      }
      log("%s: Starting daemon pid %ld port %d\n", 
	 progName, (unsigned long) getpid(), port);

      /* 
       * Must close fds 0,1, and 2 *before* openServer and becomeDaemon
       */
      close(0);
      close(1);
      close(2);

      pid = becomeDaemon();
      if (pid < 0) {
	 destroyKey(&userKey);
         return 1;		/* messages already logged */
      }

      nfd = openServer(port, rhostName, sizeof rhostName, &rport, 1+(debug < 2));
      if (nfd < 0) {
	 log("%s: open network port %d failed--%s\n", progName, port, ERRMSG);
	 destroyKey(&userKey);
	 return 1;
      }
      setProgName(); /* Openserver did a fork when it returned successfully */
      /*
       * Openserver handed us nfd as fd 1, 0 is now available
       */
   }
   log("%s: connect %s:%d\n", progName, rhostName, rport);

   protocol = serverHandshake(nfd, rhostName, rport, SETUP_TIMEOUT);
   if (!protocol) {
      destroyKey(&userKey);
      return 1;		/* log already done */
   }
   count = getString(nfd, ruser, USERNAME_SIZE-1, SETUP_TIMEOUT);
   if (count == 0) {
      log("%s: NORUSER  %s:%d\n", progName, rhostName, rport);
      destroyKey(&userKey);
      return 1;
   }
   count = getString(nfd, userName, USERNAME_SIZE-1, SETUP_TIMEOUT);
   if (count == 0) {
      log("%s: NOLUSER  %s@%s:%d\n", 
	 progName, ruser, rhostName, rport);
      destroyKey(&userKey);
      return 1;
   }
   /*
    * Minimize the the time the phrase must be visible in memory.
    */
   
   chp = userName;

   /* Modified by Federico Audano for NDBM */
   if (!use_ndbm)
   {
    res = getUserPhrase(userFile, passPhrase, PHRASE_SIZE, chp, userKey);
   }
   else
   {
    res = ndbm_getUserPhrase(userFile,passPhrase, PHRASE_SIZE, chp, userKey);
   }
   /* end of modified block */

   if (res == 0) {
      log("%s: NOPHRASE %s@%s:%d->\"%s\"\n", 
         progName, ruser, rhostName, rport, chp);
      destroyKey(&userKey);
      return 1;
   }
   if (res == -1) {
      log("%s: cannot open \"%s\" for user database\n", progName, userFile);
      destroyKey(&userKey);
      return 1;
   }
   if (res == -2) {
      log("%s: incorrect decryption key for user database \"%s\"\n", 
         progName, userFile);
      destroyKey(&userKey);
      return 1;
   }
   key = challenge(nfd, passPhrase, loginTimeout);
   if (key == (keyType) 0) {
      log("%s: BADLOGIN %s@%s:%d->\"%s\"\n", 
	 progName, ruser, rhostName, rport, userName);
      destroyKey(&userKey);
      return 1;
   }
   
   log("%s: Changing passphrase for user: %s\n",
	progName,userName);
  
   ndbm_cipherstring(buff,"ok.",4,key,0);
   res = write(nfd,buff,4);
   if (res < 0)
   {
    log("%s: network write failed.\n",progName);
    memset(passPhrase,'\0',PHRASE_SIZE);
    destroyKey(&userKey);
    destroyKey(&key);
    return(1);
   }

   res = read(nfd,temp,PHRASE_SIZE);
   if (res < 0)
   {
    log("%s: network read failed.\n",progName);
    memset(passPhrase,'\0',PHRASE_SIZE);
    destroyKey(&userKey);
    destroyKey(&key);  
    return(1);
   }
   ndbm_cipherstring(new,temp,PHRASE_SIZE,key,1);

   res = write(nfd,temp,PHRASE_SIZE);
   if (res < 0)
   {
    log("%s: Network write failed.\n",progName);
    memset(passPhrase,'\0',PHRASE_SIZE);
    destroyKey(&userKey);
    destroyKey(&key);
    return(1);
   }


   log("%s New passphrase: %s\n",progName,new);

   res = read(nfd,temp,11);
   if (res < 0)
   {
    log("%s: Network read failed.\n",progName);
    memset(passPhrase,'\0',PHRASE_SIZE);
    destroyKey(&userKey);
    destroyKey(&key);
    return(1);
   }

   ndbm_cipherstring(buff,temp,11,key,1);


   log("%s String ok: %s\n",progName,buff);

   if (!strcmp(buff,"okproceed."))
    res = ndbm_changepwd(userFile,userName,passPhrase,new,userKey);
   else
    res = -1;

   memset(new,'\0',PHRASE_SIZE);
   memset(passPhrase,'\0',PHRASE_SIZE);
   if (res < 0)
   {
    ndbm_cipherstring(buff,"unchanged.",11,key,0);
    res = write(nfd,buff,11);
    if (res < 0)
    {
     log("%s: network write failed.\n",progName);
    }
    destroyKey(&key);
    destroyKey(&userKey);
    log("%s: passphrase unchanged.\n",progName);
    return(1); 
   }

   ndbm_cipherstring(buff,"okchanged.",11,key,0);
   res = write(nfd,buff,11);
   if (res < 0)
   {
    log("%s: network write failed.\n",progName);
    destroyKey(&key);
    destroyKey(&userKey);
    return(1);
   }
   log("%s: User %s , password changed.\n",progName,userName);

   destroyKey(&userKey);
   destroyKey(&key);
   exit(0); 
}
