use v5.10.0;
package JMAP::Tester::Response::Paragraph;
# ABSTRACT: a group of sentences in a JMAP response
$JMAP::Tester::Response::Paragraph::VERSION = '0.003';
use Moo;

#pod =head1 OVERVIEW
#pod
#pod These objects represent paragraphs in the JMAP response.  That is, if your
#pod response is:
#pod
#pod   [
#pod     [ "messages", { ... }, "a" ],      # 1
#pod     [ "smellUpdates", { ... }, "b" ],  # 2
#pod     [ "smells",       { ... }, "b" ],  # 3
#pod   ]
#pod
#pod ...then #1 forms one paragraph and #2 and #3 together form another.  It goes by
#pod matching client ids.
#pod
#pod =cut

has sentences => (is => 'bare', reader => '_sentences', required => 1);

has _json_typist => (
  is => 'ro',
  handles => {
    _strip_json_types => 'strip_types',
  },
);

#pod =method sentences
#pod
#pod The C<sentences> method returns a list of
#pod L<Sentence|JMAP::Tester::Response::Sentence> objects, one for each sentence
#pod in the paragraph.
#pod
#pod =cut

sub sentences { @{ $_[0]->_sentences } }

#pod =method sentence
#pod
#pod   my $sentence = $para->sentence($n);
#pod
#pod This method returns the I<n>th sentence of the paragraph.
#pod
#pod =cut

sub sentence {
  # die on out-of-range?
  $_[0]->_sentences->[$_[1]];
}

#pod =method single
#pod
#pod   my $sentence = $para->single;
#pod   my $sentence = $para->single($name);
#pod
#pod This method throws an exception if there is more than one sentence in the
#pod paragraph.  If a C<$name> argument is given and the paragraph's single
#pod sentence doesn't have that name, an exception is raised.
#pod
#pod Otherwise, this method returns the sentence.
#pod
#pod =cut

sub single {
  my ($self, $name) = @_;

  my @sentences = $self->sentences;

  Carp::confess("more than one sentence in set, but ->single called")
    if @sentences > 1;

  Carp::confess("single sentence not of expected name <$name>")
    if defined $name && $name ne $sentences[0]->name;

  return $sentences[0];
}

#pod =method as_struct
#pod
#pod =method as_stripped_struct
#pod
#pod C<as_struct> returns an arrayref containing the result of calling C<as_struct>
#pod on each sentence in the paragraph. C<as_stripped_struct> removes JSON types.
#pod
#pod =cut

sub as_struct {
  [ map {; $_->as_struct } $_[0]->sentences ]
}

sub as_stripped_struct {
  $_[0]->_strip_json_types($_[0]->as_struct);
}

#pod =method as_pairs
#pod
#pod C<as_pairs> returns an arrayref containing the result of calling C<as_pair>
#pod on each sentence in the paragraph. C<as_stripped_pairs> removes JSON types.
#pod
#pod =cut

sub as_pairs {
  [ map {; $_->as_pair } $_[0]->sentences ]
}

sub as_stripped_pairs {
  $_[0]->_strip_json_types($_[0]->as_pairs);
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

JMAP::Tester::Response::Paragraph - a group of sentences in a JMAP response

=head1 VERSION

version 0.003

=head1 OVERVIEW

These objects represent paragraphs in the JMAP response.  That is, if your
response is:

  [
    [ "messages", { ... }, "a" ],      # 1
    [ "smellUpdates", { ... }, "b" ],  # 2
    [ "smells",       { ... }, "b" ],  # 3
  ]

...then #1 forms one paragraph and #2 and #3 together form another.  It goes by
matching client ids.

=head1 METHODS

=head2 sentences

The C<sentences> method returns a list of
L<Sentence|JMAP::Tester::Response::Sentence> objects, one for each sentence
in the paragraph.

=head2 sentence

  my $sentence = $para->sentence($n);

This method returns the I<n>th sentence of the paragraph.

=head2 single

  my $sentence = $para->single;
  my $sentence = $para->single($name);

This method throws an exception if there is more than one sentence in the
paragraph.  If a C<$name> argument is given and the paragraph's single
sentence doesn't have that name, an exception is raised.

Otherwise, this method returns the sentence.

=head2 as_struct

=head2 as_stripped_struct

C<as_struct> returns an arrayref containing the result of calling C<as_struct>
on each sentence in the paragraph. C<as_stripped_struct> removes JSON types.

=head2 as_pairs

C<as_pairs> returns an arrayref containing the result of calling C<as_pair>
on each sentence in the paragraph. C<as_stripped_pairs> removes JSON types.

=head1 AUTHOR

Ricardo SIGNES <rjbs@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by FastMail, Ltd.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
