#!perl

use strict;
use warnings;

use Test::More tests => 3833;

my $class;

BEGIN { $class = 'Math::BigInt::Lite'; }
BEGIN { use_ok($class, '0.16'); }

while (<DATA>) {
    s/#.*$//;           # remove comments
    s/\s+$//;           # remove trailing whitespace
    next unless length; # skip empty lines

    my ($in0, $out0) = split /:/;
    my $x;

    my $test = qq|\$x = $class -> from_bin("$in0");|;
    my $desc = $test;

    eval $test;
    die $@ if $@;       # this should never happen

    subtest $desc, sub {
        plan tests => 2,

        # Check output.

        # Explcitly setting Math::BigInt here is bad. We should use be able to
        # use Math::BigInt::Lite -> config("upgrade"), but that doesn't work
        # yet. FIXME!

        my $xref = ref($x);
        ok($xref eq $class || $xref eq 'Math::BigInt',
           "output arg is a $class or a Math::BigInt");

        is($x, $out0, 'output arg has the right value');
    };

}

__END__

NaN:NaN
+inf:NaN
-inf:NaN

0b10:2
0B10:2
b10:NaN
B10:NaN

###############################################################################
# Absolute value is close to 2^1 = 2
###############################################################################

# Unsigned

0b0:0
0b1:1
0b10:2
0b11:3
0b100:4

b0:NaN
b1:NaN
b10:NaN
b11:NaN
b100:NaN

0:0
1:1
10:2
11:3
100:4

# Positive

+0b0:0
+0b1:1
+0b10:2
+0b11:3
+0b100:4

+b0:NaN
+b1:NaN
+b10:NaN
+b11:NaN
+b100:NaN

+0:0
+1:1
+10:2
+11:3
+100:4

# Negative

-0b0:0
-0b1:-1
-0b10:-2
-0b11:-3
-0b100:-4

-b0:NaN
-b1:NaN
-b10:NaN
-b11:NaN
-b100:NaN

-0:0
-1:-1
-10:-2
-11:-3
-100:-4

###############################################################################
# Absolute value is close to 2^2 = 4
###############################################################################

# Unsigned

0b10:2
0b11:3
0b100:4
0b101:5
0b110:6

b10:NaN
b11:NaN
b100:NaN
b101:NaN
b110:NaN

10:2
11:3
100:4
101:5
110:6

# Positive

+0b10:2
+0b11:3
+0b100:4
+0b101:5
+0b110:6

+b10:NaN
+b11:NaN
+b100:NaN
+b101:NaN
+b110:NaN

+10:2
+11:3
+100:4
+101:5
+110:6

# Negative

-0b10:-2
-0b11:-3
-0b100:-4
-0b101:-5
-0b110:-6

-b10:NaN
-b11:NaN
-b100:NaN
-b101:NaN
-b110:NaN

-10:-2
-11:-3
-100:-4
-101:-5
-110:-6

###############################################################################
# Absolute value is close to 2^3 = 8
###############################################################################

# Unsigned

0b110:6
0b111:7
0b1000:8
0b1001:9
0b1010:10

b110:NaN
b111:NaN
b1000:NaN
b1001:NaN
b1010:NaN

110:6
111:7
1000:8
1001:9
1010:10

# Positive

+0b110:6
+0b111:7
+0b1000:8
+0b1001:9
+0b1010:10

+b110:NaN
+b111:NaN
+b1000:NaN
+b1001:NaN
+b1010:NaN

+110:6
+111:7
+1000:8
+1001:9
+1010:10

# Negative

-0b110:-6
-0b111:-7
-0b1000:-8
-0b1001:-9
-0b1010:-10

-b110:NaN
-b111:NaN
-b1000:NaN
-b1001:NaN
-b1010:NaN

-110:-6
-111:-7
-1000:-8
-1001:-9
-1010:-10

###############################################################################
# Absolute value is close to 2^4 = 16
###############################################################################

# Unsigned

0b1110:14
0b1111:15
0b10000:16
0b10001:17
0b10010:18

b1110:NaN
b1111:NaN
b10000:NaN
b10001:NaN
b10010:NaN

1110:14
1111:15
10000:16
10001:17
10010:18

# Positive

+0b1110:14
+0b1111:15
+0b10000:16
+0b10001:17
+0b10010:18

+b1110:NaN
+b1111:NaN
+b10000:NaN
+b10001:NaN
+b10010:NaN

+1110:14
+1111:15
+10000:16
+10001:17
+10010:18

# Negative

-0b1110:-14
-0b1111:-15
-0b10000:-16
-0b10001:-17
-0b10010:-18

-b1110:NaN
-b1111:NaN
-b10000:NaN
-b10001:NaN
-b10010:NaN

-1110:-14
-1111:-15
-10000:-16
-10001:-17
-10010:-18

###############################################################################
# Absolute value is close to 2^5 = 32
###############################################################################

# Unsigned

0b11110:30
0b11111:31
0b100000:32
0b100001:33
0b100010:34

b11110:NaN
b11111:NaN
b100000:NaN
b100001:NaN
b100010:NaN

11110:30
11111:31
100000:32
100001:33
100010:34

# Positive

+0b11110:30
+0b11111:31
+0b100000:32
+0b100001:33
+0b100010:34

+b11110:NaN
+b11111:NaN
+b100000:NaN
+b100001:NaN
+b100010:NaN

+11110:30
+11111:31
+100000:32
+100001:33
+100010:34

# Negative

-0b11110:-30
-0b11111:-31
-0b100000:-32
-0b100001:-33
-0b100010:-34

-b11110:NaN
-b11111:NaN
-b100000:NaN
-b100001:NaN
-b100010:NaN

-11110:-30
-11111:-31
-100000:-32
-100001:-33
-100010:-34

###############################################################################
# Absolute value is close to 2^6 = 64
###############################################################################

# Unsigned

0b111110:62
0b111111:63
0b1000000:64
0b1000001:65
0b1000010:66

b111110:NaN
b111111:NaN
b1000000:NaN
b1000001:NaN
b1000010:NaN

111110:62
111111:63
1000000:64
1000001:65
1000010:66

# Positive

+0b111110:62
+0b111111:63
+0b1000000:64
+0b1000001:65
+0b1000010:66

+b111110:NaN
+b111111:NaN
+b1000000:NaN
+b1000001:NaN
+b1000010:NaN

+111110:62
+111111:63
+1000000:64
+1000001:65
+1000010:66

# Negative

-0b111110:-62
-0b111111:-63
-0b1000000:-64
-0b1000001:-65
-0b1000010:-66

-b111110:NaN
-b111111:NaN
-b1000000:NaN
-b1000001:NaN
-b1000010:NaN

-111110:-62
-111111:-63
-1000000:-64
-1000001:-65
-1000010:-66

###############################################################################
# Absolute value is close to 2^7 = 128
###############################################################################

# Unsigned

0b1111110:126
0b1111111:127
0b10000000:128
0b10000001:129
0b10000010:130

b1111110:NaN
b1111111:NaN
b10000000:NaN
b10000001:NaN
b10000010:NaN

1111110:126
1111111:127
10000000:128
10000001:129
10000010:130

# Positive

+0b1111110:126
+0b1111111:127
+0b10000000:128
+0b10000001:129
+0b10000010:130

+b1111110:NaN
+b1111111:NaN
+b10000000:NaN
+b10000001:NaN
+b10000010:NaN

+1111110:126
+1111111:127
+10000000:128
+10000001:129
+10000010:130

# Negative

-0b1111110:-126
-0b1111111:-127
-0b10000000:-128
-0b10000001:-129
-0b10000010:-130

-b1111110:NaN
-b1111111:NaN
-b10000000:NaN
-b10000001:NaN
-b10000010:NaN

-1111110:-126
-1111111:-127
-10000000:-128
-10000001:-129
-10000010:-130

###############################################################################
# Absolute value is close to 2^8 = 256
###############################################################################

# Unsigned

0b11111110:254
0b11111111:255
0b100000000:256
0b100000001:257
0b100000010:258

b11111110:NaN
b11111111:NaN
b100000000:NaN
b100000001:NaN
b100000010:NaN

11111110:254
11111111:255
100000000:256
100000001:257
100000010:258

# Positive

+0b11111110:254
+0b11111111:255
+0b100000000:256
+0b100000001:257
+0b100000010:258

+b11111110:NaN
+b11111111:NaN
+b100000000:NaN
+b100000001:NaN
+b100000010:NaN

+11111110:254
+11111111:255
+100000000:256
+100000001:257
+100000010:258

# Negative

-0b11111110:-254
-0b11111111:-255
-0b100000000:-256
-0b100000001:-257
-0b100000010:-258

-b11111110:NaN
-b11111111:NaN
-b100000000:NaN
-b100000001:NaN
-b100000010:NaN

-11111110:-254
-11111111:-255
-100000000:-256
-100000001:-257
-100000010:-258

###############################################################################
# Absolute value is close to 2^9 = 512
###############################################################################

# Unsigned

0b111111110:510
0b111111111:511
0b1000000000:512
0b1000000001:513
0b1000000010:514

b111111110:NaN
b111111111:NaN
b1000000000:NaN
b1000000001:NaN
b1000000010:NaN

111111110:510
111111111:511
1000000000:512
1000000001:513
1000000010:514

# Positive

+0b111111110:510
+0b111111111:511
+0b1000000000:512
+0b1000000001:513
+0b1000000010:514

+b111111110:NaN
+b111111111:NaN
+b1000000000:NaN
+b1000000001:NaN
+b1000000010:NaN

+111111110:510
+111111111:511
+1000000000:512
+1000000001:513
+1000000010:514

# Negative

-0b111111110:-510
-0b111111111:-511
-0b1000000000:-512
-0b1000000001:-513
-0b1000000010:-514

-b111111110:NaN
-b111111111:NaN
-b1000000000:NaN
-b1000000001:NaN
-b1000000010:NaN

-111111110:-510
-111111111:-511
-1000000000:-512
-1000000001:-513
-1000000010:-514

###############################################################################
# Absolute value is close to 2^10 = 1024
###############################################################################

# Unsigned

0b1111111110:1022
0b1111111111:1023
0b10000000000:1024
0b10000000001:1025
0b10000000010:1026

b1111111110:NaN
b1111111111:NaN
b10000000000:NaN
b10000000001:NaN
b10000000010:NaN

1111111110:1022
1111111111:1023
10000000000:1024
10000000001:1025
10000000010:1026

# Positive

+0b1111111110:1022
+0b1111111111:1023
+0b10000000000:1024
+0b10000000001:1025
+0b10000000010:1026

+b1111111110:NaN
+b1111111111:NaN
+b10000000000:NaN
+b10000000001:NaN
+b10000000010:NaN

+1111111110:1022
+1111111111:1023
+10000000000:1024
+10000000001:1025
+10000000010:1026

# Negative

-0b1111111110:-1022
-0b1111111111:-1023
-0b10000000000:-1024
-0b10000000001:-1025
-0b10000000010:-1026

-b1111111110:NaN
-b1111111111:NaN
-b10000000000:NaN
-b10000000001:NaN
-b10000000010:NaN

-1111111110:-1022
-1111111111:-1023
-10000000000:-1024
-10000000001:-1025
-10000000010:-1026

###############################################################################
# Absolute value is close to 2^11 = 2048
###############################################################################

# Unsigned

0b11111111110:2046
0b11111111111:2047
0b100000000000:2048
0b100000000001:2049
0b100000000010:2050

b11111111110:NaN
b11111111111:NaN
b100000000000:NaN
b100000000001:NaN
b100000000010:NaN

11111111110:2046
11111111111:2047
100000000000:2048
100000000001:2049
100000000010:2050

# Positive

+0b11111111110:2046
+0b11111111111:2047
+0b100000000000:2048
+0b100000000001:2049
+0b100000000010:2050

+b11111111110:NaN
+b11111111111:NaN
+b100000000000:NaN
+b100000000001:NaN
+b100000000010:NaN

+11111111110:2046
+11111111111:2047
+100000000000:2048
+100000000001:2049
+100000000010:2050

# Negative

-0b11111111110:-2046
-0b11111111111:-2047
-0b100000000000:-2048
-0b100000000001:-2049
-0b100000000010:-2050

-b11111111110:NaN
-b11111111111:NaN
-b100000000000:NaN
-b100000000001:NaN
-b100000000010:NaN

-11111111110:-2046
-11111111111:-2047
-100000000000:-2048
-100000000001:-2049
-100000000010:-2050

###############################################################################
# Absolute value is close to 2^12 = 4096
###############################################################################

# Unsigned

0b111111111110:4094
0b111111111111:4095
0b1000000000000:4096
0b1000000000001:4097
0b1000000000010:4098

b111111111110:NaN
b111111111111:NaN
b1000000000000:NaN
b1000000000001:NaN
b1000000000010:NaN

111111111110:4094
111111111111:4095
1000000000000:4096
1000000000001:4097
1000000000010:4098

# Positive

+0b111111111110:4094
+0b111111111111:4095
+0b1000000000000:4096
+0b1000000000001:4097
+0b1000000000010:4098

+b111111111110:NaN
+b111111111111:NaN
+b1000000000000:NaN
+b1000000000001:NaN
+b1000000000010:NaN

+111111111110:4094
+111111111111:4095
+1000000000000:4096
+1000000000001:4097
+1000000000010:4098

# Negative

-0b111111111110:-4094
-0b111111111111:-4095
-0b1000000000000:-4096
-0b1000000000001:-4097
-0b1000000000010:-4098

-b111111111110:NaN
-b111111111111:NaN
-b1000000000000:NaN
-b1000000000001:NaN
-b1000000000010:NaN

-111111111110:-4094
-111111111111:-4095
-1000000000000:-4096
-1000000000001:-4097
-1000000000010:-4098

###############################################################################
# Absolute value is close to 2^13 = 8192
###############################################################################

# Unsigned

0b1111111111110:8190
0b1111111111111:8191
0b10000000000000:8192
0b10000000000001:8193
0b10000000000010:8194

b1111111111110:NaN
b1111111111111:NaN
b10000000000000:NaN
b10000000000001:NaN
b10000000000010:NaN

1111111111110:8190
1111111111111:8191
10000000000000:8192
10000000000001:8193
10000000000010:8194

# Positive

+0b1111111111110:8190
+0b1111111111111:8191
+0b10000000000000:8192
+0b10000000000001:8193
+0b10000000000010:8194

+b1111111111110:NaN
+b1111111111111:NaN
+b10000000000000:NaN
+b10000000000001:NaN
+b10000000000010:NaN

+1111111111110:8190
+1111111111111:8191
+10000000000000:8192
+10000000000001:8193
+10000000000010:8194

# Negative

-0b1111111111110:-8190
-0b1111111111111:-8191
-0b10000000000000:-8192
-0b10000000000001:-8193
-0b10000000000010:-8194

-b1111111111110:NaN
-b1111111111111:NaN
-b10000000000000:NaN
-b10000000000001:NaN
-b10000000000010:NaN

-1111111111110:-8190
-1111111111111:-8191
-10000000000000:-8192
-10000000000001:-8193
-10000000000010:-8194

###############################################################################
# Absolute value is close to 2^14 = 16384
###############################################################################

# Unsigned

0b11111111111110:16382
0b11111111111111:16383
0b100000000000000:16384
0b100000000000001:16385
0b100000000000010:16386

b11111111111110:NaN
b11111111111111:NaN
b100000000000000:NaN
b100000000000001:NaN
b100000000000010:NaN

11111111111110:16382
11111111111111:16383
100000000000000:16384
100000000000001:16385
100000000000010:16386

# Positive

+0b11111111111110:16382
+0b11111111111111:16383
+0b100000000000000:16384
+0b100000000000001:16385
+0b100000000000010:16386

+b11111111111110:NaN
+b11111111111111:NaN
+b100000000000000:NaN
+b100000000000001:NaN
+b100000000000010:NaN

+11111111111110:16382
+11111111111111:16383
+100000000000000:16384
+100000000000001:16385
+100000000000010:16386

# Negative

-0b11111111111110:-16382
-0b11111111111111:-16383
-0b100000000000000:-16384
-0b100000000000001:-16385
-0b100000000000010:-16386

-b11111111111110:NaN
-b11111111111111:NaN
-b100000000000000:NaN
-b100000000000001:NaN
-b100000000000010:NaN

-11111111111110:-16382
-11111111111111:-16383
-100000000000000:-16384
-100000000000001:-16385
-100000000000010:-16386

###############################################################################
# Absolute value is close to 2^15 = 32768
###############################################################################

# Unsigned

0b111111111111110:32766
0b111111111111111:32767
0b1000000000000000:32768
0b1000000000000001:32769
0b1000000000000010:32770

b111111111111110:NaN
b111111111111111:NaN
b1000000000000000:NaN
b1000000000000001:NaN
b1000000000000010:NaN

111111111111110:32766
111111111111111:32767
1000000000000000:32768
1000000000000001:32769
1000000000000010:32770

# Positive

+0b111111111111110:32766
+0b111111111111111:32767
+0b1000000000000000:32768
+0b1000000000000001:32769
+0b1000000000000010:32770

+b111111111111110:NaN
+b111111111111111:NaN
+b1000000000000000:NaN
+b1000000000000001:NaN
+b1000000000000010:NaN

+111111111111110:32766
+111111111111111:32767
+1000000000000000:32768
+1000000000000001:32769
+1000000000000010:32770

# Negative

-0b111111111111110:-32766
-0b111111111111111:-32767
-0b1000000000000000:-32768
-0b1000000000000001:-32769
-0b1000000000000010:-32770

-b111111111111110:NaN
-b111111111111111:NaN
-b1000000000000000:NaN
-b1000000000000001:NaN
-b1000000000000010:NaN

-111111111111110:-32766
-111111111111111:-32767
-1000000000000000:-32768
-1000000000000001:-32769
-1000000000000010:-32770

###############################################################################
# Absolute value is close to 2^16 = 65536
###############################################################################

# Unsigned

0b1111111111111110:65534
0b1111111111111111:65535
0b10000000000000000:65536
0b10000000000000001:65537
0b10000000000000010:65538

b1111111111111110:NaN
b1111111111111111:NaN
b10000000000000000:NaN
b10000000000000001:NaN
b10000000000000010:NaN

1111111111111110:65534
1111111111111111:65535
10000000000000000:65536
10000000000000001:65537
10000000000000010:65538

# Positive

+0b1111111111111110:65534
+0b1111111111111111:65535
+0b10000000000000000:65536
+0b10000000000000001:65537
+0b10000000000000010:65538

+b1111111111111110:NaN
+b1111111111111111:NaN
+b10000000000000000:NaN
+b10000000000000001:NaN
+b10000000000000010:NaN

+1111111111111110:65534
+1111111111111111:65535
+10000000000000000:65536
+10000000000000001:65537
+10000000000000010:65538

# Negative

-0b1111111111111110:-65534
-0b1111111111111111:-65535
-0b10000000000000000:-65536
-0b10000000000000001:-65537
-0b10000000000000010:-65538

-b1111111111111110:NaN
-b1111111111111111:NaN
-b10000000000000000:NaN
-b10000000000000001:NaN
-b10000000000000010:NaN

-1111111111111110:-65534
-1111111111111111:-65535
-10000000000000000:-65536
-10000000000000001:-65537
-10000000000000010:-65538

###############################################################################
# Absolute value is close to 2^17 = 131072
###############################################################################

# Unsigned

0b11111111111111110:131070
0b11111111111111111:131071
0b100000000000000000:131072
0b100000000000000001:131073
0b100000000000000010:131074

b11111111111111110:NaN
b11111111111111111:NaN
b100000000000000000:NaN
b100000000000000001:NaN
b100000000000000010:NaN

11111111111111110:131070
11111111111111111:131071
100000000000000000:131072
100000000000000001:131073
100000000000000010:131074

# Positive

+0b11111111111111110:131070
+0b11111111111111111:131071
+0b100000000000000000:131072
+0b100000000000000001:131073
+0b100000000000000010:131074

+b11111111111111110:NaN
+b11111111111111111:NaN
+b100000000000000000:NaN
+b100000000000000001:NaN
+b100000000000000010:NaN

+11111111111111110:131070
+11111111111111111:131071
+100000000000000000:131072
+100000000000000001:131073
+100000000000000010:131074

# Negative

-0b11111111111111110:-131070
-0b11111111111111111:-131071
-0b100000000000000000:-131072
-0b100000000000000001:-131073
-0b100000000000000010:-131074

-b11111111111111110:NaN
-b11111111111111111:NaN
-b100000000000000000:NaN
-b100000000000000001:NaN
-b100000000000000010:NaN

-11111111111111110:-131070
-11111111111111111:-131071
-100000000000000000:-131072
-100000000000000001:-131073
-100000000000000010:-131074

###############################################################################
# Absolute value is close to 2^18 = 262144
###############################################################################

# Unsigned

0b111111111111111110:262142
0b111111111111111111:262143
0b1000000000000000000:262144
0b1000000000000000001:262145
0b1000000000000000010:262146

b111111111111111110:NaN
b111111111111111111:NaN
b1000000000000000000:NaN
b1000000000000000001:NaN
b1000000000000000010:NaN

111111111111111110:262142
111111111111111111:262143
1000000000000000000:262144
1000000000000000001:262145
1000000000000000010:262146

# Positive

+0b111111111111111110:262142
+0b111111111111111111:262143
+0b1000000000000000000:262144
+0b1000000000000000001:262145
+0b1000000000000000010:262146

+b111111111111111110:NaN
+b111111111111111111:NaN
+b1000000000000000000:NaN
+b1000000000000000001:NaN
+b1000000000000000010:NaN

+111111111111111110:262142
+111111111111111111:262143
+1000000000000000000:262144
+1000000000000000001:262145
+1000000000000000010:262146

# Negative

-0b111111111111111110:-262142
-0b111111111111111111:-262143
-0b1000000000000000000:-262144
-0b1000000000000000001:-262145
-0b1000000000000000010:-262146

-b111111111111111110:NaN
-b111111111111111111:NaN
-b1000000000000000000:NaN
-b1000000000000000001:NaN
-b1000000000000000010:NaN

-111111111111111110:-262142
-111111111111111111:-262143
-1000000000000000000:-262144
-1000000000000000001:-262145
-1000000000000000010:-262146

###############################################################################
# Absolute value is close to 2^19 = 524288
###############################################################################

# Unsigned

0b1111111111111111110:524286
0b1111111111111111111:524287
0b10000000000000000000:524288
0b10000000000000000001:524289
0b10000000000000000010:524290

b1111111111111111110:NaN
b1111111111111111111:NaN
b10000000000000000000:NaN
b10000000000000000001:NaN
b10000000000000000010:NaN

1111111111111111110:524286
1111111111111111111:524287
10000000000000000000:524288
10000000000000000001:524289
10000000000000000010:524290

# Positive

+0b1111111111111111110:524286
+0b1111111111111111111:524287
+0b10000000000000000000:524288
+0b10000000000000000001:524289
+0b10000000000000000010:524290

+b1111111111111111110:NaN
+b1111111111111111111:NaN
+b10000000000000000000:NaN
+b10000000000000000001:NaN
+b10000000000000000010:NaN

+1111111111111111110:524286
+1111111111111111111:524287
+10000000000000000000:524288
+10000000000000000001:524289
+10000000000000000010:524290

# Negative

-0b1111111111111111110:-524286
-0b1111111111111111111:-524287
-0b10000000000000000000:-524288
-0b10000000000000000001:-524289
-0b10000000000000000010:-524290

-b1111111111111111110:NaN
-b1111111111111111111:NaN
-b10000000000000000000:NaN
-b10000000000000000001:NaN
-b10000000000000000010:NaN

-1111111111111111110:-524286
-1111111111111111111:-524287
-10000000000000000000:-524288
-10000000000000000001:-524289
-10000000000000000010:-524290

###############################################################################
# Absolute value is close to 2^20 = 1048576
###############################################################################

# Unsigned

0b11111111111111111110:1048574
0b11111111111111111111:1048575
0b100000000000000000000:1048576
0b100000000000000000001:1048577
0b100000000000000000010:1048578

b11111111111111111110:NaN
b11111111111111111111:NaN
b100000000000000000000:NaN
b100000000000000000001:NaN
b100000000000000000010:NaN

11111111111111111110:1048574
11111111111111111111:1048575
100000000000000000000:1048576
100000000000000000001:1048577
100000000000000000010:1048578

# Positive

+0b11111111111111111110:1048574
+0b11111111111111111111:1048575
+0b100000000000000000000:1048576
+0b100000000000000000001:1048577
+0b100000000000000000010:1048578

+b11111111111111111110:NaN
+b11111111111111111111:NaN
+b100000000000000000000:NaN
+b100000000000000000001:NaN
+b100000000000000000010:NaN

+11111111111111111110:1048574
+11111111111111111111:1048575
+100000000000000000000:1048576
+100000000000000000001:1048577
+100000000000000000010:1048578

# Negative

-0b11111111111111111110:-1048574
-0b11111111111111111111:-1048575
-0b100000000000000000000:-1048576
-0b100000000000000000001:-1048577
-0b100000000000000000010:-1048578

-b11111111111111111110:NaN
-b11111111111111111111:NaN
-b100000000000000000000:NaN
-b100000000000000000001:NaN
-b100000000000000000010:NaN

-11111111111111111110:-1048574
-11111111111111111111:-1048575
-100000000000000000000:-1048576
-100000000000000000001:-1048577
-100000000000000000010:-1048578

###############################################################################
# Absolute value is close to 2^21 = 2097152
###############################################################################

# Unsigned

0b111111111111111111110:2097150
0b111111111111111111111:2097151
0b1000000000000000000000:2097152
0b1000000000000000000001:2097153
0b1000000000000000000010:2097154

b111111111111111111110:NaN
b111111111111111111111:NaN
b1000000000000000000000:NaN
b1000000000000000000001:NaN
b1000000000000000000010:NaN

111111111111111111110:2097150
111111111111111111111:2097151
1000000000000000000000:2097152
1000000000000000000001:2097153
1000000000000000000010:2097154

# Positive

+0b111111111111111111110:2097150
+0b111111111111111111111:2097151
+0b1000000000000000000000:2097152
+0b1000000000000000000001:2097153
+0b1000000000000000000010:2097154

+b111111111111111111110:NaN
+b111111111111111111111:NaN
+b1000000000000000000000:NaN
+b1000000000000000000001:NaN
+b1000000000000000000010:NaN

+111111111111111111110:2097150
+111111111111111111111:2097151
+1000000000000000000000:2097152
+1000000000000000000001:2097153
+1000000000000000000010:2097154

# Negative

-0b111111111111111111110:-2097150
-0b111111111111111111111:-2097151
-0b1000000000000000000000:-2097152
-0b1000000000000000000001:-2097153
-0b1000000000000000000010:-2097154

-b111111111111111111110:NaN
-b111111111111111111111:NaN
-b1000000000000000000000:NaN
-b1000000000000000000001:NaN
-b1000000000000000000010:NaN

-111111111111111111110:-2097150
-111111111111111111111:-2097151
-1000000000000000000000:-2097152
-1000000000000000000001:-2097153
-1000000000000000000010:-2097154

###############################################################################
# Absolute value is close to 2^22 = 4194304
###############################################################################

# Unsigned

0b1111111111111111111110:4194302
0b1111111111111111111111:4194303
0b10000000000000000000000:4194304
0b10000000000000000000001:4194305
0b10000000000000000000010:4194306

b1111111111111111111110:NaN
b1111111111111111111111:NaN
b10000000000000000000000:NaN
b10000000000000000000001:NaN
b10000000000000000000010:NaN

1111111111111111111110:4194302
1111111111111111111111:4194303
10000000000000000000000:4194304
10000000000000000000001:4194305
10000000000000000000010:4194306

# Positive

+0b1111111111111111111110:4194302
+0b1111111111111111111111:4194303
+0b10000000000000000000000:4194304
+0b10000000000000000000001:4194305
+0b10000000000000000000010:4194306

+b1111111111111111111110:NaN
+b1111111111111111111111:NaN
+b10000000000000000000000:NaN
+b10000000000000000000001:NaN
+b10000000000000000000010:NaN

+1111111111111111111110:4194302
+1111111111111111111111:4194303
+10000000000000000000000:4194304
+10000000000000000000001:4194305
+10000000000000000000010:4194306

# Negative

-0b1111111111111111111110:-4194302
-0b1111111111111111111111:-4194303
-0b10000000000000000000000:-4194304
-0b10000000000000000000001:-4194305
-0b10000000000000000000010:-4194306

-b1111111111111111111110:NaN
-b1111111111111111111111:NaN
-b10000000000000000000000:NaN
-b10000000000000000000001:NaN
-b10000000000000000000010:NaN

-1111111111111111111110:-4194302
-1111111111111111111111:-4194303
-10000000000000000000000:-4194304
-10000000000000000000001:-4194305
-10000000000000000000010:-4194306

###############################################################################
# Absolute value is close to 2^23 = 8388608
###############################################################################

# Unsigned

0b11111111111111111111110:8388606
0b11111111111111111111111:8388607
0b100000000000000000000000:8388608
0b100000000000000000000001:8388609
0b100000000000000000000010:8388610

b11111111111111111111110:NaN
b11111111111111111111111:NaN
b100000000000000000000000:NaN
b100000000000000000000001:NaN
b100000000000000000000010:NaN

11111111111111111111110:8388606
11111111111111111111111:8388607
100000000000000000000000:8388608
100000000000000000000001:8388609
100000000000000000000010:8388610

# Positive

+0b11111111111111111111110:8388606
+0b11111111111111111111111:8388607
+0b100000000000000000000000:8388608
+0b100000000000000000000001:8388609
+0b100000000000000000000010:8388610

+b11111111111111111111110:NaN
+b11111111111111111111111:NaN
+b100000000000000000000000:NaN
+b100000000000000000000001:NaN
+b100000000000000000000010:NaN

+11111111111111111111110:8388606
+11111111111111111111111:8388607
+100000000000000000000000:8388608
+100000000000000000000001:8388609
+100000000000000000000010:8388610

# Negative

-0b11111111111111111111110:-8388606
-0b11111111111111111111111:-8388607
-0b100000000000000000000000:-8388608
-0b100000000000000000000001:-8388609
-0b100000000000000000000010:-8388610

-b11111111111111111111110:NaN
-b11111111111111111111111:NaN
-b100000000000000000000000:NaN
-b100000000000000000000001:NaN
-b100000000000000000000010:NaN

-11111111111111111111110:-8388606
-11111111111111111111111:-8388607
-100000000000000000000000:-8388608
-100000000000000000000001:-8388609
-100000000000000000000010:-8388610

###############################################################################
# Absolute value is close to 2^24 = 16777216
###############################################################################

# Unsigned

0b111111111111111111111110:16777214
0b111111111111111111111111:16777215
0b1000000000000000000000000:16777216
0b1000000000000000000000001:16777217
0b1000000000000000000000010:16777218

b111111111111111111111110:NaN
b111111111111111111111111:NaN
b1000000000000000000000000:NaN
b1000000000000000000000001:NaN
b1000000000000000000000010:NaN

111111111111111111111110:16777214
111111111111111111111111:16777215
1000000000000000000000000:16777216
1000000000000000000000001:16777217
1000000000000000000000010:16777218

# Positive

+0b111111111111111111111110:16777214
+0b111111111111111111111111:16777215
+0b1000000000000000000000000:16777216
+0b1000000000000000000000001:16777217
+0b1000000000000000000000010:16777218

+b111111111111111111111110:NaN
+b111111111111111111111111:NaN
+b1000000000000000000000000:NaN
+b1000000000000000000000001:NaN
+b1000000000000000000000010:NaN

+111111111111111111111110:16777214
+111111111111111111111111:16777215
+1000000000000000000000000:16777216
+1000000000000000000000001:16777217
+1000000000000000000000010:16777218

# Negative

-0b111111111111111111111110:-16777214
-0b111111111111111111111111:-16777215
-0b1000000000000000000000000:-16777216
-0b1000000000000000000000001:-16777217
-0b1000000000000000000000010:-16777218

-b111111111111111111111110:NaN
-b111111111111111111111111:NaN
-b1000000000000000000000000:NaN
-b1000000000000000000000001:NaN
-b1000000000000000000000010:NaN

-111111111111111111111110:-16777214
-111111111111111111111111:-16777215
-1000000000000000000000000:-16777216
-1000000000000000000000001:-16777217
-1000000000000000000000010:-16777218

###############################################################################
# Absolute value is close to 2^25 = 33554432
###############################################################################

# Unsigned

0b1111111111111111111111110:33554430
0b1111111111111111111111111:33554431
0b10000000000000000000000000:33554432
0b10000000000000000000000001:33554433
0b10000000000000000000000010:33554434

b1111111111111111111111110:NaN
b1111111111111111111111111:NaN
b10000000000000000000000000:NaN
b10000000000000000000000001:NaN
b10000000000000000000000010:NaN

1111111111111111111111110:33554430
1111111111111111111111111:33554431
10000000000000000000000000:33554432
10000000000000000000000001:33554433
10000000000000000000000010:33554434

# Positive

+0b1111111111111111111111110:33554430
+0b1111111111111111111111111:33554431
+0b10000000000000000000000000:33554432
+0b10000000000000000000000001:33554433
+0b10000000000000000000000010:33554434

+b1111111111111111111111110:NaN
+b1111111111111111111111111:NaN
+b10000000000000000000000000:NaN
+b10000000000000000000000001:NaN
+b10000000000000000000000010:NaN

+1111111111111111111111110:33554430
+1111111111111111111111111:33554431
+10000000000000000000000000:33554432
+10000000000000000000000001:33554433
+10000000000000000000000010:33554434

# Negative

-0b1111111111111111111111110:-33554430
-0b1111111111111111111111111:-33554431
-0b10000000000000000000000000:-33554432
-0b10000000000000000000000001:-33554433
-0b10000000000000000000000010:-33554434

-b1111111111111111111111110:NaN
-b1111111111111111111111111:NaN
-b10000000000000000000000000:NaN
-b10000000000000000000000001:NaN
-b10000000000000000000000010:NaN

-1111111111111111111111110:-33554430
-1111111111111111111111111:-33554431
-10000000000000000000000000:-33554432
-10000000000000000000000001:-33554433
-10000000000000000000000010:-33554434

###############################################################################
# Absolute value is close to 2^26 = 67108864
###############################################################################

# Unsigned

0b11111111111111111111111110:67108862
0b11111111111111111111111111:67108863
0b100000000000000000000000000:67108864
0b100000000000000000000000001:67108865
0b100000000000000000000000010:67108866

b11111111111111111111111110:NaN
b11111111111111111111111111:NaN
b100000000000000000000000000:NaN
b100000000000000000000000001:NaN
b100000000000000000000000010:NaN

11111111111111111111111110:67108862
11111111111111111111111111:67108863
100000000000000000000000000:67108864
100000000000000000000000001:67108865
100000000000000000000000010:67108866

# Positive

+0b11111111111111111111111110:67108862
+0b11111111111111111111111111:67108863
+0b100000000000000000000000000:67108864
+0b100000000000000000000000001:67108865
+0b100000000000000000000000010:67108866

+b11111111111111111111111110:NaN
+b11111111111111111111111111:NaN
+b100000000000000000000000000:NaN
+b100000000000000000000000001:NaN
+b100000000000000000000000010:NaN

+11111111111111111111111110:67108862
+11111111111111111111111111:67108863
+100000000000000000000000000:67108864
+100000000000000000000000001:67108865
+100000000000000000000000010:67108866

# Negative

-0b11111111111111111111111110:-67108862
-0b11111111111111111111111111:-67108863
-0b100000000000000000000000000:-67108864
-0b100000000000000000000000001:-67108865
-0b100000000000000000000000010:-67108866

-b11111111111111111111111110:NaN
-b11111111111111111111111111:NaN
-b100000000000000000000000000:NaN
-b100000000000000000000000001:NaN
-b100000000000000000000000010:NaN

-11111111111111111111111110:-67108862
-11111111111111111111111111:-67108863
-100000000000000000000000000:-67108864
-100000000000000000000000001:-67108865
-100000000000000000000000010:-67108866

###############################################################################
# Absolute value is close to 2^27 = 134217728
###############################################################################

# Unsigned

0b111111111111111111111111110:134217726
0b111111111111111111111111111:134217727
0b1000000000000000000000000000:134217728
0b1000000000000000000000000001:134217729
0b1000000000000000000000000010:134217730

b111111111111111111111111110:NaN
b111111111111111111111111111:NaN
b1000000000000000000000000000:NaN
b1000000000000000000000000001:NaN
b1000000000000000000000000010:NaN

111111111111111111111111110:134217726
111111111111111111111111111:134217727
1000000000000000000000000000:134217728
1000000000000000000000000001:134217729
1000000000000000000000000010:134217730

# Positive

+0b111111111111111111111111110:134217726
+0b111111111111111111111111111:134217727
+0b1000000000000000000000000000:134217728
+0b1000000000000000000000000001:134217729
+0b1000000000000000000000000010:134217730

+b111111111111111111111111110:NaN
+b111111111111111111111111111:NaN
+b1000000000000000000000000000:NaN
+b1000000000000000000000000001:NaN
+b1000000000000000000000000010:NaN

+111111111111111111111111110:134217726
+111111111111111111111111111:134217727
+1000000000000000000000000000:134217728
+1000000000000000000000000001:134217729
+1000000000000000000000000010:134217730

# Negative

-0b111111111111111111111111110:-134217726
-0b111111111111111111111111111:-134217727
-0b1000000000000000000000000000:-134217728
-0b1000000000000000000000000001:-134217729
-0b1000000000000000000000000010:-134217730

-b111111111111111111111111110:NaN
-b111111111111111111111111111:NaN
-b1000000000000000000000000000:NaN
-b1000000000000000000000000001:NaN
-b1000000000000000000000000010:NaN

-111111111111111111111111110:-134217726
-111111111111111111111111111:-134217727
-1000000000000000000000000000:-134217728
-1000000000000000000000000001:-134217729
-1000000000000000000000000010:-134217730

###############################################################################
# Absolute value is close to 2^28 = 268435456
###############################################################################

# Unsigned

0b1111111111111111111111111110:268435454
0b1111111111111111111111111111:268435455
0b10000000000000000000000000000:268435456
0b10000000000000000000000000001:268435457
0b10000000000000000000000000010:268435458

b1111111111111111111111111110:NaN
b1111111111111111111111111111:NaN
b10000000000000000000000000000:NaN
b10000000000000000000000000001:NaN
b10000000000000000000000000010:NaN

1111111111111111111111111110:268435454
1111111111111111111111111111:268435455
10000000000000000000000000000:268435456
10000000000000000000000000001:268435457
10000000000000000000000000010:268435458

# Positive

+0b1111111111111111111111111110:268435454
+0b1111111111111111111111111111:268435455
+0b10000000000000000000000000000:268435456
+0b10000000000000000000000000001:268435457
+0b10000000000000000000000000010:268435458

+b1111111111111111111111111110:NaN
+b1111111111111111111111111111:NaN
+b10000000000000000000000000000:NaN
+b10000000000000000000000000001:NaN
+b10000000000000000000000000010:NaN

+1111111111111111111111111110:268435454
+1111111111111111111111111111:268435455
+10000000000000000000000000000:268435456
+10000000000000000000000000001:268435457
+10000000000000000000000000010:268435458

# Negative

-0b1111111111111111111111111110:-268435454
-0b1111111111111111111111111111:-268435455
-0b10000000000000000000000000000:-268435456
-0b10000000000000000000000000001:-268435457
-0b10000000000000000000000000010:-268435458

-b1111111111111111111111111110:NaN
-b1111111111111111111111111111:NaN
-b10000000000000000000000000000:NaN
-b10000000000000000000000000001:NaN
-b10000000000000000000000000010:NaN

-1111111111111111111111111110:-268435454
-1111111111111111111111111111:-268435455
-10000000000000000000000000000:-268435456
-10000000000000000000000000001:-268435457
-10000000000000000000000000010:-268435458

###############################################################################
# Absolute value is close to 2^29 = 536870912
###############################################################################

# Unsigned

0b11111111111111111111111111110:536870910
0b11111111111111111111111111111:536870911
0b100000000000000000000000000000:536870912
0b100000000000000000000000000001:536870913
0b100000000000000000000000000010:536870914

b11111111111111111111111111110:NaN
b11111111111111111111111111111:NaN
b100000000000000000000000000000:NaN
b100000000000000000000000000001:NaN
b100000000000000000000000000010:NaN

11111111111111111111111111110:536870910
11111111111111111111111111111:536870911
100000000000000000000000000000:536870912
100000000000000000000000000001:536870913
100000000000000000000000000010:536870914

# Positive

+0b11111111111111111111111111110:536870910
+0b11111111111111111111111111111:536870911
+0b100000000000000000000000000000:536870912
+0b100000000000000000000000000001:536870913
+0b100000000000000000000000000010:536870914

+b11111111111111111111111111110:NaN
+b11111111111111111111111111111:NaN
+b100000000000000000000000000000:NaN
+b100000000000000000000000000001:NaN
+b100000000000000000000000000010:NaN

+11111111111111111111111111110:536870910
+11111111111111111111111111111:536870911
+100000000000000000000000000000:536870912
+100000000000000000000000000001:536870913
+100000000000000000000000000010:536870914

# Negative

-0b11111111111111111111111111110:-536870910
-0b11111111111111111111111111111:-536870911
-0b100000000000000000000000000000:-536870912
-0b100000000000000000000000000001:-536870913
-0b100000000000000000000000000010:-536870914

-b11111111111111111111111111110:NaN
-b11111111111111111111111111111:NaN
-b100000000000000000000000000000:NaN
-b100000000000000000000000000001:NaN
-b100000000000000000000000000010:NaN

-11111111111111111111111111110:-536870910
-11111111111111111111111111111:-536870911
-100000000000000000000000000000:-536870912
-100000000000000000000000000001:-536870913
-100000000000000000000000000010:-536870914

###############################################################################
# Absolute value is close to 2^30 = 1073741824
###############################################################################

# Unsigned

0b111111111111111111111111111110:1073741822
0b111111111111111111111111111111:1073741823
0b1000000000000000000000000000000:1073741824
0b1000000000000000000000000000001:1073741825
0b1000000000000000000000000000010:1073741826

b111111111111111111111111111110:NaN
b111111111111111111111111111111:NaN
b1000000000000000000000000000000:NaN
b1000000000000000000000000000001:NaN
b1000000000000000000000000000010:NaN

111111111111111111111111111110:1073741822
111111111111111111111111111111:1073741823
1000000000000000000000000000000:1073741824
1000000000000000000000000000001:1073741825
1000000000000000000000000000010:1073741826

# Positive

+0b111111111111111111111111111110:1073741822
+0b111111111111111111111111111111:1073741823
+0b1000000000000000000000000000000:1073741824
+0b1000000000000000000000000000001:1073741825
+0b1000000000000000000000000000010:1073741826

+b111111111111111111111111111110:NaN
+b111111111111111111111111111111:NaN
+b1000000000000000000000000000000:NaN
+b1000000000000000000000000000001:NaN
+b1000000000000000000000000000010:NaN

+111111111111111111111111111110:1073741822
+111111111111111111111111111111:1073741823
+1000000000000000000000000000000:1073741824
+1000000000000000000000000000001:1073741825
+1000000000000000000000000000010:1073741826

# Negative

-0b111111111111111111111111111110:-1073741822
-0b111111111111111111111111111111:-1073741823
-0b1000000000000000000000000000000:-1073741824
-0b1000000000000000000000000000001:-1073741825
-0b1000000000000000000000000000010:-1073741826

-b111111111111111111111111111110:NaN
-b111111111111111111111111111111:NaN
-b1000000000000000000000000000000:NaN
-b1000000000000000000000000000001:NaN
-b1000000000000000000000000000010:NaN

-111111111111111111111111111110:-1073741822
-111111111111111111111111111111:-1073741823
-1000000000000000000000000000000:-1073741824
-1000000000000000000000000000001:-1073741825
-1000000000000000000000000000010:-1073741826

###############################################################################
# Absolute value is close to 2^31 = 2147483648
###############################################################################

# Unsigned

0b1111111111111111111111111111110:2147483646
0b1111111111111111111111111111111:2147483647
0b10000000000000000000000000000000:2147483648
0b10000000000000000000000000000001:2147483649
0b10000000000000000000000000000010:2147483650

b1111111111111111111111111111110:NaN
b1111111111111111111111111111111:NaN
b10000000000000000000000000000000:NaN
b10000000000000000000000000000001:NaN
b10000000000000000000000000000010:NaN

1111111111111111111111111111110:2147483646
1111111111111111111111111111111:2147483647
10000000000000000000000000000000:2147483648
10000000000000000000000000000001:2147483649
10000000000000000000000000000010:2147483650

# Positive

+0b1111111111111111111111111111110:2147483646
+0b1111111111111111111111111111111:2147483647
+0b10000000000000000000000000000000:2147483648
+0b10000000000000000000000000000001:2147483649
+0b10000000000000000000000000000010:2147483650

+b1111111111111111111111111111110:NaN
+b1111111111111111111111111111111:NaN
+b10000000000000000000000000000000:NaN
+b10000000000000000000000000000001:NaN
+b10000000000000000000000000000010:NaN

+1111111111111111111111111111110:2147483646
+1111111111111111111111111111111:2147483647
+10000000000000000000000000000000:2147483648
+10000000000000000000000000000001:2147483649
+10000000000000000000000000000010:2147483650

# Negative

-0b1111111111111111111111111111110:-2147483646
-0b1111111111111111111111111111111:-2147483647
-0b10000000000000000000000000000000:-2147483648
-0b10000000000000000000000000000001:-2147483649
-0b10000000000000000000000000000010:-2147483650

-b1111111111111111111111111111110:NaN
-b1111111111111111111111111111111:NaN
-b10000000000000000000000000000000:NaN
-b10000000000000000000000000000001:NaN
-b10000000000000000000000000000010:NaN

-1111111111111111111111111111110:-2147483646
-1111111111111111111111111111111:-2147483647
-10000000000000000000000000000000:-2147483648
-10000000000000000000000000000001:-2147483649
-10000000000000000000000000000010:-2147483650

###############################################################################
# Absolute value is close to 2^32 = 4294967296
###############################################################################

# Unsigned

0b11111111111111111111111111111110:4294967294
0b11111111111111111111111111111111:4294967295
0b100000000000000000000000000000000:4294967296
0b100000000000000000000000000000001:4294967297
0b100000000000000000000000000000010:4294967298

b11111111111111111111111111111110:NaN
b11111111111111111111111111111111:NaN
b100000000000000000000000000000000:NaN
b100000000000000000000000000000001:NaN
b100000000000000000000000000000010:NaN

11111111111111111111111111111110:4294967294
11111111111111111111111111111111:4294967295
100000000000000000000000000000000:4294967296
100000000000000000000000000000001:4294967297
100000000000000000000000000000010:4294967298

# Positive

+0b11111111111111111111111111111110:4294967294
+0b11111111111111111111111111111111:4294967295
+0b100000000000000000000000000000000:4294967296
+0b100000000000000000000000000000001:4294967297
+0b100000000000000000000000000000010:4294967298

+b11111111111111111111111111111110:NaN
+b11111111111111111111111111111111:NaN
+b100000000000000000000000000000000:NaN
+b100000000000000000000000000000001:NaN
+b100000000000000000000000000000010:NaN

+11111111111111111111111111111110:4294967294
+11111111111111111111111111111111:4294967295
+100000000000000000000000000000000:4294967296
+100000000000000000000000000000001:4294967297
+100000000000000000000000000000010:4294967298

# Negative

-0b11111111111111111111111111111110:-4294967294
-0b11111111111111111111111111111111:-4294967295
-0b100000000000000000000000000000000:-4294967296
-0b100000000000000000000000000000001:-4294967297
-0b100000000000000000000000000000010:-4294967298

-b11111111111111111111111111111110:NaN
-b11111111111111111111111111111111:NaN
-b100000000000000000000000000000000:NaN
-b100000000000000000000000000000001:NaN
-b100000000000000000000000000000010:NaN

-11111111111111111111111111111110:-4294967294
-11111111111111111111111111111111:-4294967295
-100000000000000000000000000000000:-4294967296
-100000000000000000000000000000001:-4294967297
-100000000000000000000000000000010:-4294967298

###############################################################################
# Absolute value is close to 2^33 = 8589934592
###############################################################################

# Unsigned

0b111111111111111111111111111111110:8589934590
0b111111111111111111111111111111111:8589934591
0b1000000000000000000000000000000000:8589934592
0b1000000000000000000000000000000001:8589934593
0b1000000000000000000000000000000010:8589934594

b111111111111111111111111111111110:NaN
b111111111111111111111111111111111:NaN
b1000000000000000000000000000000000:NaN
b1000000000000000000000000000000001:NaN
b1000000000000000000000000000000010:NaN

111111111111111111111111111111110:8589934590
111111111111111111111111111111111:8589934591
1000000000000000000000000000000000:8589934592
1000000000000000000000000000000001:8589934593
1000000000000000000000000000000010:8589934594

# Positive

+0b111111111111111111111111111111110:8589934590
+0b111111111111111111111111111111111:8589934591
+0b1000000000000000000000000000000000:8589934592
+0b1000000000000000000000000000000001:8589934593
+0b1000000000000000000000000000000010:8589934594

+b111111111111111111111111111111110:NaN
+b111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000:NaN
+b1000000000000000000000000000000001:NaN
+b1000000000000000000000000000000010:NaN

+111111111111111111111111111111110:8589934590
+111111111111111111111111111111111:8589934591
+1000000000000000000000000000000000:8589934592
+1000000000000000000000000000000001:8589934593
+1000000000000000000000000000000010:8589934594

# Negative

-0b111111111111111111111111111111110:-8589934590
-0b111111111111111111111111111111111:-8589934591
-0b1000000000000000000000000000000000:-8589934592
-0b1000000000000000000000000000000001:-8589934593
-0b1000000000000000000000000000000010:-8589934594

-b111111111111111111111111111111110:NaN
-b111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000:NaN
-b1000000000000000000000000000000001:NaN
-b1000000000000000000000000000000010:NaN

-111111111111111111111111111111110:-8589934590
-111111111111111111111111111111111:-8589934591
-1000000000000000000000000000000000:-8589934592
-1000000000000000000000000000000001:-8589934593
-1000000000000000000000000000000010:-8589934594

###############################################################################
# Absolute value is close to 2^34 = 17179869184
###############################################################################

# Unsigned

0b1111111111111111111111111111111110:17179869182
0b1111111111111111111111111111111111:17179869183
0b10000000000000000000000000000000000:17179869184
0b10000000000000000000000000000000001:17179869185
0b10000000000000000000000000000000010:17179869186

b1111111111111111111111111111111110:NaN
b1111111111111111111111111111111111:NaN
b10000000000000000000000000000000000:NaN
b10000000000000000000000000000000001:NaN
b10000000000000000000000000000000010:NaN

1111111111111111111111111111111110:17179869182
1111111111111111111111111111111111:17179869183
10000000000000000000000000000000000:17179869184
10000000000000000000000000000000001:17179869185
10000000000000000000000000000000010:17179869186

# Positive

+0b1111111111111111111111111111111110:17179869182
+0b1111111111111111111111111111111111:17179869183
+0b10000000000000000000000000000000000:17179869184
+0b10000000000000000000000000000000001:17179869185
+0b10000000000000000000000000000000010:17179869186

+b1111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000:NaN
+b10000000000000000000000000000000001:NaN
+b10000000000000000000000000000000010:NaN

+1111111111111111111111111111111110:17179869182
+1111111111111111111111111111111111:17179869183
+10000000000000000000000000000000000:17179869184
+10000000000000000000000000000000001:17179869185
+10000000000000000000000000000000010:17179869186

# Negative

-0b1111111111111111111111111111111110:-17179869182
-0b1111111111111111111111111111111111:-17179869183
-0b10000000000000000000000000000000000:-17179869184
-0b10000000000000000000000000000000001:-17179869185
-0b10000000000000000000000000000000010:-17179869186

-b1111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000:NaN
-b10000000000000000000000000000000001:NaN
-b10000000000000000000000000000000010:NaN

-1111111111111111111111111111111110:-17179869182
-1111111111111111111111111111111111:-17179869183
-10000000000000000000000000000000000:-17179869184
-10000000000000000000000000000000001:-17179869185
-10000000000000000000000000000000010:-17179869186

###############################################################################
# Absolute value is close to 2^35 = 34359738368
###############################################################################

# Unsigned

0b11111111111111111111111111111111110:34359738366
0b11111111111111111111111111111111111:34359738367
0b100000000000000000000000000000000000:34359738368
0b100000000000000000000000000000000001:34359738369
0b100000000000000000000000000000000010:34359738370

b11111111111111111111111111111111110:NaN
b11111111111111111111111111111111111:NaN
b100000000000000000000000000000000000:NaN
b100000000000000000000000000000000001:NaN
b100000000000000000000000000000000010:NaN

11111111111111111111111111111111110:34359738366
11111111111111111111111111111111111:34359738367
100000000000000000000000000000000000:34359738368
100000000000000000000000000000000001:34359738369
100000000000000000000000000000000010:34359738370

# Positive

+0b11111111111111111111111111111111110:34359738366
+0b11111111111111111111111111111111111:34359738367
+0b100000000000000000000000000000000000:34359738368
+0b100000000000000000000000000000000001:34359738369
+0b100000000000000000000000000000000010:34359738370

+b11111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000:NaN
+b100000000000000000000000000000000001:NaN
+b100000000000000000000000000000000010:NaN

+11111111111111111111111111111111110:34359738366
+11111111111111111111111111111111111:34359738367
+100000000000000000000000000000000000:34359738368
+100000000000000000000000000000000001:34359738369
+100000000000000000000000000000000010:34359738370

# Negative

-0b11111111111111111111111111111111110:-34359738366
-0b11111111111111111111111111111111111:-34359738367
-0b100000000000000000000000000000000000:-34359738368
-0b100000000000000000000000000000000001:-34359738369
-0b100000000000000000000000000000000010:-34359738370

-b11111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000:NaN
-b100000000000000000000000000000000001:NaN
-b100000000000000000000000000000000010:NaN

-11111111111111111111111111111111110:-34359738366
-11111111111111111111111111111111111:-34359738367
-100000000000000000000000000000000000:-34359738368
-100000000000000000000000000000000001:-34359738369
-100000000000000000000000000000000010:-34359738370

###############################################################################
# Absolute value is close to 2^36 = 68719476736
###############################################################################

# Unsigned

0b111111111111111111111111111111111110:68719476734
0b111111111111111111111111111111111111:68719476735
0b1000000000000000000000000000000000000:68719476736
0b1000000000000000000000000000000000001:68719476737
0b1000000000000000000000000000000000010:68719476738

b111111111111111111111111111111111110:NaN
b111111111111111111111111111111111111:NaN
b1000000000000000000000000000000000000:NaN
b1000000000000000000000000000000000001:NaN
b1000000000000000000000000000000000010:NaN

111111111111111111111111111111111110:68719476734
111111111111111111111111111111111111:68719476735
1000000000000000000000000000000000000:68719476736
1000000000000000000000000000000000001:68719476737
1000000000000000000000000000000000010:68719476738

# Positive

+0b111111111111111111111111111111111110:68719476734
+0b111111111111111111111111111111111111:68719476735
+0b1000000000000000000000000000000000000:68719476736
+0b1000000000000000000000000000000000001:68719476737
+0b1000000000000000000000000000000000010:68719476738

+b111111111111111111111111111111111110:NaN
+b111111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000000:NaN
+b1000000000000000000000000000000000001:NaN
+b1000000000000000000000000000000000010:NaN

+111111111111111111111111111111111110:68719476734
+111111111111111111111111111111111111:68719476735
+1000000000000000000000000000000000000:68719476736
+1000000000000000000000000000000000001:68719476737
+1000000000000000000000000000000000010:68719476738

# Negative

-0b111111111111111111111111111111111110:-68719476734
-0b111111111111111111111111111111111111:-68719476735
-0b1000000000000000000000000000000000000:-68719476736
-0b1000000000000000000000000000000000001:-68719476737
-0b1000000000000000000000000000000000010:-68719476738

-b111111111111111111111111111111111110:NaN
-b111111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000000:NaN
-b1000000000000000000000000000000000001:NaN
-b1000000000000000000000000000000000010:NaN

-111111111111111111111111111111111110:-68719476734
-111111111111111111111111111111111111:-68719476735
-1000000000000000000000000000000000000:-68719476736
-1000000000000000000000000000000000001:-68719476737
-1000000000000000000000000000000000010:-68719476738

###############################################################################
# Absolute value is close to 2^37 = 137438953472
###############################################################################

# Unsigned

0b1111111111111111111111111111111111110:137438953470
0b1111111111111111111111111111111111111:137438953471
0b10000000000000000000000000000000000000:137438953472
0b10000000000000000000000000000000000001:137438953473
0b10000000000000000000000000000000000010:137438953474

b1111111111111111111111111111111111110:NaN
b1111111111111111111111111111111111111:NaN
b10000000000000000000000000000000000000:NaN
b10000000000000000000000000000000000001:NaN
b10000000000000000000000000000000000010:NaN

1111111111111111111111111111111111110:137438953470
1111111111111111111111111111111111111:137438953471
10000000000000000000000000000000000000:137438953472
10000000000000000000000000000000000001:137438953473
10000000000000000000000000000000000010:137438953474

# Positive

+0b1111111111111111111111111111111111110:137438953470
+0b1111111111111111111111111111111111111:137438953471
+0b10000000000000000000000000000000000000:137438953472
+0b10000000000000000000000000000000000001:137438953473
+0b10000000000000000000000000000000000010:137438953474

+b1111111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000000:NaN
+b10000000000000000000000000000000000001:NaN
+b10000000000000000000000000000000000010:NaN

+1111111111111111111111111111111111110:137438953470
+1111111111111111111111111111111111111:137438953471
+10000000000000000000000000000000000000:137438953472
+10000000000000000000000000000000000001:137438953473
+10000000000000000000000000000000000010:137438953474

# Negative

-0b1111111111111111111111111111111111110:-137438953470
-0b1111111111111111111111111111111111111:-137438953471
-0b10000000000000000000000000000000000000:-137438953472
-0b10000000000000000000000000000000000001:-137438953473
-0b10000000000000000000000000000000000010:-137438953474

-b1111111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000000:NaN
-b10000000000000000000000000000000000001:NaN
-b10000000000000000000000000000000000010:NaN

-1111111111111111111111111111111111110:-137438953470
-1111111111111111111111111111111111111:-137438953471
-10000000000000000000000000000000000000:-137438953472
-10000000000000000000000000000000000001:-137438953473
-10000000000000000000000000000000000010:-137438953474

###############################################################################
# Absolute value is close to 2^38 = 274877906944
###############################################################################

# Unsigned

0b11111111111111111111111111111111111110:274877906942
0b11111111111111111111111111111111111111:274877906943
0b100000000000000000000000000000000000000:274877906944
0b100000000000000000000000000000000000001:274877906945
0b100000000000000000000000000000000000010:274877906946

b11111111111111111111111111111111111110:NaN
b11111111111111111111111111111111111111:NaN
b100000000000000000000000000000000000000:NaN
b100000000000000000000000000000000000001:NaN
b100000000000000000000000000000000000010:NaN

11111111111111111111111111111111111110:274877906942
11111111111111111111111111111111111111:274877906943
100000000000000000000000000000000000000:274877906944
100000000000000000000000000000000000001:274877906945
100000000000000000000000000000000000010:274877906946

# Positive

+0b11111111111111111111111111111111111110:274877906942
+0b11111111111111111111111111111111111111:274877906943
+0b100000000000000000000000000000000000000:274877906944
+0b100000000000000000000000000000000000001:274877906945
+0b100000000000000000000000000000000000010:274877906946

+b11111111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000000:NaN
+b100000000000000000000000000000000000001:NaN
+b100000000000000000000000000000000000010:NaN

+11111111111111111111111111111111111110:274877906942
+11111111111111111111111111111111111111:274877906943
+100000000000000000000000000000000000000:274877906944
+100000000000000000000000000000000000001:274877906945
+100000000000000000000000000000000000010:274877906946

# Negative

-0b11111111111111111111111111111111111110:-274877906942
-0b11111111111111111111111111111111111111:-274877906943
-0b100000000000000000000000000000000000000:-274877906944
-0b100000000000000000000000000000000000001:-274877906945
-0b100000000000000000000000000000000000010:-274877906946

-b11111111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000000:NaN
-b100000000000000000000000000000000000001:NaN
-b100000000000000000000000000000000000010:NaN

-11111111111111111111111111111111111110:-274877906942
-11111111111111111111111111111111111111:-274877906943
-100000000000000000000000000000000000000:-274877906944
-100000000000000000000000000000000000001:-274877906945
-100000000000000000000000000000000000010:-274877906946

###############################################################################
# Absolute value is close to 2^39 = 549755813888
###############################################################################

# Unsigned

0b111111111111111111111111111111111111110:549755813886
0b111111111111111111111111111111111111111:549755813887
0b1000000000000000000000000000000000000000:549755813888
0b1000000000000000000000000000000000000001:549755813889
0b1000000000000000000000000000000000000010:549755813890

b111111111111111111111111111111111111110:NaN
b111111111111111111111111111111111111111:NaN
b1000000000000000000000000000000000000000:NaN
b1000000000000000000000000000000000000001:NaN
b1000000000000000000000000000000000000010:NaN

111111111111111111111111111111111111110:549755813886
111111111111111111111111111111111111111:549755813887
1000000000000000000000000000000000000000:549755813888
1000000000000000000000000000000000000001:549755813889
1000000000000000000000000000000000000010:549755813890

# Positive

+0b111111111111111111111111111111111111110:549755813886
+0b111111111111111111111111111111111111111:549755813887
+0b1000000000000000000000000000000000000000:549755813888
+0b1000000000000000000000000000000000000001:549755813889
+0b1000000000000000000000000000000000000010:549755813890

+b111111111111111111111111111111111111110:NaN
+b111111111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000000000:NaN
+b1000000000000000000000000000000000000001:NaN
+b1000000000000000000000000000000000000010:NaN

+111111111111111111111111111111111111110:549755813886
+111111111111111111111111111111111111111:549755813887
+1000000000000000000000000000000000000000:549755813888
+1000000000000000000000000000000000000001:549755813889
+1000000000000000000000000000000000000010:549755813890

# Negative

-0b111111111111111111111111111111111111110:-549755813886
-0b111111111111111111111111111111111111111:-549755813887
-0b1000000000000000000000000000000000000000:-549755813888
-0b1000000000000000000000000000000000000001:-549755813889
-0b1000000000000000000000000000000000000010:-549755813890

-b111111111111111111111111111111111111110:NaN
-b111111111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000000000:NaN
-b1000000000000000000000000000000000000001:NaN
-b1000000000000000000000000000000000000010:NaN

-111111111111111111111111111111111111110:-549755813886
-111111111111111111111111111111111111111:-549755813887
-1000000000000000000000000000000000000000:-549755813888
-1000000000000000000000000000000000000001:-549755813889
-1000000000000000000000000000000000000010:-549755813890

###############################################################################
# Absolute value is close to 2^40 = 1099511627776
###############################################################################

# Unsigned

0b1111111111111111111111111111111111111110:1099511627774
0b1111111111111111111111111111111111111111:1099511627775
0b10000000000000000000000000000000000000000:1099511627776
0b10000000000000000000000000000000000000001:1099511627777
0b10000000000000000000000000000000000000010:1099511627778

b1111111111111111111111111111111111111110:NaN
b1111111111111111111111111111111111111111:NaN
b10000000000000000000000000000000000000000:NaN
b10000000000000000000000000000000000000001:NaN
b10000000000000000000000000000000000000010:NaN

1111111111111111111111111111111111111110:1099511627774
1111111111111111111111111111111111111111:1099511627775
10000000000000000000000000000000000000000:1099511627776
10000000000000000000000000000000000000001:1099511627777
10000000000000000000000000000000000000010:1099511627778

# Positive

+0b1111111111111111111111111111111111111110:1099511627774
+0b1111111111111111111111111111111111111111:1099511627775
+0b10000000000000000000000000000000000000000:1099511627776
+0b10000000000000000000000000000000000000001:1099511627777
+0b10000000000000000000000000000000000000010:1099511627778

+b1111111111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000000000:NaN
+b10000000000000000000000000000000000000001:NaN
+b10000000000000000000000000000000000000010:NaN

+1111111111111111111111111111111111111110:1099511627774
+1111111111111111111111111111111111111111:1099511627775
+10000000000000000000000000000000000000000:1099511627776
+10000000000000000000000000000000000000001:1099511627777
+10000000000000000000000000000000000000010:1099511627778

# Negative

-0b1111111111111111111111111111111111111110:-1099511627774
-0b1111111111111111111111111111111111111111:-1099511627775
-0b10000000000000000000000000000000000000000:-1099511627776
-0b10000000000000000000000000000000000000001:-1099511627777
-0b10000000000000000000000000000000000000010:-1099511627778

-b1111111111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000000000:NaN
-b10000000000000000000000000000000000000001:NaN
-b10000000000000000000000000000000000000010:NaN

-1111111111111111111111111111111111111110:-1099511627774
-1111111111111111111111111111111111111111:-1099511627775
-10000000000000000000000000000000000000000:-1099511627776
-10000000000000000000000000000000000000001:-1099511627777
-10000000000000000000000000000000000000010:-1099511627778

###############################################################################
# Absolute value is close to 2^41 = 2199023255552
###############################################################################

# Unsigned

0b11111111111111111111111111111111111111110:2199023255550
0b11111111111111111111111111111111111111111:2199023255551
0b100000000000000000000000000000000000000000:2199023255552
0b100000000000000000000000000000000000000001:2199023255553
0b100000000000000000000000000000000000000010:2199023255554

b11111111111111111111111111111111111111110:NaN
b11111111111111111111111111111111111111111:NaN
b100000000000000000000000000000000000000000:NaN
b100000000000000000000000000000000000000001:NaN
b100000000000000000000000000000000000000010:NaN

11111111111111111111111111111111111111110:2199023255550
11111111111111111111111111111111111111111:2199023255551
100000000000000000000000000000000000000000:2199023255552
100000000000000000000000000000000000000001:2199023255553
100000000000000000000000000000000000000010:2199023255554

# Positive

+0b11111111111111111111111111111111111111110:2199023255550
+0b11111111111111111111111111111111111111111:2199023255551
+0b100000000000000000000000000000000000000000:2199023255552
+0b100000000000000000000000000000000000000001:2199023255553
+0b100000000000000000000000000000000000000010:2199023255554

+b11111111111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000000000:NaN
+b100000000000000000000000000000000000000001:NaN
+b100000000000000000000000000000000000000010:NaN

+11111111111111111111111111111111111111110:2199023255550
+11111111111111111111111111111111111111111:2199023255551
+100000000000000000000000000000000000000000:2199023255552
+100000000000000000000000000000000000000001:2199023255553
+100000000000000000000000000000000000000010:2199023255554

# Negative

-0b11111111111111111111111111111111111111110:-2199023255550
-0b11111111111111111111111111111111111111111:-2199023255551
-0b100000000000000000000000000000000000000000:-2199023255552
-0b100000000000000000000000000000000000000001:-2199023255553
-0b100000000000000000000000000000000000000010:-2199023255554

-b11111111111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000000000:NaN
-b100000000000000000000000000000000000000001:NaN
-b100000000000000000000000000000000000000010:NaN

-11111111111111111111111111111111111111110:-2199023255550
-11111111111111111111111111111111111111111:-2199023255551
-100000000000000000000000000000000000000000:-2199023255552
-100000000000000000000000000000000000000001:-2199023255553
-100000000000000000000000000000000000000010:-2199023255554

###############################################################################
# Absolute value is close to 2^42 = 4398046511104
###############################################################################

# Unsigned

0b111111111111111111111111111111111111111110:4398046511102
0b111111111111111111111111111111111111111111:4398046511103
0b1000000000000000000000000000000000000000000:4398046511104
0b1000000000000000000000000000000000000000001:4398046511105
0b1000000000000000000000000000000000000000010:4398046511106

b111111111111111111111111111111111111111110:NaN
b111111111111111111111111111111111111111111:NaN
b1000000000000000000000000000000000000000000:NaN
b1000000000000000000000000000000000000000001:NaN
b1000000000000000000000000000000000000000010:NaN

111111111111111111111111111111111111111110:4398046511102
111111111111111111111111111111111111111111:4398046511103
1000000000000000000000000000000000000000000:4398046511104
1000000000000000000000000000000000000000001:4398046511105
1000000000000000000000000000000000000000010:4398046511106

# Positive

+0b111111111111111111111111111111111111111110:4398046511102
+0b111111111111111111111111111111111111111111:4398046511103
+0b1000000000000000000000000000000000000000000:4398046511104
+0b1000000000000000000000000000000000000000001:4398046511105
+0b1000000000000000000000000000000000000000010:4398046511106

+b111111111111111111111111111111111111111110:NaN
+b111111111111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000000000000:NaN
+b1000000000000000000000000000000000000000001:NaN
+b1000000000000000000000000000000000000000010:NaN

+111111111111111111111111111111111111111110:4398046511102
+111111111111111111111111111111111111111111:4398046511103
+1000000000000000000000000000000000000000000:4398046511104
+1000000000000000000000000000000000000000001:4398046511105
+1000000000000000000000000000000000000000010:4398046511106

# Negative

-0b111111111111111111111111111111111111111110:-4398046511102
-0b111111111111111111111111111111111111111111:-4398046511103
-0b1000000000000000000000000000000000000000000:-4398046511104
-0b1000000000000000000000000000000000000000001:-4398046511105
-0b1000000000000000000000000000000000000000010:-4398046511106

-b111111111111111111111111111111111111111110:NaN
-b111111111111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000000000000:NaN
-b1000000000000000000000000000000000000000001:NaN
-b1000000000000000000000000000000000000000010:NaN

-111111111111111111111111111111111111111110:-4398046511102
-111111111111111111111111111111111111111111:-4398046511103
-1000000000000000000000000000000000000000000:-4398046511104
-1000000000000000000000000000000000000000001:-4398046511105
-1000000000000000000000000000000000000000010:-4398046511106

###############################################################################
# Absolute value is close to 2^43 = 8796093022208
###############################################################################

# Unsigned

0b1111111111111111111111111111111111111111110:8796093022206
0b1111111111111111111111111111111111111111111:8796093022207
0b10000000000000000000000000000000000000000000:8796093022208
0b10000000000000000000000000000000000000000001:8796093022209
0b10000000000000000000000000000000000000000010:8796093022210

b1111111111111111111111111111111111111111110:NaN
b1111111111111111111111111111111111111111111:NaN
b10000000000000000000000000000000000000000000:NaN
b10000000000000000000000000000000000000000001:NaN
b10000000000000000000000000000000000000000010:NaN

1111111111111111111111111111111111111111110:8796093022206
1111111111111111111111111111111111111111111:8796093022207
10000000000000000000000000000000000000000000:8796093022208
10000000000000000000000000000000000000000001:8796093022209
10000000000000000000000000000000000000000010:8796093022210

# Positive

+0b1111111111111111111111111111111111111111110:8796093022206
+0b1111111111111111111111111111111111111111111:8796093022207
+0b10000000000000000000000000000000000000000000:8796093022208
+0b10000000000000000000000000000000000000000001:8796093022209
+0b10000000000000000000000000000000000000000010:8796093022210

+b1111111111111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000000000000:NaN
+b10000000000000000000000000000000000000000001:NaN
+b10000000000000000000000000000000000000000010:NaN

+1111111111111111111111111111111111111111110:8796093022206
+1111111111111111111111111111111111111111111:8796093022207
+10000000000000000000000000000000000000000000:8796093022208
+10000000000000000000000000000000000000000001:8796093022209
+10000000000000000000000000000000000000000010:8796093022210

# Negative

-0b1111111111111111111111111111111111111111110:-8796093022206
-0b1111111111111111111111111111111111111111111:-8796093022207
-0b10000000000000000000000000000000000000000000:-8796093022208
-0b10000000000000000000000000000000000000000001:-8796093022209
-0b10000000000000000000000000000000000000000010:-8796093022210

-b1111111111111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000000000000:NaN
-b10000000000000000000000000000000000000000001:NaN
-b10000000000000000000000000000000000000000010:NaN

-1111111111111111111111111111111111111111110:-8796093022206
-1111111111111111111111111111111111111111111:-8796093022207
-10000000000000000000000000000000000000000000:-8796093022208
-10000000000000000000000000000000000000000001:-8796093022209
-10000000000000000000000000000000000000000010:-8796093022210

###############################################################################
# Absolute value is close to 2^44 = 17592186044416
###############################################################################

# Unsigned

0b11111111111111111111111111111111111111111110:17592186044414
0b11111111111111111111111111111111111111111111:17592186044415
0b100000000000000000000000000000000000000000000:17592186044416
0b100000000000000000000000000000000000000000001:17592186044417
0b100000000000000000000000000000000000000000010:17592186044418

b11111111111111111111111111111111111111111110:NaN
b11111111111111111111111111111111111111111111:NaN
b100000000000000000000000000000000000000000000:NaN
b100000000000000000000000000000000000000000001:NaN
b100000000000000000000000000000000000000000010:NaN

11111111111111111111111111111111111111111110:17592186044414
11111111111111111111111111111111111111111111:17592186044415
100000000000000000000000000000000000000000000:17592186044416
100000000000000000000000000000000000000000001:17592186044417
100000000000000000000000000000000000000000010:17592186044418

# Positive

+0b11111111111111111111111111111111111111111110:17592186044414
+0b11111111111111111111111111111111111111111111:17592186044415
+0b100000000000000000000000000000000000000000000:17592186044416
+0b100000000000000000000000000000000000000000001:17592186044417
+0b100000000000000000000000000000000000000000010:17592186044418

+b11111111111111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000000000000:NaN
+b100000000000000000000000000000000000000000001:NaN
+b100000000000000000000000000000000000000000010:NaN

+11111111111111111111111111111111111111111110:17592186044414
+11111111111111111111111111111111111111111111:17592186044415
+100000000000000000000000000000000000000000000:17592186044416
+100000000000000000000000000000000000000000001:17592186044417
+100000000000000000000000000000000000000000010:17592186044418

# Negative

-0b11111111111111111111111111111111111111111110:-17592186044414
-0b11111111111111111111111111111111111111111111:-17592186044415
-0b100000000000000000000000000000000000000000000:-17592186044416
-0b100000000000000000000000000000000000000000001:-17592186044417
-0b100000000000000000000000000000000000000000010:-17592186044418

-b11111111111111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000000000000:NaN
-b100000000000000000000000000000000000000000001:NaN
-b100000000000000000000000000000000000000000010:NaN

-11111111111111111111111111111111111111111110:-17592186044414
-11111111111111111111111111111111111111111111:-17592186044415
-100000000000000000000000000000000000000000000:-17592186044416
-100000000000000000000000000000000000000000001:-17592186044417
-100000000000000000000000000000000000000000010:-17592186044418

###############################################################################
# Absolute value is close to 2^45 = 35184372088832
###############################################################################

# Unsigned

0b111111111111111111111111111111111111111111110:35184372088830
0b111111111111111111111111111111111111111111111:35184372088831
0b1000000000000000000000000000000000000000000000:35184372088832
0b1000000000000000000000000000000000000000000001:35184372088833
0b1000000000000000000000000000000000000000000010:35184372088834

b111111111111111111111111111111111111111111110:NaN
b111111111111111111111111111111111111111111111:NaN
b1000000000000000000000000000000000000000000000:NaN
b1000000000000000000000000000000000000000000001:NaN
b1000000000000000000000000000000000000000000010:NaN

111111111111111111111111111111111111111111110:35184372088830
111111111111111111111111111111111111111111111:35184372088831
1000000000000000000000000000000000000000000000:35184372088832
1000000000000000000000000000000000000000000001:35184372088833
1000000000000000000000000000000000000000000010:35184372088834

# Positive

+0b111111111111111111111111111111111111111111110:35184372088830
+0b111111111111111111111111111111111111111111111:35184372088831
+0b1000000000000000000000000000000000000000000000:35184372088832
+0b1000000000000000000000000000000000000000000001:35184372088833
+0b1000000000000000000000000000000000000000000010:35184372088834

+b111111111111111111111111111111111111111111110:NaN
+b111111111111111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000000000000000:NaN
+b1000000000000000000000000000000000000000000001:NaN
+b1000000000000000000000000000000000000000000010:NaN

+111111111111111111111111111111111111111111110:35184372088830
+111111111111111111111111111111111111111111111:35184372088831
+1000000000000000000000000000000000000000000000:35184372088832
+1000000000000000000000000000000000000000000001:35184372088833
+1000000000000000000000000000000000000000000010:35184372088834

# Negative

-0b111111111111111111111111111111111111111111110:-35184372088830
-0b111111111111111111111111111111111111111111111:-35184372088831
-0b1000000000000000000000000000000000000000000000:-35184372088832
-0b1000000000000000000000000000000000000000000001:-35184372088833
-0b1000000000000000000000000000000000000000000010:-35184372088834

-b111111111111111111111111111111111111111111110:NaN
-b111111111111111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000000000000000:NaN
-b1000000000000000000000000000000000000000000001:NaN
-b1000000000000000000000000000000000000000000010:NaN

-111111111111111111111111111111111111111111110:-35184372088830
-111111111111111111111111111111111111111111111:-35184372088831
-1000000000000000000000000000000000000000000000:-35184372088832
-1000000000000000000000000000000000000000000001:-35184372088833
-1000000000000000000000000000000000000000000010:-35184372088834

###############################################################################
# Absolute value is close to 2^46 = 70368744177664
###############################################################################

# Unsigned

0b1111111111111111111111111111111111111111111110:70368744177662
0b1111111111111111111111111111111111111111111111:70368744177663
0b10000000000000000000000000000000000000000000000:70368744177664
0b10000000000000000000000000000000000000000000001:70368744177665
0b10000000000000000000000000000000000000000000010:70368744177666

b1111111111111111111111111111111111111111111110:NaN
b1111111111111111111111111111111111111111111111:NaN
b10000000000000000000000000000000000000000000000:NaN
b10000000000000000000000000000000000000000000001:NaN
b10000000000000000000000000000000000000000000010:NaN

1111111111111111111111111111111111111111111110:70368744177662
1111111111111111111111111111111111111111111111:70368744177663
10000000000000000000000000000000000000000000000:70368744177664
10000000000000000000000000000000000000000000001:70368744177665
10000000000000000000000000000000000000000000010:70368744177666

# Positive

+0b1111111111111111111111111111111111111111111110:70368744177662
+0b1111111111111111111111111111111111111111111111:70368744177663
+0b10000000000000000000000000000000000000000000000:70368744177664
+0b10000000000000000000000000000000000000000000001:70368744177665
+0b10000000000000000000000000000000000000000000010:70368744177666

+b1111111111111111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000000000000000:NaN
+b10000000000000000000000000000000000000000000001:NaN
+b10000000000000000000000000000000000000000000010:NaN

+1111111111111111111111111111111111111111111110:70368744177662
+1111111111111111111111111111111111111111111111:70368744177663
+10000000000000000000000000000000000000000000000:70368744177664
+10000000000000000000000000000000000000000000001:70368744177665
+10000000000000000000000000000000000000000000010:70368744177666

# Negative

-0b1111111111111111111111111111111111111111111110:-70368744177662
-0b1111111111111111111111111111111111111111111111:-70368744177663
-0b10000000000000000000000000000000000000000000000:-70368744177664
-0b10000000000000000000000000000000000000000000001:-70368744177665
-0b10000000000000000000000000000000000000000000010:-70368744177666

-b1111111111111111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000000000000000:NaN
-b10000000000000000000000000000000000000000000001:NaN
-b10000000000000000000000000000000000000000000010:NaN

-1111111111111111111111111111111111111111111110:-70368744177662
-1111111111111111111111111111111111111111111111:-70368744177663
-10000000000000000000000000000000000000000000000:-70368744177664
-10000000000000000000000000000000000000000000001:-70368744177665
-10000000000000000000000000000000000000000000010:-70368744177666

###############################################################################
# Absolute value is close to 2^47 = 140737488355328
###############################################################################

# Unsigned

0b11111111111111111111111111111111111111111111110:140737488355326
0b11111111111111111111111111111111111111111111111:140737488355327
0b100000000000000000000000000000000000000000000000:140737488355328
0b100000000000000000000000000000000000000000000001:140737488355329
0b100000000000000000000000000000000000000000000010:140737488355330

b11111111111111111111111111111111111111111111110:NaN
b11111111111111111111111111111111111111111111111:NaN
b100000000000000000000000000000000000000000000000:NaN
b100000000000000000000000000000000000000000000001:NaN
b100000000000000000000000000000000000000000000010:NaN

11111111111111111111111111111111111111111111110:140737488355326
11111111111111111111111111111111111111111111111:140737488355327
100000000000000000000000000000000000000000000000:140737488355328
100000000000000000000000000000000000000000000001:140737488355329
100000000000000000000000000000000000000000000010:140737488355330

# Positive

+0b11111111111111111111111111111111111111111111110:140737488355326
+0b11111111111111111111111111111111111111111111111:140737488355327
+0b100000000000000000000000000000000000000000000000:140737488355328
+0b100000000000000000000000000000000000000000000001:140737488355329
+0b100000000000000000000000000000000000000000000010:140737488355330

+b11111111111111111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000000000000000:NaN
+b100000000000000000000000000000000000000000000001:NaN
+b100000000000000000000000000000000000000000000010:NaN

+11111111111111111111111111111111111111111111110:140737488355326
+11111111111111111111111111111111111111111111111:140737488355327
+100000000000000000000000000000000000000000000000:140737488355328
+100000000000000000000000000000000000000000000001:140737488355329
+100000000000000000000000000000000000000000000010:140737488355330

# Negative

-0b11111111111111111111111111111111111111111111110:-140737488355326
-0b11111111111111111111111111111111111111111111111:-140737488355327
-0b100000000000000000000000000000000000000000000000:-140737488355328
-0b100000000000000000000000000000000000000000000001:-140737488355329
-0b100000000000000000000000000000000000000000000010:-140737488355330

-b11111111111111111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000000000000000:NaN
-b100000000000000000000000000000000000000000000001:NaN
-b100000000000000000000000000000000000000000000010:NaN

-11111111111111111111111111111111111111111111110:-140737488355326
-11111111111111111111111111111111111111111111111:-140737488355327
-100000000000000000000000000000000000000000000000:-140737488355328
-100000000000000000000000000000000000000000000001:-140737488355329
-100000000000000000000000000000000000000000000010:-140737488355330

###############################################################################
# Absolute value is close to 2^48 = 281474976710656
###############################################################################

# Unsigned

0b111111111111111111111111111111111111111111111110:281474976710654
0b111111111111111111111111111111111111111111111111:281474976710655
0b1000000000000000000000000000000000000000000000000:281474976710656
0b1000000000000000000000000000000000000000000000001:281474976710657
0b1000000000000000000000000000000000000000000000010:281474976710658

b111111111111111111111111111111111111111111111110:NaN
b111111111111111111111111111111111111111111111111:NaN
b1000000000000000000000000000000000000000000000000:NaN
b1000000000000000000000000000000000000000000000001:NaN
b1000000000000000000000000000000000000000000000010:NaN

111111111111111111111111111111111111111111111110:281474976710654
111111111111111111111111111111111111111111111111:281474976710655
1000000000000000000000000000000000000000000000000:281474976710656
1000000000000000000000000000000000000000000000001:281474976710657
1000000000000000000000000000000000000000000000010:281474976710658

# Positive

+0b111111111111111111111111111111111111111111111110:281474976710654
+0b111111111111111111111111111111111111111111111111:281474976710655
+0b1000000000000000000000000000000000000000000000000:281474976710656
+0b1000000000000000000000000000000000000000000000001:281474976710657
+0b1000000000000000000000000000000000000000000000010:281474976710658

+b111111111111111111111111111111111111111111111110:NaN
+b111111111111111111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000000000000000000:NaN
+b1000000000000000000000000000000000000000000000001:NaN
+b1000000000000000000000000000000000000000000000010:NaN

+111111111111111111111111111111111111111111111110:281474976710654
+111111111111111111111111111111111111111111111111:281474976710655
+1000000000000000000000000000000000000000000000000:281474976710656
+1000000000000000000000000000000000000000000000001:281474976710657
+1000000000000000000000000000000000000000000000010:281474976710658

# Negative

-0b111111111111111111111111111111111111111111111110:-281474976710654
-0b111111111111111111111111111111111111111111111111:-281474976710655
-0b1000000000000000000000000000000000000000000000000:-281474976710656
-0b1000000000000000000000000000000000000000000000001:-281474976710657
-0b1000000000000000000000000000000000000000000000010:-281474976710658

-b111111111111111111111111111111111111111111111110:NaN
-b111111111111111111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000000000000000000:NaN
-b1000000000000000000000000000000000000000000000001:NaN
-b1000000000000000000000000000000000000000000000010:NaN

-111111111111111111111111111111111111111111111110:-281474976710654
-111111111111111111111111111111111111111111111111:-281474976710655
-1000000000000000000000000000000000000000000000000:-281474976710656
-1000000000000000000000000000000000000000000000001:-281474976710657
-1000000000000000000000000000000000000000000000010:-281474976710658

###############################################################################
# Absolute value is close to 2^49 = 562949953421312
###############################################################################

# Unsigned

0b1111111111111111111111111111111111111111111111110:562949953421310
0b1111111111111111111111111111111111111111111111111:562949953421311
0b10000000000000000000000000000000000000000000000000:562949953421312
0b10000000000000000000000000000000000000000000000001:562949953421313
0b10000000000000000000000000000000000000000000000010:562949953421314

b1111111111111111111111111111111111111111111111110:NaN
b1111111111111111111111111111111111111111111111111:NaN
b10000000000000000000000000000000000000000000000000:NaN
b10000000000000000000000000000000000000000000000001:NaN
b10000000000000000000000000000000000000000000000010:NaN

1111111111111111111111111111111111111111111111110:562949953421310
1111111111111111111111111111111111111111111111111:562949953421311
10000000000000000000000000000000000000000000000000:562949953421312
10000000000000000000000000000000000000000000000001:562949953421313
10000000000000000000000000000000000000000000000010:562949953421314

# Positive

+0b1111111111111111111111111111111111111111111111110:562949953421310
+0b1111111111111111111111111111111111111111111111111:562949953421311
+0b10000000000000000000000000000000000000000000000000:562949953421312
+0b10000000000000000000000000000000000000000000000001:562949953421313
+0b10000000000000000000000000000000000000000000000010:562949953421314

+b1111111111111111111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000000000000000000:NaN
+b10000000000000000000000000000000000000000000000001:NaN
+b10000000000000000000000000000000000000000000000010:NaN

+1111111111111111111111111111111111111111111111110:562949953421310
+1111111111111111111111111111111111111111111111111:562949953421311
+10000000000000000000000000000000000000000000000000:562949953421312
+10000000000000000000000000000000000000000000000001:562949953421313
+10000000000000000000000000000000000000000000000010:562949953421314

# Negative

-0b1111111111111111111111111111111111111111111111110:-562949953421310
-0b1111111111111111111111111111111111111111111111111:-562949953421311
-0b10000000000000000000000000000000000000000000000000:-562949953421312
-0b10000000000000000000000000000000000000000000000001:-562949953421313
-0b10000000000000000000000000000000000000000000000010:-562949953421314

-b1111111111111111111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000000000000000000:NaN
-b10000000000000000000000000000000000000000000000001:NaN
-b10000000000000000000000000000000000000000000000010:NaN

-1111111111111111111111111111111111111111111111110:-562949953421310
-1111111111111111111111111111111111111111111111111:-562949953421311
-10000000000000000000000000000000000000000000000000:-562949953421312
-10000000000000000000000000000000000000000000000001:-562949953421313
-10000000000000000000000000000000000000000000000010:-562949953421314

###############################################################################
# Absolute value is close to 2^50 = 1125899906842624
###############################################################################

# Unsigned

0b11111111111111111111111111111111111111111111111110:1125899906842622
0b11111111111111111111111111111111111111111111111111:1125899906842623
0b100000000000000000000000000000000000000000000000000:1125899906842624
0b100000000000000000000000000000000000000000000000001:1125899906842625
0b100000000000000000000000000000000000000000000000010:1125899906842626

b11111111111111111111111111111111111111111111111110:NaN
b11111111111111111111111111111111111111111111111111:NaN
b100000000000000000000000000000000000000000000000000:NaN
b100000000000000000000000000000000000000000000000001:NaN
b100000000000000000000000000000000000000000000000010:NaN

11111111111111111111111111111111111111111111111110:1125899906842622
11111111111111111111111111111111111111111111111111:1125899906842623
100000000000000000000000000000000000000000000000000:1125899906842624
100000000000000000000000000000000000000000000000001:1125899906842625
100000000000000000000000000000000000000000000000010:1125899906842626

# Positive

+0b11111111111111111111111111111111111111111111111110:1125899906842622
+0b11111111111111111111111111111111111111111111111111:1125899906842623
+0b100000000000000000000000000000000000000000000000000:1125899906842624
+0b100000000000000000000000000000000000000000000000001:1125899906842625
+0b100000000000000000000000000000000000000000000000010:1125899906842626

+b11111111111111111111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000000000000000000:NaN
+b100000000000000000000000000000000000000000000000001:NaN
+b100000000000000000000000000000000000000000000000010:NaN

+11111111111111111111111111111111111111111111111110:1125899906842622
+11111111111111111111111111111111111111111111111111:1125899906842623
+100000000000000000000000000000000000000000000000000:1125899906842624
+100000000000000000000000000000000000000000000000001:1125899906842625
+100000000000000000000000000000000000000000000000010:1125899906842626

# Negative

-0b11111111111111111111111111111111111111111111111110:-1125899906842622
-0b11111111111111111111111111111111111111111111111111:-1125899906842623
-0b100000000000000000000000000000000000000000000000000:-1125899906842624
-0b100000000000000000000000000000000000000000000000001:-1125899906842625
-0b100000000000000000000000000000000000000000000000010:-1125899906842626

-b11111111111111111111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000000000000000000:NaN
-b100000000000000000000000000000000000000000000000001:NaN
-b100000000000000000000000000000000000000000000000010:NaN

-11111111111111111111111111111111111111111111111110:-1125899906842622
-11111111111111111111111111111111111111111111111111:-1125899906842623
-100000000000000000000000000000000000000000000000000:-1125899906842624
-100000000000000000000000000000000000000000000000001:-1125899906842625
-100000000000000000000000000000000000000000000000010:-1125899906842626

###############################################################################
# Absolute value is close to 2^51 = 2251799813685248
###############################################################################

# Unsigned

0b111111111111111111111111111111111111111111111111110:2251799813685246
0b111111111111111111111111111111111111111111111111111:2251799813685247
0b1000000000000000000000000000000000000000000000000000:2251799813685248
0b1000000000000000000000000000000000000000000000000001:2251799813685249
0b1000000000000000000000000000000000000000000000000010:2251799813685250

b111111111111111111111111111111111111111111111111110:NaN
b111111111111111111111111111111111111111111111111111:NaN
b1000000000000000000000000000000000000000000000000000:NaN
b1000000000000000000000000000000000000000000000000001:NaN
b1000000000000000000000000000000000000000000000000010:NaN

111111111111111111111111111111111111111111111111110:2251799813685246
111111111111111111111111111111111111111111111111111:2251799813685247
1000000000000000000000000000000000000000000000000000:2251799813685248
1000000000000000000000000000000000000000000000000001:2251799813685249
1000000000000000000000000000000000000000000000000010:2251799813685250

# Positive

+0b111111111111111111111111111111111111111111111111110:2251799813685246
+0b111111111111111111111111111111111111111111111111111:2251799813685247
+0b1000000000000000000000000000000000000000000000000000:2251799813685248
+0b1000000000000000000000000000000000000000000000000001:2251799813685249
+0b1000000000000000000000000000000000000000000000000010:2251799813685250

+b111111111111111111111111111111111111111111111111110:NaN
+b111111111111111111111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000000000000000000000:NaN
+b1000000000000000000000000000000000000000000000000001:NaN
+b1000000000000000000000000000000000000000000000000010:NaN

+111111111111111111111111111111111111111111111111110:2251799813685246
+111111111111111111111111111111111111111111111111111:2251799813685247
+1000000000000000000000000000000000000000000000000000:2251799813685248
+1000000000000000000000000000000000000000000000000001:2251799813685249
+1000000000000000000000000000000000000000000000000010:2251799813685250

# Negative

-0b111111111111111111111111111111111111111111111111110:-2251799813685246
-0b111111111111111111111111111111111111111111111111111:-2251799813685247
-0b1000000000000000000000000000000000000000000000000000:-2251799813685248
-0b1000000000000000000000000000000000000000000000000001:-2251799813685249
-0b1000000000000000000000000000000000000000000000000010:-2251799813685250

-b111111111111111111111111111111111111111111111111110:NaN
-b111111111111111111111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000000000000000000000:NaN
-b1000000000000000000000000000000000000000000000000001:NaN
-b1000000000000000000000000000000000000000000000000010:NaN

-111111111111111111111111111111111111111111111111110:-2251799813685246
-111111111111111111111111111111111111111111111111111:-2251799813685247
-1000000000000000000000000000000000000000000000000000:-2251799813685248
-1000000000000000000000000000000000000000000000000001:-2251799813685249
-1000000000000000000000000000000000000000000000000010:-2251799813685250

###############################################################################
# Absolute value is close to 2^52 = 4503599627370496
###############################################################################

# Unsigned

0b1111111111111111111111111111111111111111111111111110:4503599627370494
0b1111111111111111111111111111111111111111111111111111:4503599627370495
0b10000000000000000000000000000000000000000000000000000:4503599627370496
0b10000000000000000000000000000000000000000000000000001:4503599627370497
0b10000000000000000000000000000000000000000000000000010:4503599627370498

b1111111111111111111111111111111111111111111111111110:NaN
b1111111111111111111111111111111111111111111111111111:NaN
b10000000000000000000000000000000000000000000000000000:NaN
b10000000000000000000000000000000000000000000000000001:NaN
b10000000000000000000000000000000000000000000000000010:NaN

1111111111111111111111111111111111111111111111111110:4503599627370494
1111111111111111111111111111111111111111111111111111:4503599627370495
10000000000000000000000000000000000000000000000000000:4503599627370496
10000000000000000000000000000000000000000000000000001:4503599627370497
10000000000000000000000000000000000000000000000000010:4503599627370498

# Positive

+0b1111111111111111111111111111111111111111111111111110:4503599627370494
+0b1111111111111111111111111111111111111111111111111111:4503599627370495
+0b10000000000000000000000000000000000000000000000000000:4503599627370496
+0b10000000000000000000000000000000000000000000000000001:4503599627370497
+0b10000000000000000000000000000000000000000000000000010:4503599627370498

+b1111111111111111111111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000000000000000000000:NaN
+b10000000000000000000000000000000000000000000000000001:NaN
+b10000000000000000000000000000000000000000000000000010:NaN

+1111111111111111111111111111111111111111111111111110:4503599627370494
+1111111111111111111111111111111111111111111111111111:4503599627370495
+10000000000000000000000000000000000000000000000000000:4503599627370496
+10000000000000000000000000000000000000000000000000001:4503599627370497
+10000000000000000000000000000000000000000000000000010:4503599627370498

# Negative

-0b1111111111111111111111111111111111111111111111111110:-4503599627370494
-0b1111111111111111111111111111111111111111111111111111:-4503599627370495
-0b10000000000000000000000000000000000000000000000000000:-4503599627370496
-0b10000000000000000000000000000000000000000000000000001:-4503599627370497
-0b10000000000000000000000000000000000000000000000000010:-4503599627370498

-b1111111111111111111111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000000000000000000000:NaN
-b10000000000000000000000000000000000000000000000000001:NaN
-b10000000000000000000000000000000000000000000000000010:NaN

-1111111111111111111111111111111111111111111111111110:-4503599627370494
-1111111111111111111111111111111111111111111111111111:-4503599627370495
-10000000000000000000000000000000000000000000000000000:-4503599627370496
-10000000000000000000000000000000000000000000000000001:-4503599627370497
-10000000000000000000000000000000000000000000000000010:-4503599627370498

###############################################################################
# Absolute value is close to 2^53 = 9007199254740992
###############################################################################

# Unsigned

0b11111111111111111111111111111111111111111111111111110:9007199254740990
0b11111111111111111111111111111111111111111111111111111:9007199254740991
0b100000000000000000000000000000000000000000000000000000:9007199254740992
0b100000000000000000000000000000000000000000000000000001:9007199254740993
0b100000000000000000000000000000000000000000000000000010:9007199254740994

b11111111111111111111111111111111111111111111111111110:NaN
b11111111111111111111111111111111111111111111111111111:NaN
b100000000000000000000000000000000000000000000000000000:NaN
b100000000000000000000000000000000000000000000000000001:NaN
b100000000000000000000000000000000000000000000000000010:NaN

11111111111111111111111111111111111111111111111111110:9007199254740990
11111111111111111111111111111111111111111111111111111:9007199254740991
100000000000000000000000000000000000000000000000000000:9007199254740992
100000000000000000000000000000000000000000000000000001:9007199254740993
100000000000000000000000000000000000000000000000000010:9007199254740994

# Positive

+0b11111111111111111111111111111111111111111111111111110:9007199254740990
+0b11111111111111111111111111111111111111111111111111111:9007199254740991
+0b100000000000000000000000000000000000000000000000000000:9007199254740992
+0b100000000000000000000000000000000000000000000000000001:9007199254740993
+0b100000000000000000000000000000000000000000000000000010:9007199254740994

+b11111111111111111111111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000000000000000000000:NaN
+b100000000000000000000000000000000000000000000000000001:NaN
+b100000000000000000000000000000000000000000000000000010:NaN

+11111111111111111111111111111111111111111111111111110:9007199254740990
+11111111111111111111111111111111111111111111111111111:9007199254740991
+100000000000000000000000000000000000000000000000000000:9007199254740992
+100000000000000000000000000000000000000000000000000001:9007199254740993
+100000000000000000000000000000000000000000000000000010:9007199254740994

# Negative

-0b11111111111111111111111111111111111111111111111111110:-9007199254740990
-0b11111111111111111111111111111111111111111111111111111:-9007199254740991
-0b100000000000000000000000000000000000000000000000000000:-9007199254740992
-0b100000000000000000000000000000000000000000000000000001:-9007199254740993
-0b100000000000000000000000000000000000000000000000000010:-9007199254740994

-b11111111111111111111111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000000000000000000000:NaN
-b100000000000000000000000000000000000000000000000000001:NaN
-b100000000000000000000000000000000000000000000000000010:NaN

-11111111111111111111111111111111111111111111111111110:-9007199254740990
-11111111111111111111111111111111111111111111111111111:-9007199254740991
-100000000000000000000000000000000000000000000000000000:-9007199254740992
-100000000000000000000000000000000000000000000000000001:-9007199254740993
-100000000000000000000000000000000000000000000000000010:-9007199254740994

###############################################################################
# Absolute value is close to 2^54 = 18014398509481984
###############################################################################

# Unsigned

0b111111111111111111111111111111111111111111111111111110:18014398509481982
0b111111111111111111111111111111111111111111111111111111:18014398509481983
0b1000000000000000000000000000000000000000000000000000000:18014398509481984
0b1000000000000000000000000000000000000000000000000000001:18014398509481985
0b1000000000000000000000000000000000000000000000000000010:18014398509481986

b111111111111111111111111111111111111111111111111111110:NaN
b111111111111111111111111111111111111111111111111111111:NaN
b1000000000000000000000000000000000000000000000000000000:NaN
b1000000000000000000000000000000000000000000000000000001:NaN
b1000000000000000000000000000000000000000000000000000010:NaN

111111111111111111111111111111111111111111111111111110:18014398509481982
111111111111111111111111111111111111111111111111111111:18014398509481983
1000000000000000000000000000000000000000000000000000000:18014398509481984
1000000000000000000000000000000000000000000000000000001:18014398509481985
1000000000000000000000000000000000000000000000000000010:18014398509481986

# Positive

+0b111111111111111111111111111111111111111111111111111110:18014398509481982
+0b111111111111111111111111111111111111111111111111111111:18014398509481983
+0b1000000000000000000000000000000000000000000000000000000:18014398509481984
+0b1000000000000000000000000000000000000000000000000000001:18014398509481985
+0b1000000000000000000000000000000000000000000000000000010:18014398509481986

+b111111111111111111111111111111111111111111111111111110:NaN
+b111111111111111111111111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000000000000000000000000:NaN
+b1000000000000000000000000000000000000000000000000000001:NaN
+b1000000000000000000000000000000000000000000000000000010:NaN

+111111111111111111111111111111111111111111111111111110:18014398509481982
+111111111111111111111111111111111111111111111111111111:18014398509481983
+1000000000000000000000000000000000000000000000000000000:18014398509481984
+1000000000000000000000000000000000000000000000000000001:18014398509481985
+1000000000000000000000000000000000000000000000000000010:18014398509481986

# Negative

-0b111111111111111111111111111111111111111111111111111110:-18014398509481982
-0b111111111111111111111111111111111111111111111111111111:-18014398509481983
-0b1000000000000000000000000000000000000000000000000000000:-18014398509481984
-0b1000000000000000000000000000000000000000000000000000001:-18014398509481985
-0b1000000000000000000000000000000000000000000000000000010:-18014398509481986

-b111111111111111111111111111111111111111111111111111110:NaN
-b111111111111111111111111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000000000000000000000000:NaN
-b1000000000000000000000000000000000000000000000000000001:NaN
-b1000000000000000000000000000000000000000000000000000010:NaN

-111111111111111111111111111111111111111111111111111110:-18014398509481982
-111111111111111111111111111111111111111111111111111111:-18014398509481983
-1000000000000000000000000000000000000000000000000000000:-18014398509481984
-1000000000000000000000000000000000000000000000000000001:-18014398509481985
-1000000000000000000000000000000000000000000000000000010:-18014398509481986

###############################################################################
# Absolute value is close to 2^55 = 36028797018963968
###############################################################################

# Unsigned

0b1111111111111111111111111111111111111111111111111111110:36028797018963966
0b1111111111111111111111111111111111111111111111111111111:36028797018963967
0b10000000000000000000000000000000000000000000000000000000:36028797018963968
0b10000000000000000000000000000000000000000000000000000001:36028797018963969
0b10000000000000000000000000000000000000000000000000000010:36028797018963970

b1111111111111111111111111111111111111111111111111111110:NaN
b1111111111111111111111111111111111111111111111111111111:NaN
b10000000000000000000000000000000000000000000000000000000:NaN
b10000000000000000000000000000000000000000000000000000001:NaN
b10000000000000000000000000000000000000000000000000000010:NaN

1111111111111111111111111111111111111111111111111111110:36028797018963966
1111111111111111111111111111111111111111111111111111111:36028797018963967
10000000000000000000000000000000000000000000000000000000:36028797018963968
10000000000000000000000000000000000000000000000000000001:36028797018963969
10000000000000000000000000000000000000000000000000000010:36028797018963970

# Positive

+0b1111111111111111111111111111111111111111111111111111110:36028797018963966
+0b1111111111111111111111111111111111111111111111111111111:36028797018963967
+0b10000000000000000000000000000000000000000000000000000000:36028797018963968
+0b10000000000000000000000000000000000000000000000000000001:36028797018963969
+0b10000000000000000000000000000000000000000000000000000010:36028797018963970

+b1111111111111111111111111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000000000000000000000000:NaN
+b10000000000000000000000000000000000000000000000000000001:NaN
+b10000000000000000000000000000000000000000000000000000010:NaN

+1111111111111111111111111111111111111111111111111111110:36028797018963966
+1111111111111111111111111111111111111111111111111111111:36028797018963967
+10000000000000000000000000000000000000000000000000000000:36028797018963968
+10000000000000000000000000000000000000000000000000000001:36028797018963969
+10000000000000000000000000000000000000000000000000000010:36028797018963970

# Negative

-0b1111111111111111111111111111111111111111111111111111110:-36028797018963966
-0b1111111111111111111111111111111111111111111111111111111:-36028797018963967
-0b10000000000000000000000000000000000000000000000000000000:-36028797018963968
-0b10000000000000000000000000000000000000000000000000000001:-36028797018963969
-0b10000000000000000000000000000000000000000000000000000010:-36028797018963970

-b1111111111111111111111111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000000000000000000000000:NaN
-b10000000000000000000000000000000000000000000000000000001:NaN
-b10000000000000000000000000000000000000000000000000000010:NaN

-1111111111111111111111111111111111111111111111111111110:-36028797018963966
-1111111111111111111111111111111111111111111111111111111:-36028797018963967
-10000000000000000000000000000000000000000000000000000000:-36028797018963968
-10000000000000000000000000000000000000000000000000000001:-36028797018963969
-10000000000000000000000000000000000000000000000000000010:-36028797018963970

###############################################################################
# Absolute value is close to 2^56 = 72057594037927936
###############################################################################

# Unsigned

0b11111111111111111111111111111111111111111111111111111110:72057594037927934
0b11111111111111111111111111111111111111111111111111111111:72057594037927935
0b100000000000000000000000000000000000000000000000000000000:72057594037927936
0b100000000000000000000000000000000000000000000000000000001:72057594037927937
0b100000000000000000000000000000000000000000000000000000010:72057594037927938

b11111111111111111111111111111111111111111111111111111110:NaN
b11111111111111111111111111111111111111111111111111111111:NaN
b100000000000000000000000000000000000000000000000000000000:NaN
b100000000000000000000000000000000000000000000000000000001:NaN
b100000000000000000000000000000000000000000000000000000010:NaN

11111111111111111111111111111111111111111111111111111110:72057594037927934
11111111111111111111111111111111111111111111111111111111:72057594037927935
100000000000000000000000000000000000000000000000000000000:72057594037927936
100000000000000000000000000000000000000000000000000000001:72057594037927937
100000000000000000000000000000000000000000000000000000010:72057594037927938

# Positive

+0b11111111111111111111111111111111111111111111111111111110:72057594037927934
+0b11111111111111111111111111111111111111111111111111111111:72057594037927935
+0b100000000000000000000000000000000000000000000000000000000:72057594037927936
+0b100000000000000000000000000000000000000000000000000000001:72057594037927937
+0b100000000000000000000000000000000000000000000000000000010:72057594037927938

+b11111111111111111111111111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000000000000000000000000:NaN
+b100000000000000000000000000000000000000000000000000000001:NaN
+b100000000000000000000000000000000000000000000000000000010:NaN

+11111111111111111111111111111111111111111111111111111110:72057594037927934
+11111111111111111111111111111111111111111111111111111111:72057594037927935
+100000000000000000000000000000000000000000000000000000000:72057594037927936
+100000000000000000000000000000000000000000000000000000001:72057594037927937
+100000000000000000000000000000000000000000000000000000010:72057594037927938

# Negative

-0b11111111111111111111111111111111111111111111111111111110:-72057594037927934
-0b11111111111111111111111111111111111111111111111111111111:-72057594037927935
-0b100000000000000000000000000000000000000000000000000000000:-72057594037927936
-0b100000000000000000000000000000000000000000000000000000001:-72057594037927937
-0b100000000000000000000000000000000000000000000000000000010:-72057594037927938

-b11111111111111111111111111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000000000000000000000000:NaN
-b100000000000000000000000000000000000000000000000000000001:NaN
-b100000000000000000000000000000000000000000000000000000010:NaN

-11111111111111111111111111111111111111111111111111111110:-72057594037927934
-11111111111111111111111111111111111111111111111111111111:-72057594037927935
-100000000000000000000000000000000000000000000000000000000:-72057594037927936
-100000000000000000000000000000000000000000000000000000001:-72057594037927937
-100000000000000000000000000000000000000000000000000000010:-72057594037927938

###############################################################################
# Absolute value is close to 2^57 = 144115188075855872
###############################################################################

# Unsigned

0b111111111111111111111111111111111111111111111111111111110:144115188075855870
0b111111111111111111111111111111111111111111111111111111111:144115188075855871
0b1000000000000000000000000000000000000000000000000000000000:144115188075855872
0b1000000000000000000000000000000000000000000000000000000001:144115188075855873
0b1000000000000000000000000000000000000000000000000000000010:144115188075855874

b111111111111111111111111111111111111111111111111111111110:NaN
b111111111111111111111111111111111111111111111111111111111:NaN
b1000000000000000000000000000000000000000000000000000000000:NaN
b1000000000000000000000000000000000000000000000000000000001:NaN
b1000000000000000000000000000000000000000000000000000000010:NaN

111111111111111111111111111111111111111111111111111111110:144115188075855870
111111111111111111111111111111111111111111111111111111111:144115188075855871
1000000000000000000000000000000000000000000000000000000000:144115188075855872
1000000000000000000000000000000000000000000000000000000001:144115188075855873
1000000000000000000000000000000000000000000000000000000010:144115188075855874

# Positive

+0b111111111111111111111111111111111111111111111111111111110:144115188075855870
+0b111111111111111111111111111111111111111111111111111111111:144115188075855871
+0b1000000000000000000000000000000000000000000000000000000000:144115188075855872
+0b1000000000000000000000000000000000000000000000000000000001:144115188075855873
+0b1000000000000000000000000000000000000000000000000000000010:144115188075855874

+b111111111111111111111111111111111111111111111111111111110:NaN
+b111111111111111111111111111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000000000000000000000000000:NaN
+b1000000000000000000000000000000000000000000000000000000001:NaN
+b1000000000000000000000000000000000000000000000000000000010:NaN

+111111111111111111111111111111111111111111111111111111110:144115188075855870
+111111111111111111111111111111111111111111111111111111111:144115188075855871
+1000000000000000000000000000000000000000000000000000000000:144115188075855872
+1000000000000000000000000000000000000000000000000000000001:144115188075855873
+1000000000000000000000000000000000000000000000000000000010:144115188075855874

# Negative

-0b111111111111111111111111111111111111111111111111111111110:-144115188075855870
-0b111111111111111111111111111111111111111111111111111111111:-144115188075855871
-0b1000000000000000000000000000000000000000000000000000000000:-144115188075855872
-0b1000000000000000000000000000000000000000000000000000000001:-144115188075855873
-0b1000000000000000000000000000000000000000000000000000000010:-144115188075855874

-b111111111111111111111111111111111111111111111111111111110:NaN
-b111111111111111111111111111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000000000000000000000000000:NaN
-b1000000000000000000000000000000000000000000000000000000001:NaN
-b1000000000000000000000000000000000000000000000000000000010:NaN

-111111111111111111111111111111111111111111111111111111110:-144115188075855870
-111111111111111111111111111111111111111111111111111111111:-144115188075855871
-1000000000000000000000000000000000000000000000000000000000:-144115188075855872
-1000000000000000000000000000000000000000000000000000000001:-144115188075855873
-1000000000000000000000000000000000000000000000000000000010:-144115188075855874

###############################################################################
# Absolute value is close to 2^58 = 288230376151711744
###############################################################################

# Unsigned

0b1111111111111111111111111111111111111111111111111111111110:288230376151711742
0b1111111111111111111111111111111111111111111111111111111111:288230376151711743
0b10000000000000000000000000000000000000000000000000000000000:288230376151711744
0b10000000000000000000000000000000000000000000000000000000001:288230376151711745
0b10000000000000000000000000000000000000000000000000000000010:288230376151711746

b1111111111111111111111111111111111111111111111111111111110:NaN
b1111111111111111111111111111111111111111111111111111111111:NaN
b10000000000000000000000000000000000000000000000000000000000:NaN
b10000000000000000000000000000000000000000000000000000000001:NaN
b10000000000000000000000000000000000000000000000000000000010:NaN

1111111111111111111111111111111111111111111111111111111110:288230376151711742
1111111111111111111111111111111111111111111111111111111111:288230376151711743
10000000000000000000000000000000000000000000000000000000000:288230376151711744
10000000000000000000000000000000000000000000000000000000001:288230376151711745
10000000000000000000000000000000000000000000000000000000010:288230376151711746

# Positive

+0b1111111111111111111111111111111111111111111111111111111110:288230376151711742
+0b1111111111111111111111111111111111111111111111111111111111:288230376151711743
+0b10000000000000000000000000000000000000000000000000000000000:288230376151711744
+0b10000000000000000000000000000000000000000000000000000000001:288230376151711745
+0b10000000000000000000000000000000000000000000000000000000010:288230376151711746

+b1111111111111111111111111111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000000000000000000000000000:NaN
+b10000000000000000000000000000000000000000000000000000000001:NaN
+b10000000000000000000000000000000000000000000000000000000010:NaN

+1111111111111111111111111111111111111111111111111111111110:288230376151711742
+1111111111111111111111111111111111111111111111111111111111:288230376151711743
+10000000000000000000000000000000000000000000000000000000000:288230376151711744
+10000000000000000000000000000000000000000000000000000000001:288230376151711745
+10000000000000000000000000000000000000000000000000000000010:288230376151711746

# Negative

-0b1111111111111111111111111111111111111111111111111111111110:-288230376151711742
-0b1111111111111111111111111111111111111111111111111111111111:-288230376151711743
-0b10000000000000000000000000000000000000000000000000000000000:-288230376151711744
-0b10000000000000000000000000000000000000000000000000000000001:-288230376151711745
-0b10000000000000000000000000000000000000000000000000000000010:-288230376151711746

-b1111111111111111111111111111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000000000000000000000000000:NaN
-b10000000000000000000000000000000000000000000000000000000001:NaN
-b10000000000000000000000000000000000000000000000000000000010:NaN

-1111111111111111111111111111111111111111111111111111111110:-288230376151711742
-1111111111111111111111111111111111111111111111111111111111:-288230376151711743
-10000000000000000000000000000000000000000000000000000000000:-288230376151711744
-10000000000000000000000000000000000000000000000000000000001:-288230376151711745
-10000000000000000000000000000000000000000000000000000000010:-288230376151711746

###############################################################################
# Absolute value is close to 2^59 = 576460752303423488
###############################################################################

# Unsigned

0b11111111111111111111111111111111111111111111111111111111110:576460752303423486
0b11111111111111111111111111111111111111111111111111111111111:576460752303423487
0b100000000000000000000000000000000000000000000000000000000000:576460752303423488
0b100000000000000000000000000000000000000000000000000000000001:576460752303423489
0b100000000000000000000000000000000000000000000000000000000010:576460752303423490

b11111111111111111111111111111111111111111111111111111111110:NaN
b11111111111111111111111111111111111111111111111111111111111:NaN
b100000000000000000000000000000000000000000000000000000000000:NaN
b100000000000000000000000000000000000000000000000000000000001:NaN
b100000000000000000000000000000000000000000000000000000000010:NaN

11111111111111111111111111111111111111111111111111111111110:576460752303423486
11111111111111111111111111111111111111111111111111111111111:576460752303423487
100000000000000000000000000000000000000000000000000000000000:576460752303423488
100000000000000000000000000000000000000000000000000000000001:576460752303423489
100000000000000000000000000000000000000000000000000000000010:576460752303423490

# Positive

+0b11111111111111111111111111111111111111111111111111111111110:576460752303423486
+0b11111111111111111111111111111111111111111111111111111111111:576460752303423487
+0b100000000000000000000000000000000000000000000000000000000000:576460752303423488
+0b100000000000000000000000000000000000000000000000000000000001:576460752303423489
+0b100000000000000000000000000000000000000000000000000000000010:576460752303423490

+b11111111111111111111111111111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000000000000000000000000000:NaN
+b100000000000000000000000000000000000000000000000000000000001:NaN
+b100000000000000000000000000000000000000000000000000000000010:NaN

+11111111111111111111111111111111111111111111111111111111110:576460752303423486
+11111111111111111111111111111111111111111111111111111111111:576460752303423487
+100000000000000000000000000000000000000000000000000000000000:576460752303423488
+100000000000000000000000000000000000000000000000000000000001:576460752303423489
+100000000000000000000000000000000000000000000000000000000010:576460752303423490

# Negative

-0b11111111111111111111111111111111111111111111111111111111110:-576460752303423486
-0b11111111111111111111111111111111111111111111111111111111111:-576460752303423487
-0b100000000000000000000000000000000000000000000000000000000000:-576460752303423488
-0b100000000000000000000000000000000000000000000000000000000001:-576460752303423489
-0b100000000000000000000000000000000000000000000000000000000010:-576460752303423490

-b11111111111111111111111111111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000000000000000000000000000:NaN
-b100000000000000000000000000000000000000000000000000000000001:NaN
-b100000000000000000000000000000000000000000000000000000000010:NaN

-11111111111111111111111111111111111111111111111111111111110:-576460752303423486
-11111111111111111111111111111111111111111111111111111111111:-576460752303423487
-100000000000000000000000000000000000000000000000000000000000:-576460752303423488
-100000000000000000000000000000000000000000000000000000000001:-576460752303423489
-100000000000000000000000000000000000000000000000000000000010:-576460752303423490

###############################################################################
# Absolute value is close to 2^60 = 1152921504606846976
###############################################################################

# Unsigned

0b111111111111111111111111111111111111111111111111111111111110:1152921504606846974
0b111111111111111111111111111111111111111111111111111111111111:1152921504606846975
0b1000000000000000000000000000000000000000000000000000000000000:1152921504606846976
0b1000000000000000000000000000000000000000000000000000000000001:1152921504606846977
0b1000000000000000000000000000000000000000000000000000000000010:1152921504606846978

b111111111111111111111111111111111111111111111111111111111110:NaN
b111111111111111111111111111111111111111111111111111111111111:NaN
b1000000000000000000000000000000000000000000000000000000000000:NaN
b1000000000000000000000000000000000000000000000000000000000001:NaN
b1000000000000000000000000000000000000000000000000000000000010:NaN

111111111111111111111111111111111111111111111111111111111110:1152921504606846974
111111111111111111111111111111111111111111111111111111111111:1152921504606846975
1000000000000000000000000000000000000000000000000000000000000:1152921504606846976
1000000000000000000000000000000000000000000000000000000000001:1152921504606846977
1000000000000000000000000000000000000000000000000000000000010:1152921504606846978

# Positive

+0b111111111111111111111111111111111111111111111111111111111110:1152921504606846974
+0b111111111111111111111111111111111111111111111111111111111111:1152921504606846975
+0b1000000000000000000000000000000000000000000000000000000000000:1152921504606846976
+0b1000000000000000000000000000000000000000000000000000000000001:1152921504606846977
+0b1000000000000000000000000000000000000000000000000000000000010:1152921504606846978

+b111111111111111111111111111111111111111111111111111111111110:NaN
+b111111111111111111111111111111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000000000000000000000000000000:NaN
+b1000000000000000000000000000000000000000000000000000000000001:NaN
+b1000000000000000000000000000000000000000000000000000000000010:NaN

+111111111111111111111111111111111111111111111111111111111110:1152921504606846974
+111111111111111111111111111111111111111111111111111111111111:1152921504606846975
+1000000000000000000000000000000000000000000000000000000000000:1152921504606846976
+1000000000000000000000000000000000000000000000000000000000001:1152921504606846977
+1000000000000000000000000000000000000000000000000000000000010:1152921504606846978

# Negative

-0b111111111111111111111111111111111111111111111111111111111110:-1152921504606846974
-0b111111111111111111111111111111111111111111111111111111111111:-1152921504606846975
-0b1000000000000000000000000000000000000000000000000000000000000:-1152921504606846976
-0b1000000000000000000000000000000000000000000000000000000000001:-1152921504606846977
-0b1000000000000000000000000000000000000000000000000000000000010:-1152921504606846978

-b111111111111111111111111111111111111111111111111111111111110:NaN
-b111111111111111111111111111111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000000000000000000000000000000:NaN
-b1000000000000000000000000000000000000000000000000000000000001:NaN
-b1000000000000000000000000000000000000000000000000000000000010:NaN

-111111111111111111111111111111111111111111111111111111111110:-1152921504606846974
-111111111111111111111111111111111111111111111111111111111111:-1152921504606846975
-1000000000000000000000000000000000000000000000000000000000000:-1152921504606846976
-1000000000000000000000000000000000000000000000000000000000001:-1152921504606846977
-1000000000000000000000000000000000000000000000000000000000010:-1152921504606846978

###############################################################################
# Absolute value is close to 2^61 = 2305843009213693952
###############################################################################

# Unsigned

0b1111111111111111111111111111111111111111111111111111111111110:2305843009213693950
0b1111111111111111111111111111111111111111111111111111111111111:2305843009213693951
0b10000000000000000000000000000000000000000000000000000000000000:2305843009213693952
0b10000000000000000000000000000000000000000000000000000000000001:2305843009213693953
0b10000000000000000000000000000000000000000000000000000000000010:2305843009213693954

b1111111111111111111111111111111111111111111111111111111111110:NaN
b1111111111111111111111111111111111111111111111111111111111111:NaN
b10000000000000000000000000000000000000000000000000000000000000:NaN
b10000000000000000000000000000000000000000000000000000000000001:NaN
b10000000000000000000000000000000000000000000000000000000000010:NaN

1111111111111111111111111111111111111111111111111111111111110:2305843009213693950
1111111111111111111111111111111111111111111111111111111111111:2305843009213693951
10000000000000000000000000000000000000000000000000000000000000:2305843009213693952
10000000000000000000000000000000000000000000000000000000000001:2305843009213693953
10000000000000000000000000000000000000000000000000000000000010:2305843009213693954

# Positive

+0b1111111111111111111111111111111111111111111111111111111111110:2305843009213693950
+0b1111111111111111111111111111111111111111111111111111111111111:2305843009213693951
+0b10000000000000000000000000000000000000000000000000000000000000:2305843009213693952
+0b10000000000000000000000000000000000000000000000000000000000001:2305843009213693953
+0b10000000000000000000000000000000000000000000000000000000000010:2305843009213693954

+b1111111111111111111111111111111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000000000000000000000000000000:NaN
+b10000000000000000000000000000000000000000000000000000000000001:NaN
+b10000000000000000000000000000000000000000000000000000000000010:NaN

+1111111111111111111111111111111111111111111111111111111111110:2305843009213693950
+1111111111111111111111111111111111111111111111111111111111111:2305843009213693951
+10000000000000000000000000000000000000000000000000000000000000:2305843009213693952
+10000000000000000000000000000000000000000000000000000000000001:2305843009213693953
+10000000000000000000000000000000000000000000000000000000000010:2305843009213693954

# Negative

-0b1111111111111111111111111111111111111111111111111111111111110:-2305843009213693950
-0b1111111111111111111111111111111111111111111111111111111111111:-2305843009213693951
-0b10000000000000000000000000000000000000000000000000000000000000:-2305843009213693952
-0b10000000000000000000000000000000000000000000000000000000000001:-2305843009213693953
-0b10000000000000000000000000000000000000000000000000000000000010:-2305843009213693954

-b1111111111111111111111111111111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000000000000000000000000000000:NaN
-b10000000000000000000000000000000000000000000000000000000000001:NaN
-b10000000000000000000000000000000000000000000000000000000000010:NaN

-1111111111111111111111111111111111111111111111111111111111110:-2305843009213693950
-1111111111111111111111111111111111111111111111111111111111111:-2305843009213693951
-10000000000000000000000000000000000000000000000000000000000000:-2305843009213693952
-10000000000000000000000000000000000000000000000000000000000001:-2305843009213693953
-10000000000000000000000000000000000000000000000000000000000010:-2305843009213693954

###############################################################################
# Absolute value is close to 2^62 = 4611686018427387904
###############################################################################

# Unsigned

0b11111111111111111111111111111111111111111111111111111111111110:4611686018427387902
0b11111111111111111111111111111111111111111111111111111111111111:4611686018427387903
0b100000000000000000000000000000000000000000000000000000000000000:4611686018427387904
0b100000000000000000000000000000000000000000000000000000000000001:4611686018427387905
0b100000000000000000000000000000000000000000000000000000000000010:4611686018427387906

b11111111111111111111111111111111111111111111111111111111111110:NaN
b11111111111111111111111111111111111111111111111111111111111111:NaN
b100000000000000000000000000000000000000000000000000000000000000:NaN
b100000000000000000000000000000000000000000000000000000000000001:NaN
b100000000000000000000000000000000000000000000000000000000000010:NaN

11111111111111111111111111111111111111111111111111111111111110:4611686018427387902
11111111111111111111111111111111111111111111111111111111111111:4611686018427387903
100000000000000000000000000000000000000000000000000000000000000:4611686018427387904
100000000000000000000000000000000000000000000000000000000000001:4611686018427387905
100000000000000000000000000000000000000000000000000000000000010:4611686018427387906

# Positive

+0b11111111111111111111111111111111111111111111111111111111111110:4611686018427387902
+0b11111111111111111111111111111111111111111111111111111111111111:4611686018427387903
+0b100000000000000000000000000000000000000000000000000000000000000:4611686018427387904
+0b100000000000000000000000000000000000000000000000000000000000001:4611686018427387905
+0b100000000000000000000000000000000000000000000000000000000000010:4611686018427387906

+b11111111111111111111111111111111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000000000000000000000000000000:NaN
+b100000000000000000000000000000000000000000000000000000000000001:NaN
+b100000000000000000000000000000000000000000000000000000000000010:NaN

+11111111111111111111111111111111111111111111111111111111111110:4611686018427387902
+11111111111111111111111111111111111111111111111111111111111111:4611686018427387903
+100000000000000000000000000000000000000000000000000000000000000:4611686018427387904
+100000000000000000000000000000000000000000000000000000000000001:4611686018427387905
+100000000000000000000000000000000000000000000000000000000000010:4611686018427387906

# Negative

-0b11111111111111111111111111111111111111111111111111111111111110:-4611686018427387902
-0b11111111111111111111111111111111111111111111111111111111111111:-4611686018427387903
-0b100000000000000000000000000000000000000000000000000000000000000:-4611686018427387904
-0b100000000000000000000000000000000000000000000000000000000000001:-4611686018427387905
-0b100000000000000000000000000000000000000000000000000000000000010:-4611686018427387906

-b11111111111111111111111111111111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000000000000000000000000000000:NaN
-b100000000000000000000000000000000000000000000000000000000000001:NaN
-b100000000000000000000000000000000000000000000000000000000000010:NaN

-11111111111111111111111111111111111111111111111111111111111110:-4611686018427387902
-11111111111111111111111111111111111111111111111111111111111111:-4611686018427387903
-100000000000000000000000000000000000000000000000000000000000000:-4611686018427387904
-100000000000000000000000000000000000000000000000000000000000001:-4611686018427387905
-100000000000000000000000000000000000000000000000000000000000010:-4611686018427387906

###############################################################################
# Absolute value is close to 2^63 = 9223372036854775808
###############################################################################

# Unsigned

0b111111111111111111111111111111111111111111111111111111111111110:9223372036854775806
0b111111111111111111111111111111111111111111111111111111111111111:9223372036854775807
0b1000000000000000000000000000000000000000000000000000000000000000:9223372036854775808
0b1000000000000000000000000000000000000000000000000000000000000001:9223372036854775809
0b1000000000000000000000000000000000000000000000000000000000000010:9223372036854775810

b111111111111111111111111111111111111111111111111111111111111110:NaN
b111111111111111111111111111111111111111111111111111111111111111:NaN
b1000000000000000000000000000000000000000000000000000000000000000:NaN
b1000000000000000000000000000000000000000000000000000000000000001:NaN
b1000000000000000000000000000000000000000000000000000000000000010:NaN

111111111111111111111111111111111111111111111111111111111111110:9223372036854775806
111111111111111111111111111111111111111111111111111111111111111:9223372036854775807
1000000000000000000000000000000000000000000000000000000000000000:9223372036854775808
1000000000000000000000000000000000000000000000000000000000000001:9223372036854775809
1000000000000000000000000000000000000000000000000000000000000010:9223372036854775810

# Positive

+0b111111111111111111111111111111111111111111111111111111111111110:9223372036854775806
+0b111111111111111111111111111111111111111111111111111111111111111:9223372036854775807
+0b1000000000000000000000000000000000000000000000000000000000000000:9223372036854775808
+0b1000000000000000000000000000000000000000000000000000000000000001:9223372036854775809
+0b1000000000000000000000000000000000000000000000000000000000000010:9223372036854775810

+b111111111111111111111111111111111111111111111111111111111111110:NaN
+b111111111111111111111111111111111111111111111111111111111111111:NaN
+b1000000000000000000000000000000000000000000000000000000000000000:NaN
+b1000000000000000000000000000000000000000000000000000000000000001:NaN
+b1000000000000000000000000000000000000000000000000000000000000010:NaN

+111111111111111111111111111111111111111111111111111111111111110:9223372036854775806
+111111111111111111111111111111111111111111111111111111111111111:9223372036854775807
+1000000000000000000000000000000000000000000000000000000000000000:9223372036854775808
+1000000000000000000000000000000000000000000000000000000000000001:9223372036854775809
+1000000000000000000000000000000000000000000000000000000000000010:9223372036854775810

# Negative

-0b111111111111111111111111111111111111111111111111111111111111110:-9223372036854775806
-0b111111111111111111111111111111111111111111111111111111111111111:-9223372036854775807
-0b1000000000000000000000000000000000000000000000000000000000000000:-9223372036854775808
-0b1000000000000000000000000000000000000000000000000000000000000001:-9223372036854775809
-0b1000000000000000000000000000000000000000000000000000000000000010:-9223372036854775810

-b111111111111111111111111111111111111111111111111111111111111110:NaN
-b111111111111111111111111111111111111111111111111111111111111111:NaN
-b1000000000000000000000000000000000000000000000000000000000000000:NaN
-b1000000000000000000000000000000000000000000000000000000000000001:NaN
-b1000000000000000000000000000000000000000000000000000000000000010:NaN

-111111111111111111111111111111111111111111111111111111111111110:-9223372036854775806
-111111111111111111111111111111111111111111111111111111111111111:-9223372036854775807
-1000000000000000000000000000000000000000000000000000000000000000:-9223372036854775808
-1000000000000000000000000000000000000000000000000000000000000001:-9223372036854775809
-1000000000000000000000000000000000000000000000000000000000000010:-9223372036854775810

###############################################################################
# Absolute value is close to 2^64 = 18446744073709551616
###############################################################################

# Unsigned

0b1111111111111111111111111111111111111111111111111111111111111110:18446744073709551614
0b1111111111111111111111111111111111111111111111111111111111111111:18446744073709551615
0b10000000000000000000000000000000000000000000000000000000000000000:18446744073709551616
0b10000000000000000000000000000000000000000000000000000000000000001:18446744073709551617
0b10000000000000000000000000000000000000000000000000000000000000010:18446744073709551618

b1111111111111111111111111111111111111111111111111111111111111110:NaN
b1111111111111111111111111111111111111111111111111111111111111111:NaN
b10000000000000000000000000000000000000000000000000000000000000000:NaN
b10000000000000000000000000000000000000000000000000000000000000001:NaN
b10000000000000000000000000000000000000000000000000000000000000010:NaN

1111111111111111111111111111111111111111111111111111111111111110:18446744073709551614
1111111111111111111111111111111111111111111111111111111111111111:18446744073709551615
10000000000000000000000000000000000000000000000000000000000000000:18446744073709551616
10000000000000000000000000000000000000000000000000000000000000001:18446744073709551617
10000000000000000000000000000000000000000000000000000000000000010:18446744073709551618

# Positive

+0b1111111111111111111111111111111111111111111111111111111111111110:18446744073709551614
+0b1111111111111111111111111111111111111111111111111111111111111111:18446744073709551615
+0b10000000000000000000000000000000000000000000000000000000000000000:18446744073709551616
+0b10000000000000000000000000000000000000000000000000000000000000001:18446744073709551617
+0b10000000000000000000000000000000000000000000000000000000000000010:18446744073709551618

+b1111111111111111111111111111111111111111111111111111111111111110:NaN
+b1111111111111111111111111111111111111111111111111111111111111111:NaN
+b10000000000000000000000000000000000000000000000000000000000000000:NaN
+b10000000000000000000000000000000000000000000000000000000000000001:NaN
+b10000000000000000000000000000000000000000000000000000000000000010:NaN

+1111111111111111111111111111111111111111111111111111111111111110:18446744073709551614
+1111111111111111111111111111111111111111111111111111111111111111:18446744073709551615
+10000000000000000000000000000000000000000000000000000000000000000:18446744073709551616
+10000000000000000000000000000000000000000000000000000000000000001:18446744073709551617
+10000000000000000000000000000000000000000000000000000000000000010:18446744073709551618

# Negative

-0b1111111111111111111111111111111111111111111111111111111111111110:-18446744073709551614
-0b1111111111111111111111111111111111111111111111111111111111111111:-18446744073709551615
-0b10000000000000000000000000000000000000000000000000000000000000000:-18446744073709551616
-0b10000000000000000000000000000000000000000000000000000000000000001:-18446744073709551617
-0b10000000000000000000000000000000000000000000000000000000000000010:-18446744073709551618

-b1111111111111111111111111111111111111111111111111111111111111110:NaN
-b1111111111111111111111111111111111111111111111111111111111111111:NaN
-b10000000000000000000000000000000000000000000000000000000000000000:NaN
-b10000000000000000000000000000000000000000000000000000000000000001:NaN
-b10000000000000000000000000000000000000000000000000000000000000010:NaN

-1111111111111111111111111111111111111111111111111111111111111110:-18446744073709551614
-1111111111111111111111111111111111111111111111111111111111111111:-18446744073709551615
-10000000000000000000000000000000000000000000000000000000000000000:-18446744073709551616
-10000000000000000000000000000000000000000000000000000000000000001:-18446744073709551617
-10000000000000000000000000000000000000000000000000000000000000010:-18446744073709551618

###############################################################################
# Absolute value is close to 2^65 = 36893488147419103232
###############################################################################

# Unsigned

0b11111111111111111111111111111111111111111111111111111111111111110:36893488147419103230
0b11111111111111111111111111111111111111111111111111111111111111111:36893488147419103231
0b100000000000000000000000000000000000000000000000000000000000000000:36893488147419103232
0b100000000000000000000000000000000000000000000000000000000000000001:36893488147419103233
0b100000000000000000000000000000000000000000000000000000000000000010:36893488147419103234

b11111111111111111111111111111111111111111111111111111111111111110:NaN
b11111111111111111111111111111111111111111111111111111111111111111:NaN
b100000000000000000000000000000000000000000000000000000000000000000:NaN
b100000000000000000000000000000000000000000000000000000000000000001:NaN
b100000000000000000000000000000000000000000000000000000000000000010:NaN

11111111111111111111111111111111111111111111111111111111111111110:36893488147419103230
11111111111111111111111111111111111111111111111111111111111111111:36893488147419103231
100000000000000000000000000000000000000000000000000000000000000000:36893488147419103232
100000000000000000000000000000000000000000000000000000000000000001:36893488147419103233
100000000000000000000000000000000000000000000000000000000000000010:36893488147419103234

# Positive

+0b11111111111111111111111111111111111111111111111111111111111111110:36893488147419103230
+0b11111111111111111111111111111111111111111111111111111111111111111:36893488147419103231
+0b100000000000000000000000000000000000000000000000000000000000000000:36893488147419103232
+0b100000000000000000000000000000000000000000000000000000000000000001:36893488147419103233
+0b100000000000000000000000000000000000000000000000000000000000000010:36893488147419103234

+b11111111111111111111111111111111111111111111111111111111111111110:NaN
+b11111111111111111111111111111111111111111111111111111111111111111:NaN
+b100000000000000000000000000000000000000000000000000000000000000000:NaN
+b100000000000000000000000000000000000000000000000000000000000000001:NaN
+b100000000000000000000000000000000000000000000000000000000000000010:NaN

+11111111111111111111111111111111111111111111111111111111111111110:36893488147419103230
+11111111111111111111111111111111111111111111111111111111111111111:36893488147419103231
+100000000000000000000000000000000000000000000000000000000000000000:36893488147419103232
+100000000000000000000000000000000000000000000000000000000000000001:36893488147419103233
+100000000000000000000000000000000000000000000000000000000000000010:36893488147419103234

# Negative

-0b11111111111111111111111111111111111111111111111111111111111111110:-36893488147419103230
-0b11111111111111111111111111111111111111111111111111111111111111111:-36893488147419103231
-0b100000000000000000000000000000000000000000000000000000000000000000:-36893488147419103232
-0b100000000000000000000000000000000000000000000000000000000000000001:-36893488147419103233
-0b100000000000000000000000000000000000000000000000000000000000000010:-36893488147419103234

-b11111111111111111111111111111111111111111111111111111111111111110:NaN
-b11111111111111111111111111111111111111111111111111111111111111111:NaN
-b100000000000000000000000000000000000000000000000000000000000000000:NaN
-b100000000000000000000000000000000000000000000000000000000000000001:NaN
-b100000000000000000000000000000000000000000000000000000000000000010:NaN

-11111111111111111111111111111111111111111111111111111111111111110:-36893488147419103230
-11111111111111111111111111111111111111111111111111111111111111111:-36893488147419103231
-100000000000000000000000000000000000000000000000000000000000000000:-36893488147419103232
-100000000000000000000000000000000000000000000000000000000000000001:-36893488147419103233
-100000000000000000000000000000000000000000000000000000000000000010:-36893488147419103234

###############################################################################
# Absolute value is close to 10^1 = 10
###############################################################################

# Unsigned

0b1000:8
0b1001:9
0b1010:10
0b1011:11
0b1100:12

b1000:NaN
b1001:NaN
b1010:NaN
b1011:NaN
b1100:NaN

1000:8
1001:9
1010:10
1011:11
1100:12

# Positive

+0b1000:8
+0b1001:9
+0b1010:10
+0b1011:11
+0b1100:12

+b1000:NaN
+b1001:NaN
+b1010:NaN
+b1011:NaN
+b1100:NaN

+1000:8
+1001:9
+1010:10
+1011:11
+1100:12

# Negative

-0b1000:-8
-0b1001:-9
-0b1010:-10
-0b1011:-11
-0b1100:-12

-b1000:NaN
-b1001:NaN
-b1010:NaN
-b1011:NaN
-b1100:NaN

-1000:-8
-1001:-9
-1010:-10
-1011:-11
-1100:-12

###############################################################################
# Absolute value is close to 10^2 = 100
###############################################################################

# Unsigned

0b1100010:98
0b1100011:99
0b1100100:100
0b1100101:101
0b1100110:102

b1100010:NaN
b1100011:NaN
b1100100:NaN
b1100101:NaN
b1100110:NaN

1100010:98
1100011:99
1100100:100
1100101:101
1100110:102

# Positive

+0b1100010:98
+0b1100011:99
+0b1100100:100
+0b1100101:101
+0b1100110:102

+b1100010:NaN
+b1100011:NaN
+b1100100:NaN
+b1100101:NaN
+b1100110:NaN

+1100010:98
+1100011:99
+1100100:100
+1100101:101
+1100110:102

# Negative

-0b1100010:-98
-0b1100011:-99
-0b1100100:-100
-0b1100101:-101
-0b1100110:-102

-b1100010:NaN
-b1100011:NaN
-b1100100:NaN
-b1100101:NaN
-b1100110:NaN

-1100010:-98
-1100011:-99
-1100100:-100
-1100101:-101
-1100110:-102

###############################################################################
# Absolute value is close to 10^3 = 1000
###############################################################################

# Unsigned

0b1111100110:998
0b1111100111:999
0b1111101000:1000
0b1111101001:1001
0b1111101010:1002

b1111100110:NaN
b1111100111:NaN
b1111101000:NaN
b1111101001:NaN
b1111101010:NaN

1111100110:998
1111100111:999
1111101000:1000
1111101001:1001
1111101010:1002

# Positive

+0b1111100110:998
+0b1111100111:999
+0b1111101000:1000
+0b1111101001:1001
+0b1111101010:1002

+b1111100110:NaN
+b1111100111:NaN
+b1111101000:NaN
+b1111101001:NaN
+b1111101010:NaN

+1111100110:998
+1111100111:999
+1111101000:1000
+1111101001:1001
+1111101010:1002

# Negative

-0b1111100110:-998
-0b1111100111:-999
-0b1111101000:-1000
-0b1111101001:-1001
-0b1111101010:-1002

-b1111100110:NaN
-b1111100111:NaN
-b1111101000:NaN
-b1111101001:NaN
-b1111101010:NaN

-1111100110:-998
-1111100111:-999
-1111101000:-1000
-1111101001:-1001
-1111101010:-1002

###############################################################################
# Absolute value is close to 10^4 = 10000
###############################################################################

# Unsigned

0b10011100001110:9998
0b10011100001111:9999
0b10011100010000:10000
0b10011100010001:10001
0b10011100010010:10002

b10011100001110:NaN
b10011100001111:NaN
b10011100010000:NaN
b10011100010001:NaN
b10011100010010:NaN

10011100001110:9998
10011100001111:9999
10011100010000:10000
10011100010001:10001
10011100010010:10002

# Positive

+0b10011100001110:9998
+0b10011100001111:9999
+0b10011100010000:10000
+0b10011100010001:10001
+0b10011100010010:10002

+b10011100001110:NaN
+b10011100001111:NaN
+b10011100010000:NaN
+b10011100010001:NaN
+b10011100010010:NaN

+10011100001110:9998
+10011100001111:9999
+10011100010000:10000
+10011100010001:10001
+10011100010010:10002

# Negative

-0b10011100001110:-9998
-0b10011100001111:-9999
-0b10011100010000:-10000
-0b10011100010001:-10001
-0b10011100010010:-10002

-b10011100001110:NaN
-b10011100001111:NaN
-b10011100010000:NaN
-b10011100010001:NaN
-b10011100010010:NaN

-10011100001110:-9998
-10011100001111:-9999
-10011100010000:-10000
-10011100010001:-10001
-10011100010010:-10002

###############################################################################
# Absolute value is close to 10^5 = 100000
###############################################################################

# Unsigned

0b11000011010011110:99998
0b11000011010011111:99999
0b11000011010100000:100000
0b11000011010100001:100001
0b11000011010100010:100002

b11000011010011110:NaN
b11000011010011111:NaN
b11000011010100000:NaN
b11000011010100001:NaN
b11000011010100010:NaN

11000011010011110:99998
11000011010011111:99999
11000011010100000:100000
11000011010100001:100001
11000011010100010:100002

# Positive

+0b11000011010011110:99998
+0b11000011010011111:99999
+0b11000011010100000:100000
+0b11000011010100001:100001
+0b11000011010100010:100002

+b11000011010011110:NaN
+b11000011010011111:NaN
+b11000011010100000:NaN
+b11000011010100001:NaN
+b11000011010100010:NaN

+11000011010011110:99998
+11000011010011111:99999
+11000011010100000:100000
+11000011010100001:100001
+11000011010100010:100002

# Negative

-0b11000011010011110:-99998
-0b11000011010011111:-99999
-0b11000011010100000:-100000
-0b11000011010100001:-100001
-0b11000011010100010:-100002

-b11000011010011110:NaN
-b11000011010011111:NaN
-b11000011010100000:NaN
-b11000011010100001:NaN
-b11000011010100010:NaN

-11000011010011110:-99998
-11000011010011111:-99999
-11000011010100000:-100000
-11000011010100001:-100001
-11000011010100010:-100002

###############################################################################
# Absolute value is close to 10^6 = 1000000
###############################################################################

# Unsigned

0b11110100001000111110:999998
0b11110100001000111111:999999
0b11110100001001000000:1000000
0b11110100001001000001:1000001
0b11110100001001000010:1000002

b11110100001000111110:NaN
b11110100001000111111:NaN
b11110100001001000000:NaN
b11110100001001000001:NaN
b11110100001001000010:NaN

11110100001000111110:999998
11110100001000111111:999999
11110100001001000000:1000000
11110100001001000001:1000001
11110100001001000010:1000002

# Positive

+0b11110100001000111110:999998
+0b11110100001000111111:999999
+0b11110100001001000000:1000000
+0b11110100001001000001:1000001
+0b11110100001001000010:1000002

+b11110100001000111110:NaN
+b11110100001000111111:NaN
+b11110100001001000000:NaN
+b11110100001001000001:NaN
+b11110100001001000010:NaN

+11110100001000111110:999998
+11110100001000111111:999999
+11110100001001000000:1000000
+11110100001001000001:1000001
+11110100001001000010:1000002

# Negative

-0b11110100001000111110:-999998
-0b11110100001000111111:-999999
-0b11110100001001000000:-1000000
-0b11110100001001000001:-1000001
-0b11110100001001000010:-1000002

-b11110100001000111110:NaN
-b11110100001000111111:NaN
-b11110100001001000000:NaN
-b11110100001001000001:NaN
-b11110100001001000010:NaN

-11110100001000111110:-999998
-11110100001000111111:-999999
-11110100001001000000:-1000000
-11110100001001000001:-1000001
-11110100001001000010:-1000002

###############################################################################
# Absolute value is close to 10^7 = 10000000
###############################################################################

# Unsigned

0b100110001001011001111110:9999998
0b100110001001011001111111:9999999
0b100110001001011010000000:10000000
0b100110001001011010000001:10000001
0b100110001001011010000010:10000002

b100110001001011001111110:NaN
b100110001001011001111111:NaN
b100110001001011010000000:NaN
b100110001001011010000001:NaN
b100110001001011010000010:NaN

100110001001011001111110:9999998
100110001001011001111111:9999999
100110001001011010000000:10000000
100110001001011010000001:10000001
100110001001011010000010:10000002

# Positive

+0b100110001001011001111110:9999998
+0b100110001001011001111111:9999999
+0b100110001001011010000000:10000000
+0b100110001001011010000001:10000001
+0b100110001001011010000010:10000002

+b100110001001011001111110:NaN
+b100110001001011001111111:NaN
+b100110001001011010000000:NaN
+b100110001001011010000001:NaN
+b100110001001011010000010:NaN

+100110001001011001111110:9999998
+100110001001011001111111:9999999
+100110001001011010000000:10000000
+100110001001011010000001:10000001
+100110001001011010000010:10000002

# Negative

-0b100110001001011001111110:-9999998
-0b100110001001011001111111:-9999999
-0b100110001001011010000000:-10000000
-0b100110001001011010000001:-10000001
-0b100110001001011010000010:-10000002

-b100110001001011001111110:NaN
-b100110001001011001111111:NaN
-b100110001001011010000000:NaN
-b100110001001011010000001:NaN
-b100110001001011010000010:NaN

-100110001001011001111110:-9999998
-100110001001011001111111:-9999999
-100110001001011010000000:-10000000
-100110001001011010000001:-10000001
-100110001001011010000010:-10000002

###############################################################################
# Absolute value is close to 10^8 = 100000000
###############################################################################

# Unsigned

0b101111101011110000011111110:99999998
0b101111101011110000011111111:99999999
0b101111101011110000100000000:100000000
0b101111101011110000100000001:100000001
0b101111101011110000100000010:100000002

b101111101011110000011111110:NaN
b101111101011110000011111111:NaN
b101111101011110000100000000:NaN
b101111101011110000100000001:NaN
b101111101011110000100000010:NaN

101111101011110000011111110:99999998
101111101011110000011111111:99999999
101111101011110000100000000:100000000
101111101011110000100000001:100000001
101111101011110000100000010:100000002

# Positive

+0b101111101011110000011111110:99999998
+0b101111101011110000011111111:99999999
+0b101111101011110000100000000:100000000
+0b101111101011110000100000001:100000001
+0b101111101011110000100000010:100000002

+b101111101011110000011111110:NaN
+b101111101011110000011111111:NaN
+b101111101011110000100000000:NaN
+b101111101011110000100000001:NaN
+b101111101011110000100000010:NaN

+101111101011110000011111110:99999998
+101111101011110000011111111:99999999
+101111101011110000100000000:100000000
+101111101011110000100000001:100000001
+101111101011110000100000010:100000002

# Negative

-0b101111101011110000011111110:-99999998
-0b101111101011110000011111111:-99999999
-0b101111101011110000100000000:-100000000
-0b101111101011110000100000001:-100000001
-0b101111101011110000100000010:-100000002

-b101111101011110000011111110:NaN
-b101111101011110000011111111:NaN
-b101111101011110000100000000:NaN
-b101111101011110000100000001:NaN
-b101111101011110000100000010:NaN

-101111101011110000011111110:-99999998
-101111101011110000011111111:-99999999
-101111101011110000100000000:-100000000
-101111101011110000100000001:-100000001
-101111101011110000100000010:-100000002

###############################################################################
# Absolute value is close to 10^9 = 1000000000
###############################################################################

# Unsigned

0b111011100110101100100111111110:999999998
0b111011100110101100100111111111:999999999
0b111011100110101100101000000000:1000000000
0b111011100110101100101000000001:1000000001
0b111011100110101100101000000010:1000000002

b111011100110101100100111111110:NaN
b111011100110101100100111111111:NaN
b111011100110101100101000000000:NaN
b111011100110101100101000000001:NaN
b111011100110101100101000000010:NaN

111011100110101100100111111110:999999998
111011100110101100100111111111:999999999
111011100110101100101000000000:1000000000
111011100110101100101000000001:1000000001
111011100110101100101000000010:1000000002

# Positive

+0b111011100110101100100111111110:999999998
+0b111011100110101100100111111111:999999999
+0b111011100110101100101000000000:1000000000
+0b111011100110101100101000000001:1000000001
+0b111011100110101100101000000010:1000000002

+b111011100110101100100111111110:NaN
+b111011100110101100100111111111:NaN
+b111011100110101100101000000000:NaN
+b111011100110101100101000000001:NaN
+b111011100110101100101000000010:NaN

+111011100110101100100111111110:999999998
+111011100110101100100111111111:999999999
+111011100110101100101000000000:1000000000
+111011100110101100101000000001:1000000001
+111011100110101100101000000010:1000000002

# Negative

-0b111011100110101100100111111110:-999999998
-0b111011100110101100100111111111:-999999999
-0b111011100110101100101000000000:-1000000000
-0b111011100110101100101000000001:-1000000001
-0b111011100110101100101000000010:-1000000002

-b111011100110101100100111111110:NaN
-b111011100110101100100111111111:NaN
-b111011100110101100101000000000:NaN
-b111011100110101100101000000001:NaN
-b111011100110101100101000000010:NaN

-111011100110101100100111111110:-999999998
-111011100110101100100111111111:-999999999
-111011100110101100101000000000:-1000000000
-111011100110101100101000000001:-1000000001
-111011100110101100101000000010:-1000000002

###############################################################################
# Absolute value is close to 10^10 = 10000000000
###############################################################################

# Unsigned

0b1001010100000010111110001111111110:9999999998
0b1001010100000010111110001111111111:9999999999
0b1001010100000010111110010000000000:10000000000
0b1001010100000010111110010000000001:10000000001
0b1001010100000010111110010000000010:10000000002

b1001010100000010111110001111111110:NaN
b1001010100000010111110001111111111:NaN
b1001010100000010111110010000000000:NaN
b1001010100000010111110010000000001:NaN
b1001010100000010111110010000000010:NaN

1001010100000010111110001111111110:9999999998
1001010100000010111110001111111111:9999999999
1001010100000010111110010000000000:10000000000
1001010100000010111110010000000001:10000000001
1001010100000010111110010000000010:10000000002

# Positive

+0b1001010100000010111110001111111110:9999999998
+0b1001010100000010111110001111111111:9999999999
+0b1001010100000010111110010000000000:10000000000
+0b1001010100000010111110010000000001:10000000001
+0b1001010100000010111110010000000010:10000000002

+b1001010100000010111110001111111110:NaN
+b1001010100000010111110001111111111:NaN
+b1001010100000010111110010000000000:NaN
+b1001010100000010111110010000000001:NaN
+b1001010100000010111110010000000010:NaN

+1001010100000010111110001111111110:9999999998
+1001010100000010111110001111111111:9999999999
+1001010100000010111110010000000000:10000000000
+1001010100000010111110010000000001:10000000001
+1001010100000010111110010000000010:10000000002

# Negative

-0b1001010100000010111110001111111110:-9999999998
-0b1001010100000010111110001111111111:-9999999999
-0b1001010100000010111110010000000000:-10000000000
-0b1001010100000010111110010000000001:-10000000001
-0b1001010100000010111110010000000010:-10000000002

-b1001010100000010111110001111111110:NaN
-b1001010100000010111110001111111111:NaN
-b1001010100000010111110010000000000:NaN
-b1001010100000010111110010000000001:NaN
-b1001010100000010111110010000000010:NaN

-1001010100000010111110001111111110:-9999999998
-1001010100000010111110001111111111:-9999999999
-1001010100000010111110010000000000:-10000000000
-1001010100000010111110010000000001:-10000000001
-1001010100000010111110010000000010:-10000000002

###############################################################################
# Absolute value is close to 10^11 = 100000000000
###############################################################################

# Unsigned

0b1011101001000011101101110011111111110:99999999998
0b1011101001000011101101110011111111111:99999999999
0b1011101001000011101101110100000000000:100000000000
0b1011101001000011101101110100000000001:100000000001
0b1011101001000011101101110100000000010:100000000002

b1011101001000011101101110011111111110:NaN
b1011101001000011101101110011111111111:NaN
b1011101001000011101101110100000000000:NaN
b1011101001000011101101110100000000001:NaN
b1011101001000011101101110100000000010:NaN

1011101001000011101101110011111111110:99999999998
1011101001000011101101110011111111111:99999999999
1011101001000011101101110100000000000:100000000000
1011101001000011101101110100000000001:100000000001
1011101001000011101101110100000000010:100000000002

# Positive

+0b1011101001000011101101110011111111110:99999999998
+0b1011101001000011101101110011111111111:99999999999
+0b1011101001000011101101110100000000000:100000000000
+0b1011101001000011101101110100000000001:100000000001
+0b1011101001000011101101110100000000010:100000000002

+b1011101001000011101101110011111111110:NaN
+b1011101001000011101101110011111111111:NaN
+b1011101001000011101101110100000000000:NaN
+b1011101001000011101101110100000000001:NaN
+b1011101001000011101101110100000000010:NaN

+1011101001000011101101110011111111110:99999999998
+1011101001000011101101110011111111111:99999999999
+1011101001000011101101110100000000000:100000000000
+1011101001000011101101110100000000001:100000000001
+1011101001000011101101110100000000010:100000000002

# Negative

-0b1011101001000011101101110011111111110:-99999999998
-0b1011101001000011101101110011111111111:-99999999999
-0b1011101001000011101101110100000000000:-100000000000
-0b1011101001000011101101110100000000001:-100000000001
-0b1011101001000011101101110100000000010:-100000000002

-b1011101001000011101101110011111111110:NaN
-b1011101001000011101101110011111111111:NaN
-b1011101001000011101101110100000000000:NaN
-b1011101001000011101101110100000000001:NaN
-b1011101001000011101101110100000000010:NaN

-1011101001000011101101110011111111110:-99999999998
-1011101001000011101101110011111111111:-99999999999
-1011101001000011101101110100000000000:-100000000000
-1011101001000011101101110100000000001:-100000000001
-1011101001000011101101110100000000010:-100000000002

###############################################################################
# Absolute value is close to 10^12 = 1000000000000
###############################################################################

# Unsigned

0b1110100011010100101001010000111111111110:999999999998
0b1110100011010100101001010000111111111111:999999999999
0b1110100011010100101001010001000000000000:1000000000000
0b1110100011010100101001010001000000000001:1000000000001
0b1110100011010100101001010001000000000010:1000000000002

b1110100011010100101001010000111111111110:NaN
b1110100011010100101001010000111111111111:NaN
b1110100011010100101001010001000000000000:NaN
b1110100011010100101001010001000000000001:NaN
b1110100011010100101001010001000000000010:NaN

1110100011010100101001010000111111111110:999999999998
1110100011010100101001010000111111111111:999999999999
1110100011010100101001010001000000000000:1000000000000
1110100011010100101001010001000000000001:1000000000001
1110100011010100101001010001000000000010:1000000000002

# Positive

+0b1110100011010100101001010000111111111110:999999999998
+0b1110100011010100101001010000111111111111:999999999999
+0b1110100011010100101001010001000000000000:1000000000000
+0b1110100011010100101001010001000000000001:1000000000001
+0b1110100011010100101001010001000000000010:1000000000002

+b1110100011010100101001010000111111111110:NaN
+b1110100011010100101001010000111111111111:NaN
+b1110100011010100101001010001000000000000:NaN
+b1110100011010100101001010001000000000001:NaN
+b1110100011010100101001010001000000000010:NaN

+1110100011010100101001010000111111111110:999999999998
+1110100011010100101001010000111111111111:999999999999
+1110100011010100101001010001000000000000:1000000000000
+1110100011010100101001010001000000000001:1000000000001
+1110100011010100101001010001000000000010:1000000000002

# Negative

-0b1110100011010100101001010000111111111110:-999999999998
-0b1110100011010100101001010000111111111111:-999999999999
-0b1110100011010100101001010001000000000000:-1000000000000
-0b1110100011010100101001010001000000000001:-1000000000001
-0b1110100011010100101001010001000000000010:-1000000000002

-b1110100011010100101001010000111111111110:NaN
-b1110100011010100101001010000111111111111:NaN
-b1110100011010100101001010001000000000000:NaN
-b1110100011010100101001010001000000000001:NaN
-b1110100011010100101001010001000000000010:NaN

-1110100011010100101001010000111111111110:-999999999998
-1110100011010100101001010000111111111111:-999999999999
-1110100011010100101001010001000000000000:-1000000000000
-1110100011010100101001010001000000000001:-1000000000001
-1110100011010100101001010001000000000010:-1000000000002

###############################################################################
# Absolute value is close to 10^13 = 10000000000000
###############################################################################

# Unsigned

0b10010001100001001110011100101001111111111110:9999999999998
0b10010001100001001110011100101001111111111111:9999999999999
0b10010001100001001110011100101010000000000000:10000000000000
0b10010001100001001110011100101010000000000001:10000000000001
0b10010001100001001110011100101010000000000010:10000000000002

b10010001100001001110011100101001111111111110:NaN
b10010001100001001110011100101001111111111111:NaN
b10010001100001001110011100101010000000000000:NaN
b10010001100001001110011100101010000000000001:NaN
b10010001100001001110011100101010000000000010:NaN

10010001100001001110011100101001111111111110:9999999999998
10010001100001001110011100101001111111111111:9999999999999
10010001100001001110011100101010000000000000:10000000000000
10010001100001001110011100101010000000000001:10000000000001
10010001100001001110011100101010000000000010:10000000000002

# Positive

+0b10010001100001001110011100101001111111111110:9999999999998
+0b10010001100001001110011100101001111111111111:9999999999999
+0b10010001100001001110011100101010000000000000:10000000000000
+0b10010001100001001110011100101010000000000001:10000000000001
+0b10010001100001001110011100101010000000000010:10000000000002

+b10010001100001001110011100101001111111111110:NaN
+b10010001100001001110011100101001111111111111:NaN
+b10010001100001001110011100101010000000000000:NaN
+b10010001100001001110011100101010000000000001:NaN
+b10010001100001001110011100101010000000000010:NaN

+10010001100001001110011100101001111111111110:9999999999998
+10010001100001001110011100101001111111111111:9999999999999
+10010001100001001110011100101010000000000000:10000000000000
+10010001100001001110011100101010000000000001:10000000000001
+10010001100001001110011100101010000000000010:10000000000002

# Negative

-0b10010001100001001110011100101001111111111110:-9999999999998
-0b10010001100001001110011100101001111111111111:-9999999999999
-0b10010001100001001110011100101010000000000000:-10000000000000
-0b10010001100001001110011100101010000000000001:-10000000000001
-0b10010001100001001110011100101010000000000010:-10000000000002

-b10010001100001001110011100101001111111111110:NaN
-b10010001100001001110011100101001111111111111:NaN
-b10010001100001001110011100101010000000000000:NaN
-b10010001100001001110011100101010000000000001:NaN
-b10010001100001001110011100101010000000000010:NaN

-10010001100001001110011100101001111111111110:-9999999999998
-10010001100001001110011100101001111111111111:-9999999999999
-10010001100001001110011100101010000000000000:-10000000000000
-10010001100001001110011100101010000000000001:-10000000000001
-10010001100001001110011100101010000000000010:-10000000000002

###############################################################################
# Absolute value is close to 10^14 = 100000000000000
###############################################################################

# Unsigned

0b10110101111001100010000011110100011111111111110:99999999999998
0b10110101111001100010000011110100011111111111111:99999999999999
0b10110101111001100010000011110100100000000000000:100000000000000
0b10110101111001100010000011110100100000000000001:100000000000001
0b10110101111001100010000011110100100000000000010:100000000000002

b10110101111001100010000011110100011111111111110:NaN
b10110101111001100010000011110100011111111111111:NaN
b10110101111001100010000011110100100000000000000:NaN
b10110101111001100010000011110100100000000000001:NaN
b10110101111001100010000011110100100000000000010:NaN

10110101111001100010000011110100011111111111110:99999999999998
10110101111001100010000011110100011111111111111:99999999999999
10110101111001100010000011110100100000000000000:100000000000000
10110101111001100010000011110100100000000000001:100000000000001
10110101111001100010000011110100100000000000010:100000000000002

# Positive

+0b10110101111001100010000011110100011111111111110:99999999999998
+0b10110101111001100010000011110100011111111111111:99999999999999
+0b10110101111001100010000011110100100000000000000:100000000000000
+0b10110101111001100010000011110100100000000000001:100000000000001
+0b10110101111001100010000011110100100000000000010:100000000000002

+b10110101111001100010000011110100011111111111110:NaN
+b10110101111001100010000011110100011111111111111:NaN
+b10110101111001100010000011110100100000000000000:NaN
+b10110101111001100010000011110100100000000000001:NaN
+b10110101111001100010000011110100100000000000010:NaN

+10110101111001100010000011110100011111111111110:99999999999998
+10110101111001100010000011110100011111111111111:99999999999999
+10110101111001100010000011110100100000000000000:100000000000000
+10110101111001100010000011110100100000000000001:100000000000001
+10110101111001100010000011110100100000000000010:100000000000002

# Negative

-0b10110101111001100010000011110100011111111111110:-99999999999998
-0b10110101111001100010000011110100011111111111111:-99999999999999
-0b10110101111001100010000011110100100000000000000:-100000000000000
-0b10110101111001100010000011110100100000000000001:-100000000000001
-0b10110101111001100010000011110100100000000000010:-100000000000002

-b10110101111001100010000011110100011111111111110:NaN
-b10110101111001100010000011110100011111111111111:NaN
-b10110101111001100010000011110100100000000000000:NaN
-b10110101111001100010000011110100100000000000001:NaN
-b10110101111001100010000011110100100000000000010:NaN

-10110101111001100010000011110100011111111111110:-99999999999998
-10110101111001100010000011110100011111111111111:-99999999999999
-10110101111001100010000011110100100000000000000:-100000000000000
-10110101111001100010000011110100100000000000001:-100000000000001
-10110101111001100010000011110100100000000000010:-100000000000002

###############################################################################
# Absolute value is close to 10^15 = 1000000000000000
###############################################################################

# Unsigned

0b11100011010111111010100100110001100111111111111110:999999999999998
0b11100011010111111010100100110001100111111111111111:999999999999999
0b11100011010111111010100100110001101000000000000000:1000000000000000
0b11100011010111111010100100110001101000000000000001:1000000000000001
0b11100011010111111010100100110001101000000000000010:1000000000000002

b11100011010111111010100100110001100111111111111110:NaN
b11100011010111111010100100110001100111111111111111:NaN
b11100011010111111010100100110001101000000000000000:NaN
b11100011010111111010100100110001101000000000000001:NaN
b11100011010111111010100100110001101000000000000010:NaN

11100011010111111010100100110001100111111111111110:999999999999998
11100011010111111010100100110001100111111111111111:999999999999999
11100011010111111010100100110001101000000000000000:1000000000000000
11100011010111111010100100110001101000000000000001:1000000000000001
11100011010111111010100100110001101000000000000010:1000000000000002

# Positive

+0b11100011010111111010100100110001100111111111111110:999999999999998
+0b11100011010111111010100100110001100111111111111111:999999999999999
+0b11100011010111111010100100110001101000000000000000:1000000000000000
+0b11100011010111111010100100110001101000000000000001:1000000000000001
+0b11100011010111111010100100110001101000000000000010:1000000000000002

+b11100011010111111010100100110001100111111111111110:NaN
+b11100011010111111010100100110001100111111111111111:NaN
+b11100011010111111010100100110001101000000000000000:NaN
+b11100011010111111010100100110001101000000000000001:NaN
+b11100011010111111010100100110001101000000000000010:NaN

+11100011010111111010100100110001100111111111111110:999999999999998
+11100011010111111010100100110001100111111111111111:999999999999999
+11100011010111111010100100110001101000000000000000:1000000000000000
+11100011010111111010100100110001101000000000000001:1000000000000001
+11100011010111111010100100110001101000000000000010:1000000000000002

# Negative

-0b11100011010111111010100100110001100111111111111110:-999999999999998
-0b11100011010111111010100100110001100111111111111111:-999999999999999
-0b11100011010111111010100100110001101000000000000000:-1000000000000000
-0b11100011010111111010100100110001101000000000000001:-1000000000000001
-0b11100011010111111010100100110001101000000000000010:-1000000000000002

-b11100011010111111010100100110001100111111111111110:NaN
-b11100011010111111010100100110001100111111111111111:NaN
-b11100011010111111010100100110001101000000000000000:NaN
-b11100011010111111010100100110001101000000000000001:NaN
-b11100011010111111010100100110001101000000000000010:NaN

-11100011010111111010100100110001100111111111111110:-999999999999998
-11100011010111111010100100110001100111111111111111:-999999999999999
-11100011010111111010100100110001101000000000000000:-1000000000000000
-11100011010111111010100100110001101000000000000001:-1000000000000001
-11100011010111111010100100110001101000000000000010:-1000000000000002

###############################################################################
# Absolute value is close to 10^16 = 10000000000000000
###############################################################################

# Unsigned

0b100011100001101111001001101111110000001111111111111110:9999999999999998
0b100011100001101111001001101111110000001111111111111111:9999999999999999
0b100011100001101111001001101111110000010000000000000000:10000000000000000
0b100011100001101111001001101111110000010000000000000001:10000000000000001
0b100011100001101111001001101111110000010000000000000010:10000000000000002

b100011100001101111001001101111110000001111111111111110:NaN
b100011100001101111001001101111110000001111111111111111:NaN
b100011100001101111001001101111110000010000000000000000:NaN
b100011100001101111001001101111110000010000000000000001:NaN
b100011100001101111001001101111110000010000000000000010:NaN

100011100001101111001001101111110000001111111111111110:9999999999999998
100011100001101111001001101111110000001111111111111111:9999999999999999
100011100001101111001001101111110000010000000000000000:10000000000000000
100011100001101111001001101111110000010000000000000001:10000000000000001
100011100001101111001001101111110000010000000000000010:10000000000000002

# Positive

+0b100011100001101111001001101111110000001111111111111110:9999999999999998
+0b100011100001101111001001101111110000001111111111111111:9999999999999999
+0b100011100001101111001001101111110000010000000000000000:10000000000000000
+0b100011100001101111001001101111110000010000000000000001:10000000000000001
+0b100011100001101111001001101111110000010000000000000010:10000000000000002

+b100011100001101111001001101111110000001111111111111110:NaN
+b100011100001101111001001101111110000001111111111111111:NaN
+b100011100001101111001001101111110000010000000000000000:NaN
+b100011100001101111001001101111110000010000000000000001:NaN
+b100011100001101111001001101111110000010000000000000010:NaN

+100011100001101111001001101111110000001111111111111110:9999999999999998
+100011100001101111001001101111110000001111111111111111:9999999999999999
+100011100001101111001001101111110000010000000000000000:10000000000000000
+100011100001101111001001101111110000010000000000000001:10000000000000001
+100011100001101111001001101111110000010000000000000010:10000000000000002

# Negative

-0b100011100001101111001001101111110000001111111111111110:-9999999999999998
-0b100011100001101111001001101111110000001111111111111111:-9999999999999999
-0b100011100001101111001001101111110000010000000000000000:-10000000000000000
-0b100011100001101111001001101111110000010000000000000001:-10000000000000001
-0b100011100001101111001001101111110000010000000000000010:-10000000000000002

-b100011100001101111001001101111110000001111111111111110:NaN
-b100011100001101111001001101111110000001111111111111111:NaN
-b100011100001101111001001101111110000010000000000000000:NaN
-b100011100001101111001001101111110000010000000000000001:NaN
-b100011100001101111001001101111110000010000000000000010:NaN

-100011100001101111001001101111110000001111111111111110:-9999999999999998
-100011100001101111001001101111110000001111111111111111:-9999999999999999
-100011100001101111001001101111110000010000000000000000:-10000000000000000
-100011100001101111001001101111110000010000000000000001:-10000000000000001
-100011100001101111001001101111110000010000000000000010:-10000000000000002

###############################################################################
# Absolute value is close to 10^17 = 100000000000000000
###############################################################################

# Unsigned

0b101100011010001010111100001011101100010011111111111111110:99999999999999998
0b101100011010001010111100001011101100010011111111111111111:99999999999999999
0b101100011010001010111100001011101100010100000000000000000:100000000000000000
0b101100011010001010111100001011101100010100000000000000001:100000000000000001
0b101100011010001010111100001011101100010100000000000000010:100000000000000002

b101100011010001010111100001011101100010011111111111111110:NaN
b101100011010001010111100001011101100010011111111111111111:NaN
b101100011010001010111100001011101100010100000000000000000:NaN
b101100011010001010111100001011101100010100000000000000001:NaN
b101100011010001010111100001011101100010100000000000000010:NaN

101100011010001010111100001011101100010011111111111111110:99999999999999998
101100011010001010111100001011101100010011111111111111111:99999999999999999
101100011010001010111100001011101100010100000000000000000:100000000000000000
101100011010001010111100001011101100010100000000000000001:100000000000000001
101100011010001010111100001011101100010100000000000000010:100000000000000002

# Positive

+0b101100011010001010111100001011101100010011111111111111110:99999999999999998
+0b101100011010001010111100001011101100010011111111111111111:99999999999999999
+0b101100011010001010111100001011101100010100000000000000000:100000000000000000
+0b101100011010001010111100001011101100010100000000000000001:100000000000000001
+0b101100011010001010111100001011101100010100000000000000010:100000000000000002

+b101100011010001010111100001011101100010011111111111111110:NaN
+b101100011010001010111100001011101100010011111111111111111:NaN
+b101100011010001010111100001011101100010100000000000000000:NaN
+b101100011010001010111100001011101100010100000000000000001:NaN
+b101100011010001010111100001011101100010100000000000000010:NaN

+101100011010001010111100001011101100010011111111111111110:99999999999999998
+101100011010001010111100001011101100010011111111111111111:99999999999999999
+101100011010001010111100001011101100010100000000000000000:100000000000000000
+101100011010001010111100001011101100010100000000000000001:100000000000000001
+101100011010001010111100001011101100010100000000000000010:100000000000000002

# Negative

-0b101100011010001010111100001011101100010011111111111111110:-99999999999999998
-0b101100011010001010111100001011101100010011111111111111111:-99999999999999999
-0b101100011010001010111100001011101100010100000000000000000:-100000000000000000
-0b101100011010001010111100001011101100010100000000000000001:-100000000000000001
-0b101100011010001010111100001011101100010100000000000000010:-100000000000000002

-b101100011010001010111100001011101100010011111111111111110:NaN
-b101100011010001010111100001011101100010011111111111111111:NaN
-b101100011010001010111100001011101100010100000000000000000:NaN
-b101100011010001010111100001011101100010100000000000000001:NaN
-b101100011010001010111100001011101100010100000000000000010:NaN

-101100011010001010111100001011101100010011111111111111110:-99999999999999998
-101100011010001010111100001011101100010011111111111111111:-99999999999999999
-101100011010001010111100001011101100010100000000000000000:-100000000000000000
-101100011010001010111100001011101100010100000000000000001:-100000000000000001
-101100011010001010111100001011101100010100000000000000010:-100000000000000002

###############################################################################
# Absolute value is close to 10^18 = 1000000000000000000
###############################################################################

# Unsigned

0b110111100000101101101011001110100111011000111111111111111110:999999999999999998
0b110111100000101101101011001110100111011000111111111111111111:999999999999999999
0b110111100000101101101011001110100111011001000000000000000000:1000000000000000000
0b110111100000101101101011001110100111011001000000000000000001:1000000000000000001
0b110111100000101101101011001110100111011001000000000000000010:1000000000000000002

b110111100000101101101011001110100111011000111111111111111110:NaN
b110111100000101101101011001110100111011000111111111111111111:NaN
b110111100000101101101011001110100111011001000000000000000000:NaN
b110111100000101101101011001110100111011001000000000000000001:NaN
b110111100000101101101011001110100111011001000000000000000010:NaN

110111100000101101101011001110100111011000111111111111111110:999999999999999998
110111100000101101101011001110100111011000111111111111111111:999999999999999999
110111100000101101101011001110100111011001000000000000000000:1000000000000000000
110111100000101101101011001110100111011001000000000000000001:1000000000000000001
110111100000101101101011001110100111011001000000000000000010:1000000000000000002

# Positive

+0b110111100000101101101011001110100111011000111111111111111110:999999999999999998
+0b110111100000101101101011001110100111011000111111111111111111:999999999999999999
+0b110111100000101101101011001110100111011001000000000000000000:1000000000000000000
+0b110111100000101101101011001110100111011001000000000000000001:1000000000000000001
+0b110111100000101101101011001110100111011001000000000000000010:1000000000000000002

+b110111100000101101101011001110100111011000111111111111111110:NaN
+b110111100000101101101011001110100111011000111111111111111111:NaN
+b110111100000101101101011001110100111011001000000000000000000:NaN
+b110111100000101101101011001110100111011001000000000000000001:NaN
+b110111100000101101101011001110100111011001000000000000000010:NaN

+110111100000101101101011001110100111011000111111111111111110:999999999999999998
+110111100000101101101011001110100111011000111111111111111111:999999999999999999
+110111100000101101101011001110100111011001000000000000000000:1000000000000000000
+110111100000101101101011001110100111011001000000000000000001:1000000000000000001
+110111100000101101101011001110100111011001000000000000000010:1000000000000000002

# Negative

-0b110111100000101101101011001110100111011000111111111111111110:-999999999999999998
-0b110111100000101101101011001110100111011000111111111111111111:-999999999999999999
-0b110111100000101101101011001110100111011001000000000000000000:-1000000000000000000
-0b110111100000101101101011001110100111011001000000000000000001:-1000000000000000001
-0b110111100000101101101011001110100111011001000000000000000010:-1000000000000000002

-b110111100000101101101011001110100111011000111111111111111110:NaN
-b110111100000101101101011001110100111011000111111111111111111:NaN
-b110111100000101101101011001110100111011001000000000000000000:NaN
-b110111100000101101101011001110100111011001000000000000000001:NaN
-b110111100000101101101011001110100111011001000000000000000010:NaN

-110111100000101101101011001110100111011000111111111111111110:-999999999999999998
-110111100000101101101011001110100111011000111111111111111111:-999999999999999999
-110111100000101101101011001110100111011001000000000000000000:-1000000000000000000
-110111100000101101101011001110100111011001000000000000000001:-1000000000000000001
-110111100000101101101011001110100111011001000000000000000010:-1000000000000000002

###############################################################################
# Absolute value is close to 10^19 = 10000000000000000000
###############################################################################

# Unsigned

0b1000101011000111001000110000010010001001111001111111111111111110:9999999999999999998
0b1000101011000111001000110000010010001001111001111111111111111111:9999999999999999999
0b1000101011000111001000110000010010001001111010000000000000000000:10000000000000000000
0b1000101011000111001000110000010010001001111010000000000000000001:10000000000000000001
0b1000101011000111001000110000010010001001111010000000000000000010:10000000000000000002

b1000101011000111001000110000010010001001111001111111111111111110:NaN
b1000101011000111001000110000010010001001111001111111111111111111:NaN
b1000101011000111001000110000010010001001111010000000000000000000:NaN
b1000101011000111001000110000010010001001111010000000000000000001:NaN
b1000101011000111001000110000010010001001111010000000000000000010:NaN

1000101011000111001000110000010010001001111001111111111111111110:9999999999999999998
1000101011000111001000110000010010001001111001111111111111111111:9999999999999999999
1000101011000111001000110000010010001001111010000000000000000000:10000000000000000000
1000101011000111001000110000010010001001111010000000000000000001:10000000000000000001
1000101011000111001000110000010010001001111010000000000000000010:10000000000000000002

# Positive

+0b1000101011000111001000110000010010001001111001111111111111111110:9999999999999999998
+0b1000101011000111001000110000010010001001111001111111111111111111:9999999999999999999
+0b1000101011000111001000110000010010001001111010000000000000000000:10000000000000000000
+0b1000101011000111001000110000010010001001111010000000000000000001:10000000000000000001
+0b1000101011000111001000110000010010001001111010000000000000000010:10000000000000000002

+b1000101011000111001000110000010010001001111001111111111111111110:NaN
+b1000101011000111001000110000010010001001111001111111111111111111:NaN
+b1000101011000111001000110000010010001001111010000000000000000000:NaN
+b1000101011000111001000110000010010001001111010000000000000000001:NaN
+b1000101011000111001000110000010010001001111010000000000000000010:NaN

+1000101011000111001000110000010010001001111001111111111111111110:9999999999999999998
+1000101011000111001000110000010010001001111001111111111111111111:9999999999999999999
+1000101011000111001000110000010010001001111010000000000000000000:10000000000000000000
+1000101011000111001000110000010010001001111010000000000000000001:10000000000000000001
+1000101011000111001000110000010010001001111010000000000000000010:10000000000000000002

# Negative

-0b1000101011000111001000110000010010001001111001111111111111111110:-9999999999999999998
-0b1000101011000111001000110000010010001001111001111111111111111111:-9999999999999999999
-0b1000101011000111001000110000010010001001111010000000000000000000:-10000000000000000000
-0b1000101011000111001000110000010010001001111010000000000000000001:-10000000000000000001
-0b1000101011000111001000110000010010001001111010000000000000000010:-10000000000000000002

-b1000101011000111001000110000010010001001111001111111111111111110:NaN
-b1000101011000111001000110000010010001001111001111111111111111111:NaN
-b1000101011000111001000110000010010001001111010000000000000000000:NaN
-b1000101011000111001000110000010010001001111010000000000000000001:NaN
-b1000101011000111001000110000010010001001111010000000000000000010:NaN

-1000101011000111001000110000010010001001111001111111111111111110:-9999999999999999998
-1000101011000111001000110000010010001001111001111111111111111111:-9999999999999999999
-1000101011000111001000110000010010001001111010000000000000000000:-10000000000000000000
-1000101011000111001000110000010010001001111010000000000000000001:-10000000000000000001
-1000101011000111001000110000010010001001111010000000000000000010:-10000000000000000002

###############################################################################
# Absolute value is close to 10^20 = 100000000000000000000
###############################################################################

# Unsigned

0b1010110101111000111010111100010110101100011000011111111111111111110:99999999999999999998
0b1010110101111000111010111100010110101100011000011111111111111111111:99999999999999999999
0b1010110101111000111010111100010110101100011000100000000000000000000:100000000000000000000
0b1010110101111000111010111100010110101100011000100000000000000000001:100000000000000000001
0b1010110101111000111010111100010110101100011000100000000000000000010:100000000000000000002

b1010110101111000111010111100010110101100011000011111111111111111110:NaN
b1010110101111000111010111100010110101100011000011111111111111111111:NaN
b1010110101111000111010111100010110101100011000100000000000000000000:NaN
b1010110101111000111010111100010110101100011000100000000000000000001:NaN
b1010110101111000111010111100010110101100011000100000000000000000010:NaN

1010110101111000111010111100010110101100011000011111111111111111110:99999999999999999998
1010110101111000111010111100010110101100011000011111111111111111111:99999999999999999999
1010110101111000111010111100010110101100011000100000000000000000000:100000000000000000000
1010110101111000111010111100010110101100011000100000000000000000001:100000000000000000001
1010110101111000111010111100010110101100011000100000000000000000010:100000000000000000002

# Positive

+0b1010110101111000111010111100010110101100011000011111111111111111110:99999999999999999998
+0b1010110101111000111010111100010110101100011000011111111111111111111:99999999999999999999
+0b1010110101111000111010111100010110101100011000100000000000000000000:100000000000000000000
+0b1010110101111000111010111100010110101100011000100000000000000000001:100000000000000000001
+0b1010110101111000111010111100010110101100011000100000000000000000010:100000000000000000002

+b1010110101111000111010111100010110101100011000011111111111111111110:NaN
+b1010110101111000111010111100010110101100011000011111111111111111111:NaN
+b1010110101111000111010111100010110101100011000100000000000000000000:NaN
+b1010110101111000111010111100010110101100011000100000000000000000001:NaN
+b1010110101111000111010111100010110101100011000100000000000000000010:NaN

+1010110101111000111010111100010110101100011000011111111111111111110:99999999999999999998
+1010110101111000111010111100010110101100011000011111111111111111111:99999999999999999999
+1010110101111000111010111100010110101100011000100000000000000000000:100000000000000000000
+1010110101111000111010111100010110101100011000100000000000000000001:100000000000000000001
+1010110101111000111010111100010110101100011000100000000000000000010:100000000000000000002

# Negative

-0b1010110101111000111010111100010110101100011000011111111111111111110:-99999999999999999998
-0b1010110101111000111010111100010110101100011000011111111111111111111:-99999999999999999999
-0b1010110101111000111010111100010110101100011000100000000000000000000:-100000000000000000000
-0b1010110101111000111010111100010110101100011000100000000000000000001:-100000000000000000001
-0b1010110101111000111010111100010110101100011000100000000000000000010:-100000000000000000002

-b1010110101111000111010111100010110101100011000011111111111111111110:NaN
-b1010110101111000111010111100010110101100011000011111111111111111111:NaN
-b1010110101111000111010111100010110101100011000100000000000000000000:NaN
-b1010110101111000111010111100010110101100011000100000000000000000001:NaN
-b1010110101111000111010111100010110101100011000100000000000000000010:NaN

-1010110101111000111010111100010110101100011000011111111111111111110:-99999999999999999998
-1010110101111000111010111100010110101100011000011111111111111111111:-99999999999999999999
-1010110101111000111010111100010110101100011000100000000000000000000:-100000000000000000000
-1010110101111000111010111100010110101100011000100000000000000000001:-100000000000000000001
-1010110101111000111010111100010110101100011000100000000000000000010:-100000000000000000002
