#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2020 -- leonerd@leonerd.org.uk

package Metrics::Any::Adapter::Prometheus;

use strict;
use warnings;

our $VERSION = '0.01';

use Carp;

use Net::Prometheus::Registry;

use Net::Prometheus::Counter;

=head1 NAME

C<Metrics::Any::Adapter::Prometheus> - a metrics reporting adapter for Prometheus

=head1 SYNOPSIS

   use Metrics::Any::Adapter 'Prometheus';

=head1 DESCRIPTION

This L<Metrics::Any> adapter type reports metrics to Prometheus by using
L<Net::Prometheus>. Each metric added to the adapter will be registered with
the global L<Net::Prometheus::Registry> instance.

It becomes the calling program's responsibility to arrange for these to be
HTTP accessible by using the C<Net::Prometheus> API.

=cut

sub new
{
   my $class = shift;
   return bless {
      stats => {},
   }, $class;
}

sub make_counter
{
   my $self = shift;
   my ( $handle, %args ) = @_;

   my $name = delete $args{name} // $handle;
   my $help = delete $args{description} // "Metrics::Any counter $handle";

   $self->{stats}{$handle} = Net::Prometheus::Registry->register(
      Net::Prometheus::Counter->new(
         name => $name,
         help => $help,
         %args,
      )
   );
}

sub inc_counter
{
   my $self = shift;
   my ( $handle, @labelvalues ) = @_;

   ( $self->{stats}{$handle} or croak "No such counter named '$handle'" )
      ->inc( @labelvalues );
}

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
