package Pod::Weaver::Plugin::Bencher::Scenario;

our $DATE = '2016-01-05'; # DATE
our $VERSION = '0.02'; # VERSION

use 5.010001;
use Moose;
with 'Pod::Weaver::Role::AddTextToSection';
with 'Pod::Weaver::Role::Section';

use Perinci::Result::Format::Lite;
use Bencher;

sub _md2pod {
    require Markdown::To::POD;

    my ($self, $md) = @_;
    my $pod = Markdown::To::POD::markdown_to_pod($md);
    # make sure we add a couple of blank lines in the end
    $pod =~ s/\s+\z//s;
    $pod . "\n\n\n";
}

sub _process_module {
    no strict 'refs';

    my ($self, $document, $input, $package) = @_;

    my $filename = $input->{filename};

    # XXX handle dynamically generated module (if there is such thing in the
    # future)
    local @INC = ("lib", @INC);

    {
        my $package_pm = $package;
        $package_pm =~ s!::!/!g;
        $package_pm .= ".pm";
        require $package_pm;
    }

    my $scenario = Bencher::parse_scenario(
        scenario => ${"$package\::scenario"});

    my $scenario_name = $package;
    $scenario_name =~ s/\ABencher::Scenario:://;

    # add Synopsis section
    {
        my @pod;
        push @pod, "To run benchmark with default option:\n\n",
            " % bencher -m $scenario_name\n\n";
        my @pmodules = Bencher::_get_participant_modules($scenario);
        if (@pmodules && !$scenario->{module_startup}) {
            push @pod, "To run module startup overhead benchmark:\n\n",
                " % bencher --module-startup -m $scenario_name\n\n";
        }
        push @pod, "For more options (dump scenario, list/include/exclude/add ",
            "participants, list/include/exclude/add datasets, etc), ",
            "see L<bencher> or run C<bencher --help>.\n\n";

        $self->add_text_to_section(
            $document, join("", @pod), 'SYNOPSIS',
            {
                after_section => ['VERSION', 'NAME'],
                before_section => 'DESCRIPTION',
                ignore => 1,
            });
    }

    # add Description section
    if ($scenario->{description}) {
        my @pod;

        push @pod, $self->_md2pod($scenario->{description});
        # blurb about Bencher
        push @pod, "Packaging a benchmark script as a Bencher scenario makes ",
            "it convenient to include/exclude/add participants/datasets (either ",
            "via CLI or Perl code), send the result to a central repository, ",
            "among others . See L<Bencher> and L<bencher> (CLI) ",
            "for more details.\n\n";

        $self->add_text_to_section(
            $document, join("", @pod), 'DESCRIPTION',
            {
                after_section => ['SYNOPSIS'],
                ignore => 1,
            });
    }

    my @modules = Bencher::_get_participant_modules($scenario);

    # add Benchmarked Modules section
    {
        last unless @modules;
        my $pod = join('', map {"L<$_>\n\n"} @modules);
        $self->add_text_to_section(
            $document, $pod, 'BENCHMARKED MODULES',
            {
                after_section => 'SYNOPSIS',
                before_section => 'DESCRIPTION',
            });
        # XXX if each participant is a unique module, then list with BENCHMARKED
        # MODULES as above. if there is a module which has two+ participants,
        # list like: *) L<Foo::Bar>'s C<routine1()>; *) C<Foo::Bar>'s
        # C<routine2()>.
    }

    # add Sample Benchmark Results section
    {
        my @pod;
        my ($res, $fres, $f2res);

        $res = Bencher::bencher(
            action => 'bench',
            scenario_module => $scenario_name,
        );
        $fres = Bencher::format_result($res);
        $f2res = Perinci::Result::Format::Lite::format($res, 'text-pretty');
        $f2res =~ s/^/ /gm;

        my $num_cores = $res->[3]{'func.cpu_info'}[0]{number_of_cores};
        push @pod, "Run on: ",
            "CPU: I<< ", $res->[3]{'func.cpu_info'}[0]{name}, " ($num_cores cores) >>, ",
            "OS: I<< ", $res->[3]{'func.platform_info'}{osname}, " ", $res->[3]{'func.platform_info'}{oslabel}, " version ", $res->[3]{'func.platform_info'}{osvers}, " >>, ",
            "OS kernel: I<< ", $res->[3]{'func.platform_info'}{kname}, " version ", $res->[3]{'func.platform_info'}{kvers}, " >>",
            ".\n\n";

        push @pod, "Benchmark with default option:\n\n", $f2res, "\n\n";

        if (@modules && !$scenario->{module_startup}) {
            $res = Bencher::bencher(
                action => 'bench',
                module_startup => 1,
                scenario_module => $scenario_name,
            );
            $fres = Bencher::format_result($res);
            $f2res = Perinci::Result::Format::Lite::format($res, 'text-pretty');
            $f2res =~ s/^/ /gm;
            push @pod, "Benchmark module startup overhead:\n\n", $f2res, "\n\n";
        }

        $self->add_text_to_section(
            $document, join("", @pod), 'SAMPLE BENCHMARK RESULTS',
            {
                after_section => ['BENCHMARKED MODULES', 'SYNOPSIS'],
                before_section => 'DESCRIPTION',
            });
    }

    $self->log(["Generated POD for '%s'", $filename]);
}

sub weave_section {
    my ($self, $document, $input) = @_;

    my $filename = $input->{filename};

    my $package;
    if ($filename =~ m!^lib/(Bencher/Scenario/.+)\.pm$!) {
        $package = $1;
        $package =~ s!/!::!g;
        $self->_process_module($document, $input, $package);
    }
}

1;
# ABSTRACT: Put various information from scenario into POD

__END__

=pod

=encoding UTF-8

=head1 NAME

Pod::Weaver::Plugin::Bencher::Scenario - Put various information from scenario into POD

=head1 VERSION

This document describes version 0.02 of Pod::Weaver::Plugin::Bencher::Scenario (from Perl distribution Pod-Weaver-Plugin-Bencher-Scenario), released on 2016-01-05.

=head1 SYNOPSIS

In your C<weaver.ini>:

 [-Bencher::Scenario]

=head1 DESCRIPTION

This plugin is to be used when building C<Bencher::Scenario::*> modules.

=for Pod::Coverage ^(weave_section)$

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Pod-Weaver-Plugin-Bencher-Scenario>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Pod-Weaver-Plugin-Bencher-Scenario>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Pod-Weaver-Plugin-Bencher-Scenario>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
