package Bencher::Scenario::RandomLineModules;

our $DATE = '2016-06-26'; # DATE
our $VERSION = '0.03'; # VERSION

use 5.010001;
use strict;
use warnings;
use Log::Any::IfLOG '$log';

sub _create_file {
    my ($num_lines) = @_;

    require File::Temp;
    my ($fh, $filename) = File::Temp::tempfile();
    for (1..$num_lines) {
        print $fh sprintf("%049d\n", $_);
    }
    $filename;
}

our $scenario = {
    summary => 'Benchmark modules which pick random line(s) from a file',
    participants => [
        {
            fcall_template => 'File::Random::Pick::random_line(<filename>)',
        },
        {
            module => 'File::RandomLine',
            code_template => 'my $rl = File::RandomLine->new(<filename>); $rl->next',
        },
    ],

    datasets => [
        {name=>'1k_line'  , _lines=>1_000     , args=>{filename=>undef}},
        {name=>'10k_line' , _lines=>10_000    , args=>{filename=>undef}},
        {name=>'100k_line', _lines=>100_000   , args=>{filename=>undef}, include_by_default=>0},
        {name=>'1M_line'  , _lines=>1_000_000 , args=>{filename=>undef}, include_by_default=>0},
        {name=>'10M_line' , _lines=>10_000_000, args=>{filename=>undef}, include_by_default=>0},
    ],

    before_gen_items => sub {
        my %args = @_;
        my $sc    = $args{scenario};

        my $dss = $sc->{datasets};
        for my $ds (@$dss) {
            $log->infof("Creating temporary file with %d lines ...", $ds->{_lines});
            my $filename = _create_file($ds->{_lines});
            $log->infof("Created file %s", $filename);
            $ds->{args}{filename} = $filename;
        }
    },

    before_return => sub {
        my %args = @_;
        my $sc    = $args{scenario};

        my $dss = $sc->{datasets};
        for my $ds (@$dss) {
            my $filename = $ds->{args}{filename};
            next unless $filename;
            $log->infof("Unlinking %s", $filename);
            unlink $filename;
        }
    },
};

1;
# ABSTRACT: Benchmark modules which pick random line(s) from a file

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::RandomLineModules - Benchmark modules which pick random line(s) from a file

=head1 VERSION

This document describes version 0.03 of Bencher::Scenario::RandomLineModules (from Perl distribution Bencher-Scenario-RandomLineModules), released on 2016-06-26.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m RandomLineModules

To run module startup overhead benchmark:

 % bencher --module-startup -m RandomLineModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<File::Random::Pick> 0.02

L<File::RandomLine> 0.20

=head1 BENCHMARK PARTICIPANTS

=over

=item * File::Random::Pick::random_line (perl_code)

Function call template:

 File::Random::Pick::random_line(<filename>)



=item * File::RandomLine (perl_code)

Code template:

 my $rl = File::RandomLine->new(<filename>); $rl->next



=back

=head1 BENCHMARK DATASETS

=over

=item * 1k_line

=item * 10k_line

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.2 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m RandomLineModules >>):

 +---------------------------------+----------+-----------+-----------+------------+---------+---------+
 | participant                     | dataset  | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +---------------------------------+----------+-----------+-----------+------------+---------+---------+
 | File::Random::Pick::random_line | 10k_line |       591 |   1.69    |       1    | 2.7e-07 |      20 |
 | File::Random::Pick::random_line | 1k_line  |      5720 |   0.175   |       9.67 | 5.3e-08 |      20 |
 | File::RandomLine                | 1k_line  |    135000 |   0.00742 |     228    | 3.3e-09 |      20 |
 | File::RandomLine                | 10k_line |    140000 |   0.0073  |     230    |   1e-08 |      20 |
 +---------------------------------+----------+-----------+-----------+------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m RandomLineModules --module-startup >>):

 +---------------------+-----------+------------------------+------------+---------+---------+
 | participant         | time (ms) | mod_overhead_time (ms) | vs_slowest |  errors | samples |
 +---------------------+-----------+------------------------+------------+---------+---------+
 | File::RandomLine    |       9   |                    6.4 |        1   |   3e-05 |      20 |
 | File::Random::Pick  |       4.6 |                    2   |        2   | 1.6e-05 |      20 |
 | perl -e1 (baseline) |       2.6 |                    0   |        3.5 | 1.2e-05 |      20 |
 +---------------------+-----------+------------------------+------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-RandomLineModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-RandomLineModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-RandomLineModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
