package CXC::Form::Tiny::OptArgs2;

# ABSTRACT: a really awesome library

use v5.26;

use strict;
use warnings;
use experimental 'signatures', 'declared_refs', 'lexical_subs';

our $VERSION = '0.01';

use Class::Load 'load_first_existing_class', 'load_optional_class';
use Ref::Util    qw( is_arrayref is_hashref );
use Scalar::Util qw( blessed );
use List::Util   qw( pairs reduce );
use Hash::Fold 'fold', unfold => { -as => 'hash_unfold' };
use Log::Any '$log';
use OptArgs2 ();

use namespace::clean;

my sub croak {
    require Carp;
    goto \&Carp::croak;
}









sub cached ( $self, $thing ) {
    return $self->cache->{$thing} // croak( "no cache for $thing?" );
}







sub cache ( $self ) {
    $self->{cache};
}







sub form ( $self ) {
    $self->{form};
}




















sub new {
    my ( $class, %args ) = @_;

    my $self = bless {
        cache  => {},
        form   => {},
        config => undef,
    }, $class;

    for ( 'command_class', 'config_option', 'read_config' ) {
        my $value = delete $args{$_};
        $self->{$_} = $value
          if defined $value;
    }

    croak( 'unknown arguments: ' . join( q{, }, keys %args ) )
      if %args;

    return $self;
}












sub config_option ( $self, @rest ) {

    return
        @rest == 0 ? $self->{config_option}
      : @rest == 1 ? ( $self->{config_option} = $rest[0] )
      :              croak( 'too many arguments to config_option' );
}











sub cmd_class ( $self, @rest ) {

    return
        @rest == 0 ? $self->{command_class}
      : @rest == 1 ? ( $self->{command_class} = $rest[0] )
      :              croak( 'too many arguments to cmd_class' );
}











sub subcmd_class ( $self, $subcmd ) {
    return join( q{::}, ( $self->cmd_class // () ), $subcmd );
}
































sub form_class ( $self, $class ) {

    my $command_class = $self->cmd_class;

    return $class . '::form'
      if !defined $command_class;

    my $form_class = $command_class . '::form';

    substr( $class, 0, length( $command_class ), $form_class );
    return $class;
}













sub cmd ( $self, @args ) {

    my $class = @args % 2 ? shift @args : undef;

    croak( q{command class not specified in call to constructor} )
      if !defined $class && !defined( $class = $self->cmd_class );

    $self->cmd_class( $class );

    $self->cache->{$class} = {};

    my %args = @args;
    $args{optargs} //= [];
    unshift $args{optargs}->@*, $self->optspec( $class )->@*;

    OptArgs2::cmd( $class, %args, );
}













sub subcmd ( $self, $class, %args ) {

    $class = $self->subcmd_class( $class );

    $self->cache->{$class} = {};

    $args{optargs} //= [];
    unshift $args{optargs}->@*, $self->optspec( $class )->@*;

    OptArgs2::subcmd( $class, %args, );
}











sub optargs ( $self, @args ) {
    my $target = caller;

    my \%opts = @args % 2 ? shift @args : {};
    my %args = @args;
    $self->cmd( $target, %args );

    ( $self->class_optargs( %opts ) )[1];
}




















sub class_optargs ( $self, %args ) {

    my ( $class, $opts, $file ) = OptArgs2::class_optargs( $self->cmd_class );

    $opts = $self->validate( $class, $opts )
      if $args{validate};

    return ( $class, $opts, $file );
}










sub optspec ( $self, $class ) {

    my $form_class = $self->form_class( $class );

    return [] unless defined load_optional_class( $form_class );

    my \%cache = $self->cached( $class );

    # form objects are reusable, so keep them in their own registry so
    # they can be found easily, and make a copy in the class cache so
    # they can easily be found for a class
    my $form    = $cache{form} = $self->form->{$form_class} //= $form_class->new;
    my $optargs = $form->optargs;

    my @optargs_defaults = grep { exists $_->value->{default} } pairs $optargs->@*;

    croak( 'do not use OptArgs2 defaults; use Form::Tiny default attribute for: ',
        join( q{, }, map { $_->key } @optargs_defaults ) )
      if @optargs_defaults;

    return $form->optargs;
}

























sub _iterate_command_class_hierarchy ( $self, $code, $class ) {

    my $command_class = $self->cmd_class;
    my $qclass        = $class;

    while ( !!1 ) {

        return !!1 if $self->$code( $qclass );

        last if
          # we just checked against the top level command
          $qclass eq $command_class;

        my $idx = rindex( $qclass, q{::} );
        # this should not happen: we've run out of components in the
        # class path
        croak( "internal error; iteration over command path ($class) hit root namespace: $qclass" )
          if $idx < 0;

        ## no critic( BuiltinFunctions::ProhibitLvalueSubstr)
        substr( $qclass, $idx ) = q{};
    }

    return !!0;
}










sub _cached_form ( $self, $class ) {

    my $form;

    return $form
      if $self->_iterate_command_class_hierarchy(
        sub ( $self, $fragment ) {
            $form = $self->cached( $fragment )->{form};
            return defined $form;
        },
        $class,
      );

    croak( "no form for $class?" );
}












sub validate ( $self, $class, $optargs ) {

    my $form   = $self->_cached_form( $class );
    my $inputs = $self->load_inputs( $class, $optargs );
    $form->set_input( $inputs );

    $form->valid or do {
        require Data::Dump;
        croak( Data::Dump::pp( $form->errors_hash ) );
    };
    return $form->fields;
}










sub load_cmd ( $self, $class ) {

    my @classes_to_load;

    $self->_iterate_class_hierarchy(
        sub ( $, $fragment ) {
            push @classes_to_load, $fragment;
            return 0;
        },
        $class,
    );

    $log->trace( "Classes to load for $class: " . join( ', ', @classes_to_load ) );
    load_first_existing_class @classes_to_load;
}









sub merge ( $self, @hashes ) {

    my sub croak_merge ( $src, $dest, @ ) {
        croak( "attempt to merge $src into $dest" );
    }

    ## no critic (Subroutines::ProtectPrivateSubs)
    require Hash::Merge;
    state $spec = {
        'SCALAR' => {
            'SCALAR' => sub { defined $_[0] ? $_[0] : $_[1] },
            'ARRAY'  => sub { croak_merge( 'ARRAY', 'SCALAR', @_ ) },
            'HASH'   => sub { croak_merge( 'HASH',  'SCALAR', @_ ) },
        },
        'ARRAY' => {
            'SCALAR' => sub { croak_merge( 'SCALAR', 'ARRAY' ) },
            'ARRAY'  => sub { $_[0]->@* ? $_[0] : $_[1] },
            'HASH'   => sub { croak_merge( 'HASH', 'ARRAY', @_ ) },
        },
        'HASH' => {
            'SCALAR' => sub { croak_merge( 'SCALAR', 'HASH', @_ ) },
            'ARRAY'  => sub { croak_merge( 'ARRAY',  'HASH', @_ ) },
            'HASH'   => sub { Hash::Merge::_merge_hashes( @_ ) },
        },
    };

    my $merger = Hash::Merge->new();
    $merger->add_behavior_spec( $spec );

    reduce { $merger->merge( $a, $b ) } grep { defined } @hashes;
}











sub load_inputs ( $self, $class, $optargs ) {

    my @options;

    if ( defined( my $config_option = $self->config_option ) ) {
        my $config = $optargs->{$config_option};
        $config  = [$config] unless is_arrayref( $config );
        @options = map { $self->load_config( $_, $class ) } $config->@*;
    }

    push @options, $self->_cached_form( $class )->inflate_optargs( $optargs );

    # merger object implements left precedence, and the command
    # line should have the highest priority.
    return $self->merge( reverse @options );
}

































# sub read_config ( $slf, $file ) {
#     croak( 'unimplemented class method: read_config' );
# }















sub load_config ( $self, $file, $class, $fold = !!0 ) {

    $log->info( "reading configuration from $file" );

    my $reader = $self->{read_config} // $self->can( 'read_config' );
    croak( 'read_config method or constructor argument not provided' )
      unless defined $reader;
    my $cfg = $self->$reader( $file );

    my @configs;

    my $command_class = $self->cmd_class;

    # if we're not the top level command and there are config values
    # for subcommands, slurp 'em up.
    if ( $class ne $command_class && defined( my $subcmds = $cfg->{subcommands} ) ) {

        croak( "$file: 'subcommands' entry is not a hash" )
          unless is_hashref( $subcmds );

        my \%config = $subcmds;
        $log->debug( 'read subcommands config', \%config );

        # this creates sequence of hashes from nested command levels, with
        # deepest level at start of list.
        $self->_iterate_class_hierarchy(
            sub ( $, $element ) {
                push @configs, delete( $config{$element} ) // {};
                return 0;
            },
            $class,
        );
    }

    # always pull in config for top level
    if ( defined( my $cmd = $cfg->{command} ) ) {
        croak( "$file: 'command' entry is not a hash" )
          unless is_hashref( $cmd );
        push @configs, $cmd;
    }

    # now merge hashes, with left precedence
    my $config = $self->merge( @configs );
    $log->debug( 'merged config', $config );

    return $fold ? fold( $config ) : $config;
}


1;

#
# This file is part of CXC-Form-Tiny-OptArgs2
#
# This software is Copyright (c) 2023 by Smithsonian Astrophysical Observatory.
#
# This is free software, licensed under:
#
#   The GNU General Public License, Version 3, June 2007
#

__END__

=pod

=for :stopwords Diab Jerius Smithsonian Astrophysical Observatory cmd optargs optspec
subcmd subcommand subcommands

=head1 NAME

CXC::Form::Tiny::OptArgs2 - a really awesome library

=head1 VERSION

version 0.01

=head1 SYNOPSIS

=head1 DESCRIPTION

This class, in conjunction with L<CXC::Form::Tiny::Plugin::OptArgs2>,
provide a solution to using L<Form::Tiny> and L<OptArgs2> to manage
options and arguments derived from the command line and from
configuration files.

This class is a wrapper around L<OptArgs2>,  automating some of
the linkage between it and L<Form::Tiny>.

=head2 Extension

All of the functionality in this module is performed via
methods, which can be overridden in a subclass to alter

Here are some common things to customize

=head3 Form Classes

Mapping between a class implementing a command or subcommand and its
associated L<Form::Tiny> class is done via the L</form_class> method.

=head3 Configuration files

If the application should read configuration files specified on the
command line, override the C<config_option> method to return the name
of the configuration option.  The option may be scalar or an array.

A modestly complex application will be configured via default values,
configuration files, command line options and arguments, and possibly
from environment variables.

Validation of the configuration can only be performed after all of the
sources have been queried, in particular because configuration files may
be specified on the command line.

The stages in processing command line arguments and configuration
files are

=over

=item 1.

parsing the command line

=item 2.

parsing configuration files

=item 3.

merging default values, values from configuration files, values from the command line and values from the
environment.

=item 4.

validation of configuration values.

=back

This module concerns itself with the first three stages, and uses
L<CXC::Form::Tiny::Plugin::OptArgs2> to generate the command line
specifications for feeding L<OptArgs2> and for the last stage,
validation.

=head2 USAGE

=head2 Command class hierarchy

L<OptArgs2> maps nested commands onto class hierarchies.  An
application provides a single command which may have multiple nested
sub-commands, each of which must reside in a nested class below
their parent command or subcommand.

For example, if an application is called C<App::demo>, running it as

  % demo

will result in a call to

  [???]::demo->new(...)->run;

and running a subcommand

  % demo foo

will result in a call to

  [???]demo::foo->new(...)->run;

and so on.  It doesn't matter what class C<[???]> is named, or where in
the hierarchy the code is, just that it be properly nested and that the
final component in the class implementing the top-level command is the
name of the executable.

When specifying commands and subcommands to
B<CXC::Form::Tiny::OptArgs2>, you can specify paths relative to
a root class, (unlike L<OptArgs2>, which requires full qualified
class names).

The common root, which is the name of the class which would implement
the top-level command, may be passed either to the constructor (via
the C<command_class> option) or as the first argument to the L</cmd> method.
=head1 EXAMPLES

=head3 Reading configuration files

The L</read_config> class method must be overridden.  Here's an
example which uses L<YAML::PP> and L<Path::Tiny>:

  sub read_config ( $class, $file ) {
      require Path::Tiny;
      $file = Path::Tiny::path($file);
      $file->exists or die( "configuration file '$file' doesn't exist" );

      require YAML::PP;
      my $yp = YAML::PP->new( schema => [qw/ + Merge Include /] );
      return $yp->load_string( $file->slurp_utf8 );
  }

=head1 OBJECT ATTRIBUTES

=head2 cache

read only

=head2 form

  read only

=head1 CONSTRUCTORS

=head2 new

  $optargs = CXC::Form::Tiny::OptArgs2->new( %options );

Construct an object.  The options are

=over

=item command_class

The command base class, e.g. the class which implements the C<run>
method for the top level command.  If set, then the classes
implementing subcommands are specified relative to this class.
Otherwise they must be specified as fully qualified classes.

=back

=head1 METHODS

=head2 cached

   $cached_value = $optargs->cached( $label );

Return the value associated with C<$label>, or throw.

=head2 config_option

  my $config_option = $optargs->config_option( ?$new_option_name );

Retrieve or set the name of the option which is used to specify
configuration files on the command line.  If this is defined,
then L</validate> will load the configuration data and merge them
with the defaults and command line data.

=head2 cmd_class

  my $cmd_class = $optargs->cmd_class( ?$new_cmd_class );

Retrieve or set the name of the class which implements the top-level
command functionality.  Even if there is no top level command, this
should be specified.

=head2 subcmd_class

  $class = $optargs->subcmd_class( $subcommand )

Return a fully qualified class for C<$subcommand>, which is specified
relative to L</cmd_class>.

=head2 form_class

  $form_class_name = $optargs->form_class( $class );

Return a form class name for the passed fully qualified command class.

This replaces the base component of C<$class> (e.g. L</cmd_class>)
with C<$cmd_clss . '::form'>.

Override this method to change this.

For example, if the command base class is

   App::myapp

and the passed command is

   App::myapp::subcommand

the resulting form class is

  App::myapp::form::subcommand

If no arguments are passed (indicating the top level command), then
the resulting form class name is

  App::myapp::form

=head2 cmd

  # specify the class which implements the top level command
  $optargs->cmd( $class, %args );

  # rely upon a the value passed to the constructor:
  $optargs->cmd( %args );

C<%args> is the same as passed to L/OptArgs2>'s B<cmd> subroutine.

=head2 subcmd

  # specify the sub command class relative to the command's class
  $optargs->subcmd( $class, %args );

Takes the same arguments as L<OptArgs2>'s C<subcmd> subroutine, except
that C<$class> is relative to the class provided
via L</cmd> or via the constructor.

=head2 optargs

  my \%args =  $optargs->optargs( ?\%opts, %args );

C<%args> are the same arguments as for L<OptArgs2>'s C<optargs>
subroutine.  C<%opts> are passed to L</class_optargs>.

=head2 class_optargs

  my ( $class, $opts, $file ) =  $class->class_optargs( %opts );

Processes the command line and returns the same results as
L<OptArgs2>'s class_optargs command.  C<%opts> may include:

=over

=item validate

If true, the options retrieved from the command line are pass to
L</validate>, which performs validation against the forms, but
also will load and merge data read from configuration files.

=back

=head2 optspec

   $optspec = $optargs->optspec;

Returns a L<OptArgs2> compatible specification suitable to be provided
to it via it's C<optargs> argument.

=head2 validate

   \%options = $optargs->validate( $class, \%options )

Validate C<%options> against the form associated with C<$class>.  If
the return value of L</config_option> is defined, additional options
are read from configuration files and merged before validation.  See
L</load_inputs>.

=head2 load_cmd

  $optargs->load_cmd( $class );

Loads the first class which exists in $class or its higher name hierarchy,
up to and including L</cmd_class>.

=head2 merge

  \%merged = $self->merge( @configs );

Merge a list of hashes.

=head2 load_inputs

  \%options = $optargs->load_inputs( $class, \%optargs );

Merge C<%optargs>, which should be the options read from the command
line, with values read from configuration files (if L</config_option>
returns defined).

=head2 read_config

  \%config = $class->read_config( $filename );

Read a configuration file.  This method must be implemented or a
coderef may be passed to the class constructor. It is only used if the
L</config_option> method returns a defined value.

The returned hash must have the following structure:

  {
     command => {...},
     subcommands => {
        $subcommand1 => {...},
        $subcommand2 => {...},
     },
  }

The values in the C<command> hash are options and values for the top-level command.

The keys in the C<subcommands> hash are the sub-command names relative
to the top level. For example, the

  myapp db create

subcommand would have a key of C<db::create>.

The elements in the C<subcommands> hash are options and values for
that subcommand.

=head2 load_config

   \%config = $optargs->load_config( $file, $class, ?$fold );

Load configuration data from C<$file> for the command/subcommand class
$class.  required either that the C<read_config> method be implemented
in a subclass, or the C<read_config> option be specified to the
constructor.

If C<$fold> is true, L<Hash::Fold::fold> is run on the configuration
data.

=head1 INTERNALS

=head2 Methods

=head3 _iterate_command_class_hierarchy

   $optargs->_iterate_command_class_hierarchy ( $code, $class );

Call C<$optargs->$code( )> on C<$class> and each successively higher
class in its name hierarchy, including, but not above L</cmd_class>.

Immediately returns true if C<$optargs->$code( $class )> returns true,
otherwise moves on to the next higher class.  It returns false if it
has iterated over all possibilities

For example, if C<cmd_class> is C<App::foo>, calling

   $optargs->_iterate_command_class_heirarch ( $code, 'App::foo::bar::baz' );

is equivalent to:

    return !!1 if $optargs->$code( 'App::foo::bar::baz');
    return !!1 if $optargs->$code( 'App::foo::bar');
    return !!1 if $optargs->$code( 'App::foo');
    return !!0;

=head3 _cached_form

  $form_object = $optargs->_cached_form( $class );

Retrieves a previously stored L<Form::Tiny> object for
C<$class>. Throws if it can't find one.

=head1 SUPPORT

=head2 Bugs

Please report any bugs or feature requests to bug-cxc-form-tiny-optargs2@rt.cpan.org  or through the web interface at: L<https://rt.cpan.org/Public/Dist/Display.html?Name=CXC-Form-Tiny-Optargs2>

=head2 Source

Source is available at

  https://gitlab.com/djerius/cxc-form-tiny-optargs2

and may be cloned from

  https://gitlab.com/djerius/cxc-form-tiny-optargs2.git

=head1 AUTHOR

Diab Jerius <djerius@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2023 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut
