#!/usr/bin/perl
use strict;
use warnings;
use lib '../devel/lib';
use lib 'devel/lib';

use Getopt::Long;
use Boilerplater::Hierarchy;

my ( $base_dir, $dest_dir, $use_lucy, $use_kino );

GetOptions(
    'in=s'  => \$base_dir,
    'out=s' => \$dest_dir,
    'lucy'  => \$use_lucy,
    'kino'  => \$use_kino,
);

die "Must specify --in=SOURCE_DIR" unless defined $base_dir;
die "Must specify --out=DEST_DIR"  unless defined $dest_dir;
die "Must specify either --lucy or --kino"
    unless ( defined $use_lucy
    or defined $use_kino );

my $header = <<END_STUFF;
/** \@file */
/***********************************************

 !!!! DO NOT EDIT THIS FILE !!!!

 All content has been auto-generated by 
 the boilerplater.pl utility.

 See boilerplater's documentation for details.

 ***********************************************/

END_STUFF

my $footer;
if ($use_kino) {
    Boilerplater->init_prefixes(qw( kino_ Kino_ KINO_ ));
    $footer = <<END_STUFF;
/* Copyright 2007 Marvin Humphrey
 *
 * This program is free software; you can redistribute it and/or modify
 * under the same terms as Perl itself.
 */
END_STUFF
}
elsif ($use_lucy) {
    Boilerplater->init_prefixes(qw( lucy_ Lucy_ LUCY_ ));
    $footer = <<END_STUFF;
/**
 * Copyright 2006 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
END_STUFF
}

die "'$base_dir' isn't a directory" unless -d $base_dir;
if ( !-d $dest_dir ) {
    mkdir $dest_dir or die "Can't mkdir '$dest_dir': $!";
}

my $hierarchy = Boilerplater::Hierarchy->new(
    base_dir => $base_dir,
    dest_dir => $dest_dir,
    header   => $header,
    footer   => $footer,
);
$hierarchy->build;
$hierarchy->write_all_modified;

__END__

=head1 NAME

boilerplater.pl - Invoke Boilerplater code generator.

=head1 SYNOPSIS

   perl ../devel/boilerplater.pl --in=SOURCE_DIR --out=DEST_DIR --lucy

=head1 DESCRIPTION

This script is a wrapper around Boilerplater, so see Boilerplater's docs for
an explanation.

=head2 ARGUMENTS

boilerplater.pl must be invoked with valid values for C<in> and C<out>, plus
one of either C<kino> or <lucy>.

=over

=item --in=SOURCE_DIR

Directory to be scanned recursively for .h files containing Boilerplater code.

=item --out=DEST_DIR

The directory where the generated .r files will be written.

=item --lucy

Generate code for Lucy.

=item --kino

Generate code for KinoSearch.

=back

=head1 AUTHOR

Marvin Humphrey

=head1
