use strict;
use warnings;
use Test;

BEGIN { plan(tests => 107) }

use Net::Gopher;
use Net::Gopher::Request;
use Net::Gopher::Constants qw(:item_types);

require './t/serverfunctions.pl';







################################################################################
#
# These tests check the integrity of the Net::Gopher object itself, making
# sure initialization via the constructor and later modification via the
# accessor methods all work properly:
#

{
	my $ng = new Net::Gopher;

	ok($ng->buffer_size, 4096);                             # 1
	ok($ng->timeout, 30);                                   # 2
	ok($ng->upward_compatible, 1);                          # 3
	ok($ng->warn_handler ==
		$Net::Gopher::Exception::DEFAULT_WARN_HANDLER); # 4
	ok($ng->die_handler == 
		$Net::Gopher::Exception::DEFAULT_DIE_HANDLER);  # 5
	ok($ng->silent, 0);                                     # 6
	ok($ng->debug, 0);                                      # 7
	ok(!defined $ng->log_file);                             # 8
	ok(!defined $ng->_buffer);                              # 9
	ok(!defined $ng->_socket);                              # 10
	ok(!defined $ng->_select);                              # 11
}

{
	my $warn = sub { print "a warning" };
	my $die  = sub { print "a fatal error" };

	my $ng = new Net::Gopher (
		BufferSize       => 777777,
		TIMEOUT          => 60,
		upwardcompatible => 'true',
		WaRNHaNDleR      => $warn,
		DieHandler       => $die,
		SILENT           => undef,
		debug            => 'also true',
		LoGFiLe          => 'a_filename.txt'
	);

	ok($ng->buffer_size, 777777);         # 12
	ok($ng->timeout, 60);                 # 13
	ok($ng->upward_compatible, 1);        # 14
	ok($ng->warn_handler == $warn);       # 15
	ok($ng->die_handler  == $die);        # 16
	ok($ng->silent, 0);                   # 17
	ok($ng->debug, 1);                    # 18
	ok($ng->log_file, 'a_filename.txt');  # 19
	ok(!defined $ng->_buffer);            # 20
	ok(!defined $ng->_socket);            # 21
	ok(!defined $ng->_select);            # 22
}

{
	my $ng = new Net::Gopher;

	$ng->buffer_size(1234567);
	ok($ng->buffer_size, 1234567); # 23

	$ng->timeout(100);
	ok($ng->timeout, 100);         # 24

	$ng->upward_compatible('true');
	ok($ng->upward_compatible, 1); # 25

	$ng->upward_compatible(0);
	ok($ng->upward_compatible, 0); # 26

	$ng->debug(100);
	ok($ng->debug, 1);             # 27

	$ng->debug(0);
	ok($ng->debug, 0);             # 28
}







{
	# this runs testserver.pl with -e to echo back each request:
	ok(launch_echo_server()); # 29

	my $ng = new Net::Gopher;



	#######################################################################
	#
	# These tests are used to make sure that the Net::Gopher request()
	# method properly sends the request strings generated by the
	# Net::Gopher::Request as_string() method to the server:
	#

	{
		my $request = new Net::Gopher::Request (
			Gopher => {
				Host        => 'localhost',
				Port        => 7070,
				Selector    => '/something',
				SearchWords => ['red', 'green', 'blue']
			}
		);

		my $response = $ng->request($request);

		ok($response->is_success);                        # 30
		ok($response->raw_response, $request->as_string); # 31
	}

	{
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host           => 'localhost',
				Port           => 7070,
				Selector       => '/something_else',
				Representation => 'text/plain'
			}
		);

		my $response = $ng->request($request);

		ok($response->is_success);                        # 32
		ok($response->raw_response, $request->as_string); # 33
	}

	{
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host           => 'localhost',
				Port           => 7070,
				Selector       => '/something_else',
				Representation => 'text/plain',
				DataBlock      => 'This is a single-line block'
			}
		);

		my $response = $ng->request($request);

		ok($response->is_success);                        # 34
		ok($response->raw_response, $request->as_string); # 35
	}

	{
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host           => 'localhost',
				Port           => 7070,
				Selector       => '/something_else',
				Representation => 'text/plain',
				DataBlock      => 'This is a big single-line block ' x 2000
			}
		);

		my $response = $ng->request($request);

		ok($response->is_success);                        # 36
		ok($response->raw_response, $request->as_string); # 37
	}

	{ 
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host           => 'localhost',
				Port           => 7070,
				Selector       => '/something_else',
				Representation => 'text/plain',
				DataBlock      =>
				"This\015\012is\012a\015\012multi-line\012block"
			}
		);

		my $response = $ng->request($request);

		ok($response->is_success);                        # 38
		ok($response->raw_response, $request->as_string); # 39
	}

	{
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host           => 'localhost',
				Port           => 7070,
				Selector       => '/something_else',
				Representation => 'text/plain',
				DataBlock      =>
				"This\015\012is\012a\015\012multi-line\012block " x 2000,
			}
		);

		my $response = $ng->request($request);

		ok($response->is_success);                        # 40
		ok($response->raw_response, $request->as_string); # 41
	}

	{
		my $request = new Net::Gopher::Request (
			ItemAttribute => {
				Host           => 'localhost',
				Port           => 7070,
				Selector       => '/some_item',
				Attributes     => '+ATTR'
			}
		);

		my $response = $ng->request($request);

		ok($response->is_success);                        # 42
		ok($response->raw_response, $request->as_string); # 43
	}

	{
		my $request = new Net::Gopher::Request (
			DirectoryAttribute => {
				Host           => 'localhost',
				Port           => 7070,
				Selector       => '/some_dir',
				Attributes     => '+ATTR'
			}
		);

		my $response = $ng->request($request);

		ok($response->is_success);                        # 44
		ok($response->raw_response, $request->as_string); # 45
	}







	########################################################################
	# 
	# These tests are used to make sure that the named request methods
	# create the proper request objects:
	#

	{
		my $request = new Net::Gopher::Request (
			Gopher => {
				Host        => 'localhost',
				Port        => 7070,
				Selector    => '/something',
				SearchWords => ['red', 'green', 'blue']
			}
		);

		my $response = $ng->gopher(
			Host        => 'localhost',
			Port        => 7070,
			Selector    => '/something',
			SearchWords => ['red', 'green', 'blue']
		);

		ok($response->is_success);                                    # 46
		ok($response->request->request_type, $request->request_type); # 47
		ok($response->request->as_string, $request->as_string);       # 48
	}

	{
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host           => 'localhost',
				Port           => 7070,
				Selector       => '/something_else',
				Representation => 'text/plain'
			}
		);

		my $response = $ng->gopher_plus(
			Host           => 'localhost',
			Port           => 7070,
			Selector       => '/something_else',
			Representation => 'text/plain'
		);

		ok($response->is_success);                                    # 49
		ok($response->request->request_type, $request->request_type); # 50
		ok($response->request->as_string, $request->as_string);       # 51
	}

	{
		my $request = new Net::Gopher::Request (
			ItemAttribute => {
				Host           => 'localhost',
				Port           => 7070,
				Selector       => '/some_dir',
				Attributes     => '+ATTR'
			}
		);

		my $response = $ng->item_attribute(
			Host           => 'localhost',
			Port           => 7070,
			Selector       => '/some_dir',
			Attributes     => '+ATTR'
		);

		ok($response->is_success);                                    # 52
		ok($response->request->request_type, $request->request_type); # 53
		ok($response->request->as_string, $request->as_string);       # 54
	}

	{
		my $request = new Net::Gopher::Request (
			DirectoryAttribute => {
				Host           => 'localhost',
				Port           => 7070,
				Selector       => '/some_dir',
				Attributes     => '+ATTR'
			}
		);

		my $response = $ng->directory_attribute(
			Host           => 'localhost',
			Port           => 7070,
			Selector       => '/some_dir',
			Attributes     => '+ATTR'
		);

		ok($response->is_success);                                    # 55
		ok($response->request->request_type, $request->request_type); # 56
		ok($response->request->as_string, $request->as_string);       # 57
	}
}







{
	ok(launch_item_server()); # 58

	my $ng = new Net::Gopher;



	########################################################################
	# 
	# These tests are used to make sure that the request method handles
	# response handlers correctly:
	#

	{
		# see the "index" file in the ./t/items directory:
		my $request = new Net::Gopher::Request (
			Gopher => {
				Host        => 'localhost',
				Selector    => '/index'
			}
		);

		my $content;
		my $content_matches;
		my $last_request_obj;
		my $last_response_obj;
		my $response = $ng->request($request,
			Handler => sub {
				my $buffer = shift;
				($last_request_obj, $last_response_obj) = @_;

				$content .= $buffer;
				$content_matches++
					if ($content eq $last_response_obj->content);
			}
		);

		ok($response->is_success);          # 59
		ok($request == $last_request_obj);  # 60
		ok($response== $last_response_obj); # 61
		ok($content_matches);               # 62
	}

	{
		# see the "gp_period_term" file in the ./t/items directory:
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host        => 'localhost',
				Selector    => '/gp_period_term'
			}
		);

		my $content;
		my $content_matches;
		my $last_request_obj;
		my $last_response_obj;
		my $response = $ng->request($request,
			Handler => sub {
				my $buffer = shift;
				($last_request_obj, $last_response_obj) = @_;

				$content .= $buffer;
				$content_matches++
					if ($content eq $last_response_obj->content);
			}
		);

		ok($response->is_success);           # 63
		ok($request == $last_request_obj);   # 64
		ok($response == $last_response_obj); # 65
		ok($content_matches);                # 66
	}

	{
		# see the "gp_no_term" file in the ./t/items directory:
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host        => 'localhost',
				Selector    => '/gp_no_term'
			}
		);

		my $content;
		my $content_matches;
		my $last_request_obj;
		my $last_response_obj;
		my $response = $ng->request($request,
			Handler => sub {
				my $buffer = shift;
				($last_request_obj, $last_response_obj) = @_;

				$content .= $buffer;
				$content_matches++
					if ($content eq $last_response_obj->content);
			}
		);

		ok($response->is_success);         # 67
		ok($request, $last_request_obj);   # 68
		ok($response, $last_response_obj); # 69
		ok($content_matches);              # 70
	}

	{
		# see the "gp_byte_term" file in the ./t/items directory:
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host        => 'localhost',
				Selector    => '/gp_byte_term'
			}
		);

		my $content;
		my $content_matches;
		my $last_request_obj;
		my $last_response_obj;
		my $response = $ng->request($request,
			Handler => sub {
				my $buffer = shift;
				($last_request_obj, $last_response_obj) = @_;

				$content .= $buffer;
				$content_matches++
					if ($content eq $last_response_obj->content);
			}
		);

		ok($response->is_success);           # 71
		ok($request == $last_request_obj);   # 72
		ok($response == $last_response_obj); # 73
		ok($content_matches);                # 74
	}

	{
		# see the "gp_s_period_term" file in the ./t/items directory:
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host        => 'localhost',
				Selector    => '/gp_s_period_term'
			}
		);

		my $content;
		my $content_matches;
		my $last_request_obj;
		my $last_response_obj;
		my $response = $ng->request($request,
			Handler => sub {
				my $buffer = shift;
				($last_request_obj, $last_response_obj) = @_;

				$content .= $buffer;
				$content_matches++
					if ($content eq $last_response_obj->content);
			}
		);

		ok($response->is_success);           # 75
		ok($request == $last_request_obj);   # 76
		ok($response == $last_response_obj); # 77
		ok($content_matches);                # 78
	}

	{
		# see the "gp_s_no_term" file in the ./t/items directory:
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host        => 'localhost',
				Selector    => '/gp_s_no_term'
			}
		);

		my $content;
		my $content_matches;
		my $last_request_obj;
		my $last_response_obj;
		my $response = $ng->request($request,
			Handler => sub {
				my $buffer = shift;
				($last_request_obj, $last_response_obj) = @_;

				$content .= $buffer;
				$content_matches++
					if ($content eq $last_response_obj->content);
			}
		);

		ok($response->is_success);           # 79
		ok($request == $last_request_obj);   # 80
		ok($response == $last_response_obj); # 81
		ok($content_matches);                # 82
	}

	{
		# see the "gp_s_byte_term" file in the ./t/items directory:
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host        => 'localhost',
				Selector    => '/gp_s_byte_term'
			}
		);

		my $content;
		my $content_matches;
		my $last_request_obj;
		my $last_response_obj;
		my $response = $ng->request($request,
			Handler => sub {
				my $buffer = shift;
				($last_request_obj, $last_response_obj) = @_;

				$content .= $buffer;
				$content_matches++
					if ($content eq $last_response_obj->content);
			}
		);

		ok($response->is_success);           # 83
		ok($request == $last_request_obj);   # 84
		ok($response == $last_response_obj); # 85
		ok($content_matches);                # 86
	}

	{
		# see the "index" file in the ./t/items directory:
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host        => 'localhost',
				Selector    => '/index'
			}
		);

		my $content;
		my $content_matches;
		my $last_request_obj;
		my $last_response_obj;
		my $response = $ng->request($request,
			Handler => sub {
				my $buffer = shift;
				($last_request_obj, $last_response_obj) = @_;

				$content .= $buffer;
				$content_matches++
					if ($content eq $last_response_obj->content);
			}
		);

		ok($response->is_success);           # 87
		ok($request == $last_request_obj);   # 88
		ok($response == $last_response_obj); # 89
		ok($content_matches);                # 90
	}






	{
		my $request = new Net::Gopher::Request (
			Gopher => {
				Host        => 'localhost',
				Selector    => '/index'
			}
		);

		my $response = $ng->request($request, File => 'test.txt');

		ok($response->is_success);                # 91
		ok(open(TEST, 'test.txt'));               # 92
		ok(join('', <TEST>), $response->content); # 93
		close TEST;
		ok(unlink('test.txt'));                   # 94
		ok(!-e 'test.txt');                       # 95
	}

	{
		my $request = new Net::Gopher::Request (
			GopherPlus => {
				Host        => 'localhost',
				Selector    => '/gp_index'
			}
		);

		my $response = $ng->request($request, File => 'test2.txt');

		ok($response->is_success);                 # 96
		ok(open(TEST2, 'test2.txt'));              # 97
		ok(join('', <TEST2>), $response->content); # 98
		close TEST2;
		ok(unlink('test2.txt'));                   # 99
		ok(!-e 'test2.txt');                       # 100
	}







	########################################################################
	# 
	# These tests make sure Net::Gopher raises exceptions in the proper
	# places:
	#

	{
		my (@warnings, @fatal_errors);

		my $ng = new Net::Gopher(
			WarnHandler => sub { push(@warnings, @_) },
			DieHandler  => sub { push(@fatal_errors, @_) }
		);

		$ng->request();

		ok(scalar @warnings, 0);        # 101
		ok(scalar @fatal_errors, 1);    # 102
		ok($fatal_errors[0],
			'A Net::Gopher::Request object was not supplied as ' .
			'the first argument.'); # 103
	}

	{
		my (@warnings, @fatal_errors);

		my $ng = new Net::Gopher(
			WarnHandler => sub { push(@warnings, @_) },
			DieHandler  => sub { push(@fatal_errors, @_) }
		);

		$ng->request(new Net::Gopher::Request('Gopher') );

		ok(@warnings, 0);     # 104
		ok(@fatal_errors, 1); # 105
		ok($fatal_errors[0],
			join(' ',
				"You never specified a hostname; it's",
				"impossible to send your request without one.",
				"Specify it during object creation or later on",
				"with the host() method."
			));           # 106
	}



	ok(kill_servers()); # 107
}
