#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   file: lib/Test/Dist/Zilla/Build.pm
#

#pod =encoding UTF-8
#pod
#pod =head1 COPYRIGHT AND LICENSE
#pod
#pod Copyright © 2015 Van de Bugger
#pod
#pod This file is part of perl-Test-Dist-Zilla.
#pod
#pod perl-Test-Dist-Zilla is free software: you can redistribute it and/or modify it under the terms
#pod of the GNU General Public License as published by the Free Software Foundation, either version
#pod 3 of the License, or (at your option) any later version.
#pod
#pod perl-Test-Dist-Zilla is distributed in the hope that it will be useful, but WITHOUT ANY
#pod WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#pod PURPOSE. See the GNU General Public License for more details.
#pod
#pod You should have received a copy of the GNU General Public License along with
#pod perl-Test-Dist-Zilla. If not, see <http://www.gnu.org/licenses/>.
#pod
#pod =cut

#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#pod =head1 SYNOPSIS
#pod
#pod     package ManifestTester;
#pod
#pod     use Moose;
#pod     with 'Test::Dist::Zilla::Build';
#pod     use Test::Deep qw{ cmp_deeply re };
#pod     use Test::More;
#pod
#pod     has options => (
#pod         is          => 'ro',
#pod         isa         => 'HashRef',
#pod         default     => sub { {} },
#pod     );
#pod
#pod     sub _build_message_filter {
#pod         sub { grep( { $_ =~ qr{^\[Manifest\] } @_ ) };
#pod     };
#pod
#pod     sub _build_plugins {
#pod         my ( $self ) = @_;
#pod         [
#pod             'GatherDir',
#pod             [ 'Manifest' => $self->options ],
#pod             'MetaJSON',
#pod         ];
#pod     };
#pod
#pod     sub _build_files { {
#pod         'lib/Dummy.pm' => 'package Dummy; 1;',
#pod     } };
#pod
#pod     test 'Manifest' => sub {
#pod
#pod         my ( $self ) = @_;
#pod         my $expected = $self->{ expected };
#pod
#pod         if ( $self->exception ) {
#pod             plan skip_all => 'exception occurred';
#pod         };
#pod         if ( not exists( $expected->{ manifest } ) ) {
#pod             plan skip_all => 'no expected manifest';
#pod         };
#pod
#pod         my $name = $self->options->{ filename };
#pod         my $built_in = path( $self->tzil->built_in );
#pod         my @manifest = $built_in->child( $name )->lines;
#pod         cmp_deeply( \@manifest, $expected->{ manifest } );
#pod
#pod     };
#pod
#pod     run_me 'Positive test' => {
#pod         options => {
#pod             filename => 'Manifest',
#pod         },
#pod         expected => {
#pod             manifest => [
#pod                 'lib/Dunny.pm',
#pod                 'MANIFEST',
#pod                 'META.json',
#pod             ],
#pod         };
#pod     };
#pod
#pod     run_me 'Negative test' => {
#pod         options => {
#pod             filename = '/dev/null',
#pod         },
#pod         expected => {
#pod             exception => re( qr{^Abort\.\.\.} ),
#pod             messages => [
#pod                 '[Manifest] Input file /dev/null is empty',
#pod             ],
#pod         };
#pod     };
#pod
#pod     exit( 0 );
#pod
#pod =head1 DESCRIPTION
#pod
#pod This is a C<Test::Routine>-based role for testing C<Dist::Zilla> and its plugins. It creates
#pod F<dist.ini> file with specified content in a temporary directory, populates the directory with
#pod specified files, runs "build" command with testing version of C<Dist::Zilla> in the temporary
#pod directory, checks actual exception and log messages do match expected ones, and let you write other
#pod checks specific for your plugin.
#pod
#pod =cut

package Test::Dist::Zilla::Build;

use strict;
use warnings;
use namespace::autoclean;

# ABSTRACT: Test your Dist::Zilla plugin in I<build> action
our $VERSION = 'v0.3.0'; # VERSION

use Test::Routine;

with 'Test::Dist::Zilla';

use Test::Deep qw{ cmp_deeply };
use Test::More;

# --------------------------------------------------------------------------------------------------

#pod =method Build
#pod
#pod It is a test routine. It runs "build" command, then checks actual exception and log messages match
#pod expected ones. Expected exception and log messages should be specified as keys in C<expected> hash,
#pod e. g.:
#pod
#pod     run_me {
#pod         …
#pod         expected => {
#pod             exception => $exception,
#pod             messages => [
#pod                 $message,
#pod                 …
#pod             ],
#pod         },
#pod     };
#pod
#pod If C<exception> key is not specified (or exception value is C<undef>), build is expected to
#pod complete successfully (i. e. with no exception), otherwise build is expected to fail with the
#pod specified exception.
#pod
#pod If C<messages> key is not specified, log messages are not checked. Actual log messages retrieved
#pod with C<messages> method so you can filter them before comparison with expected messages by defining
#pod C<message_filter> attribute and/or by overriding C<messages> method.
#pod
#pod Exception (if not C<undef>) and log messages are compared expected counterparts with C<cmp_deeply>
#pod (from C<Test::Deep> module).
#pod
#pod =cut

test 'Build' => sub {

    my ( $self ) = @_;
    my $expected = $self->expected;

    plan 'no_plan';     # Some checks may be run inside `$self->build()`, I do not know how many.

    $self->build();
    $self->_anno_text( 'Full log', @{ $self->tzil->log_messages } );
    if ( $self->exception ) {
        $self->_anno_line( 'Exception: ' . $self->exception );
    };

    if ( exists( $expected->{ exception } ) and defined( $expected->{ exception } ) ) {
        cmp_deeply( $self->exception, $expected->{ exception }, 'build must fail' );
    } else {
        is( $self->exception, undef, 'build must pass' );
    };
    if ( exists( $expected->{ messages } ) ) {
        cmp_deeply( $self->messages, $expected->{ messages }, 'messages' );
    };

};

# --------------------------------------------------------------------------------------------------

1;

# --------------------------------------------------------------------------------------------------

#pod =head1 SEE ALSO
#pod
#pod =for :list
#pod = L<Test::Dist::Zilla>
#pod = L<Test::Deep/"$ok = cmp_deeply($got, $expected, $name)">
#pod = L<Test::Routine>
#pod
#pod =cut

# end of file #

__END__

=pod

=encoding UTF-8

=head1 NAME

Test::Dist::Zilla::Build - Test your Dist::Zilla plugin in I<build> action

=head1 VERSION

Version v0.3.0, released on 2015-09-19 23:00 UTC.

=head1 SYNOPSIS

    package ManifestTester;

    use Moose;
    with 'Test::Dist::Zilla::Build';
    use Test::Deep qw{ cmp_deeply re };
    use Test::More;

    has options => (
        is          => 'ro',
        isa         => 'HashRef',
        default     => sub { {} },
    );

    sub _build_message_filter {
        sub { grep( { $_ =~ qr{^\[Manifest\] } @_ ) };
    };

    sub _build_plugins {
        my ( $self ) = @_;
        [
            'GatherDir',
            [ 'Manifest' => $self->options ],
            'MetaJSON',
        ];
    };

    sub _build_files { {
        'lib/Dummy.pm' => 'package Dummy; 1;',
    } };

    test 'Manifest' => sub {

        my ( $self ) = @_;
        my $expected = $self->{ expected };

        if ( $self->exception ) {
            plan skip_all => 'exception occurred';
        };
        if ( not exists( $expected->{ manifest } ) ) {
            plan skip_all => 'no expected manifest';
        };

        my $name = $self->options->{ filename };
        my $built_in = path( $self->tzil->built_in );
        my @manifest = $built_in->child( $name )->lines;
        cmp_deeply( \@manifest, $expected->{ manifest } );

    };

    run_me 'Positive test' => {
        options => {
            filename => 'Manifest',
        },
        expected => {
            manifest => [
                'lib/Dunny.pm',
                'MANIFEST',
                'META.json',
            ],
        };
    };

    run_me 'Negative test' => {
        options => {
            filename = '/dev/null',
        },
        expected => {
            exception => re( qr{^Abort\.\.\.} ),
            messages => [
                '[Manifest] Input file /dev/null is empty',
            ],
        };
    };

    exit( 0 );

=head1 DESCRIPTION

This is a C<Test::Routine>-based role for testing C<Dist::Zilla> and its plugins. It creates
F<dist.ini> file with specified content in a temporary directory, populates the directory with
specified files, runs "build" command with testing version of C<Dist::Zilla> in the temporary
directory, checks actual exception and log messages do match expected ones, and let you write other
checks specific for your plugin.

=head1 OBJECT METHODS

=head2 Build

It is a test routine. It runs "build" command, then checks actual exception and log messages match
expected ones. Expected exception and log messages should be specified as keys in C<expected> hash,
e. g.:

    run_me {
        …
        expected => {
            exception => $exception,
            messages => [
                $message,
                …
            ],
        },
    };

If C<exception> key is not specified (or exception value is C<undef>), build is expected to
complete successfully (i. e. with no exception), otherwise build is expected to fail with the
specified exception.

If C<messages> key is not specified, log messages are not checked. Actual log messages retrieved
with C<messages> method so you can filter them before comparison with expected messages by defining
C<message_filter> attribute and/or by overriding C<messages> method.

Exception (if not C<undef>) and log messages are compared expected counterparts with C<cmp_deeply>
(from C<Test::Deep> module).

=head1 SEE ALSO

=over 4

=item L<Test::Dist::Zilla>

=item L<Test::Deep/"$ok = cmp_deeply($got, $expected, $name)">

=item L<Test::Routine>

=back

=head1 AUTHOR

Van de Bugger <van.de.bugger@gmail.com>

=head1 COPYRIGHT AND LICENSE

Copyright © 2015 Van de Bugger

This file is part of perl-Test-Dist-Zilla.

perl-Test-Dist-Zilla is free software: you can redistribute it and/or modify it under the terms
of the GNU General Public License as published by the Free Software Foundation, either version
3 of the License, or (at your option) any later version.

perl-Test-Dist-Zilla is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
perl-Test-Dist-Zilla. If not, see <http://www.gnu.org/licenses/>.

=cut
