#!/usr/bin/perl
#

use strict;
use warnings;
use IO::File;
use Test::More tests => 11;

use Net::DNS::Resolver::Unbound;


ok( Net::DNS::Resolver::Unbound->string(), 'default configuration' );

my $recursive = Net::DNS::Resolver::Unbound->new( nameservers => [] );
ok( $recursive, 'create fully recursive resolver instance' );
is( scalar( $recursive->nameservers ), 0, 'empty nameserver list' );

my $resolver = Net::DNS::Resolver::Unbound->new( debug_level => 0, prefer_v4 => 1 );
ok( $resolver, 'create stub resolver instance' );


my $option = 'outgoing-port-avoid';
my $value  = '3200-3201';
my $return = $resolver->option( $option, $value );
is( $return, undef, "resolver->option( $option, $value )" );
my $single = $resolver->option($option);
is( $single, $value, 'single-valued resolver option' );

$resolver->option( $option, '3202-3203' );
$resolver->option( $option, '3204-3205' );
my @multi = $resolver->option($option);
is( scalar(@multi), 3, 'multi-valued resolver option' );

my $bogus = $resolver->option('bogus');
is( $bogus, undef, 'nonexistent resolver option' );

ok( $resolver->string(), '$resolver->string' );


my $index = $$ % 10000;
my @filename;

sub filename {
	my $filename = join '', 'file', $index++;
	close( IO::File->new( $filename, '>' ) or die $! );
	push @filename, $filename;
	return $filename;
}

END {
	unlink $_ foreach @filename;
}


## exercise context methods, some of which may fail
eval { $resolver->option( 'verbosity',		 undef ) };
eval { $resolver->option( 'outgoing-port-avoid', '3206-3208' ) };
eval { $resolver->config(filename) };
eval { $resolver->set_fwd('::1') };
eval { $resolver->set_fwd('127.0.0.1') };
eval { $resolver->set_fwd('127.0.0.53') };
eval { $resolver->set_tls(0) };
eval { $resolver->set_stub( 'zone', '10.1.2.3', 0 ) };
eval { $resolver->resolv_conf(filename) };
eval { $resolver->hosts(filename) };
eval { $resolver->add_ta('zone DS') };
eval { $resolver->add_ta_file(filename) };
eval { $resolver->add_ta_autr(filename) };
eval { $resolver->trusted_keys(filename) };
eval { $resolver->debug_out(filename) };
eval { $resolver->debug_level(1) };
eval { $resolver->async_thread(0) };


my %IP_conf = (
	force_v4  => ['do-ip6:'	    => 'no'],
	force_v6  => ['do-ip4:'	    => 'no'],
	prefer_v4 => ['prefer-ip4:' => 'yes'],
	prefer_v6 => ['prefer-ip6:' => 'yes'] );
my @IPpref = map {@$_} values %IP_conf;
delete @IP_conf{qw(prefer_v4 prefer_v6)};
my @IPconf = sort keys %IP_conf;

for my $self ($resolver) {					# exploded _finalise_config()
	my $update = delete $self->{update};
	return unless $update;
	delete $self->{test_ctx};
	diag('');
	diag('delete self->{test_ctx}');
	delete $self->{ub_ctx};
	diag('delete self->{ub_ctx}');

	my $config = $self->{config};
	my $cfgopt = delete $config->{set_option};		# extract option lists
	my $updopt = delete $update->{set_option};
	my %config = ( %$config, %$update );			# merge config updates

	my %option = map {$_} @$cfgopt, @$updopt;		# merge option updates
	delete @option{@IPpref};				# expunge IP preference
	foreach my $key ( grep { $self->$_ } @IPconf ) {	# insert IP preference
		my ( $key, $value ) = @{$IP_conf{$key}};
		$option{$key} = $value;
		last;
	}
	my @option = map { ( $_, $option{$_} ) } sort keys %option;    # rebuild option list

	my $ctx = $self->{ub_ctx} = Net::DNS::Resolver::Unbound::Context->new();
	ok( $ctx, 'create ub_ctx' );
	diag('create self->{ub_ctx}');
	foreach my $name ( keys %option ) {			# set unbound options
		foreach my $value ( map { ref($_) ? @$_ : $_ } $option{$name} ) {
			$ctx->set_option( $name, $value );
			diag("ctx->set_option( $name, $value");
		}
	}

	foreach my $name ( keys %config ) {			# rebuild unbound context
		foreach my $value ( map { ref($_) ? @$_ : $_ } $config{$name} ) {
			my @arg = map { ref($_) ? @$_ : $_ } $value;
			$ctx->$name(@arg);
			diag("ctx->$name(@arg)");
		}
	}

	$config{set_option} = \@option if @option;
	$self->{config} = \%config;
}


ok( $resolver->string(), 'context rebuilt' );

exit;

