package Bio::Phylo::Annotation;
use strict;
use Bio::Phylo::Util::XMLWritable;
use Bio::Phylo::Util::CONSTANT qw(_ANNOTATION_ _DICTIONARY_);
use vars '@ISA';
use UNIVERSAL qw'can isa';
use Bio::Phylo::Util::Exceptions 'throw';
@ISA=qw(Bio::Phylo::Util::XMLWritable);
{
    my @fields = \( my ( %key, %datatype, %value ) );
    my $TYPE_CONSTANT = _ANNOTATION_;
    my $CONTAINER_CONSTANT = _DICTIONARY_;

=head1 NAME

Bio::Phylo::Annotation - Single key/value annotation, used within dictionary

=head1 SYNOPSIS

 use Bio::Phylo::Factory;
 my $fac = Bio::Phylo::Factory->new;
 # there are several data types: string, integer, float, uri, any, etc.
 # refer to nexml.org for a complete list
 my $dic = $fac->create_dictionary( -tag => 'string' );
 my $ann = $fac->create_annotation( -value => 'This is a description' );
 $dic->insert( $ann );
 print $dic->to_xml;

 # prints: <dict><string id="string2">This is a description</string></dict>

=head1 DESCRIPTION

To comply with the NeXML standard (L<http://www.nexml.org>), Bio::Phylo implements
dictionary annotations, which consist conceptually of key/value pairs where the
key is an xml id (which is either autogenerated or can be set using the set_xml_id
method) and the value is the containing element (whose name specifies the data
type of its text contents, i.e. string, integer, float and so on).

This class implements a single key/value pair, of which multiples can be inserted
in a dictionary to create a data structure that is serialized to something akin
to a hash. The dictionary can be attached to any of the subclasses of
L<Bio::Phylo::Util::XMLWritable>.

Of note is the fact that annotations can has different xml tag names, where the 
name specifies the data type of the annotation text content. For example, if you 
set the tag name to 'string' (i.e. $ann->set_tag('string')) then the value is 
interpreted to be a string (i.e. $ann->set_value('some kind of string')). Also,
a common tag name is 'any', which means that the value is any kind of xml structure,
which can be provided as a raw string, or as an xml element tree object of one of 
the following distributions: L<XML::DOM>, L<XML::GDOME>, L<XML::Twig>, L<XML::DOM2>,
L<XML::DOMBacked>, L<XML::Handler>, L<XML::Element>, L<XML::API>, L<XML::Code> or
L<XML::XMLWriter>. Alternatively, you can provide a L<RDF::Core::Model> for
semantic annotations or a L<Bio::Phylo::Dictionary> to create recursively nested
dictionaries. 

=head1 METHODS

=head2 CONSTRUCTOR

=over

=item new()

 Type    : Constructor
 Title   : new
 Usage   : my $anno = Bio::Phylo::Annotation->new;
 Function: Initializes a Bio::Phylo::Annotation object.
 Returns : A Bio::Phylo::Annotation object.
 Args    : optional constructor arguments are key/value
 		   pairs where the key corresponds with any of
 		   the methods that starts with set_ (i.e. mutators) 
 		   and the value is the permitted argument for such 
 		   a method. The method name is changed such that,
 		   in order to access the set_value($val) method
 		   in the constructor, you would pass -value => $val

=cut

    sub new {
        return shift->SUPER::new( '-tag' => 'string', @_ );     
    }

=back

=head2 MUTATORS

=over

=item set_value()

Sets the annotation value, e.g. for an annotation with tag 'string',
this would be a free form string, such as a description.

 Type    : Mutator
 Title   : set_value
 Usage   : $anno->set_value('this is a description');
 Function: Sets the annotation value
 Returns : Modified object.
 Args    : A valid argument is whatever is valid for the annotation
           data type.

=cut

    sub set_value {
        my ( $self, $value ) = @_;
        $value{ $self->get_id } = $value;
        return $self;
    }

=back

=head2 ACCESSORS

=over

=item get_value()

Gets invocant's value.

 Type    : Accessor
 Title   : get_value
 Usage   : my $val = $anno->get_value;
 Function: Gets invocant's value.
 Returns : The specified value, whose data type depends on the 
           xml tag name.
 Args    : NONE

=cut

    sub get_value {
        my $self = shift;
        return $value{ $self->get_id };
    }

=back

=head2 SERIALIZERS

=over

=item to_xml()

Serializes object to an xml string

 Type    : Serializer
 Title   : to_xml()
 Usage   : print $ann->to_xml();
 Function: Serializes object to xml string
 Returns : String 
 Args    : None
 Comments:

=cut

    sub to_xml {
        my $self = shift;
        my $key = $self->get_xml_id;
        my $xml = '';
        my $value = $self->get_value;
        if ( ref($value) ) {

        	# for RDF::Core::Model objects
        	if ( isa($value, 'RDF::Core::Model') ) {
        		eval {
        			require RDF::Core::Model::Serializer;
        			my $serialized_model = '';
  					my $serializer = RDF::Core::Model::Serializer->new(
  						'Model'  => $value,
                        'Output' => \$serialized_model,
                        # BaseURI => 'URI://BASE/',
                    );   
                    $value = $serialized_model;     			
        		};
        		if ( $@ ) {
        			throw 'API' => $@;
        		}
        	}         
        	
        	# for XML::XMLWriter object
        	elsif ( isa($value, 'XML::XMLWriter') ) {
        		$value = $value->get;
        	}
        	
        	else {
        		my $concatenated = '';
        		my ( @values, @raw_values );
        		if ( ref($value) eq 'ARRAY' ) {
        			@values = @{ $value };
        		}
        		else {
        			push @values, $value;
        		}
        		for my $v ( @values ) {
	        		# duck-typing
		        	# Bio::Phylo => to_xml, XML::DOM,XML::GDOME => toString, XML::Twig => sprint
		        	# XML::DOM2 => xmlify, XML::DOMBacked => as_xml,
		        	# XML::Handler => dump_tree, XML::Element => as_XML
		        	# XML::API => _as_string, XML::Code => code
		        	
		        	if ( ref($v) ) {
			        	my @methods = qw(to_xml toString sprint _as_string code xmlify as_xml dump_tree as_XML);
			        	SERIALIZER: for my $method ( @methods ) {
			        		if ( can($v,$method) ) {
			        			$concatenated .= $v->$method;
			        			last SERIALIZER;
			        		}
			        	}
		        	}
		        	else {
		        		push @raw_values, $v;
		        	}
        		}
        		$value = scalar(@raw_values) ? join(' ',@raw_values) : $concatenated;
        	}
        }
        my $type = $self->get_tag;
        $xml .= "<${type} id=\"${key}\">" . $value . "</${type}>";
        return $xml;
    }

=back

=cut

# podinherit_insert_token
# podinherit_start_token_do_not_remove
# AUTOGENERATED pod created by /Users/rvosa/Applications/podinherit on Wed Mar  4 17:13:19 2009
# DO NOT EDIT the code below, rerun /Users/rvosa/Applications/podinherit instead.

=pod

=head1 INHERITED METHODS

Bio::Phylo::Annotation inherits from one or more superclasses. This means that objects of 
class Bio::Phylo::Annotation also "do" the methods from the superclasses in addition to the 
ones implemented in this class. Below is the documentation for those additional 
methods, organized by superclass.

=head2 SUPERCLASS Bio::Phylo::Util::XMLWritable

Bio::Phylo::Annotation inherits from superclass L<Bio::Phylo::Util::XMLWritable>. 
Below are the public methods (if any) from this superclass.

=over

=item add_dictionary()

 Type    : Mutator
 Title   : add_dictionary
 Usage   : $obj->add_dictionary($dict);
 Function: Adds a dictionary attachment to the object
 Returns : $self
 Args    : Bio::Phylo::Dictionary

=item get_attributes()

Retrieves attributes for the element.

 Type    : Accessor
 Title   : get_attributes
 Usage   : my %attrs = %{ $obj->get_attributes };
 Function: Gets the xml attributes for the object;
 Returns : A hash reference
 Args    : None.
 Comments: throws ObjectMismatch if no linked taxa object 
           can be found

=item get_dictionaries()

Retrieves the dictionaries for the element.

 Type    : Accessor
 Title   : get_dictionaries
 Usage   : my @dicts = @{ $obj->get_dictionaries };
 Function: Retrieves the dictionaries for the element.
 Returns : An array ref of Bio::Phylo::Dictionary objects
 Args    : None.

=item get_namespaces()

 Type    : Accessor
 Title   : get_namespaces
 Usage   : my %ns = %{ $obj->get_namespaces };
 Function: Retrieves the known namespaces
 Returns : A hash of prefix/namespace key/value pairs, or
           a single namespace if a single, optional
           prefix was provided as argument
 Args    : Optional - a namespace prefix

=item get_tag()

Retrieves tag name for the element.

 Type    : Accessor
 Title   : get_tag
 Usage   : my $tag = $obj->get_tag;
 Function: Gets the xml tag name for the object;
 Returns : A tag name
 Args    : None.

=item get_xml_id()

Retrieves xml id for the element.

 Type    : Accessor
 Title   : get_xml_id
 Usage   : my $id = $obj->get_xml_id;
 Function: Gets the xml id for the object;
 Returns : An xml id
 Args    : None.

=item get_xml_tag()

Retrieves tag string

 Type    : Accessor
 Title   : get_xml_tag
 Usage   : my $str = $obj->get_xml_tag;
 Function: Gets the xml tag for the object;
 Returns : A tag, i.e. pointy brackets
 Args    : Optional: a true value, to close an empty tag

=item is_identifiable()

By default, all XMLWritable objects are identifiable when serialized,
i.e. they have a unique id attribute. However, in some cases a serialized
object may not have an id attribute (governed by the nexml schema). This
method indicates whether that is the case.

 Type    : Test
 Title   : is_identifiable
 Usage   : if ( $obj->is_identifiable ) { ... }
 Function: Indicates whether IDs are generated
 Returns : BOOLEAN
 Args    : NONE

=item remove_dictionary()

 Type    : Mutator
 Title   : remove_dictionary
 Usage   : $obj->remove_dictionary($dict);
 Function: Removes a dictionary attachment from the object
 Returns : $self
 Args    : Bio::Phylo::Dictionary

=item set_attributes()

Assigns attributes for the element.

 Type    : Mutator
 Title   : set_attributes
 Usage   : $obj->set_attributes( 'foo' => 'bar' )
 Function: Sets the xml attributes for the object;
 Returns : $self
 Args    : key/value pairs or a hash ref

=item set_identifiable()

By default, all XMLWritable objects are identifiable when serialized,
i.e. they have a unique id attribute. However, in some cases a serialized
object may not have an id attribute (governed by the nexml schema). For
such objects, id generation can be explicitly disabled using this method.
Typically, this is done internally - you will probably never use this method.

 Type    : Mutator
 Title   : set_identifiable
 Usage   : $obj->set_tag(0);
 Function: Enables/disables id generation
 Returns : $self
 Args    : BOOLEAN

=item set_namespaces()

 Type    : Mutator
 Title   : set_namespaces
 Usage   : $obj->set_namespaces( 'dwc' => 'http://www.namespaceTBD.org/darwin2' );
 Function: Adds one or more prefix/namespace pairs
 Returns : $self
 Args    : One or more prefix/namespace pairs, as even-sized list, 
           or as a hash reference, i.e.:
           $obj->set_namespaces( 'dwc' => 'http://www.namespaceTBD.org/darwin2' );
           or
           $obj->set_namespaces( { 'dwc' => 'http://www.namespaceTBD.org/darwin2' } );
 Notes   : This is a global for the XMLWritable class, so that in a recursive
 		   to_xml call the outermost element contains the namespace definitions.
 		   This method can also be called as a static class method, i.e.
 		   Bio::Phylo::Util::XMLWritable->set_namespaces(
 		   'dwc' => 'http://www.namespaceTBD.org/darwin2');

=item set_tag()

This method is usually only used internally, to define or alter the
name of the tag into which the object is serialized. For example,
for a Bio::Phylo::Forest::Node object, this method would be called 
with the 'node' argument, so that the object is serialized into an
xml element structure called <node/>

 Type    : Mutator
 Title   : set_tag
 Usage   : $obj->set_tag('node');
 Function: Sets the tag name
 Returns : $self
 Args    : A tag name (must be a valid xml element name)

=item set_xml_id()

This method is usually only used internally, to store the xml id
of an object as it is parsed out of a nexml file - this is for
the purpose of round-tripping nexml info sets.

 Type    : Mutator
 Title   : set_xml_id
 Usage   : $obj->set_xml_id('node345');
 Function: Sets the xml id
 Returns : $self
 Args    : An xml id (must be a valid xml NCName)

=item to_xml()

Serializes invocant to XML.

 Type    : XML serializer
 Title   : to_xml
 Usage   : my $xml = $obj->to_xml;
 Function: Serializes $obj to xml
 Returns : An xml string
 Args    : None

=back

=head2 SUPERCLASS Bio::Phylo

Bio::Phylo::Annotation inherits from superclass L<Bio::Phylo>. 
Below are the public methods (if any) from this superclass.

=over

=item clone()

Clones invocant.

 Type    : Utility method
 Title   : clone
 Usage   : my $clone = $object->clone;
 Function: Creates a copy of the invocant object.
 Returns : A copy of the invocant.
 Args    : None.
 Comments: Cloning is currently experimental, use with caution.

=item get()

Attempts to execute argument string as method on invocant.

 Type    : Accessor
 Title   : get
 Usage   : my $treename = $tree->get('get_name');
 Function: Alternative syntax for safely accessing
           any of the object data; useful for
           interpolating runtime $vars.
 Returns : (context dependent)
 Args    : a SCALAR variable, e.g. $var = 'get_name';

=item get_desc()

Gets invocant description.

 Type    : Accessor
 Title   : get_desc
 Usage   : my $desc = $obj->get_desc;
 Function: Returns the object's description (if any).
 Returns : A string
 Args    : None

=item get_generic()

Gets generic hashref or hash value(s).

 Type    : Accessor
 Title   : get_generic
 Usage   : my $value = $obj->get_generic($key);
           or
           my %hash = %{ $obj->get_generic() };
 Function: Returns the object's generic data. If an
           argument is used, it is considered a key
           for which the associated value is returned.
           Without arguments, a reference to the whole
           hash is returned.
 Returns : A string or hash reference.
 Args    : None

=item get_id()

Gets invocant's UID.

 Type    : Accessor
 Title   : get_id
 Usage   : my $id = $obj->get_id;
 Function: Returns the object's unique ID
 Returns : INT
 Args    : None

=item get_internal_name()

Gets invocant's 'fallback' name (possibly autogenerated).

 Type    : Accessor
 Title   : get_internal_name
 Usage   : my $name = $obj->get_internal_name;
 Function: Returns the object's name (if none was set, the name
           is a combination of the $obj's class and its UID).
 Returns : A string
 Args    : None

=item get_logger()

Gets a logger object.

 Type    : Accessor
 Title   : get_logger
 Usage   : my $logger = $obj->get_logger;
 Function: Returns a Bio::Phylo::Util::Logger object
 Returns : Bio::Phylo::Util::Logger
 Args    : None

=item get_name()

Gets invocant's name.

 Type    : Accessor
 Title   : get_name
 Usage   : my $name = $obj->get_name;
 Function: Returns the object's name.
 Returns : A string
 Args    : None

=item get_obj_by_id()

Attempts to fetch an in-memory object by its UID

 Type    : Accessor
 Title   : get_obj_by_id
 Usage   : my $obj = Bio::Phylo->get_obj_by_id($uid);
 Function: Fetches an object from the IDPool cache
 Returns : A Bio::Phylo object 
 Args    : A unique id

=item get_score()

Gets invocant's score.

 Type    : Accessor
 Title   : get_score
 Usage   : my $score = $obj->get_score;
 Function: Returns the object's numerical score (if any).
 Returns : A number
 Args    : None

=item new()

The Bio::Phylo root constructor, is rarely used directly. Rather, many other 
objects in Bio::Phylo internally go up the inheritance tree to this constructor. 
The arguments shown here can therefore also be passed to any of the child 
classes' constructors, which will pass them on up the inheritance tree. Generally, 
constructors in Bio::Phylo subclasses can process as arguments all methods that 
have set_* in their names. The arguments are named for the methods, but "set_" 
has been replaced with a dash "-", e.g. the method "set_name" becomes the 
argument "-name" in the constructor.

 Type    : Constructor
 Title   : new
 Usage   : my $phylo = Bio::Phylo->new;
 Function: Instantiates Bio::Phylo object
 Returns : a Bio::Phylo object 
 Args    : Optional, any number of setters. For example,
 		   Bio::Phylo->new( -name => $name )
 		   will call set_name( $name ) internally

=item set_desc()

Sets invocant description.

 Type    : Mutator
 Title   : set_desc
 Usage   : $obj->set_desc($desc);
 Function: Assigns an object's description.
 Returns : Modified object.
 Args    : Argument must be a string.

=item set_generic()

Sets generic key/value pair(s).

 Type    : Mutator
 Title   : set_generic
 Usage   : $obj->set_generic( %generic );
 Function: Assigns generic key/value pairs to the invocant.
 Returns : Modified object.
 Args    : Valid arguments constitute:

           * key/value pairs, for example:
             $obj->set_generic( '-lnl' => 0.87565 );

           * or a hash ref, for example:
             $obj->set_generic( { '-lnl' => 0.87565 } );

           * or nothing, to reset the stored hash, e.g.
                $obj->set_generic( );

=item set_name()

Sets invocant name.

 Type    : Mutator
 Title   : set_name
 Usage   : $obj->set_name($name);
 Function: Assigns an object's name.
 Returns : Modified object.
 Args    : Argument must be a string, will be single 
           quoted if it contains [;|,|:\(|\)] 
           or spaces. Preceding and trailing spaces
           will be removed.

=item set_score()

Sets invocant score.

 Type    : Mutator
 Title   : set_score
 Usage   : $obj->set_score($score);
 Function: Assigns an object's numerical score.
 Returns : Modified object.
 Args    : Argument must be any of
           perl's number formats, or undefined
           to reset score.

=item to_json()

Serializes object to JSON string

 Type    : Serializer
 Title   : to_json()
 Usage   : print $obj->to_json();
 Function: Serializes object to JSON string
 Returns : String 
 Args    : None
 Comments:

=item to_string()

Serializes object to general purpose string

 Type    : Serializer
 Title   : to_string()
 Usage   : print $obj->to_string();
 Function: Serializes object to general purpose string
 Returns : String 
 Args    : None
 Comments: This is YAML

=back

=cut

# podinherit_stop_token_do_not_remove

=head1 SEE ALSO

=over

=item L<Bio::Phylo::Dictionary>

Annotation objects are combined into a dictionary.

=item L<Bio::Phylo::Util::XMLWritable>

This object inherits from L<Bio::Phylo::Util::XMLWritable>, so methods
defined there are also applicable here.

=item L<Bio::Phylo::Manual>

Also see the manual: L<Bio::Phylo::Manual> and L<http://rutgervos.blogspot.com>.

=back

=head1 REVISION

 $Id: Annotation.pm 844 2009-03-05 00:07:26Z rvos $

=cut
    
    sub _type { $TYPE_CONSTANT }
    sub _container { $CONTAINER_CONSTANT }
    sub _cleanup {
        my $self = shift;
        my $id = $self->get_id;
        for my $field ( @fields ) {
            delete $field->{$id};
        }
    }    
}
1;