#!/usr/bin/perl

# Copyright 2001-2004, Paul Johnson (pjcj@cpan.org)

# This software is free.  It is licensed under the same terms as Perl itself.

# The latest version of this software should be available from my homepage:
# http://www.pjcj.net

require 5.6.1;

use strict;
use warnings;

use Cwd;
use ExtUtils::MakeMaker;
use ExtUtils::Manifest "maniread";
use File::Copy;

$| = 1;

my $Version  = "0.41";
my $Date     = "29th April 2004";
my $Author   = 'pjcj@cpan.org';

my @perlbug  = ("perlbug", "-a", $Author,
                           "-s", "Installation of Devel::Cover $Version");
my $Perlbug  = join " ", map { / / ? "'$_'" : $_ } @perlbug;

my @files    = sort keys %{maniread()}, "lib/Devel/Cover/Inc.pm";
my @versions = grep { $_ ne "README" && $_ ne "Makefile.PL" } @files;

my $base     = getcwd;

my %inc      = map { $_ => 1 } @INC;
my @inc      = sort keys %inc;

open I, ">lib/Devel/Cover/Inc.pm"
    or die "Cannot open lib/Devel/Cover/Inc.pm: $!";
print I <<"EOI";
# Copyright 2001-2004, Paul Johnson (pjcj\@cpan.org)

# This software is free.  It is licensed under the same terms as Perl itself.

# The latest version of this software should be available from my homepage:
# http://www.pjcj.net

# This file was automatically generated by Makefile.PL.

package Devel::Cover::Inc;

use strict;
use warnings;

our \$VERSION = "$Version";
our \$Perl    = '$^X';
our \$Base    = '$base';
our \@Inc     = qw( @inc );

1
EOI
close I or die "Cannot close lib/Devel/Cover/Inc.pm: $!";

print "Writing tests ........ ";

unless (-d "t")
{
    mkdir "t" or die "Cannot mkdir t: $!";
}

opendir D, "tests" or die "Cannot opendir tests: $!";
for my $t (readdir D)
{
    next unless -f "tests/$t";
    next if $t =~ /\.(pm|pl|version|org|bak)$/;
    next if $t =~ /~$/;

    if ($t =~ /\.t/)
    {
        copy("tests/$t", "t/$t") or die "Cannot copy tests/$t to t/$t: $!";
        next
    }

    open T, ">t/a$t.t" or die "Cannot open t/a$t.t: $!";
    print T <<EOT;
#!$^X

# Copyright 2002-2004, Paul Johnson (pjcj\@cpan.org)

# This software is free.  It is licensed under the same terms as Perl itself.

# The latest version of this software should be available from my homepage:
# http://www.pjcj.net

use strict;
use warnings;

use lib "$base/lib";
use lib "$base/blib/lib";
use lib "$base/blib/arch";
use lib "$base/t";

use Devel::Cover::Test $Version;

my \$test = Devel::Cover::Test->new("$t");
\$test->run_test
EOT
    close T or die "Cannot open t/a$t.t: $!";
}
closedir D or die "Cannot closedir tests: $!";

print "done\n\n";

my $e;

print "checking for Storable.pm                   ........ ";

$e = <<EOM;

Storable.pm is required to store the coverage database.  You can
download Storable from CPAN.

EOM

eval "use Storable";
if (my $m = $INC{"Storable.pm"})
{
    my $v = eval { no warnings; $Storable::VERSION };
    print "$v $m\n";
}
else
{
    print "not found\n\n$e\n";
}

print "checking for Digest::MD5.pm                ........ ";

$e = <<EOM;

Digest::MD5.pm is required to check whether covered files have changed.
You can download Digest::MD5 from CPAN.

EOM

eval "use Digest::MD5";
if (my $m = $INC{"Digest/MD5.pm"})
{
    my $v = eval { no warnings; $Digest::MD5::VERSION };
    print "$v $m\n";
}
else
{
    print "not found\n\n$e\n";
}

# print "checking for Template.pm      version 2.00 ........ ";
# 
# $e = <<EOM;
# 
# Template.pm 2.00 is required to run the HTML backend to cover and for
# cpancover.  You will not be able to generate HTML output until you
# install the Template Toolkit, available from CPAN.  In the meantime you
# may continue to use the rest of Devel::Cover.
# 
# EOM
# 
# eval "use Template";
# if (my $m = $INC{"Template.pm"})
# {
    # my $v = eval { no warnings; $Template::VERSION };
    # print $v < 2.00 ? "$v\n\n$e\n" : "$v $m\n";
# }
# else
# {
    # print "not found\n\n$e\n";
# }

print "checking for Pod::Coverage.pm version 0.06 ........ ";

$e = <<EOM;

Pod::Coverage.pm 0.06 is required to do pod coverage.  This will tell
you how well you have documented your modules.  Pod coverage will be
unavailable until you install this module, available from CPAN.  In the
meantime, you may continue to use the rest of Devel::Cover.
EOM

eval "use Pod::Coverage";
if (my $m = $INC{"Pod/Coverage.pm"})
{
    my $v = eval { no warnings; $Pod::Coverage::VERSION };
    print $v < 0.06 ? "$v\n\n$e\n" : "$v $m\n";
}
else
{
    print "not found\n\n$e\n";
}

print "checking for Test::Differences.pm          ........ ";

$e = <<EOM;

Test::Differences.pm is to display output from failed tests.  Hopefully
there won't be any failed tests, but if there are you will get output
that may not be a model of clarity.  If you do get test failures and you
fancy helping me by debugging them, then you might like to consider
installing Test::Differences.  You can download Test::Differences from
CPAN.

EOM

eval "use Test::Differences";
if (my $m = $INC{"Test/Differences.pm"})
{
    my $v = eval { no warnings; $Test::Differences::VERSION };
    print "$v $m\n";
}
else
{
    print "not found\n\n$e\n";
}

my $latest_tested = 5.008004;
print <<EOM if $] > $latest_tested;

Devel::Cover $Version has not been tested with perl $].
Testing will take place against expected output from perl $latest_tested.
You may well find failing tests.

EOM

print "\n" x 3;

$ExtUtils::MakeMaker::Verbose = 0;

WriteMakefile
(
    NAME      => "Devel::Cover",
    VERSION   => $Version,
    AUTHOR    => 'Paul Johnson (pjcj@cpan.org)',
    ABSTRACT  => "Code coverage metrics for Perl",
    DIR       => [],
    EXE_FILES => [ "cover", "gcov2perl" ],
    PREREQ_PM => {
                     Storable      => 0,
                     "Digest::MD5" => 0,
                 },
    dist      => { COMPRESS => "gzip --best --force" },
    clean     => { FILES => join " ",  "t/*", map "$_.version", @versions },
    realclean => { FILES => "lib/Devel/Cover/Inc.pm cover_db t" },
    depend    => { distdir => "@files" },
);

print "\n";
print <<EOM if 0;

--------------------------------------------------------------------------------

I like to have some idea of the distribution of this software.  To that end I
would be very grateful for mail from you.  This will be used only to satisfy my
curiosity and to help me make decisions which may involve changes to the
module.

If you can run perlbug you can send me a success report with "make ok".
Failure reports with "make nok" are also appreciated.

If you have any questions or comments, mailing list details are available in the
README file, or send mail to me directly.

EOM

sub MY::libscan
{
    my ($self, $path) = @_;
    (my $p = $path) =~ s/^\$\(INST_LIB\)/lib/;  # 5.6.1
    $p =~ s|\\|/|g if $^O eq "MSWin32";
    # print "$path $p\n";
    my $wanted = -d $p;                         # 5.9.0
    for my $f (@files)
    {
        # print "$p - $f\n";
        last if $wanted ||= $p =~ /$f$/;
    }
    $wanted && $path;
}

sub MY::postamble
{
    qq[
SET_VERSION = \$(PERL) -pi.version \\
    -e 's/(^\\s*(?:our\\s+)\\\$\$VERSION = ")\\d+\\.\\d+(";)/\$\${1}$Version\$\$2/;' \\
    -e 's/(Version )\\d+\\.\\d+( - ).*/\$\${1}$Version\$\${2}$Date/;' \\
    -e 's/(Devel::Cover Version )\\d+\\.\\d+/\$\${1}$Version/;' \\
    -e 's/(\\buse Devel::Cover(?:::\\w+)*\\s+)\\d+\\.\\d+/\$\${1}$Version/;'

tags : @files
\t ptags @files

@versions : Makefile.PL
\t \$(SET_VERSION) @versions

README : lib/Devel/Cover.pm
\t TERMCAP= COLUMNS=80 pod2text lib/Devel/Cover.pm | \\
    \$(PERL) -n \\
    -e 'print if (/NAME/ ... /^[A-Z ]+\$\$/) =~ /^\\d+\$\$/;' \\
    -e 'print if (/DESCRIPTION/ ... /^[A-Z ]+\$\$/) =~ /^\\d+\$\$/;' \\
    -e 'print if (/REQUIREMENTS/ ... /^[A-Z ]+\$\$/) =~ /^\\d+\$\$/;' \\
    > README

ppm : ppd pure_all
\t tar cf Devel-Cover.tar blib
\t gzip --best --force Devel-Cover.tar
\t \$(PERL) -pi.bak \\
    -e 's/(OS NAME=")[^"]*/\$\$1MSWin32/;' \\
    -e 's/(ARCHITECTURE NAME=")[^"]*/\$\$1MSWin32-x86-multi-thread/;' \\
    -e 's/(CODEBASE HREF=")[^"]*/\$\$1Gedcom.tar.gz/;' \\
    Devel-Cover.ppd

TAINT = -T

_run : pure_all
\t \$(PERL) \$(TAINT) -Iblib/lib -Iblib/arch -MDevel::Cover=-ignore,blib,-merge,0,`\$(PERL) -e 'qx|grep __COVER__ \$\$ARGV[0]| =~ /__COVER__\\s+criteria\\s+(.*)/; (\$\$c = \$\$1 || "all") =~ s/\\s+/,/g; print "-coverage,\$\$c"' tests/\$(TEST)` tests/\$(TEST) && \\
   \$(PERL) -Mblib cover -report text > \$(TEST).out

run : _run
\t \$(PERL) -Mblib cover && \\
   gvim -d \$(TEST).out

DB = cover_db

dump :
\t \$(PERL) -Mblib cover -dump_db \$(DB)

FONT = 8x13
FONT = -sun-screen-medium-r-normal-*-*-70-*-*-m-*-sun-fontspecific
FONT = "Bitstream Vera Sans Mono 8"

diff : _run
\t \$(PERL) -i -pe '\$\$t = index \$\$_, "time   code" if !defined \$\$t || \$\$t < 0; substr \$\$_, \$\$t, 7, "" if /^line  err   stmt/ ..  /^Branches/ and length > \$\$t' \$(TEST).out
\t gvim -d -font \$(FONT) `\$(PERL) -Mblib -MDevel::Cover::Test -e '\$\$t = Devel::Cover::Test->new("\$(TEST)"); print \$\$t->cover_gold'` \$(TEST).out

gold : pure_all
\t \$(PERL) create_gold \$(TEST)

all_test :
\t exec \$(PERL) all_versions make test

all_gold :
\t rm -rf test_output
\t exec \$(PERL) all_versions make gold

mani :
\t \$(PERL) -i.bak -ne '\$\$d = "test_output/cover"; print unless /^\$\$d/; END { open M, ">>MANIFEST"; opendir D, \$\$d; print M "\$\$d/\$\$_\\n" for sort grep -f "\$\$d/\$\$_" && !/\.version\$\$/, readdir D }' MANIFEST

ok :
\t \@$Perlbug -okay  || echo "Please send your report manually to $Author"

nok :
\t \@$Perlbug -nokay || echo "Please send your report manually to $Author"
    ]
}
