#!/usr/bin/perl

use strict;
use warnings;

use Test::More;

use Net::Prometheus;

my $client = Net::Prometheus->new;

# No labels
{
   my $gauge = $client->new_gauge(
      name => "test_gauge",
      help => "A testing gauge",
   );

   ok( defined $gauge, 'defined $gauge' );

   is( $client->render, <<'EOF', '$client->render contains gauge' );
# HELP test_gauge A testing gauge
# TYPE test_gauge gauge
EOF

   $gauge->set( 0 );

   is( $client->render, <<'EOF', '$client->render gives gauge value' );
# HELP test_gauge A testing gauge
# TYPE test_gauge gauge
test_gauge 0
EOF

   $gauge->inc;

   is( $client->render, <<'EOF', 'render after $gauge->inc' );
# HELP test_gauge A testing gauge
# TYPE test_gauge gauge
test_gauge 1
EOF

   $gauge->inc( 10 );

   is( $client->render, <<'EOF', 'render after $gauge->inc( 10 )' );
# HELP test_gauge A testing gauge
# TYPE test_gauge gauge
test_gauge 11
EOF

   $gauge->dec( 5 );

   is( $client->render, <<'EOF', 'render after $gauge->dec( 5 )' );
# HELP test_gauge A testing gauge
# TYPE test_gauge gauge
test_gauge 6
EOF

   $client->unregister( $gauge );
}

# One label
{
   my $gauge = $client->new_gauge(
      name => "labeled_gauge",
      help => "A gauge with a label",
      labels => [qw( lab )],
   );

   $gauge->set( one => 1 );
   $gauge->set( two => 2 );

   is( $client->render, <<'EOF', '$client->render gives gauge value' );
# HELP labeled_gauge A gauge with a label
# TYPE labeled_gauge gauge
labeled_gauge{lab="one"} 1
labeled_gauge{lab="two"} 2
EOF

   $gauge->labels( "three" )->set( 3 );

   is( $client->render, <<'EOF', '$client->render gives gauge value' );
# HELP labeled_gauge A gauge with a label
# TYPE labeled_gauge gauge
labeled_gauge{lab="one"} 1
labeled_gauge{lab="three"} 3
labeled_gauge{lab="two"} 2
EOF

   $client->unregister( $gauge );
}

# Two labels
{
   my $gauge = $client->new_gauge(
      name => "multidimensional_gauge",
      help => "A gauge with two labels",
      labels => [qw( x y )],
   );

   $gauge->set( 0 => 0 => 10 );
   $gauge->set( 0 => 1 => 20 );
   $gauge->set( 1 => 0 => 30 );
   $gauge->set( 1 => 1 => 40 );

   is( $client->render, <<'EOF', '$client->render gives gauge value' );
# HELP multidimensional_gauge A gauge with two labels
# TYPE multidimensional_gauge gauge
multidimensional_gauge{x="0",y="0"} 10
multidimensional_gauge{x="0",y="1"} 20
multidimensional_gauge{x="1",y="0"} 30
multidimensional_gauge{x="1",y="1"} 40
EOF

   $client->unregister( $gauge );
}

done_testing;
