#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2016-2017 -- leonerd@leonerd.org.uk

package Future::AsyncAwait;

use strict;
use warnings;

our $VERSION = '0.01';

use Carp;

require XSLoader;
XSLoader::load( __PACKAGE__, $VERSION );

=head1 NAME

C<Future::AsyncAwait> - deferred subroutine syntax for futures

=head1 SYNOPSIS

 use Future::AsyncAwait;

 async sub do_a_thing
 {
    my $first = await do_first_thing();

    my $second = await do_second_thing();

    return combine_things( $first, $second );
 }

 do_a_thing()->get;

This module provides syntax for deferring and resuming subroutines while
waiting for L<Future>s to complete.

B<WARNING>: The actual semantics in this module are not yet implemented. This
is released purely to demonstrate the syntax parts of its operation, to
reserve the name on CPAN, and to provide something that actually exists in
order to look at it. Don't expect to be able to use this module in any real
code yet.

That said, the only part that isn't actually implemented currently is the part
that suspends and resumes subroutines while waiting for a future to complete.
The syntax parsing, as well as semantics for immediate futures, are already
defined and working now. So it is already very slightly useful for writing
simple functions that return immediate futures.

Instead of writing

 sub foo
 {
    ...
    return Future->done( @result );
 }

you can now simply write

 async sub
 {
    ...
    return @result;
 }

with the added side-benefit that any exceptions thrown by the elided code will
be turned into an immediate-failed C<Future> rather than making the call
itself propagate the exception, which is usually what you wanted when dealing
with futures.

=cut

sub import
{
   my $class = shift;
   my $caller = caller;

   $class->import_into( $caller, @_ );
}

sub import_into
{
   my $class = shift;
   my ( $caller, @syms ) = @_;

   my %syms = map { $_ => 1 } @syms;
   $^H{"Future::AsyncAwait/async"}++ if delete $syms{async};

   croak "Unrecognised import symbols @{[ keys %syms ]}" if keys %syms;
}

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
