package CPAN::Module::FromURL;

our $DATE = '2016-02-21'; # DATE
our $VERSION = '0.02'; # VERSION

use 5.010001;
use strict;
use warnings;

use Exporter qw(import);
our @EXPORT_OK = qw(extract_module_from_cpan_url);

our %SPEC;

$SPEC{extract_module_from_cpan_url} = {
    v => 1.1,
    summary => 'Detect and extract module name from some CPAN-related URL',
    args => {
        url => {
            schema => 'str*',
            req => 1,
            pos => 0,
        },
    },
    args_as => 'array',
    result => {
        schema => 'str',
    },
    result_naked => 1,
    examples => [
        # metacpan.org/pod/PKG
        {
            args => {url=>'https://metacpan.org/pod/Foo::Bar'},
            result => 'Foo::Bar',
        },
        {
            args => {url=>'http://metacpan.org/pod/Foo'},
            result => 'Foo',
        },

        # metacpan.org/module/PKG
        {
            args => {url=>'https://metacpan.org/module/Foo::Bar'},
            result => 'Foo::Bar',
        },

        # metacpan.org/pod/release/PAUSEID/RELEASE/lib/PKG.pm
        {
            args => {url=>'https://metacpan.org/pod/release/SMONF/Dependencies-Searcher-0.066_001/lib/Dependencies/Searcher.pm'},
            result => 'Dependencies::Searcher',
        },
        {
            args => {url=>'http://metacpan.org/pod/release/SRI/Mojolicious-6.46/lib/Mojo.pm'},
            result => 'Mojo',
        },

        # search.cpan.org/~PAUSEID/RELEASE/lib/PKG.pm
        {
            args => {url=>'http://search.cpan.org/~unera/DR-SunDown-0.02/lib/DR/SunDown.pm'},
            result => 'DR::SunDown',
        },
        {
            args => {url=>'https://search.cpan.org/~sri/Mojolicious-6.47/lib/Mojo.pm'},
            result => 'Mojo',
        },

        # search.cpan.org/dist/DIST
        {
            args => {url=>'http://search.cpan.org/dist/Foo-Bar/'},
            result => 'Foo::Bar',
        },

        # search.cpan.org/perldoc?MOD
        {
            args => {url=>'http://search.cpan.org/perldoc?Foo::Bar'},
            result => 'Foo::Bar',
        },
        {
            args => {url=>'http://search.cpan.org/perldoc?Foo'},
            result => 'Foo',
        },

        {
            args => {url=>'https://www.google.com/'},
            result => undef,
        },
    ],
};
sub extract_module_from_cpan_url {
    my $url = shift;

    # note: /module is the old URL
    if ($url =~ m!\Ahttps?://metacpan\.org/(?:pod|module)/(\w+(?:::\w+)*)\z!) {
        return $1;
    }

    if ($url =~ m!\Ahttps?://metacpan\.org/pod/release/[^/]+/[^/]+/lib/((?:[^/]+/)*\w+)\.pm\z!) {
        my $mod = $1;
        $mod =~ s!/!::!g;
        return $mod;
    }

    if ($url =~ m!\Ahttps?://search\.cpan\.org/~[^/]+/[^/]+/lib/((?:[^/]+/)*\w+).pm\z!) {
        my $mod = $1;
        $mod =~ s!/!::!g;
        return $mod;
    }

    if ($url =~ m!\Ahttps?://search\.cpan\.org/dist/([A-Za-z0-9_-]+)/?\z!) {
        my $mod = $1;
        $mod =~ s!-!::!g;
        return $mod;
    }

    if ($url =~ m!\Ahttps?://search\.cpan\.org/perldoc\?(\w+(?:::\w+)*)\z!) {
        return $1;
    }

    undef;
}

1;
# ABSTRACT: Detect and extract module name from some CPAN-related URL

__END__

=pod

=encoding UTF-8

=head1 NAME

CPAN::Module::FromURL - Detect and extract module name from some CPAN-related URL

=head1 VERSION

This document describes version 0.02 of CPAN::Module::FromURL (from Perl distribution CPAN-Module-FromURL), released on 2016-02-21.

=head1 FUNCTIONS


=head2 extract_module_from_cpan_url($url) -> str

Detect and extract module name from some CPAN-related URL.

Examples:

=over

=item * Example #1:

 extract_module_from_cpan_url( url => "https://metacpan.org/pod/Foo::Bar"); # -> "Foo::Bar"

=item * Example #2:

 extract_module_from_cpan_url( url => "http://metacpan.org/pod/Foo"); # -> "Foo"

=item * Example #3:

 extract_module_from_cpan_url( url => "https://metacpan.org/module/Foo::Bar"); # -> "Foo::Bar"

=item * Example #4:

 extract_module_from_cpan_url(  url => "https://metacpan.org/pod/release/SMONF/Dependencies-Searcher-0.066_001/lib/Dependencies/Searcher.pm");

Result:

 "Dependencies::Searcher"

=item * Example #5:

 extract_module_from_cpan_url(  url => "http://metacpan.org/pod/release/SRI/Mojolicious-6.46/lib/Mojo.pm");

Result:

 "Mojo"

=item * Example #6:

 extract_module_from_cpan_url(  url => "http://search.cpan.org/~unera/DR-SunDown-0.02/lib/DR/SunDown.pm");

Result:

 "DR::SunDown"

=item * Example #7:

 extract_module_from_cpan_url(  url => "https://search.cpan.org/~sri/Mojolicious-6.47/lib/Mojo.pm"); # -> "Mojo"

=item * Example #8:

 extract_module_from_cpan_url( url => "http://search.cpan.org/dist/Foo-Bar/"); # -> "Foo::Bar"

=item * Example #9:

 extract_module_from_cpan_url( url => "http://search.cpan.org/perldoc?Foo::Bar"); # -> "Foo::Bar"

=item * Example #10:

 extract_module_from_cpan_url( url => "http://search.cpan.org/perldoc?Foo"); # -> "Foo"

=item * Example #11:

 extract_module_from_cpan_url( url => "https://www.google.com/"); # -> undef

=back

This function is not exported by default, but exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<url>* => I<str>

=back

Return value:  (str)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/CPAN-Module-FromURL>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-CPAN-Module-FromURL>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=CPAN-Module-FromURL>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
