package Padre::Task::Outline::Perl6;

use strict;
use warnings;

our $VERSION = '0.31';

use base 'Padre::Task::Outline';

=pod

=head1 NAME

Padre::Task::Outline::Perl6 - Perl6 document outline structure info 
gathering in the background

=head1 SYNOPSIS

  # by default, the text of the current document
  # will be fetched as will the document's notebook page.
  my $task = Padre::Task::Outline::Perl6->new;
  $task->schedule;
  
  my $task2 = Padre::Task::Outline::Perl6->new(
    text          => Padre::Current->document->text_get,
    editor        => Padre::Current->editor,
  );
  $task2->schedule;

=head1 DESCRIPTION

This class implements structure info gathering of Perl6 documents in
the background.
Also the updating of the GUI is implemented here, because other 
languages might have different outline structures.
It inherits from L<Padre::Task::Outline>.
Please read its documentation!

=cut

sub run {
	my $self = shift;
	$self->_get_outline;
	return 1;
}

sub _get_outline {
	my $self = shift;

	my $outline = [];
	
	if($self->{tokens}) {
		my $cur_pkg = {};
		my $not_first_time = 0;
		my @tokens = @{$self->{tokens}};
		for my $htoken (@tokens) {
			my %token = %{$htoken};
			my $tree = $token{tree};
			if($tree) {
				if($tree =~ /package_declarator.+package_def.+def_module_name/) {
					#XXX implement classes, grammars, modules, packages, roles
					if($not_first_time) {
						if ( not $cur_pkg->{name} ) {
							$cur_pkg->{name} = 'main';
						}
						push @{$outline}, $cur_pkg;
						$cur_pkg = {};
					}
					$not_first_time = 1;
					$cur_pkg->{name} = $token{buffer};
					$cur_pkg->{line} = $token{lineno};
				} elsif($tree =~ /routine_declarator__S_\d+sub routine_def (deflongname)/) {
					# a subroutine
					push @{ $cur_pkg->{subroutines} }, { 
						name => $token{buffer}, 
						line => $token{lineno} 
					};
				} elsif($tree =~ /routine_declarator__\w+_\d+method method_def (longname)/) {
					# a method
					push @{ $cur_pkg->{methods} }, { 
						name => $token{buffer}, 
						line => $token{lineno} 
					};
				} elsif($tree =~ /routine_declarator__\w+_\d+submethod method_def (longname)/) {
					# a submethod
					push @{ $cur_pkg->{submethods} }, { 
						name => $token{buffer}, 
						line => $token{lineno} 
					};
				} elsif($tree =~ /routine_declarator__\w+_\d+macro macro_def (deflongname)/) {
					# a macro
					push @{ $cur_pkg->{macros} }, { 
						name => $token{buffer}, 
						line => $token{lineno} 
					};
				} elsif($tree =~ /regex_declarator__\w+_\d+(regex|token|rule) (regex_def) (deflongname)/) {
					# a regex, token or rule declaration
					push @{ $cur_pkg->{regexes} }, { 
						name => $token{buffer}, 
						line => $token{lineno} 
					};
				} 
			}
		}

		if ( not $cur_pkg->{name} ) {
			$cur_pkg->{name} = 'main';
		}
		push @{$outline}, $cur_pkg;
	}

	$self->{outline} = $outline;
	return;
}

sub update_gui {
	my $self         = shift;
	my $last_outline = shift;
	my $outline      = $self->{outline};
	my $outlinebar   = Padre->ide->wx->main->outline;
	my $editor       = $self->{main_thread_only}->{editor};

	$outlinebar->Freeze;
	$outlinebar->clear;

	require Padre::Wx;

	# If there is no structure, clear the outline pane and return.
	unless ($outline) {
		return;
	}

	# Again, slightly differently
	unless (@$outline) {
		return 1;
	}

	# Add the hidden unused root
	my $root = $outlinebar->AddRoot(
		Wx::gettext('Outline'),
		-1,
		-1,
		Wx::TreeItemData->new('')
	);

	# Update the outline pane
	_update_treectrl( $outlinebar, $outline, $root );

	# Set Perl6 specific event handler
	Wx::Event::EVT_TREE_ITEM_RIGHT_CLICK(
		$outlinebar,
		$outlinebar,
		\&_on_tree_item_right_click,
	);

	$outlinebar->GetBestSize;

	$outlinebar->Thaw;
	return 1;
}

sub _on_tree_item_right_click {
	my ( $outlinebar, $event ) = @_;
	my $showMenu = 0;

	my $menu     = Wx::Menu->new;
	my $itemData = $outlinebar->GetPlData( $event->GetItem );

	if (   defined($itemData)
		&& defined( $itemData->{type} )
		&& $itemData->{type} eq 'modules' )
	{
		my $pod = $menu->Append( -1, Wx::gettext("Open &Documentation") );
		Wx::Event::EVT_MENU(
			$outlinebar,
			$pod,
			sub {

				# TODO Fix this wasting of objects (cf. Padre::Wx::Menu::Help)
				my $help = Padre::Wx::DocBrowser->new;
				$help->help( $itemData->{name} );
				$help->SetFocus;
				$help->Show(1);
				return;
			},
		);
		$showMenu++;
	}

	if ( $showMenu > 0 ) {
		my $x = $event->GetPoint->x;
		my $y = $event->GetPoint->y;
		$outlinebar->PopupMenu( $menu, $x, $y );
	}
	return;
}

sub _update_treectrl {
	my ( $outlinebar, $outline, $root ) = @_;

	foreach my $pkg ( @{$outline} ) {
		my $branch = $outlinebar->AppendItem(
			$root,
			$pkg->{name},
			-1, -1,
			Wx::TreeItemData->new(
				{   line => $pkg->{line},
					name => $pkg->{name},
					type => 'package',
				}
			)
		);
		foreach my $type (qw(modules subroutines methods submethods macros regexes)) {
			_add_subtree( $outlinebar, $pkg, $type, $branch );
		}
		$outlinebar->Expand($branch);
	}

	return;
}

sub _add_subtree {
	my ( $outlinebar, $pkg, $type, $root ) = @_;

	my $type_elem = undef;
	if ( defined( $pkg->{$type} ) && scalar( @{ $pkg->{$type} } ) > 0 ) {
		$type_elem = $outlinebar->AppendItem(
			$root,
			ucfirst($type),
			-1,
			-1,
			Wx::TreeItemData->new()
		);

		my @sorted_entries = ();
		if ( $type eq 'subroutines' || $type eq 'methods' || $type eq 'submethods' || $type eq 'macros') {
			my $config = Padre->ide->config;
			if ( $config->main_functions_order eq 'original' ) {

				# That should be the one we got
				@sorted_entries = @{ $pkg->{$type} };
			} elsif ( $config->main_functions_order eq 'alphabetical_private_last' ) {

				# ~ comes after \w
				my @pre = map { $_->{name} =~ s/^_/~/; $_ } @{ $pkg->{$type} };
				@pre = sort { $a->{name} cmp $b->{name} } @pre;
				@sorted_entries = map { $_->{name} =~ s/^~/_/; $_ } @pre;
			} else {

				# Alphabetical (aka 'abc')
				@sorted_entries = sort { $a->{name} cmp $b->{name} } @{ $pkg->{$type} };
			}
		} else {
			@sorted_entries = sort { $a->{name} cmp $b->{name} } @{ $pkg->{$type} };
		}

		foreach my $item (@sorted_entries) {
			$outlinebar->AppendItem(
				$type_elem,
				$item->{name},
				-1, -1,
				Wx::TreeItemData->new(
					{   line => $item->{line},
						name => $item->{name},
						type => $type,
					}
				)
			);
		}
	}
	if ( defined $type_elem ) {
		if ( $type eq 'subroutines' || 
			$type eq 'methods' || 
			$type eq 'submethods' || 
			$type eq 'macros' || 
			$type eq 'regexes') 
		{
			$outlinebar->Expand($type_elem);
		} else {
			$outlinebar->Collapse($type_elem);
		}
	}

	return;
}

1;

__END__

=head1 SEE ALSO

This class inherits from L<Padre::Task::Outline> which
in turn is a L<Padre::Task> and its instances can be scheduled
using L<Padre::TaskManager>.

=head1 AUTHOR

Ahmad M. Zawawi C<ahmad.zawawi@gmail.com>

=head1 COPYRIGHT AND LICENSE

Copyright 2008-2009 The Padre development team as listed in Padre.pm.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl 5 itself.

=cut

# Copyright 2008-2009 The Padre development team as listed in Padre.pm.
# LICENSE
# This program is free software; you can redistribute it and/or
# modify it under the same terms as Perl 5 itself.
