#!perl

use warnings;
use strict;

use Lab::Data::Analysis;
use Clone qw(clone);

#use Data::Dumper;
use Test::More;
use lib 't';
use Lab::Test import => ['skip_on_broken_printf'];
use Config;

skip_on_broken_printf();

if ( $Config{nvsize} > 8 ) {

    # See http://www.cpantesters.org/cpan/report/b9f8053e-1def-11e7-99c9-a3d8554a9f2d
    plan skip_all => "Test does not work if perl uses long doubles.";
}

plan tests => 5;

my $t = Lab::Data::Analysis->new('t/Analysis/TDS2024TraceData.dat');

my $ev1 = $t->ReadEvent();

my $ev1expect = {
    "STREAM" => {
        "1" => {
            "COMMENT" => [],
            "CONFIG"  => {
                "clone_headers" => 1,
                "combined_gpib" => 0,
                "use_ftell"     => 0
            },
            "NUMBER" => 1,
            "GPIB"   => [
                ">DAT:SOU CH1",
                ">*ESR?",
                "<0",
                ">DAT?",
                "<:DAT:ENC RPB;DEST REFC;SOU CH1;STAR 1;STOP 2500;WID 1",
                ">*ESR?",
                "<0",
                ">WAVF?",
                "<:WFMP:BYT_N 1;BIT_N 8;ENC BIN;BN_F RP;BYT_O MSB;NR_P 2500;WFI \"Ch1, DC coupling, 2.0E-3 V/div, 1.0E-8 s/div, 2500 points, Sample mode\";PT_F Y;XIN 4.0E-11;PT_O 0;XZE -5.0E-8;XUN \"s\";YMU 8.0E-5;YZE 0.0E0;YOF 1.28E2;YUN \"Volts\";:CURV #42500\201\201\201\201\201\201\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\206\206\206\206\206\206\207\207\207\207\206\206\206\206\206\206\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\205\205\205\205\205\205\206\206\206\206\205\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\206\206\206\206\206\206\206\206\206\206\207\207\207\207\207\206\206\206\206\206\206\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\206\206\206\206\206\206\205\205\205\205\205\205\204\204\204\204\204\203\203\203\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\206\206\206\206\206\206\206\205\205\205\205\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\205\205\205\205\205\205\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\206\206\206\206\206\206\206\206\206\206\206\206\206\206\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\205\206\206\206\206\206\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\206\206\206\206\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\204\204\204\204\204\204\204\204\203\203\203\203\202\202\202\202\201\201\201\201\201\200\200\200\200\200\200\200\200\200\200\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\204\204\204\203\203\203\203\202\202\202\202\201\201\201\201\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\201\201\201\201\201\201\202\202\202\202\202\202\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\200\200\200\200\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\203\203\203\203\203\202\202\202\202\202\202\202\201\201\201\201\201\201\201\201\201\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\202\202\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\201\200\200\200\200\200\200\200\200\201\201\201\201\201\201\201\201\202\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\202\202\202\202\201\201\201\201\201\201\201\201\201\202\202\202\202\203\203\203\204\204\204\204\204\204\204\203\203\203\203\203\203\203\202\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\201\201\201\201\201\201\200\200\200\200\200\200\177\177\177\177\177\177\177\177\177\177\200\200\200\200\200\201\201\201\201\202\202\202\202\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\200\200\200\200\177\177\177\177~~~~~}}}}}}}}}}}}}}}}}~~~~~~\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\201\201\201\201\201\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\203\203\203\203\202\202\202\202\201\201\201\201\200\200\200\200\177\177\177\177\177\177\177\177\177\177\177\177\177\177\200\200\200\200\200\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\204\204\204\204\204\204\203\203\203\203\203\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\204\204\204\204\204\204\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\204\204\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\206\206\206\206\206\206\207\207\207\207\207\206\206\206\206\206\206\206\206\206\206\206\206\206\206\205\205\205\205\206\206\206\206\206\206\206\206\206\206\206\206\206\206\207\207\207\207\207\207\206\206\206\206\206\206\206\206\206\206\206\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\201\201\201\201\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\201\201\201\201\201\201\201\201\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\202\202\202\202\202\202\201\201\201\201",
                ">*ESR?",
                "<0",
                ">*ESR?",
                "<0",
                ">ACQ:STATE 1",
                ">*ESR?",
                "<0",
                ">TRIG FORC",
                ">*ESR?",
                "<0",
                ">BUSY?",
                "<:BUSY 0"
            ]
        }
    },
    "EVENT"     => 1,
    "GPIB"      => [],
    "RUNHEADER" => undef,
    "COMMENT"   => [],
    "CONFIG"    => {
        "clone_headers" => 1,
        "combined_gpib" => 0,
        "use_ftell"     => 0
    },
    "FILEHEADER" => undef,
    "TIME"       => "1478740432",
    "POSITION"   => 3180,
    "RUN"        => 1
};

is_deeply( $ev1, $ev1expect, "event 1" );

$t->MakeIndex();
my $tgot = clone($t);

my $texpect = bless(
    {
        "FSTAT" => [
            2054,
            14680079,
            33204,
            1,
            10001,
            10001,
            0,
            22740,
            1481032012,
            1480703558,
            1481031849,
            4096,
            48
        ],
        "FILEHEADER" => undef,
        "EVENT"      => 0,
        "RUNHEADER"  => undef,
        "RUN"        => 1,
        "FH"         => bless( \*Symbol::GEN0, 'IO::File' ),
        "FILE"       => "t/Analysis/TDS2024TraceData.dat",
        "INDEX"      => {
            "RUN" => {
                "1" => {
                    "STARTTIME" => "1478740429",
                    "STOPTIME"  => "1478740440",
                    "POSITION"  => 3059,
                    "EVENT"     => {
                        "2" => {
                            "TIME"     => "1478740437",
                            "POSITION" => 13631
                        },
                        "1" => {
                            "TIME"     => "1478740432",
                            "POSITION" => 3180
                        }
                    }
                }
            },
            "STREAM" => {
                "1" => {
                    "CONNECT" =>
                        "Lab::Instrument::TDS2024B::new ({\"debug\" => 1,\"timeout\" => 1,\"usb_serial\" => undef,\"usb_vendor\" => 1689,\"usb_product\" => 874,\"connection_type\" => \"USBtmc::Trace\"})",
                    "NUMBER"  => 1,
                    "COMMENT" => { "222" => "FORCED_TRIGGER delay=2" }
                },
                "0" => {
                    "COMMENT" => {},
                    "NUMBER"  => 0
                }
            }
        },
        "CONFIG" => {
            "use_ftell"     => 0,
            "clone_headers" => 1,
            "combined_gpib" => 0
        },
        "FILE_BEGIN" => 0
    },
    'Lab::Data::Analysis'
);

# delete the stuff that varies because of filesystem issues
foreach my $k (qw(FH FSTAT)) {
    delete( $texpect->{$k} );
    delete( $tgot->{$k} );
}

is_deeply( $tgot, $texpect, "File index" );

$ev1 = $t->ReadEvent( run => 1, event => 1 );

is_deeply( $ev1, $ev1expect, "Event, via index" );

$t->ConnectAnalyzer();

my $ev_an = $t->Analyze( event => $ev1 );

my (@yint);
my $f = $ev_an->{ANALYZE}->{1}->{TekTDS}->{CHAN}->{CH1}->{Yfunc};

for ( my $t = 1.64e-9; $t <= 1.684e-9; $t += 5e-12 ) {
    push( @yint, $f->($t) );
}

my (@yexp) = (
    0.00016, 0.00015, 0.00014, 0.00013, 0.00012, 0.00011, 0.0001, 9e-05,
    8e-05,
);

is_deeply( \@yint, \@yexp, "waveform interpolation" );

foreach my $k ( keys( %{$ev_an} ) ) {
    if ( $k ne 'ANALYZE' ) {
        delete( $ev_an->{$k} );
    }
}
delete( $ev_an->{ANALYZE}->{1}->{TekTDS}->{CHAN}->{CH1}->{Yfunc} );

my $ev_an_expect = {
    "ANALYZE" => {
        "1" => {
            "TekTDS" => {
                "EVENT" => 1,
                "RUN"   => 1,
                "CHAN"  => {
                    "CH1" => {
                        "DX"    => "4.0E-11",
                        "YMIN"  => "-0.00024",
                        "Xunit" => "s",
                        "XMAX"  => "4.996e-08",
                        "YMAX"  => "0.00056",

                        # Yfunc => CODE(),
                        "ID" =>
                            "Ch1, DC coupling, 2.0E-3 V/div, 1.0E-8 s/div, 2500 points, Sample mode",
                        "CHAN"  => "CH1",
                        "STOP"  => 2500,
                        "START" => 1,
                        "XMIN"  => "-5e-08",
                        "Yunit" => "Volts",
                        "Y"     => [
                            undef,   "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.0004",  "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004",  "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.0004",  "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00056",
                            "0.00056", "0.00056", "0.00056", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.0004",  "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004",  "0.0004",  "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.0004", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00056", "0.00056", "0.00056",
                            "0.00056", "0.00056",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.0004",  "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.0004",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.0004",  "0.0004",  "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00024", "0.00024",
                            "0.00024", "0.00016", "0.00016", "0.00016",
                            "0.00016", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "0.00016",
                            "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.0004",  "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.00032", "0.00032", "0.00032", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.0004",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.0004",  "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.0004",  "0.0004",  "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004",  "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.0004",  "0.0004",  "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004",  "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.0004",  "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.0004",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.0004",  "0.0004", "0.0004",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.0004", "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.0004",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.0004",  "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004",  "0.0004",  "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00016", "0.00016", "0.00016",
                            "0.00016", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", 0, 0, 0, 0,
                            0,       0,       0,       0,       0, 0,
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.0004",  "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "8e-05", "8e-05", "8e-05", "8e-05", 0, 0, 0, 0,
                            0,       0,       0,       0,       0, 0,
                            0, 0, 0, 0, 0, 0, "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05",   "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "8e-05",   "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05", 0,
                            0, 0, 0, "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05",   "8e-05",   "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05",   "8e-05",   "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05",   "8e-05",
                            "8e-05",   "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05", 0,
                            0, 0, 0, 0, 0, 0, 0, "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "0.00016", "0.00016",
                            "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.0004",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.0004",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "8e-05", "8e-05", "8e-05",
                            "8e-05",   "8e-05",
                            "8e-05",   "8e-05",   "8e-05", "8e-05", "0.00016",
                            "0.00016",
                            "0.00016", "0.00016", "0.00024", "0.00024",
                            "0.00024", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05",   "8e-05",
                            "8e-05",   "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05",   "8e-05",
                            "8e-05",   "8e-05", "8e-05", "8e-05", "0.00016",
                            "0.00016",
                            "0.00016", "0.00016", "0.00016", "8e-05",
                            "8e-05",   "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05", 0,
                            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                            0, 0, 0, 0, 0,
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05",   "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", 0,       0,       0,       0,
                            0, 0, "-8e-05", "-8e-05", "-8e-05", "-8e-05",
                            "-8e-05", "-8e-05",
                            "-8e-05", "-8e-05", "-8e-05", "-8e-05", 0, 0, 0,
                            0,        0,        "8e-05",
                            "8e-05", "8e-05", "8e-05", "0.00016", "0.00016",
                            "0.00016",
                            "0.00016", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "8e-05",   "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05",  0, 0, 0, 0, "-8e-05", "-8e-05",
                            "-8e-05", "-8e-05",
                            "-0.00016", "-0.00016", "-0.00016", "-0.00016",
                            "-0.00016",
                            "-0.00024", "-0.00024", "-0.00024", "-0.00024",
                            "-0.00024",
                            "-0.00024", "-0.00024", "-0.00024", "-0.00024",
                            "-0.00024",
                            "-0.00024", "-0.00024", "-0.00024", "-0.00024",
                            "-0.00024",
                            "-0.00024", "-0.00024", "-0.00016", "-0.00016",
                            "-0.00016",
                            "-0.00016", "-0.00016", "-0.00016", "-8e-05",
                            "-8e-05",   "-8e-05",
                            "-8e-05", "-8e-05", "-8e-05", "-8e-05", "-8e-05",
                            "-8e-05",
                            "-8e-05", "-8e-05", "-8e-05", "-8e-05", "-8e-05",
                            "-8e-05",
                            "-8e-05", "-8e-05", "-8e-05", "-8e-05", "-8e-05",
                            "-8e-05",
                            "-8e-05", "-8e-05", "-8e-05", 0, 0, 0, 0, 0, 0,
                            0,        0,        0,        0, 0, 0,
                            0, 0, 0, 0, 0, 0, 0, 0, 0, "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05",   "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "8e-05", "8e-05", "8e-05", "8e-05", 0, 0, 0, 0,
                            "-8e-05",
                            "-8e-05", "-8e-05", "-8e-05", "-8e-05", "-8e-05",
                            "-8e-05",
                            "-8e-05", "-8e-05", "-8e-05", "-8e-05", "-8e-05",
                            "-8e-05",
                            "-8e-05", 0, 0, 0, 0, 0, "8e-05", "8e-05",
                            "8e-05",  "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "0.00016",
                            "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.0004",  "0.0004",
                            "0.0004",  "0.0004",  "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05", "8e-05", "8e-05", "8e-05", "0.00016",
                            "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.0004",  "0.0004",  "0.0004",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.0004",  "0.0004",  "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004",  "0.0004",  "0.0004",  "0.0004",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.0004",  "0.0004",  "0.0004",
                            "0.0004",  "0.0004",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00056", "0.00056", "0.00056", "0.00056",
                            "0.00056", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.0004",  "0.0004",  "0.0004",
                            "0.0004",  "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00056", "0.00056", "0.00056",
                            "0.00056", "0.00056",
                            "0.00056", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.00048", "0.00048", "0.00048", "0.00048",
                            "0.00048", "0.00048",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.0004", "0.0004", "0.0004", "0.0004", "0.0004",
                            "0.0004",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.0004",  "0.0004",
                            "0.0004",  "0.0004",
                            "0.0004",  "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "8e-05", "8e-05", "8e-05", "8e-05",
                            "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "8e-05", "8e-05", "8e-05", "8e-05", "8e-05",
                            "8e-05", "8e-05",
                            "8e-05",   "0.00016", "0.00016", "0.00016",
                            "0.00016", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00024",
                            "0.00024", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00032",
                            "0.00032", "0.00032", "0.00032", "0.00032",
                            "0.00032", "0.00024",
                            "0.00024", "0.00024", "0.00024", "0.00024",
                            "0.00024", "0.00016",
                            "0.00016", "0.00016", "0.00016", "0.00016",
                            "0.00016", "8e-05",
                            "8e-05", "8e-05", "8e-05"
                        ],
                        "X" => [
                            undef,
                            "-5e-08",     "-4.996e-08", "-4.992e-08",
                            "-4.988e-08", "-4.984e-08",
                            "-4.98e-08",  "-4.976e-08", "-4.972e-08",
                            "-4.968e-08", "-4.964e-08",
                            "-4.96e-08",  "-4.956e-08", "-4.952e-08",
                            "-4.948e-08", "-4.944e-08",
                            "-4.94e-08",  "-4.936e-08", "-4.932e-08",
                            "-4.928e-08", "-4.924e-08",
                            "-4.92e-08",  "-4.916e-08", "-4.912e-08",
                            "-4.908e-08", "-4.904e-08",
                            "-4.9e-08",   "-4.896e-08", "-4.892e-08",
                            "-4.888e-08", "-4.884e-08",
                            "-4.88e-08",  "-4.876e-08", "-4.872e-08",
                            "-4.868e-08", "-4.864e-08",
                            "-4.86e-08",  "-4.856e-08", "-4.852e-08",
                            "-4.848e-08", "-4.844e-08",
                            "-4.84e-08",  "-4.836e-08", "-4.832e-08",
                            "-4.828e-08", "-4.824e-08",
                            "-4.82e-08",  "-4.816e-08", "-4.812e-08",
                            "-4.808e-08", "-4.804e-08",
                            "-4.8e-08",   "-4.796e-08", "-4.792e-08",
                            "-4.788e-08", "-4.784e-08",
                            "-4.78e-08",  "-4.776e-08", "-4.772e-08",
                            "-4.768e-08", "-4.764e-08",
                            "-4.76e-08",  "-4.756e-08", "-4.752e-08",
                            "-4.748e-08", "-4.744e-08",
                            "-4.74e-08",  "-4.736e-08", "-4.732e-08",
                            "-4.728e-08", "-4.724e-08",
                            "-4.72e-08",  "-4.716e-08", "-4.712e-08",
                            "-4.708e-08", "-4.704e-08",
                            "-4.7e-08",   "-4.696e-08", "-4.692e-08",
                            "-4.688e-08", "-4.684e-08",
                            "-4.68e-08",  "-4.676e-08", "-4.672e-08",
                            "-4.668e-08", "-4.664e-08",
                            "-4.66e-08",  "-4.656e-08", "-4.652e-08",
                            "-4.648e-08", "-4.644e-08",
                            "-4.64e-08",  "-4.636e-08", "-4.632e-08",
                            "-4.628e-08", "-4.624e-08",
                            "-4.62e-08",  "-4.616e-08", "-4.612e-08",
                            "-4.608e-08", "-4.604e-08",
                            "-4.6e-08",   "-4.596e-08", "-4.592e-08",
                            "-4.588e-08", "-4.584e-08",
                            "-4.58e-08",  "-4.576e-08", "-4.572e-08",
                            "-4.568e-08", "-4.564e-08",
                            "-4.56e-08",  "-4.556e-08", "-4.552e-08",
                            "-4.548e-08", "-4.544e-08",
                            "-4.54e-08",  "-4.536e-08", "-4.532e-08",
                            "-4.528e-08", "-4.524e-08",
                            "-4.52e-08",  "-4.516e-08", "-4.512e-08",
                            "-4.508e-08", "-4.504e-08",
                            "-4.5e-08",   "-4.496e-08", "-4.492e-08",
                            "-4.488e-08", "-4.484e-08",
                            "-4.48e-08",  "-4.476e-08", "-4.472e-08",
                            "-4.468e-08", "-4.464e-08",
                            "-4.46e-08",  "-4.456e-08", "-4.452e-08",
                            "-4.448e-08", "-4.444e-08",
                            "-4.44e-08",  "-4.436e-08", "-4.432e-08",
                            "-4.428e-08", "-4.424e-08",
                            "-4.42e-08",  "-4.416e-08", "-4.412e-08",
                            "-4.408e-08", "-4.404e-08",
                            "-4.4e-08",   "-4.396e-08", "-4.392e-08",
                            "-4.388e-08", "-4.384e-08",
                            "-4.38e-08",  "-4.376e-08", "-4.372e-08",
                            "-4.368e-08", "-4.364e-08",
                            "-4.36e-08",  "-4.356e-08", "-4.352e-08",
                            "-4.348e-08", "-4.344e-08",
                            "-4.34e-08",  "-4.336e-08", "-4.332e-08",
                            "-4.328e-08", "-4.324e-08",
                            "-4.32e-08",  "-4.316e-08", "-4.312e-08",
                            "-4.308e-08", "-4.304e-08",
                            "-4.3e-08",   "-4.296e-08", "-4.292e-08",
                            "-4.288e-08", "-4.284e-08",
                            "-4.28e-08",  "-4.276e-08", "-4.272e-08",
                            "-4.268e-08", "-4.264e-08",
                            "-4.26e-08",  "-4.256e-08", "-4.252e-08",
                            "-4.248e-08", "-4.244e-08",
                            "-4.24e-08",  "-4.236e-08", "-4.232e-08",
                            "-4.228e-08", "-4.224e-08",
                            "-4.22e-08",  "-4.216e-08", "-4.212e-08",
                            "-4.208e-08", "-4.204e-08",
                            "-4.2e-08",   "-4.196e-08", "-4.192e-08",
                            "-4.188e-08", "-4.184e-08",
                            "-4.18e-08",  "-4.176e-08", "-4.172e-08",
                            "-4.168e-08", "-4.164e-08",
                            "-4.16e-08",  "-4.156e-08", "-4.152e-08",
                            "-4.148e-08", "-4.144e-08",
                            "-4.14e-08",  "-4.136e-08", "-4.132e-08",
                            "-4.128e-08", "-4.124e-08",
                            "-4.12e-08",  "-4.116e-08", "-4.112e-08",
                            "-4.108e-08", "-4.104e-08",
                            "-4.1e-08",   "-4.096e-08", "-4.092e-08",
                            "-4.088e-08", "-4.084e-08",
                            "-4.08e-08",  "-4.076e-08", "-4.072e-08",
                            "-4.068e-08", "-4.064e-08",
                            "-4.06e-08",  "-4.056e-08", "-4.052e-08",
                            "-4.048e-08", "-4.044e-08",
                            "-4.04e-08",  "-4.036e-08", "-4.032e-08",
                            "-4.028e-08", "-4.024e-08",
                            "-4.02e-08",  "-4.016e-08", "-4.012e-08",
                            "-4.008e-08", "-4.004e-08",
                            "-4e-08",     "-3.996e-08", "-3.992e-08",
                            "-3.988e-08", "-3.984e-08",
                            "-3.98e-08",  "-3.976e-08", "-3.972e-08",
                            "-3.968e-08", "-3.964e-08",
                            "-3.96e-08",  "-3.956e-08", "-3.952e-08",
                            "-3.948e-08", "-3.944e-08",
                            "-3.94e-08",  "-3.936e-08", "-3.932e-08",
                            "-3.928e-08", "-3.924e-08",
                            "-3.92e-08",  "-3.916e-08", "-3.912e-08",
                            "-3.908e-08", "-3.904e-08",
                            "-3.9e-08",   "-3.896e-08", "-3.892e-08",
                            "-3.888e-08", "-3.884e-08",
                            "-3.88e-08",  "-3.876e-08", "-3.872e-08",
                            "-3.868e-08", "-3.864e-08",
                            "-3.86e-08",  "-3.856e-08", "-3.852e-08",
                            "-3.848e-08", "-3.844e-08",
                            "-3.84e-08",  "-3.836e-08", "-3.832e-08",
                            "-3.828e-08", "-3.824e-08",
                            "-3.82e-08",  "-3.816e-08", "-3.812e-08",
                            "-3.808e-08", "-3.804e-08",
                            "-3.8e-08",   "-3.796e-08", "-3.792e-08",
                            "-3.788e-08", "-3.784e-08",
                            "-3.78e-08",  "-3.776e-08", "-3.772e-08",
                            "-3.768e-08", "-3.764e-08",
                            "-3.76e-08",  "-3.756e-08", "-3.752e-08",
                            "-3.748e-08", "-3.744e-08",
                            "-3.74e-08",  "-3.736e-08", "-3.732e-08",
                            "-3.728e-08", "-3.724e-08",
                            "-3.72e-08",  "-3.716e-08", "-3.712e-08",
                            "-3.708e-08", "-3.704e-08",
                            "-3.7e-08",   "-3.696e-08", "-3.692e-08",
                            "-3.688e-08", "-3.684e-08",
                            "-3.68e-08",  "-3.676e-08", "-3.672e-08",
                            "-3.668e-08", "-3.664e-08",
                            "-3.66e-08",  "-3.656e-08", "-3.652e-08",
                            "-3.648e-08", "-3.644e-08",
                            "-3.64e-08",  "-3.636e-08", "-3.632e-08",
                            "-3.628e-08", "-3.624e-08",
                            "-3.62e-08",  "-3.616e-08", "-3.612e-08",
                            "-3.608e-08", "-3.604e-08",
                            "-3.6e-08",   "-3.596e-08", "-3.592e-08",
                            "-3.588e-08", "-3.584e-08",
                            "-3.58e-08",  "-3.576e-08", "-3.572e-08",
                            "-3.568e-08", "-3.564e-08",
                            "-3.56e-08",  "-3.556e-08", "-3.552e-08",
                            "-3.548e-08", "-3.544e-08",
                            "-3.54e-08",  "-3.536e-08", "-3.532e-08",
                            "-3.528e-08", "-3.524e-08",
                            "-3.52e-08",  "-3.516e-08", "-3.512e-08",
                            "-3.508e-08", "-3.504e-08",
                            "-3.5e-08",   "-3.496e-08", "-3.492e-08",
                            "-3.488e-08", "-3.484e-08",
                            "-3.48e-08",  "-3.476e-08", "-3.472e-08",
                            "-3.468e-08", "-3.464e-08",
                            "-3.46e-08",  "-3.456e-08", "-3.452e-08",
                            "-3.448e-08", "-3.444e-08",
                            "-3.44e-08",  "-3.436e-08", "-3.432e-08",
                            "-3.428e-08", "-3.424e-08",
                            "-3.42e-08",  "-3.416e-08", "-3.412e-08",
                            "-3.408e-08", "-3.404e-08",
                            "-3.4e-08",   "-3.396e-08", "-3.392e-08",
                            "-3.388e-08", "-3.384e-08",
                            "-3.38e-08",  "-3.376e-08", "-3.372e-08",
                            "-3.368e-08", "-3.364e-08",
                            "-3.36e-08",  "-3.356e-08", "-3.352e-08",
                            "-3.348e-08", "-3.344e-08",
                            "-3.34e-08",  "-3.336e-08", "-3.332e-08",
                            "-3.328e-08", "-3.324e-08",
                            "-3.32e-08",  "-3.316e-08", "-3.312e-08",
                            "-3.308e-08", "-3.304e-08",
                            "-3.3e-08",   "-3.296e-08", "-3.292e-08",
                            "-3.288e-08", "-3.284e-08",
                            "-3.28e-08",  "-3.276e-08", "-3.272e-08",
                            "-3.268e-08", "-3.264e-08",
                            "-3.26e-08",  "-3.256e-08", "-3.252e-08",
                            "-3.248e-08", "-3.244e-08",
                            "-3.24e-08",  "-3.236e-08", "-3.232e-08",
                            "-3.228e-08", "-3.224e-08",
                            "-3.22e-08",  "-3.216e-08", "-3.212e-08",
                            "-3.208e-08", "-3.204e-08",
                            "-3.2e-08",   "-3.196e-08", "-3.192e-08",
                            "-3.188e-08", "-3.184e-08",
                            "-3.18e-08",  "-3.176e-08", "-3.172e-08",
                            "-3.168e-08", "-3.164e-08",
                            "-3.16e-08",  "-3.156e-08", "-3.152e-08",
                            "-3.148e-08", "-3.144e-08",
                            "-3.14e-08",  "-3.136e-08", "-3.132e-08",
                            "-3.128e-08", "-3.124e-08",
                            "-3.12e-08",  "-3.116e-08", "-3.112e-08",
                            "-3.108e-08", "-3.104e-08",
                            "-3.1e-08",   "-3.096e-08", "-3.092e-08",
                            "-3.088e-08", "-3.084e-08",
                            "-3.08e-08",  "-3.076e-08", "-3.072e-08",
                            "-3.068e-08", "-3.064e-08",
                            "-3.06e-08",  "-3.056e-08", "-3.052e-08",
                            "-3.048e-08", "-3.044e-08",
                            "-3.04e-08",  "-3.036e-08", "-3.032e-08",
                            "-3.028e-08", "-3.024e-08",
                            "-3.02e-08",  "-3.016e-08", "-3.012e-08",
                            "-3.008e-08", "-3.004e-08",
                            "-3e-08",     "-2.996e-08", "-2.992e-08",
                            "-2.988e-08", "-2.984e-08",
                            "-2.98e-08",  "-2.976e-08", "-2.972e-08",
                            "-2.968e-08", "-2.964e-08",
                            "-2.96e-08",  "-2.956e-08", "-2.952e-08",
                            "-2.948e-08", "-2.944e-08",
                            "-2.94e-08",  "-2.936e-08", "-2.932e-08",
                            "-2.928e-08", "-2.924e-08",
                            "-2.92e-08",  "-2.916e-08", "-2.912e-08",
                            "-2.908e-08", "-2.904e-08",
                            "-2.9e-08",   "-2.896e-08", "-2.892e-08",
                            "-2.888e-08", "-2.884e-08",
                            "-2.88e-08",  "-2.876e-08", "-2.872e-08",
                            "-2.868e-08", "-2.864e-08",
                            "-2.86e-08",  "-2.856e-08", "-2.852e-08",
                            "-2.848e-08", "-2.844e-08",
                            "-2.84e-08",  "-2.836e-08", "-2.832e-08",
                            "-2.828e-08", "-2.824e-08",
                            "-2.82e-08",  "-2.816e-08", "-2.812e-08",
                            "-2.808e-08", "-2.804e-08",
                            "-2.8e-08",   "-2.796e-08", "-2.792e-08",
                            "-2.788e-08", "-2.784e-08",
                            "-2.78e-08",  "-2.776e-08", "-2.772e-08",
                            "-2.768e-08", "-2.764e-08",
                            "-2.76e-08",  "-2.756e-08", "-2.752e-08",
                            "-2.748e-08", "-2.744e-08",
                            "-2.74e-08",  "-2.736e-08", "-2.732e-08",
                            "-2.728e-08", "-2.724e-08",
                            "-2.72e-08",  "-2.716e-08", "-2.712e-08",
                            "-2.708e-08", "-2.704e-08",
                            "-2.7e-08",   "-2.696e-08", "-2.692e-08",
                            "-2.688e-08", "-2.684e-08",
                            "-2.68e-08",  "-2.676e-08", "-2.672e-08",
                            "-2.668e-08", "-2.664e-08",
                            "-2.66e-08",  "-2.656e-08", "-2.652e-08",
                            "-2.648e-08", "-2.644e-08",
                            "-2.64e-08",  "-2.636e-08", "-2.632e-08",
                            "-2.628e-08", "-2.624e-08",
                            "-2.62e-08",  "-2.616e-08", "-2.612e-08",
                            "-2.608e-08", "-2.604e-08",
                            "-2.6e-08",   "-2.596e-08", "-2.592e-08",
                            "-2.588e-08", "-2.584e-08",
                            "-2.58e-08",  "-2.576e-08", "-2.572e-08",
                            "-2.568e-08", "-2.564e-08",
                            "-2.56e-08",  "-2.556e-08", "-2.552e-08",
                            "-2.548e-08", "-2.544e-08",
                            "-2.54e-08",  "-2.536e-08", "-2.532e-08",
                            "-2.528e-08", "-2.524e-08",
                            "-2.52e-08",  "-2.516e-08", "-2.512e-08",
                            "-2.508e-08", "-2.504e-08",
                            "-2.5e-08",   "-2.496e-08", "-2.492e-08",
                            "-2.488e-08", "-2.484e-08",
                            "-2.48e-08",  "-2.476e-08", "-2.472e-08",
                            "-2.468e-08", "-2.464e-08",
                            "-2.46e-08",  "-2.456e-08", "-2.452e-08",
                            "-2.448e-08", "-2.444e-08",
                            "-2.44e-08",  "-2.436e-08", "-2.432e-08",
                            "-2.428e-08", "-2.424e-08",
                            "-2.42e-08",  "-2.416e-08", "-2.412e-08",
                            "-2.408e-08", "-2.404e-08",
                            "-2.4e-08",   "-2.396e-08", "-2.392e-08",
                            "-2.388e-08", "-2.384e-08",
                            "-2.38e-08",  "-2.376e-08", "-2.372e-08",
                            "-2.368e-08", "-2.364e-08",
                            "-2.36e-08",  "-2.356e-08", "-2.352e-08",
                            "-2.348e-08", "-2.344e-08",
                            "-2.34e-08",  "-2.336e-08", "-2.332e-08",
                            "-2.328e-08", "-2.324e-08",
                            "-2.32e-08",  "-2.316e-08", "-2.312e-08",
                            "-2.308e-08", "-2.304e-08",
                            "-2.3e-08",   "-2.296e-08", "-2.292e-08",
                            "-2.288e-08", "-2.284e-08",
                            "-2.28e-08",  "-2.276e-08", "-2.272e-08",
                            "-2.268e-08", "-2.264e-08",
                            "-2.26e-08",  "-2.256e-08", "-2.252e-08",
                            "-2.248e-08", "-2.244e-08",
                            "-2.24e-08",  "-2.236e-08", "-2.232e-08",
                            "-2.228e-08", "-2.224e-08",
                            "-2.22e-08",  "-2.216e-08", "-2.212e-08",
                            "-2.208e-08", "-2.204e-08",
                            "-2.2e-08",   "-2.196e-08", "-2.192e-08",
                            "-2.188e-08", "-2.184e-08",
                            "-2.18e-08",  "-2.176e-08", "-2.172e-08",
                            "-2.168e-08", "-2.164e-08",
                            "-2.16e-08",  "-2.156e-08", "-2.152e-08",
                            "-2.148e-08", "-2.144e-08",
                            "-2.14e-08",  "-2.136e-08", "-2.132e-08",
                            "-2.128e-08", "-2.124e-08",
                            "-2.12e-08",  "-2.116e-08", "-2.112e-08",
                            "-2.108e-08", "-2.104e-08",
                            "-2.1e-08",   "-2.096e-08", "-2.092e-08",
                            "-2.088e-08", "-2.084e-08",
                            "-2.08e-08",  "-2.076e-08", "-2.072e-08",
                            "-2.068e-08", "-2.064e-08",
                            "-2.06e-08",  "-2.056e-08", "-2.052e-08",
                            "-2.048e-08", "-2.044e-08",
                            "-2.04e-08",  "-2.036e-08", "-2.032e-08",
                            "-2.028e-08", "-2.024e-08",
                            "-2.02e-08",  "-2.016e-08", "-2.012e-08",
                            "-2.008e-08", "-2.004e-08",
                            "-2e-08",     "-1.996e-08", "-1.992e-08",
                            "-1.988e-08", "-1.984e-08",
                            "-1.98e-08",  "-1.976e-08", "-1.972e-08",
                            "-1.968e-08", "-1.964e-08",
                            "-1.96e-08",  "-1.956e-08", "-1.952e-08",
                            "-1.948e-08", "-1.944e-08",
                            "-1.94e-08",  "-1.936e-08", "-1.932e-08",
                            "-1.928e-08", "-1.924e-08",
                            "-1.92e-08",  "-1.916e-08", "-1.912e-08",
                            "-1.908e-08", "-1.904e-08",
                            "-1.9e-08",   "-1.896e-08", "-1.892e-08",
                            "-1.888e-08", "-1.884e-08",
                            "-1.88e-08",  "-1.876e-08", "-1.872e-08",
                            "-1.868e-08", "-1.864e-08",
                            "-1.86e-08",  "-1.856e-08", "-1.852e-08",
                            "-1.848e-08", "-1.844e-08",
                            "-1.84e-08",  "-1.836e-08", "-1.832e-08",
                            "-1.828e-08", "-1.824e-08",
                            "-1.82e-08",  "-1.816e-08", "-1.812e-08",
                            "-1.808e-08", "-1.804e-08",
                            "-1.8e-08",   "-1.796e-08", "-1.792e-08",
                            "-1.788e-08", "-1.784e-08",
                            "-1.78e-08",  "-1.776e-08", "-1.772e-08",
                            "-1.768e-08", "-1.764e-08",
                            "-1.76e-08",  "-1.756e-08", "-1.752e-08",
                            "-1.748e-08", "-1.744e-08",
                            "-1.74e-08",  "-1.736e-08", "-1.732e-08",
                            "-1.728e-08", "-1.724e-08",
                            "-1.72e-08",  "-1.716e-08", "-1.712e-08",
                            "-1.708e-08", "-1.704e-08",
                            "-1.7e-08",   "-1.696e-08", "-1.692e-08",
                            "-1.688e-08", "-1.684e-08",
                            "-1.68e-08",  "-1.676e-08", "-1.672e-08",
                            "-1.668e-08", "-1.664e-08",
                            "-1.66e-08",  "-1.656e-08", "-1.652e-08",
                            "-1.648e-08", "-1.644e-08",
                            "-1.64e-08",  "-1.636e-08", "-1.632e-08",
                            "-1.628e-08", "-1.624e-08",
                            "-1.62e-08",  "-1.616e-08", "-1.612e-08",
                            "-1.608e-08", "-1.604e-08",
                            "-1.6e-08",   "-1.596e-08", "-1.592e-08",
                            "-1.588e-08", "-1.584e-08",
                            "-1.58e-08",  "-1.576e-08", "-1.572e-08",
                            "-1.568e-08", "-1.564e-08",
                            "-1.56e-08",  "-1.556e-08", "-1.552e-08",
                            "-1.548e-08", "-1.544e-08",
                            "-1.54e-08",  "-1.536e-08", "-1.532e-08",
                            "-1.528e-08", "-1.524e-08",
                            "-1.52e-08",  "-1.516e-08", "-1.512e-08",
                            "-1.508e-08", "-1.504e-08",
                            "-1.5e-08",   "-1.496e-08", "-1.492e-08",
                            "-1.488e-08", "-1.484e-08",
                            "-1.48e-08",  "-1.476e-08", "-1.472e-08",
                            "-1.468e-08", "-1.464e-08",
                            "-1.46e-08",  "-1.456e-08", "-1.452e-08",
                            "-1.448e-08", "-1.444e-08",
                            "-1.44e-08",  "-1.436e-08", "-1.432e-08",
                            "-1.428e-08", "-1.424e-08",
                            "-1.42e-08",  "-1.416e-08", "-1.412e-08",
                            "-1.408e-08", "-1.404e-08",
                            "-1.4e-08",   "-1.396e-08", "-1.392e-08",
                            "-1.388e-08", "-1.384e-08",
                            "-1.38e-08",  "-1.376e-08", "-1.372e-08",
                            "-1.368e-08", "-1.364e-08",
                            "-1.36e-08",  "-1.356e-08", "-1.352e-08",
                            "-1.348e-08", "-1.344e-08",
                            "-1.34e-08",  "-1.336e-08", "-1.332e-08",
                            "-1.328e-08", "-1.324e-08",
                            "-1.32e-08",  "-1.316e-08", "-1.312e-08",
                            "-1.308e-08", "-1.304e-08",
                            "-1.3e-08",   "-1.296e-08", "-1.292e-08",
                            "-1.288e-08", "-1.284e-08",
                            "-1.28e-08",  "-1.276e-08", "-1.272e-08",
                            "-1.268e-08", "-1.264e-08",
                            "-1.26e-08",  "-1.256e-08", "-1.252e-08",
                            "-1.248e-08", "-1.244e-08",
                            "-1.24e-08",  "-1.236e-08", "-1.232e-08",
                            "-1.228e-08", "-1.224e-08",
                            "-1.22e-08",  "-1.216e-08", "-1.212e-08",
                            "-1.208e-08", "-1.204e-08",
                            "-1.2e-08",   "-1.196e-08", "-1.192e-08",
                            "-1.188e-08", "-1.184e-08",
                            "-1.18e-08",  "-1.176e-08", "-1.172e-08",
                            "-1.168e-08", "-1.164e-08",
                            "-1.16e-08",  "-1.156e-08", "-1.152e-08",
                            "-1.148e-08", "-1.144e-08",
                            "-1.14e-08",  "-1.136e-08", "-1.132e-08",
                            "-1.128e-08", "-1.124e-08",
                            "-1.12e-08",  "-1.116e-08", "-1.112e-08",
                            "-1.108e-08", "-1.104e-08",
                            "-1.1e-08",   "-1.096e-08", "-1.092e-08",
                            "-1.088e-08", "-1.084e-08",
                            "-1.08e-08",  "-1.076e-08", "-1.072e-08",
                            "-1.068e-08", "-1.064e-08",
                            "-1.06e-08",  "-1.056e-08", "-1.052e-08",
                            "-1.048e-08", "-1.044e-08",
                            "-1.04e-08",  "-1.036e-08", "-1.032e-08",
                            "-1.028e-08", "-1.024e-08",
                            "-1.02e-08",  "-1.016e-08", "-1.012e-08",
                            "-1.008e-08", "-1.004e-08",
                            "-1e-08", "-9.96e-09", "-9.92e-09", "-9.88e-09",
                            "-9.84e-09",
                            "-9.8e-09",  "-9.76e-09", "-9.72e-09",
                            "-9.68e-09", "-9.64e-09",
                            "-9.6e-09",  "-9.56e-09", "-9.52e-09",
                            "-9.48e-09", "-9.44e-09",
                            "-9.4e-09",  "-9.36e-09", "-9.32e-09",
                            "-9.28e-09", "-9.24e-09",
                            "-9.2e-09",  "-9.16e-09", "-9.12e-09",
                            "-9.08e-09", "-9.04e-09",
                            "-9e-09", "-8.96e-09", "-8.92e-09", "-8.88e-09",
                            "-8.84e-09",
                            "-8.8e-09",  "-8.76e-09", "-8.72e-09",
                            "-8.68e-09", "-8.64e-09",
                            "-8.6e-09",  "-8.56e-09", "-8.52e-09",
                            "-8.48e-09", "-8.44e-09",
                            "-8.4e-09",  "-8.36e-09", "-8.32e-09",
                            "-8.28e-09", "-8.24e-09",
                            "-8.2e-09",  "-8.16e-09", "-8.12e-09",
                            "-8.08e-09", "-8.04e-09",
                            "-8e-09", "-7.96e-09", "-7.92e-09", "-7.88e-09",
                            "-7.84e-09",
                            "-7.8e-09",  "-7.76e-09", "-7.72e-09",
                            "-7.68e-09", "-7.64e-09",
                            "-7.6e-09",  "-7.56e-09", "-7.52e-09",
                            "-7.48e-09", "-7.44e-09",
                            "-7.4e-09",  "-7.36e-09", "-7.32e-09",
                            "-7.28e-09", "-7.24e-09",
                            "-7.2e-09",  "-7.16e-09", "-7.12e-09",
                            "-7.08e-09", "-7.04e-09",
                            "-7e-09", "-6.96e-09", "-6.92e-09", "-6.88e-09",
                            "-6.84e-09",
                            "-6.8e-09",  "-6.76e-09", "-6.72e-09",
                            "-6.68e-09", "-6.64e-09",
                            "-6.6e-09",  "-6.56e-09", "-6.52e-09",
                            "-6.48e-09", "-6.44e-09",
                            "-6.4e-09",  "-6.36e-09", "-6.32e-09",
                            "-6.28e-09", "-6.24e-09",
                            "-6.2e-09",  "-6.16e-09", "-6.12e-09",
                            "-6.08e-09", "-6.04e-09",
                            "-6e-09", "-5.96e-09", "-5.92e-09", "-5.88e-09",
                            "-5.84e-09",
                            "-5.8e-09",  "-5.76e-09", "-5.72e-09",
                            "-5.68e-09", "-5.64e-09",
                            "-5.6e-09",  "-5.56e-09", "-5.52e-09",
                            "-5.48e-09", "-5.44e-09",
                            "-5.4e-09",  "-5.36e-09", "-5.32e-09",
                            "-5.28e-09", "-5.24e-09",
                            "-5.2e-09",  "-5.16e-09", "-5.12e-09",
                            "-5.08e-09", "-5.04e-09",
                            "-5e-09", "-4.96e-09", "-4.92e-09", "-4.88e-09",
                            "-4.84e-09",
                            "-4.8e-09",  "-4.76e-09", "-4.72e-09",
                            "-4.68e-09", "-4.64e-09",
                            "-4.6e-09",  "-4.56e-09", "-4.52e-09",
                            "-4.48e-09", "-4.44e-09",
                            "-4.4e-09",  "-4.36e-09", "-4.32e-09",
                            "-4.28e-09", "-4.24e-09",
                            "-4.2e-09",  "-4.16e-09", "-4.12e-09",
                            "-4.08e-09", "-4.04e-09",
                            "-4e-09", "-3.96e-09", "-3.92e-09", "-3.88e-09",
                            "-3.84e-09",
                            "-3.8e-09",  "-3.76e-09", "-3.72e-09",
                            "-3.68e-09", "-3.64e-09",
                            "-3.6e-09",  "-3.56e-09", "-3.52e-09",
                            "-3.48e-09", "-3.44e-09",
                            "-3.4e-09",  "-3.36e-09", "-3.32e-09",
                            "-3.28e-09", "-3.24e-09",
                            "-3.2e-09",  "-3.16e-09", "-3.12e-09",
                            "-3.08e-09", "-3.04e-09",
                            "-3e-09", "-2.96e-09", "-2.92e-09", "-2.88e-09",
                            "-2.84e-09",
                            "-2.8e-09",  "-2.76e-09", "-2.72e-09",
                            "-2.68e-09", "-2.64e-09",
                            "-2.6e-09",  "-2.56e-09", "-2.52e-09",
                            "-2.48e-09", "-2.44e-09",
                            "-2.4e-09",  "-2.36e-09", "-2.32e-09",
                            "-2.28e-09", "-2.24e-09",
                            "-2.2e-09",  "-2.16e-09", "-2.12e-09",
                            "-2.08e-09", "-2.04e-09",
                            "-2e-09", "-1.96e-09", "-1.92e-09", "-1.88e-09",
                            "-1.84e-09",
                            "-1.8e-09",  "-1.76e-09", "-1.72e-09",
                            "-1.68e-09", "-1.64e-09",
                            "-1.6e-09",  "-1.56e-09", "-1.52e-09",
                            "-1.48e-09", "-1.44e-09",
                            "-1.4e-09",  "-1.36e-09", "-1.32e-09",
                            "-1.28e-09", "-1.24e-09",
                            "-1.2e-09",  "-1.16e-09", "-1.12e-09",
                            "-1.08e-09", "-1.04e-09",
                            "-1e-09",    "-9.60000000000003e-10",
                            "-9.19999999999998e-10",
                            "-8.79999999999999e-10", "-8.4e-10",
                            "-8.00000000000001e-10",
                            "-7.60000000000002e-10", "-7.20000000000002e-10",
                            "-6.80000000000003e-10", "-6.39999999999998e-10",
                            "-5.99999999999999e-10", "-5.6e-10",
                            "-5.20000000000001e-10",
                            "-4.80000000000002e-10", "-4.40000000000003e-10",
                            "-4.00000000000004e-10", "-3.59999999999998e-10",
                            "-3.19999999999999e-10", "-2.8e-10",
                            "-2.40000000000001e-10",
                            "-2.00000000000002e-10", "-1.60000000000003e-10",
                            "-1.20000000000004e-10", "-7.99999999999981e-11",
                            "-3.9999999999999e-11", 0, "3.9999999999999e-11",
                            "7.99999999999981e-11", "1.19999999999997e-10",
                            "1.59999999999996e-10", "2.00000000000002e-10",
                            "2.40000000000001e-10", "2.8e-10",
                            "3.19999999999999e-10",
                            "3.59999999999998e-10", "3.99999999999997e-10",
                            "4.39999999999996e-10", "4.80000000000002e-10",
                            "5.20000000000001e-10", "5.6e-10",
                            "5.99999999999999e-10",
                            "6.39999999999998e-10", "6.79999999999997e-10",
                            "7.20000000000002e-10", "7.60000000000002e-10",
                            "8.00000000000001e-10", "8.4e-10",
                            "8.79999999999999e-10",
                            "9.19999999999998e-10", "9.59999999999997e-10",
                            "1e-09",                "1.04e-09",
                            "1.08e-09", "1.12e-09", "1.16e-09", "1.2e-09",
                            "1.24e-09", "1.28e-09",
                            "1.32e-09", "1.36e-09", "1.4e-09",  "1.44e-09",
                            "1.48e-09", "1.52e-09",
                            "1.56e-09", "1.6e-09",  "1.64e-09", "1.68e-09",
                            "1.72e-09", "1.76e-09",
                            "1.8e-09",  "1.84e-09", "1.88e-09", "1.92e-09",
                            "1.96e-09", "2e-09",
                            "2.04e-09", "2.08e-09", "2.12e-09", "2.16e-09",
                            "2.2e-09",  "2.24e-09",
                            "2.28e-09", "2.32e-09", "2.36e-09", "2.4e-09",
                            "2.44e-09", "2.48e-09",
                            "2.52e-09", "2.56e-09", "2.6e-09",  "2.64e-09",
                            "2.68e-09", "2.72e-09",
                            "2.76e-09", "2.8e-09",  "2.84e-09", "2.88e-09",
                            "2.92e-09", "2.96e-09",
                            "3e-09",    "3.04e-09", "3.08e-09", "3.12e-09",
                            "3.16e-09", "3.2e-09",
                            "3.24e-09", "3.28e-09", "3.32e-09", "3.36e-09",
                            "3.4e-09",  "3.44e-09",
                            "3.48e-09", "3.52e-09", "3.56e-09", "3.6e-09",
                            "3.64e-09", "3.68e-09",
                            "3.72e-09", "3.76e-09", "3.8e-09",  "3.84e-09",
                            "3.88e-09", "3.92e-09",
                            "3.96e-09", "4e-09",    "4.04e-09", "4.08e-09",
                            "4.12e-09", "4.16e-09",
                            "4.2e-09",  "4.24e-09", "4.28e-09", "4.32e-09",
                            "4.36e-09", "4.4e-09",
                            "4.44e-09", "4.48e-09", "4.52e-09", "4.56e-09",
                            "4.6e-09",  "4.64e-09",
                            "4.68e-09", "4.72e-09", "4.76e-09", "4.8e-09",
                            "4.84e-09", "4.88e-09",
                            "4.92e-09", "4.96e-09", "5e-09",    "5.04e-09",
                            "5.08e-09", "5.12e-09",
                            "5.16e-09", "5.2e-09",  "5.24e-09", "5.28e-09",
                            "5.32e-09", "5.36e-09",
                            "5.4e-09",  "5.44e-09", "5.48e-09", "5.52e-09",
                            "5.56e-09", "5.6e-09",
                            "5.64e-09", "5.68e-09", "5.72e-09", "5.76e-09",
                            "5.8e-09",  "5.84e-09",
                            "5.88e-09", "5.92e-09", "5.96e-09", "6e-09",
                            "6.04e-09", "6.08e-09",
                            "6.12e-09", "6.16e-09", "6.2e-09",  "6.24e-09",
                            "6.28e-09", "6.32e-09",
                            "6.36e-09", "6.4e-09",  "6.44e-09", "6.48e-09",
                            "6.52e-09", "6.56e-09",
                            "6.6e-09",  "6.64e-09", "6.68e-09", "6.72e-09",
                            "6.76e-09", "6.8e-09",
                            "6.84e-09", "6.88e-09", "6.92e-09", "6.96e-09",
                            "7e-09",    "7.04e-09",
                            "7.08e-09", "7.12e-09", "7.16e-09", "7.2e-09",
                            "7.24e-09", "7.28e-09",
                            "7.32e-09", "7.36e-09", "7.4e-09",  "7.44e-09",
                            "7.48e-09", "7.52e-09",
                            "7.56e-09", "7.6e-09",  "7.64e-09", "7.68e-09",
                            "7.72e-09", "7.76e-09",
                            "7.8e-09",  "7.84e-09", "7.88e-09", "7.92e-09",
                            "7.96e-09", "8e-09",
                            "8.04e-09", "8.08e-09", "8.12e-09", "8.16e-09",
                            "8.2e-09",  "8.24e-09",
                            "8.28e-09", "8.32e-09", "8.36e-09", "8.4e-09",
                            "8.44e-09", "8.48e-09",
                            "8.52e-09", "8.56e-09", "8.6e-09",  "8.64e-09",
                            "8.68e-09", "8.72e-09",
                            "8.76e-09", "8.8e-09",  "8.84e-09", "8.88e-09",
                            "8.92e-09", "8.96e-09",
                            "9e-09",    "9.04e-09", "9.08e-09", "9.12e-09",
                            "9.16e-09", "9.2e-09",
                            "9.24e-09", "9.28e-09", "9.32e-09", "9.36e-09",
                            "9.4e-09",  "9.44e-09",
                            "9.48e-09", "9.52e-09", "9.56e-09", "9.6e-09",
                            "9.64e-09", "9.68e-09",
                            "9.72e-09", "9.76e-09", "9.8e-09",  "9.84e-09",
                            "9.87999999999999e-09",
                            "9.92e-09", "9.96e-09", "1e-08", "1.004e-08",
                            "1.008e-08",
                            "1.012e-08", "1.016e-08", "1.02e-08",
                            "1.024e-08", "1.028e-08",
                            "1.032e-08", "1.036e-08", "1.04e-08",
                            "1.044e-08", "1.048e-08",
                            "1.052e-08", "1.056e-08", "1.06e-08",
                            "1.064e-08", "1.068e-08",
                            "1.072e-08", "1.076e-08", "1.08e-08",
                            "1.084e-08", "1.088e-08",
                            "1.092e-08", "1.096e-08", "1.1e-08", "1.104e-08",
                            "1.108e-08",
                            "1.112e-08", "1.116e-08", "1.12e-08",
                            "1.124e-08", "1.128e-08",
                            "1.132e-08", "1.136e-08", "1.14e-08",
                            "1.144e-08", "1.148e-08",
                            "1.152e-08", "1.156e-08", "1.16e-08",
                            "1.164e-08", "1.168e-08",
                            "1.172e-08", "1.176e-08", "1.18e-08",
                            "1.184e-08", "1.188e-08",
                            "1.192e-08", "1.196e-08", "1.2e-08", "1.204e-08",
                            "1.208e-08",
                            "1.212e-08", "1.216e-08", "1.22e-08",
                            "1.224e-08", "1.228e-08",
                            "1.232e-08", "1.236e-08", "1.24e-08",
                            "1.244e-08", "1.248e-08",
                            "1.252e-08", "1.256e-08", "1.26e-08",
                            "1.264e-08", "1.268e-08",
                            "1.272e-08", "1.276e-08", "1.28e-08",
                            "1.284e-08", "1.288e-08",
                            "1.292e-08", "1.296e-08", "1.3e-08", "1.304e-08",
                            "1.308e-08",
                            "1.312e-08", "1.316e-08", "1.32e-08",
                            "1.324e-08", "1.328e-08",
                            "1.332e-08", "1.336e-08", "1.34e-08",
                            "1.344e-08", "1.348e-08",
                            "1.352e-08", "1.356e-08", "1.36e-08",
                            "1.364e-08", "1.368e-08",
                            "1.372e-08", "1.376e-08", "1.38e-08",
                            "1.384e-08", "1.388e-08",
                            "1.392e-08", "1.396e-08", "1.4e-08", "1.404e-08",
                            "1.408e-08",
                            "1.412e-08", "1.416e-08", "1.42e-08",
                            "1.424e-08", "1.428e-08",
                            "1.432e-08", "1.436e-08", "1.44e-08",
                            "1.444e-08", "1.448e-08",
                            "1.452e-08", "1.456e-08", "1.46e-08",
                            "1.464e-08", "1.468e-08",
                            "1.472e-08", "1.476e-08", "1.48e-08",
                            "1.484e-08", "1.488e-08",
                            "1.492e-08", "1.496e-08", "1.5e-08", "1.504e-08",
                            "1.508e-08",
                            "1.512e-08", "1.516e-08", "1.52e-08",
                            "1.524e-08", "1.528e-08",
                            "1.532e-08", "1.536e-08", "1.54e-08",
                            "1.544e-08", "1.548e-08",
                            "1.552e-08", "1.556e-08", "1.56e-08",
                            "1.564e-08", "1.568e-08",
                            "1.572e-08", "1.576e-08", "1.58e-08",
                            "1.584e-08", "1.588e-08",
                            "1.592e-08", "1.596e-08", "1.6e-08", "1.604e-08",
                            "1.608e-08",
                            "1.612e-08", "1.616e-08", "1.62e-08",
                            "1.624e-08", "1.628e-08",
                            "1.632e-08", "1.636e-08", "1.64e-08",
                            "1.644e-08", "1.648e-08",
                            "1.652e-08", "1.656e-08", "1.66e-08",
                            "1.664e-08", "1.668e-08",
                            "1.672e-08", "1.676e-08", "1.68e-08",
                            "1.684e-08", "1.688e-08",
                            "1.692e-08", "1.696e-08", "1.7e-08", "1.704e-08",
                            "1.708e-08",
                            "1.712e-08", "1.716e-08", "1.72e-08",
                            "1.724e-08", "1.728e-08",
                            "1.732e-08", "1.736e-08", "1.74e-08",
                            "1.744e-08", "1.748e-08",
                            "1.752e-08", "1.756e-08", "1.76e-08",
                            "1.764e-08", "1.768e-08",
                            "1.772e-08", "1.776e-08", "1.78e-08",
                            "1.784e-08", "1.788e-08",
                            "1.792e-08", "1.796e-08", "1.8e-08", "1.804e-08",
                            "1.808e-08",
                            "1.812e-08", "1.816e-08", "1.82e-08",
                            "1.824e-08", "1.828e-08",
                            "1.832e-08", "1.836e-08", "1.84e-08",
                            "1.844e-08", "1.848e-08",
                            "1.852e-08", "1.856e-08", "1.86e-08",
                            "1.864e-08", "1.868e-08",
                            "1.872e-08", "1.876e-08", "1.88e-08",
                            "1.884e-08", "1.888e-08",
                            "1.892e-08", "1.896e-08", "1.9e-08", "1.904e-08",
                            "1.908e-08",
                            "1.912e-08", "1.916e-08", "1.92e-08",
                            "1.924e-08", "1.928e-08",
                            "1.932e-08", "1.936e-08", "1.94e-08",
                            "1.944e-08", "1.948e-08",
                            "1.952e-08", "1.956e-08", "1.96e-08",
                            "1.964e-08", "1.968e-08",
                            "1.972e-08", "1.976e-08", "1.98e-08",
                            "1.984e-08", "1.988e-08",
                            "1.992e-08", "1.996e-08", "2e-08", "2.004e-08",
                            "2.008e-08",
                            "2.012e-08", "2.016e-08", "2.02e-08",
                            "2.024e-08", "2.028e-08",
                            "2.032e-08", "2.036e-08", "2.04e-08",
                            "2.044e-08", "2.048e-08",
                            "2.052e-08", "2.056e-08", "2.06e-08",
                            "2.064e-08", "2.068e-08",
                            "2.072e-08", "2.076e-08", "2.08e-08",
                            "2.084e-08", "2.088e-08",
                            "2.092e-08", "2.096e-08", "2.1e-08", "2.104e-08",
                            "2.108e-08",
                            "2.112e-08", "2.116e-08", "2.12e-08",
                            "2.124e-08", "2.128e-08",
                            "2.132e-08", "2.136e-08", "2.14e-08",
                            "2.144e-08", "2.148e-08",
                            "2.152e-08", "2.156e-08", "2.16e-08",
                            "2.164e-08", "2.168e-08",
                            "2.172e-08", "2.176e-08", "2.18e-08",
                            "2.184e-08", "2.188e-08",
                            "2.192e-08", "2.196e-08", "2.2e-08", "2.204e-08",
                            "2.208e-08",
                            "2.212e-08", "2.216e-08", "2.22e-08",
                            "2.224e-08", "2.228e-08",
                            "2.232e-08", "2.236e-08", "2.24e-08",
                            "2.244e-08", "2.248e-08",
                            "2.252e-08", "2.256e-08", "2.26e-08",
                            "2.264e-08", "2.268e-08",
                            "2.272e-08", "2.276e-08", "2.28e-08",
                            "2.284e-08", "2.288e-08",
                            "2.292e-08", "2.296e-08", "2.3e-08", "2.304e-08",
                            "2.308e-08",
                            "2.312e-08", "2.316e-08", "2.32e-08",
                            "2.324e-08", "2.328e-08",
                            "2.332e-08", "2.336e-08", "2.34e-08",
                            "2.344e-08", "2.348e-08",
                            "2.352e-08", "2.356e-08", "2.36e-08",
                            "2.364e-08", "2.368e-08",
                            "2.372e-08", "2.376e-08", "2.38e-08",
                            "2.384e-08", "2.388e-08",
                            "2.392e-08", "2.396e-08", "2.4e-08", "2.404e-08",
                            "2.408e-08",
                            "2.412e-08", "2.416e-08", "2.42e-08",
                            "2.424e-08", "2.428e-08",
                            "2.432e-08", "2.436e-08", "2.44e-08",
                            "2.444e-08", "2.448e-08",
                            "2.452e-08", "2.456e-08", "2.46e-08",
                            "2.464e-08", "2.468e-08",
                            "2.472e-08", "2.476e-08", "2.48e-08",
                            "2.484e-08", "2.488e-08",
                            "2.492e-08", "2.496e-08", "2.5e-08", "2.504e-08",
                            "2.508e-08",
                            "2.512e-08", "2.516e-08", "2.52e-08",
                            "2.524e-08", "2.528e-08",
                            "2.532e-08", "2.536e-08", "2.54e-08",
                            "2.544e-08", "2.548e-08",
                            "2.552e-08", "2.556e-08", "2.56e-08",
                            "2.564e-08", "2.568e-08",
                            "2.572e-08", "2.576e-08", "2.58e-08",
                            "2.584e-08", "2.588e-08",
                            "2.592e-08", "2.596e-08", "2.6e-08", "2.604e-08",
                            "2.608e-08",
                            "2.612e-08", "2.616e-08", "2.62e-08",
                            "2.624e-08", "2.628e-08",
                            "2.632e-08", "2.636e-08", "2.64e-08",
                            "2.644e-08", "2.648e-08",
                            "2.652e-08", "2.656e-08", "2.66e-08",
                            "2.664e-08", "2.668e-08",
                            "2.672e-08", "2.676e-08", "2.68e-08",
                            "2.684e-08", "2.688e-08",
                            "2.692e-08", "2.696e-08", "2.7e-08", "2.704e-08",
                            "2.708e-08",
                            "2.712e-08", "2.716e-08", "2.72e-08",
                            "2.724e-08", "2.728e-08",
                            "2.732e-08", "2.736e-08", "2.74e-08",
                            "2.744e-08", "2.748e-08",
                            "2.752e-08", "2.756e-08", "2.76e-08",
                            "2.764e-08", "2.768e-08",
                            "2.772e-08", "2.776e-08", "2.78e-08",
                            "2.784e-08", "2.788e-08",
                            "2.792e-08", "2.796e-08", "2.8e-08", "2.804e-08",
                            "2.808e-08",
                            "2.812e-08", "2.816e-08", "2.82e-08",
                            "2.824e-08", "2.828e-08",
                            "2.832e-08", "2.836e-08", "2.84e-08",
                            "2.844e-08", "2.848e-08",
                            "2.852e-08", "2.856e-08", "2.86e-08",
                            "2.864e-08", "2.868e-08",
                            "2.872e-08", "2.876e-08", "2.88e-08",
                            "2.884e-08", "2.888e-08",
                            "2.892e-08", "2.896e-08", "2.9e-08", "2.904e-08",
                            "2.908e-08",
                            "2.912e-08", "2.916e-08", "2.92e-08",
                            "2.924e-08", "2.928e-08",
                            "2.932e-08", "2.936e-08", "2.94e-08",
                            "2.944e-08", "2.948e-08",
                            "2.952e-08", "2.956e-08", "2.96e-08",
                            "2.964e-08", "2.968e-08",
                            "2.972e-08", "2.976e-08", "2.98e-08",
                            "2.984e-08", "2.988e-08",
                            "2.992e-08", "2.996e-08", "3e-08", "3.004e-08",
                            "3.008e-08",
                            "3.012e-08", "3.016e-08", "3.02e-08",
                            "3.024e-08", "3.028e-08",
                            "3.032e-08", "3.036e-08", "3.04e-08",
                            "3.044e-08", "3.048e-08",
                            "3.052e-08", "3.056e-08", "3.06e-08",
                            "3.064e-08", "3.068e-08",
                            "3.072e-08", "3.076e-08", "3.08e-08",
                            "3.084e-08", "3.088e-08",
                            "3.092e-08", "3.096e-08", "3.1e-08", "3.104e-08",
                            "3.108e-08",
                            "3.112e-08", "3.116e-08", "3.12e-08",
                            "3.124e-08", "3.128e-08",
                            "3.132e-08", "3.136e-08", "3.14e-08",
                            "3.144e-08", "3.148e-08",
                            "3.152e-08", "3.156e-08", "3.16e-08",
                            "3.164e-08", "3.168e-08",
                            "3.172e-08", "3.176e-08", "3.18e-08",
                            "3.184e-08", "3.188e-08",
                            "3.192e-08", "3.196e-08", "3.2e-08", "3.204e-08",
                            "3.208e-08",
                            "3.212e-08", "3.216e-08", "3.22e-08",
                            "3.224e-08", "3.228e-08",
                            "3.232e-08", "3.236e-08", "3.24e-08",
                            "3.244e-08", "3.248e-08",
                            "3.252e-08", "3.256e-08", "3.26e-08",
                            "3.264e-08", "3.268e-08",
                            "3.272e-08", "3.276e-08", "3.28e-08",
                            "3.284e-08", "3.288e-08",
                            "3.292e-08", "3.296e-08", "3.3e-08", "3.304e-08",
                            "3.308e-08",
                            "3.312e-08", "3.316e-08", "3.32e-08",
                            "3.324e-08", "3.328e-08",
                            "3.332e-08", "3.336e-08", "3.34e-08",
                            "3.344e-08", "3.348e-08",
                            "3.352e-08", "3.356e-08", "3.36e-08",
                            "3.364e-08", "3.368e-08",
                            "3.372e-08", "3.376e-08", "3.38e-08",
                            "3.384e-08", "3.388e-08",
                            "3.392e-08", "3.396e-08", "3.4e-08", "3.404e-08",
                            "3.408e-08",
                            "3.412e-08", "3.416e-08", "3.42e-08",
                            "3.424e-08", "3.428e-08",
                            "3.432e-08", "3.436e-08", "3.44e-08",
                            "3.444e-08", "3.448e-08",
                            "3.452e-08", "3.456e-08", "3.46e-08",
                            "3.464e-08", "3.468e-08",
                            "3.472e-08", "3.476e-08", "3.48e-08",
                            "3.484e-08", "3.488e-08",
                            "3.492e-08", "3.496e-08", "3.5e-08", "3.504e-08",
                            "3.508e-08",
                            "3.512e-08", "3.516e-08", "3.52e-08",
                            "3.524e-08", "3.528e-08",
                            "3.532e-08", "3.536e-08", "3.54e-08",
                            "3.544e-08", "3.548e-08",
                            "3.552e-08", "3.556e-08", "3.56e-08",
                            "3.564e-08", "3.568e-08",
                            "3.572e-08", "3.576e-08", "3.58e-08",
                            "3.584e-08", "3.588e-08",
                            "3.592e-08", "3.596e-08", "3.6e-08", "3.604e-08",
                            "3.608e-08",
                            "3.612e-08", "3.616e-08", "3.62e-08",
                            "3.624e-08", "3.628e-08",
                            "3.632e-08", "3.636e-08", "3.64e-08",
                            "3.644e-08", "3.648e-08",
                            "3.652e-08", "3.656e-08", "3.66e-08",
                            "3.664e-08", "3.668e-08",
                            "3.672e-08", "3.676e-08", "3.68e-08",
                            "3.684e-08", "3.688e-08",
                            "3.692e-08", "3.696e-08", "3.7e-08", "3.704e-08",
                            "3.708e-08",
                            "3.712e-08", "3.716e-08", "3.72e-08",
                            "3.724e-08", "3.728e-08",
                            "3.732e-08", "3.736e-08", "3.74e-08",
                            "3.744e-08", "3.748e-08",
                            "3.752e-08", "3.756e-08", "3.76e-08",
                            "3.764e-08", "3.768e-08",
                            "3.772e-08", "3.776e-08", "3.78e-08",
                            "3.784e-08", "3.788e-08",
                            "3.792e-08", "3.796e-08", "3.8e-08", "3.804e-08",
                            "3.808e-08",
                            "3.812e-08", "3.816e-08", "3.82e-08",
                            "3.824e-08", "3.828e-08",
                            "3.832e-08", "3.836e-08", "3.84e-08",
                            "3.844e-08", "3.848e-08",
                            "3.852e-08", "3.856e-08", "3.86e-08",
                            "3.864e-08", "3.868e-08",
                            "3.872e-08", "3.876e-08", "3.88e-08",
                            "3.884e-08", "3.888e-08",
                            "3.892e-08", "3.896e-08", "3.9e-08", "3.904e-08",
                            "3.908e-08",
                            "3.912e-08", "3.916e-08", "3.92e-08",
                            "3.924e-08", "3.928e-08",
                            "3.932e-08", "3.936e-08", "3.94e-08",
                            "3.944e-08", "3.948e-08",
                            "3.952e-08", "3.956e-08", "3.96e-08",
                            "3.964e-08", "3.968e-08",
                            "3.972e-08", "3.976e-08", "3.98e-08",
                            "3.984e-08", "3.988e-08",
                            "3.992e-08", "3.996e-08", "4e-08", "4.004e-08",
                            "4.008e-08",
                            "4.012e-08", "4.016e-08", "4.02e-08",
                            "4.024e-08", "4.028e-08",
                            "4.032e-08", "4.036e-08", "4.04e-08",
                            "4.044e-08", "4.048e-08",
                            "4.052e-08", "4.056e-08", "4.06e-08",
                            "4.064e-08", "4.068e-08",
                            "4.072e-08", "4.076e-08", "4.08e-08",
                            "4.084e-08", "4.088e-08",
                            "4.092e-08", "4.096e-08", "4.1e-08", "4.104e-08",
                            "4.108e-08",
                            "4.112e-08", "4.116e-08", "4.12e-08",
                            "4.124e-08", "4.128e-08",
                            "4.132e-08", "4.136e-08", "4.14e-08",
                            "4.144e-08", "4.148e-08",
                            "4.152e-08", "4.156e-08", "4.16e-08",
                            "4.164e-08", "4.168e-08",
                            "4.172e-08", "4.176e-08", "4.18e-08",
                            "4.184e-08", "4.188e-08",
                            "4.192e-08", "4.196e-08", "4.2e-08", "4.204e-08",
                            "4.208e-08",
                            "4.212e-08", "4.216e-08", "4.22e-08",
                            "4.224e-08", "4.228e-08",
                            "4.232e-08", "4.236e-08", "4.24e-08",
                            "4.244e-08", "4.248e-08",
                            "4.252e-08", "4.256e-08", "4.26e-08",
                            "4.264e-08", "4.268e-08",
                            "4.272e-08", "4.276e-08", "4.28e-08",
                            "4.284e-08", "4.288e-08",
                            "4.292e-08", "4.296e-08", "4.3e-08", "4.304e-08",
                            "4.308e-08",
                            "4.312e-08", "4.316e-08", "4.32e-08",
                            "4.324e-08", "4.328e-08",
                            "4.332e-08", "4.336e-08", "4.34e-08",
                            "4.344e-08", "4.348e-08",
                            "4.352e-08", "4.356e-08", "4.36e-08",
                            "4.364e-08", "4.368e-08",
                            "4.372e-08", "4.376e-08", "4.38e-08",
                            "4.384e-08", "4.388e-08",
                            "4.392e-08", "4.396e-08", "4.4e-08", "4.404e-08",
                            "4.408e-08",
                            "4.412e-08", "4.416e-08", "4.42e-08",
                            "4.424e-08", "4.428e-08",
                            "4.432e-08", "4.436e-08", "4.44e-08",
                            "4.444e-08", "4.448e-08",
                            "4.452e-08", "4.456e-08", "4.46e-08",
                            "4.464e-08", "4.468e-08",
                            "4.472e-08", "4.476e-08", "4.48e-08",
                            "4.484e-08", "4.488e-08",
                            "4.492e-08", "4.496e-08", "4.5e-08", "4.504e-08",
                            "4.508e-08",
                            "4.512e-08", "4.516e-08", "4.52e-08",
                            "4.524e-08", "4.528e-08",
                            "4.532e-08", "4.536e-08", "4.54e-08",
                            "4.544e-08", "4.548e-08",
                            "4.552e-08", "4.556e-08", "4.56e-08",
                            "4.564e-08", "4.568e-08",
                            "4.572e-08", "4.576e-08", "4.58e-08",
                            "4.584e-08", "4.588e-08",
                            "4.592e-08", "4.596e-08", "4.6e-08", "4.604e-08",
                            "4.608e-08",
                            "4.612e-08", "4.616e-08", "4.62e-08",
                            "4.624e-08", "4.628e-08",
                            "4.632e-08", "4.636e-08", "4.64e-08",
                            "4.644e-08", "4.648e-08",
                            "4.652e-08", "4.656e-08", "4.66e-08",
                            "4.664e-08", "4.668e-08",
                            "4.672e-08", "4.676e-08", "4.68e-08",
                            "4.684e-08", "4.688e-08",
                            "4.692e-08", "4.696e-08", "4.7e-08", "4.704e-08",
                            "4.708e-08",
                            "4.712e-08", "4.716e-08", "4.72e-08",
                            "4.724e-08", "4.728e-08",
                            "4.732e-08", "4.736e-08", "4.74e-08",
                            "4.744e-08", "4.748e-08",
                            "4.752e-08", "4.756e-08", "4.76e-08",
                            "4.764e-08", "4.768e-08",
                            "4.772e-08", "4.776e-08", "4.78e-08",
                            "4.784e-08", "4.788e-08",
                            "4.792e-08", "4.796e-08", "4.8e-08", "4.804e-08",
                            "4.808e-08",
                            "4.812e-08", "4.816e-08", "4.82e-08",
                            "4.824e-08", "4.828e-08",
                            "4.832e-08", "4.836e-08", "4.84e-08",
                            "4.844e-08", "4.848e-08",
                            "4.852e-08", "4.856e-08", "4.86e-08",
                            "4.864e-08", "4.868e-08",
                            "4.872e-08", "4.876e-08", "4.88e-08",
                            "4.884e-08", "4.888e-08",
                            "4.892e-08", "4.896e-08", "4.9e-08", "4.904e-08",
                            "4.908e-08",
                            "4.912e-08", "4.916e-08", "4.92e-08",
                            "4.924e-08", "4.928e-08",
                            "4.932e-08", "4.936e-08", "4.94e-08",
                            "4.944e-08", "4.948e-08",
                            "4.952e-08", "4.956e-08", "4.96e-08",
                            "4.964e-08", "4.968e-08",
                            "4.972e-08", "4.976e-08", "4.98e-08",
                            "4.984e-08", "4.988e-08",
                            "4.992e-08", "4.996e-08"
                        ],
                    }
                },
                "STREAM" => 1,
                "RAW"    => {
                    "CHAN" => {
                        "CH1" => {
                            "DAT:STOP" => 2500,
                            "CURV" =>
                                "#42500\201\201\201\201\201\201\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\206\206\206\206\206\206\207\207\207\207\206\206\206\206\206\206\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\205\205\205\205\205\205\206\206\206\206\205\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\206\206\206\206\206\206\206\206\206\206\207\207\207\207\207\206\206\206\206\206\206\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\206\206\206\206\206\206\205\205\205\205\205\205\204\204\204\204\204\203\203\203\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\206\206\206\206\206\206\206\205\205\205\205\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\206\205\205\205\205\205\205\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\206\206\206\206\206\206\206\206\206\206\206\206\206\206\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\205\206\206\206\206\206\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\206\206\206\206\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\204\204\204\204\204\204\204\204\203\203\203\203\202\202\202\202\201\201\201\201\201\200\200\200\200\200\200\200\200\200\200\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\204\204\204\203\203\203\203\202\202\202\202\201\201\201\201\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\201\201\201\201\201\201\202\202\202\202\202\202\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\200\200\200\200\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\203\203\203\203\203\202\202\202\202\202\202\202\201\201\201\201\201\201\201\201\201\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\202\202\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\201\200\200\200\200\200\200\200\200\201\201\201\201\201\201\201\201\202\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\202\202\202\202\201\201\201\201\201\201\201\201\201\202\202\202\202\203\203\203\204\204\204\204\204\204\204\203\203\203\203\203\203\203\202\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\201\201\201\201\201\201\200\200\200\200\200\200\177\177\177\177\177\177\177\177\177\177\200\200\200\200\200\201\201\201\201\202\202\202\202\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\200\200\200\200\177\177\177\177~~~~~}}}}}}}}}}}}}}}}}~~~~~~\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\177\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\200\201\201\201\201\201\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\203\203\203\203\202\202\202\202\201\201\201\201\200\200\200\200\177\177\177\177\177\177\177\177\177\177\177\177\177\177\200\200\200\200\200\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\204\204\204\204\204\204\203\203\203\203\203\202\202\202\202\202\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\201\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\204\204\204\204\204\204\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\205\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\204\204\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\203\203\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\203\203\203\203\203\203\203\203\204\204\204\204\204\204\205\205\205\205\205\206\206\206\206\206\206\207\207\207\207\207\206\206\206\206\206\206\206\206\206\206\206\206\206\206\205\205\205\205\206\206\206\206\206\206\206\206\206\206\206\206\206\206\207\207\207\207\207\207\206\206\206\206\206\206\206\206\206\206\206\205\205\205\205\205\205\205\205\205\205\205\205\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\205\205\205\205\205\204\204\204\204\204\204\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\202\201\201\201\201\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\203\203\203\203\203\202\202\202\202\202\202\202\202\202\202\202\202\202\202\201\201\201\201\201\201\201\201\202\202\202\202\202\202\202\202\203\203\203\203\203\203\204\204\204\204\204\204\204\204\204\204\204\204\204\204\203\203\203\203\203\203\202\202\202\202\202\202\201\201\201\201",
                            "WFMP:YMU" => "8.0E-5",
                            "WFMP:YUN" => "\"Volts\"",
                            "CHAN"     => "CH1",
                            "WFMP:XIN" => "4.0E-11",
                            "WFMP:WFI" =>
                                "\"Ch1, DC coupling, 2.0E-3 V/div, 1.0E-8 s/div, 2500 points, Sample mode\"",
                            "DAT:STAR"   => 1,
                            "WFMP:XUN"   => "\"s\"",
                            "WFMP:BYT_N" => 1,
                            "WFMP:XZE"   => "-5.0E-8",
                            "WFMP:YZE"   => "0.0E0",
                            "DAT?"       => undef,
                            "WFMP:NR_P"  => 2500,
                            "DAT:WID"    => 1,
                            "WFMP:ENC"   => "BIN",
                            "DAT:DEST"   => "REFC",
                            "DAT:SOU"    => "CH1",
                            "DAT:ENC"    => "RPB",
                            "WFMP:BYT_O" => "MSB",
                            "WFMP:BIT_N" => 8,
                            "WFMP:YOF"   => "1.28E2",
                            "WFMP:BN_F"  => "RP",
                            "WFMP:PT_F"  => "Y",
                            "WFMP:PT_O"  => 0
                        }
                    }
                },
                "MODULE"  => "TekTDS",
                "COMMENT" => [],
                "OPTIONS" => {
                    "dropraw"       => 0,
                    "interpolate"   => 1,
                    "print_summary" => 0,
                }
            }
        }
    },
};

is_deeply( $ev_an, $ev_an_expect, "Event, analyzed" );

