/*
 *  Copyright 1999 Vizdom Software, Inc. All Rights Reserved.
 * 
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the same terms as the Perl Kit, namely, under 
 *  the terms of either:
 *
 *      a) the GNU General Public License as published by the Free
 *      Software Foundation; either version 1 of the License, or 
 *      (at your option) any later version, or
 *
 *      b) the "Artistic License" that comes with the Perl Kit.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See either
 *  the GNU General Public License or the Artistic License for more 
 *  details.
 */

package com.vizdom.ber;

import com.vizdom.util.CharacterEncoder;
import com.vizdom.util.Debug;
import com.vizdom.util.UnreachableCodeException;

/**
 * A BER OCTET STRING.
 *
 * @author: John Lacey
 * @version: $Revision: 14 $
 */
public class BerOctetString extends BerObject
{
    /** The OCTET STRING identifier, [UNIVERSAL 4]. */
    static final BerIdentifier gIDENTIFIER = 
        new BerIdentifier(BerTypes.PRIMITIVE, BerTypes.OCTET_STRING);


    /** The underlying byte array of this BER OCTET STRING. */
    private byte[] mByteArray;


    /** 
     * The String value associated with <code>mByteArray</code>. This is 
     * assigned as early as possible because the conversion throws
     * <code>UnsupportedEncodingException</code>, which <code>toString</code>
     * can't do.
     */
    private String mString;


    /** The character encoding in use for BER OCTET STRING values. */
    /* private String mCharacterEncoding; */


    /** The decoding constructor. */
    protected BerOctetString()
    {
    }


    /** 
     * An encoding constructor. The given character encoding is
     * used to convert the string to an array of bytes. Using 
     * <code>new BerOctetString(s, enc)</code> is a shortcut for
     * <code>new BerOctetString(s.getBytes(enc))</code>.
     * 
     * @param aString a string
     * @param aCharacterEncoding a character encoding
     * @exception java.io.UnsupportedEncodingException if the 
     *      encoding name is unknown or unsupported on the current platform
     */
    public BerOctetString(String aString, String aCharacterEncoding)
        throws java.io.UnsupportedEncodingException
    {
        mByteArray = CharacterEncoder.toByteArray(aString, aCharacterEncoding);
        mString = aString;
        /* mCharacterEncoding = aCharacterEncoding; */
    }


    /** 
     * An encoding constructor. The given character encoding is used
     * to convert the byte array to a String (for <code>toString()</code>).
     * 
     * @param aByteArray a byte array
     * @param aCharacterEncoding a character encoding
     * @exception java.io.UnsupportedEncodingException if the 
     *      encoding name is unknown or unsupported on the current platform
     */
    public BerOctetString(byte[] aByteArray, String aCharacterEncoding)
        throws java.io.UnsupportedEncodingException
    {
        mByteArray = aByteArray;
        mString = (aByteArray.length == 0) ?
            "" : CharacterEncoder.toString(aByteArray, aCharacterEncoding);
        /* mCharacterEncoding = aCharacterEncoding; */
    }


    /** 
     * An encoding constructor. The default character encoding is used
     * to convert the byte array to a String (for <code>toString()</code>).
     * 
     * @param aByteArray a byte array
     */
    public BerOctetString(byte[] aByteArray)
    {
        mByteArray = aByteArray;

        // We will probably never need a string, so avoid creating it now.
        // This works here because the default character encoding doesn't
        // throw any exceptions.
        mString = null;

        /* mCharacterEncoding = aCharacterEncoding; */
    }


    /**
     * Returns the BER identifier for BER OCTET STRING, [UNIVERSAL 4].
     * 
     * @return the BER identifier for BER OCTET STRING, [UNIVERSAL 4]
     */
    /*
     * This may return one of two identifiers (with the same tag) if
     * constructed string encodings are implemented.
     */
    public BerIdentifier getIdentifier()
    {
        return gIDENTIFIER;
    }


    /**
     * Returns the size of the encoded contents, in bytes.
     * 
     * @return the size of the encoded contents, in bytes
     */
    protected final int mGetLength()
    {
        return mByteArray.length;
    }


    /** 
     * Writes the encoded contents to the output stream.
     * 
     * @param anOut an output stream
     * @exception IOException if an I/O error occurs
     */
    protected final void mWriteContents(java.io.OutputStream anOut)
        throws java.io.IOException
    {
        anOut.write(mByteArray);
    }


    /** 
     * Reads the encoded contents from the input stream.
     * 
     * @param anIn an input stream
     * @param aModule a BER module for reading constructed encodings
     * @param anIdentifier the BER identifier of the encoded object
     * @param aLength the length in bytes of the encoded contents
     *
     * @exception IOException if an I/O error occurs. In particular,
     *     an <code>EOFException</code> may be thrown if the end of
     *     stream is reached before the contents have been fully read
     */
    protected void mReadContents(java.io.InputStream anIn, 
        BerModule aModule, BerIdentifier anIdentifier, int aLength) 
        throws java.io.IOException
    {
        if (Debug.ASSERT && getClass() == BerOctetString.class)
            Debug.assert(anIdentifier == gIDENTIFIER); // [sic; not equals]

        // Read exactly aLength bytes (which may be more than are
        // available all at once). Any EOF is unexpected.
        mByteArray = new byte[aLength];
        int count;
        for (int offset = 0; offset < aLength; offset += count)
        {
            count = anIn.read(mByteArray, offset, aLength - offset);
            if (count == -1)
                throw new java.io.EOFException(); // ??? BerException?
        }

        /* mCharacterEncoding = aModule.getCharacterEncoding(); */
        mString = CharacterEncoder.toString(mByteArray, 
            aModule.getCharacterEncoding());
    }


    /**
     * Returns a copy of the byte array held by this object.
     * 
     * @return a copy of the byte array held by this object
     */
    public byte[] toByteArray()
    {
        byte[] bytes = new byte[mByteArray.length];
        System.arraycopy(mByteArray, 0, bytes, 0, mByteArray.length);
        return bytes;
    }


    /**
     * Returns a string representation of the object.
     * 
     * @return a string representation of the object
     */
    /* 
     * This string had to be created earlier because we can't throw
     * UnsupportedEncodingException here.
     */
    public String toString()
    {
        if (mString == null)
        {
            synchronized (this)
            {
                if (mString == null)
                {
                    mString = 
                        (mByteArray.length == 0) ? "" : new String(mByteArray);
                }
            }
        }

        return mString;
    }


    /**
     * Returns a string representation of the object. The given
     * character encoding is used to create a new String object
     * from the bytes.
     * 
     * @param aCharacterEncoding a character encoding
     * @return a string representation of the object
     * @exception java.io.UnsupportedEncodingException if the 
     *      encoding name is unknown or unsupported on the current platform
     */
    /* 
     * We don't try to cache these, which may be called with different
     * character encodings.
     */
    public String toString(String aCharacterEncoding) 
        throws java.io.UnsupportedEncodingException
    {
        return CharacterEncoder.toString(mByteArray, aCharacterEncoding);
    }
}
