package UML::Sequence::SimpleSeq;
use     strict;
use     warnings;

our $VERSION = '0.01';

=head1 NAME

UML::Sequence::SimpleSeq - turns simple outlines (see below) into UML sequence diagrams

=head1 SYNOPSIS

    genericseq.pl UML::Sequence::SimpleSeq outline_file > sequence.xml
    seq2svg sequence.xml > sequence.svg

OR

    genericseq.pl UML::Sequence::SimpleSeq outline_file | seq2svg.pl > sequence.svg

=head1 DESCRIPTION

This file may be used directly by a script (as shown above) or as a base class
for other sequencers (see UML::Sequence::JavaSeq).  It supplies routines for
handling simple outlines like a user could be expected to type by hand.  Such
outlines look like this:

    At Home.Wash Car
        Garage.retrieve bucket
        Kitchen.prepare bucket
            Kitchen.pour soap in bucket
            Kitchen.fill bucket
        Garage.get sponge
        Garage.open door
        Driveway.apply soapy water
        Driveway.rinse
        Driveway.empty bucket
        Garage.close door
        Garage.replace sponge
        Garage.replace bucket

The "class" name and "method" name are separated by a dot.  If there are
multiple dots, the method name is everything after the last dot.  Classes
and methods in this context are elements of a UML sequence diagram.  Classes
get boxes at the top of the page.  Method calls are labeled lines from one
class to another.  If you want two classes with the same name, you must
append a suffix or prefix (try instanceName:ClassName).

=head1 grab_outline_text

Call this first with the outline file (in the format described above).
Pass it the name of the file to read.
Returns a reference to an array whose elements are lines from the outline
with spacing preserved.

=cut

sub grab_outline_text {
    shift;  # discard class
    my $file = shift;
    my @outline;

    open FILE, "$file" or die "Couldn't open $file\n";
    while (<FILE>) {
        push @outline, $_;
    }
    close FILE;

    return \@outline;
}

=head1 grab_methods

Call this with an outline (possibly generated by grab_outline).  It will return
a hash reference.  Each method mentioned in the outline will appear as a key
in the hash (the values are less important, they count the occurances of
the method).

=cut

sub grab_methods {
    shift;  # discard class
    my $outline         = shift;
    my %methods;

    foreach (@$outline) {
        my $line = $_;
        $line    =~ s/^\s+//;
        $methods{$line}++;
    }
    return \%methods;
}

=head1 parse_signature

This method is a call back used by the UML::Sequence constructor.  It accepts
a signature and returns the "class" name (in scalar context) or the "class"
and "method" names (in list context).  It splits the signature on the
last dot it sees after removing any argument list and associated parentheses.
Thus, it works for simple and Java outlines.

=cut

sub parse_signature {
    chomp(my $class = shift);
    $class =~ s/\(.*//;
    $class =~ s/\.([^.]*)$//;
    my $method = $1;
    if (defined $method) {
        $method    =~ s/</&lt;/g;
        $method    =~ s/>/&gt;/g;
    }
    return wantarray ? ($class, $method) : $class;
}

1;

=head1 EDIT HISTORY

  0.01 Jan 2003  Initial release.

=head1 AUTHOR
 
Phil Crow, <philcrow2000@yahoo.com>
  
=head1 COPYRIGHT
   
Copyright 2003, Philip Crow, all rights reserved.  You may modify and/or
redistribute this code in the same manner as Perl itself.

=cut
